/**
 * Utility functions for handling fullscreen functionality across different devices and browsers
 */

/**
 * Request fullscreen with enhanced mobile support
 * @param {HTMLElement} element - Element to make fullscreen (defaults to document.documentElement)
 * @returns {Promise<boolean>} - Returns true if successful, false otherwise
 */
export const requestFullscreen = async (element = document.documentElement) => {
  try {
    // Check if fullscreen is already active
    if (document.fullscreenElement || 
        document.webkitFullscreenElement || 
        document.mozFullScreenElement || 
        document.msFullscreenElement) {
      return true;
    }

    // Try different fullscreen methods
    if (element.requestFullscreen) {
      await element.requestFullscreen();
    } else if (element.webkitRequestFullscreen) {
      // Safari and older Chrome
      await element.webkitRequestFullscreen();
    } else if (element.webkitRequestFullScreen) {
      // Older Safari
      await element.webkitRequestFullScreen();
    } else if (element.mozRequestFullScreen) {
      // Firefox
      await element.mozRequestFullScreen();
    } else if (element.msRequestFullscreen) {
      // IE/Edge
      await element.msRequestFullscreen();
    } else {
      console.warn('Fullscreen API not supported');
      return false;
    }

    return true;
  } catch (error) {
    console.warn('Failed to enter fullscreen:', error);
    return false;
  }
};

/**
 * Exit fullscreen
 * @returns {Promise<boolean>} - Returns true if successful, false otherwise
 */
export const exitFullscreen = async () => {
  try {
    if (document.exitFullscreen) {
      await document.exitFullscreen();
    } else if (document.webkitExitFullscreen) {
      await document.webkitExitFullscreen();
    } else if (document.webkitCancelFullScreen) {
      await document.webkitCancelFullScreen();
    } else if (document.mozCancelFullScreen) {
      await document.mozCancelFullScreen();
    } else if (document.msExitFullscreen) {
      await document.msExitFullscreen();
    } else {
      return false;
    }
    return true;
  } catch (error) {
    console.warn('Failed to exit fullscreen:', error);
    return false;
  }
};

/**
 * Check if currently in fullscreen mode
 * @returns {boolean}
 */
export const isFullscreen = () => {
  return !!(
    document.fullscreenElement ||
    document.webkitFullscreenElement ||
    document.mozFullScreenElement ||
    document.msFullscreenElement
  );
};

/**
 * Check if fullscreen is supported
 * @returns {boolean}
 */
export const isFullscreenSupported = () => {
  const element = document.documentElement;
  return !!(
    element.requestFullscreen ||
    element.webkitRequestFullscreen ||
    element.webkitRequestFullScreen ||
    element.mozRequestFullScreen ||
    element.msRequestFullscreen
  );
};

/**
 * Enhanced fullscreen request with mobile-specific handling
 * This function should be called within a user gesture (click, touch, etc.)
 * @param {HTMLElement} element - Element to make fullscreen
 * @param {Object} options - Additional options
 * @returns {Promise<boolean>}
 */
export const requestFullscreenEnhanced = async (element = document.documentElement, options = {}) => {
  const { 
    showWarning = true, 
    fallbackToViewport = true,
    mobileOptimization = true 
  } = options;

  // Check if fullscreen is supported
  if (!isFullscreenSupported()) {
    if (showWarning) {
      console.warn('Fullscreen not supported on this device');
    }
    return false;
  }

  // Mobile-specific optimizations
  if (mobileOptimization && isMobileDevice()) {
    // Hide address bar on mobile browsers
    try {
      window.scrollTo(0, 1);
      setTimeout(() => window.scrollTo(0, 0), 100);
    } catch (e) {
      // Ignore errors
    }

    // Set viewport meta tag for better mobile experience
    let viewport = document.querySelector('meta[name="viewport"]');
    if (!viewport) {
      viewport = document.createElement('meta');
      viewport.name = 'viewport';
      document.head.appendChild(viewport);
    }
    viewport.content = 'width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no';
  }

  // Request fullscreen
  const success = await requestFullscreen(element);
  
  if (!success && fallbackToViewport) {
    // Fallback: maximize viewport on mobile
    if (isMobileDevice()) {
      document.body.style.margin = '0';
      document.body.style.padding = '0';
      document.documentElement.style.margin = '0';
      document.documentElement.style.padding = '0';
      
      // Hide browser UI elements
      const style = document.createElement('style');
      style.textContent = `
        body { 
          overflow: hidden !important; 
          position: fixed !important;
          width: 100vw !important;
          height: 100vh !important;
          top: 0 !important;
          left: 0 !important;
        }
        html {
          overflow: hidden !important;
        }
      `;
      document.head.appendChild(style);
      
      return true;
    }
  }

  return success;
};

/**
 * Detect if device is mobile
 * @returns {boolean}
 */
export const isMobileDevice = () => {
  return /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent) ||
         (navigator.maxTouchPoints && navigator.maxTouchPoints > 2 && /MacIntel/.test(navigator.platform));
};

/**
 * Add fullscreen change event listener
 * @param {Function} callback - Callback function to execute on fullscreen change
 * @returns {Function} - Cleanup function to remove listeners
 */
export const onFullscreenChange = (callback) => {
  const events = [
    'fullscreenchange',
    'webkitfullscreenchange',
    'mozfullscreenchange',
    'MSFullscreenChange'
  ];

  const handler = () => callback(isFullscreen());
  
  events.forEach(event => {
    document.addEventListener(event, handler);
  });

  // Return cleanup function
  return () => {
    events.forEach(event => {
      document.removeEventListener(event, handler);
    });
  };
};