import axios from "axios";

// Tentukan baseURL: gunakan proxy "/api" saat dev untuk menghindari CORS,
// gunakan VITE_BACKEND_URL di production bila disediakan, fallback tetap "/api".
const backendBase = (import.meta?.env?.DEV)
  ? '/api'
  : (import.meta?.env?.VITE_BACKEND_URL
    ? new URL('api', import.meta.env.VITE_BACKEND_URL).toString()
    : '/api');

const api = axios.create({
  baseURL: backendBase,
  withCredentials: false,
});

// Tambahkan token otomatis jika ada, dengan preferensi token peserta untuk endpoint tertentu
api.interceptors.request.use((config) => {
  const url = String(config?.url || "");
  const skipAuth = [
    "/login",
    "/participant/login",
    "/participant/login-token",
  ].some((p) => url.endsWith(p) || url.includes(p));

  if (!skipAuth) {
    const adminToken = localStorage.getItem("token");
    const participantToken = localStorage.getItem("participantToken");

    // Endpoint yang HARUS pakai token peserta
    const participantOnly = (
      url.endsWith('/exams/today') ||
      url.includes('/exam-sessions/') ||
      /\/exams\/\d+\/start(\?|$)/.test(url) ||
      url.includes('/exam-participants/update-activity') ||
      url.includes('/participant/')
    );

    const token = participantOnly
      ? (participantToken || adminToken)
      : (adminToken || participantToken);
    if (token) config.headers.Authorization = `Bearer ${token}`;
  }

  // Pastikan header Accept JSON agar respons API tidak diarahkan ke web flow
  config.headers.Accept = "application/json";
  // Tandai sebagai permintaan AJAX standar (membantu beberapa konfigurasi Laravel)
  config.headers["X-Requested-With"] = "XMLHttpRequest";
  // Hindari cache pada metode tulis untuk mencegah 304 yang tidak diharapkan
  if (config.method && ['post','put','patch','delete'].includes(String(config.method).toLowerCase())) {
    config.headers['Cache-Control'] = 'no-store';
    config.headers['Pragma'] = 'no-cache';
  }
  return config;
});

export default api;
