import React, { useEffect, useState } from 'react';
import { useParams } from 'react-router-dom';
import api from '../utils/axios';

export default function TodayPublicDetails() {
  const { examId } = useParams();
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [exam, setExam] = useState(null);
  const [inProgress, setInProgress] = useState([]);
  const [finished, setFinished] = useState([]);
  const [notStarted, setNotStarted] = useState([]);
  const [inactive, setInactive] = useState([]);
  const [counts, setCounts] = useState({ in_progress: 0, finished: 0, not_started: 0, inactive: 0 });
  const [search, setSearch] = useState('');
  const [classFilter, setClassFilter] = useState('');
  const [schoolFilter, setSchoolFilter] = useState('');
  const [statusFilter, setStatusFilter] = useState('all');

  useEffect(() => {
    const load = async () => {
      setLoading(true); setError('');
      try {
        const res = await api.get(`/exams/${examId}/participants-status`);
        setExam(res.data.exam || null);
        setInProgress(Array.isArray(res.data.in_progress) ? res.data.in_progress : []);
        setFinished(Array.isArray(res.data.finished) ? res.data.finished : []);
        setNotStarted(Array.isArray(res.data.not_started) ? res.data.not_started : []);
        setInactive(Array.isArray(res.data.inactive) ? res.data.inactive : []);
        setCounts(res.data.counts || { in_progress: 0, finished: 0, not_started: 0, inactive: 0 });
      } catch (e) {
        console.error('Gagal memuat detail peserta ujian', e);
        setError('Gagal memuat detail peserta ujian. Pastikan Anda login.');
      } finally {
        setLoading(false);
      }
    };
    load();
  }, [examId]);

  const normalize = (s) => (s || '').toString().toLowerCase();
  const applyFilters = (rows) => {
    const q = normalize(search);
    return rows.filter((row) => {
      const name = normalize(row.participant?.nama);
      const nisn = normalize(row.participant?.nisn);
      const className = row.class?.name || '';
      const schoolName = row.school?.nama || '';
      const matchQ = !q || name.includes(q) || nisn.includes(q);
      const matchClass = !classFilter || className === classFilter;
      const matchSchool = !schoolFilter || schoolName === schoolFilter;
      return matchQ && matchClass && matchSchool;
    });
  };

  const allRows = [...inProgress, ...finished, ...notStarted, ...inactive];
  const uniqueClasses = Array.from(new Set(allRows.map(r => r.class?.name).filter(Boolean))).sort();
  const uniqueSchools = Array.from(new Set(allRows.map(r => r.school?.nama).filter(Boolean))).sort();

  const filteredInProgress = applyFilters(inProgress);
  const filteredFinished = applyFilters(finished);
  const filteredNotStarted = applyFilters(notStarted);
  const filteredInactive = applyFilters(inactive);

  const resetFilters = () => {
    setSearch('');
    setClassFilter('');
    setSchoolFilter('');
    setStatusFilter('all');
  };

  return (
    <div className="container mt-4">
      <h3>Detail Peserta Ujian</h3>
      {exam && (
        <div className="mb-3">
          <div><strong>Ujian:</strong> {exam.name} ({exam.code || '-'})</div>
          <div><strong>Mata Pelajaran:</strong> {exam.subject_name || '-'} ({exam.subject_code || '-'})</div>
          <div><strong>Jadwal:</strong> {String(exam.scheduled_at || '-')}</div>
          <div><strong>Ringkasan:</strong> Sedang: {counts.in_progress} • Selesai: {counts.finished} • Belum Mulai: {counts.not_started} • Tidak Aktif: {counts.inactive}</div>
          <div className="text-muted small">Terapan filter tidak mengubah ringkasan, hanya tampilan tabel.</div>
        </div>
      )}

      {/* Filter Bar */}
      <div className="card mb-3">
        <div className="card-body">
          <div className="row g-2 align-items-end">
            <div className="col-md-4">
              <label className="form-label">Cari (Nama/NISN)</label>
              <input
                type="text"
                className="form-control"
                placeholder="mis. Budi / 1234567890"
                value={search}
                onChange={(e) => setSearch(e.target.value)}
              />
            </div>
            <div className="col-md-3">
              <label className="form-label">Kelas</label>
              <select className="form-select" value={classFilter} onChange={(e) => setClassFilter(e.target.value)}>
                <option value="">Semua</option>
                {uniqueClasses.map((c) => (
                  <option key={c} value={c}>{c}</option>
                ))}
              </select>
            </div>
            <div className="col-md-3">
              <label className="form-label">Sekolah</label>
              <select className="form-select" value={schoolFilter} onChange={(e) => setSchoolFilter(e.target.value)}>
                <option value="">Semua</option>
                {uniqueSchools.map((s) => (
                  <option key={s} value={s}>{s}</option>
                ))}
              </select>
            </div>
            <div className="col-md-2">
              <label className="form-label">Status</label>
              <select className="form-select" value={statusFilter} onChange={(e) => setStatusFilter(e.target.value)}>
                <option value="all">Semua</option>
                <option value="in_progress">Sedang</option>
                <option value="finished">Selesai</option>
                <option value="not_started">Belum Mulai</option>
                <option value="inactive">Tidak Aktif</option>
              </select>
            </div>
            <div className="col-12 mt-2">
              <button className="btn btn-sm btn-outline-secondary" onClick={resetFilters}>Reset Filter</button>
            </div>
          </div>
        </div>
      </div>

      {loading ? (
        <div className="text-muted">Memuat data...</div>
      ) : error ? (
        <div className="alert alert-danger">{error}</div>
      ) : (
        <div className="row">
          {(statusFilter === 'all' || statusFilter === 'in_progress') && (
            <div className="col-md-6">
              <div className="card">
                <div className="card-header">Sedang Mengerjakan ({filteredInProgress.length})</div>
                <div className="card-body p-0">
                  <div className="table-responsive">
                    <table className="table table-sm mb-0">
                      <thead>
                        <tr>
                          <th>NISN</th>
                          <th>Nama</th>
                          <th>Kelas</th>
                          <th>Sekolah</th>
                          <th>Mulai</th>
                          <th>Aktivitas</th>
                        </tr>
                      </thead>
                      <tbody>
                        {filteredInProgress.map(row => (
                          <tr key={row.session_id}>
                            <td>{row.participant?.nisn || '-'}</td>
                            <td>{row.participant?.nama || '-'}</td>
                            <td>{row.class?.name || '-'}</td>
                            <td>{row.school?.nama || '-'}</td>
                            <td>{row.started_at || '-'}</td>
                            <td>{row.participant?.last_activity || '-'}</td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                </div>
              </div>
            </div>
          )}
          {(statusFilter === 'all' || statusFilter === 'finished') && (
            <div className="col-md-6">
              <div className="card">
                <div className="card-header">Sudah Selesai ({filteredFinished.length})</div>
                <div className="card-body p-0">
                  <div className="table-responsive">
                    <table className="table table-sm mb-0">
                      <thead>
                        <tr>
                          <th>NISN</th>
                          <th>Nama</th>
                          <th>Kelas</th>
                          <th>Sekolah</th>
                          <th>Mulai</th>
                          <th>Selesai</th>
                        </tr>
                      </thead>
                      <tbody>
                        {filteredFinished.map(row => (
                          <tr key={row.session_id}>
                            <td>{row.participant?.nisn || '-'}</td>
                            <td>{row.participant?.nama || '-'}</td>
                            <td>{row.class?.name || '-'}</td>
                            <td>{row.school?.nama || '-'}</td>
                            <td>{row.started_at || '-'}</td>
                            <td>{row.finished_at || '-'}</td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                </div>
              </div>
            </div>
          )}
          {(statusFilter === 'all' || statusFilter === 'not_started') && (
            <div className="col-md-6 mt-3">
              <div className="card">
                <div className="card-header">Belum Mulai ({filteredNotStarted.length})</div>
                <div className="card-body p-0">
                  <div className="table-responsive">
                    <table className="table table-sm mb-0">
                      <thead>
                        <tr>
                          <th>NISN</th>
                          <th>Nama</th>
                          <th>Kelas</th>
                          <th>Sekolah</th>
                          <th>Aktivitas Terakhir</th>
                        </tr>
                      </thead>
                      <tbody>
                        {filteredNotStarted.map(row => (
                          <tr key={row.participant?.id || `${row.participant?.nisn}-ns`}>
                            <td>{row.participant?.nisn || '-'}</td>
                            <td>{row.participant?.nama || '-'}</td>
                            <td>{row.class?.name || '-'}</td>
                            <td>{row.school?.nama || '-'}</td>
                            <td>{row.participant?.last_activity || '-'}</td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                </div>
              </div>
            </div>
          )}
          {(statusFilter === 'all' || statusFilter === 'inactive') && (
            <div className="col-md-6 mt-3">
              <div className="card">
                <div className="card-header">Tidak Aktif ({filteredInactive.length})</div>
                <div className="card-body p-0">
                  <div className="table-responsive">
                    <table className="table table-sm mb-0">
                      <thead>
                        <tr>
                          <th>NISN</th>
                          <th>Nama</th>
                          <th>Kelas</th>
                          <th>Sekolah</th>
                          <th>Aktivitas Terakhir</th>
                        </tr>
                      </thead>
                      <tbody>
                        {filteredInactive.map(row => (
                          <tr key={row.participant?.id || `${row.participant?.nisn}-inactive`}>
                            <td>{row.participant?.nisn || '-'}</td>
                            <td>{row.participant?.nama || '-'}</td>
                            <td>{row.class?.name || '-'}</td>
                            <td>{row.school?.nama || '-'}</td>
                            <td>{row.participant?.last_activity || '-'}</td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                </div>
              </div>
            </div>
          )}
        </div>
      )}
    </div>
  );
}