import React, { useEffect, useMemo, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import api from '../utils/axios';
import DataTable from 'react-data-table-component';
import Swal from 'sweetalert2';
import { FaPlus, FaEdit, FaTrash, FaInfoCircle } from 'react-icons/fa';
import { useAuth } from '../context/AuthContext.jsx';
import { canAccess } from '../utils/permissionHelper';

export default function Subjects() {
  const navigate = useNavigate();
  const { user, hasRole } = useAuth();
  const isSuperadmin = !!(hasRole && hasRole('superadmin'));
  const [subjects, setSubjects] = useState([]);
  const [loading, setLoading] = useState(true);
  const [filterText, setFilterText] = useState('');
  const [selectedSchoolId, setSelectedSchoolId] = useState('');
  const [selectedGradeId, setSelectedGradeId] = useState('');
  const [showAddModal, setShowAddModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [showDetailModal, setShowDetailModal] = useState(false);
  const [form, setForm] = useState({ name: '', code: '', description: '', id_school: '', id_grade: '' });
  const [editing, setEditing] = useState(null);
  const [detailItem, setDetailItem] = useState(null);
  const [schools, setSchools] = useState([]);
  const [gradesForm, setGradesForm] = useState([]);
  const [gradesAll, setGradesAll] = useState([]);
  const [gradesFilter, setGradesFilter] = useState([]);
  const tokenHeader = { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } };

  const canManageSubjects = canAccess(user, 'manage-subjects');
  const canViewSubject = canAccess(user, 'view-subject') || canManageSubjects || canAccess(user, 'view-all-subjects') || canAccess(user, 'view-own-subjects');
  const canCreateSubject = canAccess(user, 'create-subject') || canManageSubjects;
  const canEditSubject = canAccess(user, 'edit-subject') || canManageSubjects;
  const canDeleteSubject = canAccess(user, 'delete-subject') || canManageSubjects;
  const canViewSchool = canAccess(user, 'view-school');
  const canViewGrade = canAccess(user, 'view-grade');
  const canViewQuestion = canAccess(user, 'view-question') || canAccess(user, 'manage-questions');
  const canCreateQuestion = canAccess(user, 'create-question') || canAccess(user, 'manage-questions');
  const canCreateEssay = canAccess(user, 'create-question') || canAccess(user, 'manage-questions');

  const fetchSubjects = async (schoolId = '', gradeId = '') => {
    setLoading(true);
    try {
      const res = await api.get('/subjects', { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }, params: { id_school: schoolId || '', id_grade: gradeId || '' } });
      setSubjects(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      Swal.fire('Error', 'Gagal memuat daftar mata pelajaran', 'error');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => { if (canViewSubject) fetchSubjects(); }, [canViewSubject]);
  useEffect(() => { if (canViewSchool) fetchSchools(); }, [canViewSchool]);
  useEffect(() => { if (canViewGrade) fetchGradesForForm(form.id_school || ''); }, [form.id_school, canViewGrade]);
  useEffect(() => { if (canViewGrade) fetchGradesAll(); }, [canViewGrade]);
  useEffect(() => { if (canViewGrade) fetchGradesForFilter(''); }, [canViewGrade]);

  // Re-fetch grade list and subjects when school filter changes
  useEffect(() => {
    setSelectedGradeId('');
    if (canViewGrade) fetchGradesForFilter(selectedSchoolId || '');
    if (canViewSubject) fetchSubjects(selectedSchoolId || '', '');
  }, [selectedSchoolId, canViewGrade, canViewSubject]);

  // Re-fetch subjects when grade filter changes
  useEffect(() => {
    if (canViewSubject) fetchSubjects(selectedSchoolId || '', selectedGradeId || '');
  }, [selectedGradeId, canViewSubject]);

  const fetchSchools = async () => {
    try {
      const res = await api.get('/schools', tokenHeader);
      setSchools(Array.isArray(res.data) ? res.data : []);
    } catch (err) { console.error(err); }
  };

  const fetchGradesForForm = async (schoolId) => {
    try {
      const res = await api.get('/grades', { ...tokenHeader, params: { id_school: schoolId || '' } });
      setGradesForm(Array.isArray(res.data) ? res.data : []);
    } catch (err) { console.error(err); }
  };

  const fetchGradesAll = async () => {
    try {
      const res = await api.get('/grades', tokenHeader);
      setGradesAll(Array.isArray(res.data) ? res.data : []);
    } catch (err) { console.error(err); }
  };

  const fetchGradesForFilter = async (schoolId = '') => {
    try {
      const url = schoolId ? '/grades' : '/grades';
      const res = await api.get(url, schoolId ? { ...tokenHeader, params: { id_school: schoolId } } : tokenHeader);
      setGradesFilter(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      Swal.fire('Error', 'Gagal memuat daftar grade untuk filter', 'error');
    }
  };

  const filtered = useMemo(() => {
    // Urutkan terbaru dulu berdasarkan created_at (fallback ke id)
    const base = (() => {
      const arr = Array.isArray(subjects) ? [...subjects] : [];
      arr.sort((a, b) => {
        const aDate = a?.created_at ? new Date(a.created_at).getTime() : 0;
        const bDate = b?.created_at ? new Date(b.created_at).getTime() : 0;
        if (aDate !== bDate) return bDate - aDate;
        const aId = typeof a?.id === 'number' ? a.id : 0;
        const bId = typeof b?.id === 'number' ? b.id : 0;
        return bId - aId;
      });
      return arr;
    })();

    if (!filterText) return base;
    const q = filterText.toLowerCase();
    return base.filter(s =>
      (s.name || '').toLowerCase().includes(q) ||
      (s.code || '').toLowerCase().includes(q)
    );
  }, [filterText, subjects]);

  const getSchoolName = (id) => schools.find(s => s.id === id)?.nama || '-';
  const getGradeNumber = (id) => gradesAll.find(g => g.id === id)?.grade || '-';

  const columns = [
    { name: '#', selector: (row, index) => index + 1, width: '60px' },
    { name: 'Nama', selector: row => row.name, sortable: true },
    { name: 'Sekolah', selector: row => getSchoolName(row.id_school), sortable: true, wrap: true },
    { name: 'Grade', selector: row => getGradeNumber(row.id_grade), sortable: true, width: '120px' },
    { name: 'Jumlah Soal', selector: row => (typeof row.questions_count === 'number' ? row.questions_count : '-'), sortable: true, width: '140px' },
    {
      name: 'Aksi',
      cell: row => (
        <div className="d-flex gap-2 flex-wrap">
          {canViewQuestion && (
            <button className="btn btn-sm btn-outline-secondary" title="Kelola Soal" onClick={() => navigate(`/subjects/${row.id}/questions`)}>
              Soal
            </button>
          )}
          {canCreateQuestion && (
            <button className="btn btn-sm btn-outline-success" title="Tambah Soal" onClick={() => navigate(`/subjects/${row.id}/questions?add=1`)}>
              + Soal
            </button>
          )}
          {canCreateEssay && (
            <button className="btn btn-sm btn-outline-primary" title="Tambah Essay" onClick={() => navigate(`/subjects/${row.id}/essay-keys`)}>
              + Essay
            </button>
          )}
          {canEditSubject && (
            <button className="btn btn-sm btn-primary" title="Edit" onClick={() => openEdit(row)}>
              <FaEdit />
            </button>
          )}
          {canViewSubject && (
            <button className="btn btn-sm btn-outline-info" title="Detail" onClick={() => openDetail(row)}>
              <FaInfoCircle />
            </button>
          )}
          {canDeleteSubject && (
            <button className="btn btn-sm btn-danger" title="Hapus" onClick={() => handleDelete(row)}>
              <FaTrash />
            </button>
          )}
        </div>
      ),
      width: '360px'
    }
  ];

  if (!canViewSubject) {
    return (
      <div className="container mt-4">
        <h3>Akses Ditolak</h3>
        <p>Anda tidak memiliki izin untuk melihat daftar mata pelajaran.</p>
      </div>
    );
  }

  const openAdd = () => {
    setForm({ name: '', code: '', description: '', id_school: '', id_grade: '' });
    setShowAddModal(true);
  };

  const openEdit = (row) => {
    setEditing(row);
    setForm({ name: row.name || '', code: row.code || '', description: row.description || '', id_school: row.id_school || '', id_grade: row.id_grade || '' });
    setShowEditModal(true);
  };

  const openDetail = (row) => {
    setDetailItem(row);
    setShowDetailModal(true);
  };

  const handleAdd = async () => {
    if (!form.name.trim()) return;
    if (!String(form.id_school).trim() || !String(form.id_grade).trim()) return;
    try {
      await api.post('/subjects', { name: form.name, code: form.code, description: form.description, id_school: Number(form.id_school), id_grade: Number(form.id_grade) }, tokenHeader);
      setShowAddModal(false);
      await fetchSubjects();
      Swal.fire('Berhasil', 'Mata pelajaran ditambahkan', 'success');
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', err.response?.data?.message || 'Gagal menambah mata pelajaran', 'error');
    }
  };

  const handleUpdate = async () => {
    if (!editing) return;
    if (!form.name.trim()) return;
    if (!String(form.id_school).trim() || !String(form.id_grade).trim()) return;
    try {
      await api.put(`/subjects/${editing.id}`, { name: form.name, code: form.code, description: form.description, id_school: Number(form.id_school), id_grade: Number(form.id_grade) }, tokenHeader);
      setShowEditModal(false);
      setEditing(null);
      await fetchSubjects();
      Swal.fire('Berhasil', 'Mata pelajaran diperbarui', 'success');
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', err.response?.data?.message || 'Gagal memperbarui mata pelajaran', 'error');
    }
  };

  const handleDelete = async (row) => {
    const confirm = await Swal.fire({
      title: 'Hapus Mata Pelajaran?',
      text: `Anda akan menghapus: ${row.name}`,
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Ya, hapus',
      cancelButtonText: 'Batal'
    });
    if (!confirm.isConfirmed) return;
    try {
      await api.delete(`/subjects/${row.id}`, { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } });
      await fetchSubjects();
      Swal.fire('Berhasil', 'Mata pelajaran dihapus', 'success');
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', err.response?.data?.message || 'Gagal menghapus mata pelajaran', 'error');
    }
  };

  const handleDeleteAll = async () => {
    const scopeParts = [];
    if (selectedSchoolId) scopeParts.push(`sekolah: ${getSchoolName(Number(selectedSchoolId))}`);
    if (selectedGradeId) scopeParts.push(`grade: ${getGradeNumber(Number(selectedGradeId))}`);
    const scopeText = scopeParts.length ? `mapel sesuai filter (${scopeParts.join(', ')})` : 'semua mapel';
    const confirm = await Swal.fire({
      title: 'Hapus Massal Mata Pelajaran?',
      text: `Anda akan menghapus ${scopeText}. Tindakan ini akan menghapus semua soal dan gambar terkait serta ujian yang berhubungan. Tindakan tidak dapat dibatalkan.`,
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Ya, hapus semua',
      cancelButtonText: 'Batal'
    });
    if (!confirm.isConfirmed) return;
    try {
      const params = {};
      if (selectedSchoolId) params.id_school = Number(selectedSchoolId);
      if (selectedGradeId) params.id_grade = Number(selectedGradeId);
      const res = await api.delete('/subjects', { ...tokenHeader, params });
      await fetchSubjects(selectedSchoolId || '', selectedGradeId || '');
      const msg = res?.data?.message || 'Berhasil menghapus mata pelajaran massal';
      Swal.fire('Berhasil', msg, 'success');
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', err.response?.data?.message || 'Gagal menghapus mata pelajaran massal', 'error');
    }
  };

  if (loading) return <div className="text-center mt-5">Memuat mata pelajaran...</div>;

  return (
    <div className="container mt-4">
      <h3>Daftar Mata Pelajaran</h3>

      <div className="d-flex justify-content-between align-items-center mb-3 gap-2 flex-wrap">
        <div className="d-flex gap-2 align-items-center">
          {canCreateSubject && (
            <button className="btn btn-sm btn-primary d-flex align-items-center" onClick={openAdd}>
              <FaPlus className="me-1" /> Tambah
            </button>
          )}
          {canViewSchool && (
          <select
            className="form-select"
            style={{ minWidth: '220px' }}
            value={selectedSchoolId}
            onChange={e => setSelectedSchoolId(e.target.value)}
          >
            <option value="">Semua Sekolah</option>
            {schools.map(s => (
              <option key={s.id} value={s.id}>{s.nama}</option>
            ))}
          </select>
          )}
          {canViewGrade && (
          <select
            className="form-select"
            style={{ minWidth: '160px' }}
            value={selectedGradeId}
            onChange={e => setSelectedGradeId(e.target.value)}
          >
            <option value="">Semua</option>
            {gradesFilter.map(g => (
              <option key={g.id} value={g.id}>{g.grade}</option>
            ))}
          </select>
          )}
          {isSuperadmin && (
            <button className="btn btn-sm btn-outline-danger d-flex align-items-center" onClick={handleDeleteAll} title={selectedSchoolId || selectedGradeId ? 'Hapus mapel sesuai filter' : 'Hapus semua mapel'}>
              <FaTrash className="me-1" />Semua
            </button>
          )}
        </div>

        <input
          type="text"
          placeholder="Cari nama/kode..."
          className="form-control"
          style={{ maxWidth: '300px' }}
          value={filterText}
          onChange={e => setFilterText(e.target.value)}
        />
      </div>

      <DataTable
        columns={columns}
        data={filtered}
        pagination
        highlightOnHover
        striped
        noHeader
      />

      {/* Modal Add */}
      {showAddModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Tambah Mata Pelajaran</h5>
                <button type="button" className="btn-close" onClick={() => setShowAddModal(false)}></button>
              </div>
              <div className="modal-body">
                <select
                  className="form-select mb-2"
                  value={form.id_school}
                  onChange={e => setForm({ ...form, id_school: e.target.value, id_grade: '' })}
                >
                  <option value="">Pilih Sekolah</option>
                  {schools.map(s => (
                    <option key={s.id} value={s.id}>{s.nama}</option>
                  ))}
                </select>
                <select
                  className="form-select mb-2"
                  value={form.id_grade}
                  onChange={e => setForm({ ...form, id_grade: e.target.value })}
                  disabled={!form.id_school}
                >
                  <option value="">Pilih Grade</option>
                  {gradesForm.map(g => (
                    <option key={g.id} value={g.id}>{g.grade}</option>
                  ))}
                </select>
                <input className="form-control mb-2" placeholder="Nama" value={form.name} onChange={e => setForm({ ...form, name: e.target.value })} />
                <input className="form-control mb-2" placeholder="Kode (opsional)" value={form.code} onChange={e => setForm({ ...form, code: e.target.value })} />
                <textarea className="form-control" rows="3" placeholder="Deskripsi (opsional)" value={form.description} onChange={e => setForm({ ...form, description: e.target.value })}></textarea>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowAddModal(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleAdd}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Edit */}
      {showEditModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Edit Mata Pelajaran</h5>
                <button type="button" className="btn-close" onClick={() => setShowEditModal(false)}></button>
              </div>
              <div className="modal-body">
                <select
                  className="form-select mb-2"
                  value={form.id_school}
                  onChange={e => setForm({ ...form, id_school: e.target.value, id_grade: '' })}
                >
                  <option value="">Pilih Sekolah</option>
                  {schools.map(s => (
                    <option key={s.id} value={s.id}>{s.nama}</option>
                  ))}
                </select>
                <select
                  className="form-select mb-2"
                  value={form.id_grade}
                  onChange={e => setForm({ ...form, id_grade: e.target.value })}
                  disabled={!form.id_school}
                >
                  <option value="">Pilih Grade</option>
                  {gradesForm.map(g => (
                    <option key={g.id} value={g.id}>{g.grade}</option>
                  ))}
                </select>
                <input className="form-control mb-2" placeholder="Nama" value={form.name} onChange={e => setForm({ ...form, name: e.target.value })} />
                <input className="form-control mb-2" placeholder="Kode (opsional)" value={form.code} onChange={e => setForm({ ...form, code: e.target.value })} />
                <textarea className="form-control" rows="3" placeholder="Deskripsi (opsional)" value={form.description} onChange={e => setForm({ ...form, description: e.target.value })}></textarea>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowEditModal(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleUpdate}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Detail */}
      {showDetailModal && detailItem && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Detail Mata Pelajaran</h5>
                <button type="button" className="btn-close" onClick={() => { setShowDetailModal(false); setDetailItem(null); }}></button>
              </div>
              <div className="modal-body">
                <div className="mb-2">
                  <div className="fw-semibold">{detailItem.name || '-'}</div>
                  <div className="d-flex flex-wrap gap-2 mt-1">
                    <span className="badge bg-secondary">Kode: {detailItem.code || '-'}</span>
                    <span className="badge bg-info text-dark">Sekolah: {getSchoolName(detailItem.id_school)}</span>
                    <span className="badge bg-warning text-dark">Grade: {getGradeNumber(detailItem.id_grade)}</span>
                    <span className="badge bg-light text-dark">Jumlah Soal: {typeof detailItem.questions_count === 'number' ? detailItem.questions_count : '-'}</span>
                  </div>
                </div>
                <div className="mb-2"><strong>Deskripsi:</strong></div>
                <div className="p-2 bg-light rounded text-secondary" style={{ whiteSpace: 'pre-wrap' }}>
                  {detailItem.description || '-'}
                </div>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => { setShowDetailModal(false); setDetailItem(null); }}>Tutup</button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}