import React, { useEffect, useMemo, useState } from 'react';
import DataTable from 'react-data-table-component';
import Swal from 'sweetalert2';
import { FaPlus, FaEdit, FaTrash, FaEye } from 'react-icons/fa';
import api from '../utils/axios';
import { Link } from 'react-router-dom';
import { useAuth } from '../context/AuthContext.jsx';
import { canAccess } from '../utils/permissionHelper';

export default function Schools() {
  const { user } = useAuth();
  const [schools, setSchools] = useState([]);
  const [loading, setLoading] = useState(true);
  const [filterText, setFilterText] = useState('');
  const [showAddModal, setShowAddModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [form, setForm] = useState({
    nama: '', alamat: '', kepsek: '',
    kode: '', kota: '', provinsi: '', no: '', status: 'Aktif', note: '',
    logo: null, signature: null, stamp: null,
    logoPath: '', signaturePath: '', stampPath: ''
  });
  const [editing, setEditing] = useState(null);

  const canViewSchool = canAccess(user, 'view-school') || canAccess(user, 'manage-schools');
  const canCreateSchool = canAccess(user, 'create-school') || canAccess(user, 'manage-schools');
  const canEditSchool = canAccess(user, 'edit-school') || canAccess(user, 'manage-schools');
  const canDeleteSchool = canAccess(user, 'delete-school') || canAccess(user, 'manage-schools');

  const fetchSchools = async () => {
    setLoading(true);
    try {
      const res = await api.get('/schools', { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } });
      setSchools(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk melihat daftar sekolah.',
        });
      } else {
        Swal.fire('Error', 'Gagal memuat daftar sekolah', 'error');
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (canViewSchool) {
      fetchSchools();
    } else {
      setLoading(false);
    }
  }, [canViewSchool]);

  const filtered = useMemo(() => {
    if (!filterText) return schools;
    const q = filterText.toLowerCase();
    return schools.filter(s => (s.nama || '').toLowerCase().includes(q) || (s.alamat || '').toLowerCase().includes(q));
  }, [filterText, schools]);

  const openAdd = () => {
    setForm({
      nama: '', alamat: '', kepsek: '',
      kode: '', kota: '', provinsi: '', no: '', status: 'Aktif', note: '',
      logo: null, signature: null, stamp: null,
      logoPath: '', signaturePath: '', stampPath: ''
    });
    setShowAddModal(true);
  };

  const openEdit = (row) => {
    setEditing(row);
    setForm({
      nama: row.nama || '', alamat: row.alamat || '', kepsek: row.kepsek || '',
      kode: row.kode || '', kota: row.kota || '', provinsi: row.provinsi || '',
      no: row.no || '', status: row.status || 'Aktif', note: row.note || '',
      logo: null, signature: null, stamp: null,
      logoPath: row.logo_path || '', signaturePath: row.signature_path || '', stampPath: row.stamp_path || ''
    });
    setShowEditModal(true);
  };

  const handleAdd = async () => {
    if (!form.nama.trim()) return;
    try {
      const fd = new FormData();
      fd.append('nama', form.nama);
      fd.append('alamat', form.alamat || '');
      if (form.kode) fd.append('kode', form.kode || '');
      if (form.kota) fd.append('kota', form.kota || '');
      if (form.provinsi) fd.append('provinsi', form.provinsi || '');
      if (form.no) fd.append('no', form.no || '');
      if (form.status) fd.append('status', form.status || 'Aktif');
      if (form.note) fd.append('note', form.note || '');
      if (form.kepsek) fd.append('kepsek', form.kepsek || '');
      if (form.logo) fd.append('logo', form.logo);
      if (form.signature) fd.append('signature', form.signature);
      if (form.stamp) fd.append('stamp', form.stamp);
      // Path langsung ke folder images/school jika diisi
      if (form.logoPath?.trim()) fd.append('logo_path', form.logoPath.trim());
      if (form.signaturePath?.trim()) fd.append('signature_path', form.signaturePath.trim());
      if (form.stampPath?.trim()) fd.append('stamp_path', form.stampPath.trim());
      await api.post('/schools', fd, { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } });
      setShowAddModal(false);
      await fetchSchools();
      Swal.fire('Berhasil', 'Sekolah ditambahkan', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk menambah sekolah.',
        });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Gagal menambah sekolah', 'error');
      }
    }
  };

  const handleEditSave = async () => {
    if (!editing) return;
    if (!form.nama.trim()) return;
    try {
      const fd = new FormData();
      fd.append('_method', 'PUT');
      fd.append('nama', form.nama);
      fd.append('alamat', form.alamat || '');
      if (form.kode) fd.append('kode', form.kode || '');
      if (form.kota) fd.append('kota', form.kota || '');
      if (form.provinsi) fd.append('provinsi', form.provinsi || '');
      if (form.no) fd.append('no', form.no || '');
      if (form.status) fd.append('status', form.status || 'Aktif');
      if (form.note) fd.append('note', form.note || '');
      if (form.kepsek) fd.append('kepsek', form.kepsek || '');
      if (form.logo) fd.append('logo', form.logo);
      if (form.signature) fd.append('signature', form.signature);
      if (form.stamp) fd.append('stamp', form.stamp);
      // Path langsung ke folder images/school jika diisi
      if (form.logoPath?.trim()) fd.append('logo_path', form.logoPath.trim());
      if (form.signaturePath?.trim()) fd.append('signature_path', form.signaturePath.trim());
      if (form.stampPath?.trim()) fd.append('stamp_path', form.stampPath.trim());
      await api.post(`/schools/${editing.id}`, fd, { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } });
      setShowEditModal(false);
      setEditing(null);
      await fetchSchools();
      Swal.fire('Berhasil', 'Sekolah diperbarui', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk memperbarui sekolah.',
        });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Gagal memperbarui sekolah', 'error');
      }
    }
  };

  const handleDelete = async (row) => {
    const ok = await Swal.fire({
      title: 'Hapus sekolah?',
      text: `Anda yakin menghapus "${row.nama}"?`,
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Ya, hapus',
      cancelButtonText: 'Batal'
    }).then(r => r.isConfirmed);
    if (!ok) return;
    try {
      await api.delete(`/schools/${row.id}`, { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } });
      await fetchSchools();
      Swal.fire('Berhasil', 'Sekolah dihapus', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk menghapus sekolah.',
        });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Gagal menghapus sekolah', 'error');
      }
    }
  };

  const columns = [
    { name: '#', selector: (row, index) => index + 1, width: '60px' },
    { name: 'Nama', selector: row => row.nama, sortable: true },
    { name: 'Alamat', selector: row => row.alamat || '-', wrap: true },
    { name: 'Nama Kepala Sekolah', selector: row => row.kepsek || '-', wrap: true },
    {
      name: 'Aksi',
      cell: row => (
        <div className="d-flex gap-2 flex-wrap">
          {canEditSchool && (
            <button className="btn btn-sm btn-primary" title="Edit" onClick={() => openEdit(row)}>
              <FaEdit />
            </button>
          )}
          {canDeleteSchool && (
            <button className="btn btn-sm btn-danger" title="Hapus" onClick={() => handleDelete(row)}>
              <FaTrash />
            </button>
          )}
          {canViewSchool && (
            <Link to={`/schools/${row.id}`} className="btn btn-sm btn-outline-primary" title="Detail">
              <FaEye />
            </Link>
          )}
        </div>
      ),
      width: '160px'
    }
  ];
  if (!canViewSchool) {
    return (
      <div className="container mt-4">
        <h3>Akses Ditolak</h3>
        <p>Anda tidak memiliki izin untuk melihat daftar sekolah.</p>
      </div>
    );
  }
  if (loading) return <div className="text-center mt-5">Memuat sekolah...</div>;

  return (
    <div className="container mt-4">
      <h3>Daftar Sekolah</h3>

      <div className="d-flex justify-content-between align-items-center mb-3">
        {canCreateSchool && (
          <button className="btn btn-primary" onClick={openAdd}>
            <FaPlus className="me-1" /> Tambah
          </button>
        )}
        <input
          type="text"
          placeholder="Cari nama/alamat..."
          className="form-control"
          style={{ maxWidth: '300px' }}
          value={filterText}
          onChange={e => setFilterText(e.target.value)}
        />
      </div>

      <DataTable
        columns={columns}
        data={filtered}
        pagination
        highlightOnHover
        striped
        noHeader
      />

      {/* Modal Add */}
      {showAddModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Tambah Sekolah</h5>
                <button type="button" className="btn-close" onClick={() => setShowAddModal(false)}></button>
              </div>
      <div className="modal-body">
        <input className="form-control mb-2" placeholder="Nama" value={form.nama} onChange={e => setForm({ ...form, nama: e.target.value })} />
        <input className="form-control mb-2" placeholder="Alamat (opsional)" value={form.alamat} onChange={e => setForm({ ...form, alamat: e.target.value })} />
        <input className="form-control mb-2" placeholder="Nama Kepala Sekolah (opsional)" value={form.kepsek} onChange={e => setForm({ ...form, kepsek: e.target.value })} />
        <div className="row g-2 mt-1">
          <div className="col-6">
            <input className="form-control" placeholder="KODE/NUPTK/NIDN/NIP" value={form.kode} onChange={e => setForm({ ...form, kode: e.target.value })} />
          </div>
          <div className="col-6">
            <input className="form-control" placeholder="Nomor Sekolah (opsional)" value={form.no} onChange={e => setForm({ ...form, no: e.target.value })} />
          </div>
          <div className="col-6 mt-2">
            <input className="form-control" placeholder="Kota (opsional)" value={form.kota} onChange={e => setForm({ ...form, kota: e.target.value })} />
          </div>
          <div className="col-6 mt-2">
            <input className="form-control" placeholder="Provinsi (opsional)" value={form.provinsi} onChange={e => setForm({ ...form, provinsi: e.target.value })} />
          </div>
          <div className="col-6 mt-2">
            <select className="form-select" value={form.status} onChange={e => setForm({ ...form, status: e.target.value })}>
              <option value="Aktif">Aktif</option>
              <option value="Tidak Aktif">Tidak Aktif</option>
            </select>
          </div>
          <div className="col-12 mt-2">
            <textarea className="form-control" rows={3} placeholder="Catatan (opsional)" value={form.note} onChange={e => setForm({ ...form, note: e.target.value })}></textarea>
          </div>
        </div>
        <label className="form-label mt-2">Logo Sekolah (PNG/JPG, maks 2MB)</label>
        <input type="file" accept="image/png,image/jpeg" className="form-control" onChange={e => setForm({ ...form, logo: e.target.files?.[0] || null })} />
        <small className="text-muted">Atau isi path langsung (contoh: images/school/logo.png)</small>
        <input type="text" className="form-control mt-1" placeholder="images/school/logo.png" value={form.logoPath} onChange={e => setForm({ ...form, logoPath: e.target.value })} />
        <label className="form-label mt-2">TTD Kepala Sekolah (PNG/JPG)</label>
        <input type="file" accept="image/png,image/jpeg" className="form-control" onChange={e => setForm({ ...form, signature: e.target.files?.[0] || null })} />
        <small className="text-muted">Atau isi path langsung (contoh: images/school/signature.png)</small>
        <input type="text" className="form-control mt-1" placeholder="images/school/signature.png" value={form.signaturePath} onChange={e => setForm({ ...form, signaturePath: e.target.value })} />
        <label className="form-label mt-2">Cap/Stempel Sekolah (PNG/JPG)</label>
        <input type="file" accept="image/png,image/jpeg" className="form-control" onChange={e => setForm({ ...form, stamp: e.target.files?.[0] || null })} />
        <small className="text-muted">Atau isi path langsung (contoh: images/school/stamp.png)</small>
        <input type="text" className="form-control mt-1" placeholder="images/school/stamp.png" value={form.stampPath} onChange={e => setForm({ ...form, stampPath: e.target.value })} />
      </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowAddModal(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleAdd}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Edit */}
      {showEditModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Edit Sekolah</h5>
                <button type="button" className="btn-close" onClick={() => setShowEditModal(false)}></button>
              </div>
      <div className="modal-body">
        <input className="form-control mb-2" placeholder="Nama" value={form.nama} onChange={e => setForm({ ...form, nama: e.target.value })} />
        <input className="form-control mb-2" placeholder="Alamat (opsional)" value={form.alamat} onChange={e => setForm({ ...form, alamat: e.target.value })} />
        <input className="form-control mb-2" placeholder="Nama Kepala Sekolah (opsional)" value={form.kepsek} onChange={e => setForm({ ...form, kepsek: e.target.value })} />
        <div className="row g-2 mt-1">
          <div className="col-6">
            <input className="form-control" placeholder="KODE/NUPTK/NIDN/NIP" value={form.kode} onChange={e => setForm({ ...form, kode: e.target.value })} />
          </div>
          <div className="col-6">
            <input className="form-control" placeholder="Nomor Sekolah (opsional)" value={form.no} onChange={e => setForm({ ...form, no: e.target.value })} />
          </div>
          <div className="col-6 mt-2">
            <input className="form-control" placeholder="Kota (opsional)" value={form.kota} onChange={e => setForm({ ...form, kota: e.target.value })} />
          </div>
          <div className="col-6 mt-2">
            <input className="form-control" placeholder="Provinsi (opsional)" value={form.provinsi} onChange={e => setForm({ ...form, provinsi: e.target.value })} />
          </div>
          <div className="col-6 mt-2">
            <select className="form-select" value={form.status} onChange={e => setForm({ ...form, status: e.target.value })}>
              <option value="Aktif">Aktif</option>
              <option value="Tidak Aktif">Tidak Aktif</option>
            </select>
          </div>
          <div className="col-12 mt-2">
            <textarea className="form-control" rows={3} placeholder="Catatan (opsional)" value={form.note} onChange={e => setForm({ ...form, note: e.target.value })}></textarea>
          </div>
        </div>
        <div className="row g-2 mt-2">
          <div className="col-12">
            <label className="form-label">Logo Sekolah (PNG/JPG)</label>
            <input type="file" accept="image/png,image/jpeg" className="form-control" onChange={e => setForm({ ...form, logo: e.target.files?.[0] || null })} />
            <small className="text-muted">Atau isi path langsung (contoh: images/school/logo.png)</small>
            <input type="text" className="form-control mt-1" placeholder="images/school/logo.png" value={form.logoPath} onChange={e => setForm({ ...form, logoPath: e.target.value })} />
          </div>
          <div className="col-12">
            <label className="form-label">TTD Kepala Sekolah (PNG/JPG)</label>
            <input type="file" accept="image/png,image/jpeg" className="form-control" onChange={e => setForm({ ...form, signature: e.target.files?.[0] || null })} />
            <small className="text-muted">Atau isi path langsung (contoh: images/school/signature.png)</small>
            <input type="text" className="form-control mt-1" placeholder="images/school/signature.png" value={form.signaturePath} onChange={e => setForm({ ...form, signaturePath: e.target.value })} />
          </div>
          <div className="col-12">
            <label className="form-label">Cap/Stempel Sekolah (PNG/JPG)</label>
            <input type="file" accept="image/png,image/jpeg" className="form-control" onChange={e => setForm({ ...form, stamp: e.target.files?.[0] || null })} />
            <small className="text-muted">Atau isi path langsung (contoh: images/school/stamp.png)</small>
            <input type="text" className="form-control mt-1" placeholder="images/school/stamp.png" value={form.stampPath} onChange={e => setForm({ ...form, stampPath: e.target.value })} />
          </div>
        </div>
      </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowEditModal(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleEditSave}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}