import React, { useEffect, useState } from 'react';
import { useParams, Link } from 'react-router-dom';
import api from '../utils/axios';

// Pastikan akses file asset mengarah ke backend (dukungan /images dan /storage)
const BACKEND_BASE = import.meta.env.VITE_BACKEND_URL || 'http://127.0.0.1:8000';
const resolveImageUrl = (urlOrPath) => {
  if (!urlOrPath) return null;
  const s = String(urlOrPath).trim();
  const base = String(BACKEND_BASE || '').replace(/\/+$/, '');
  if (/^(data|blob):/.test(s)) return s;
  // Rebase absolut ke /storage/ atau /images/ supaya origin mengikuti BACKEND_BASE
  if (/^https?:\/\/[^/]+\/(storage|images)\//.test(s)) {
    return s.replace(/^https?:\/\/[^/]+/, base);
  }
  if (/^https?:\/\//.test(s)) return s;
  if (/^\/storage\//.test(s)) return `${base}${s}`;
  const clean = s.replace(/^\/+/, '');
  if (/^images\//.test(clean)) {
    return `${base}/${clean}`;
  }
  if (/^storage\//.test(clean)) {
    return `${base}/storage/${clean.replace(/^storage\//,'')}`;
  }
  return `${base}/storage/${clean.replace(/^public\//,'')}`;
};

export default function SchoolDetail() {
  const { schoolId } = useParams();
  const [school, setSchool] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');

  useEffect(() => {
    const load = async () => {
      setLoading(true);
      setError('');
      try {
        const res = await api.get(`/schools/${schoolId}`, { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } });
        setSchool(res.data);
      } catch (e) {
        setError(e.response?.data?.message || 'Gagal memuat detail sekolah');
      } finally {
        setLoading(false);
      }
    };
    load();
  }, [schoolId]);

  if (loading) return <div className="container mt-4">Memuat detail sekolah...</div>;
  if (error) return <div className="container mt-4"><div className="alert alert-danger">{error}</div></div>;
  if (!school) return <div className="container mt-4">Data sekolah tidak ditemukan.</div>;

  return (
    <div className="container mt-4">
      <div className="d-flex justify-content-between align-items-center mb-3">
        <h3>Detail Sekolah</h3>
        <Link className="btn btn-outline-secondary btn-sm" to="/schools">Kembali ke Daftar</Link>
      </div>

      <div className="card">
        <div className="card-body">
          <div className="row g-3">
            <div className="col-md-6">
              <div><strong>Nama</strong><div>{school.nama || '-'}</div></div>
              <div className="mt-2"><strong>Alamat</strong><div>{school.alamat || '-'}</div></div>
              <div className="mt-2"><strong>Nama Kepala Sekolah</strong><div>{school.kepsek || '-'}</div></div>
              <div className="mt-2"><strong>Kode</strong><div>{school.kode || '-'}</div></div>
              <div className="mt-2"><strong>Kota</strong><div>{school.kota || '-'}</div></div>
              <div className="mt-2"><strong>Provinsi</strong><div>{school.provinsi || '-'}</div></div>
              <div className="mt-2"><strong>No</strong><div>{school.no || '-'}</div></div>
              <div className="mt-2"><strong>Status</strong><div>{school.status || '-'}</div></div>
              <div className="mt-2"><strong>Catatan</strong><div>{school.note || '-'}</div></div>
            </div>
            <div className="col-md-6">
              <div className="mb-3">
                <strong>Logo Sekolah</strong>
                <div>
                  {school.logo_path ? (
                    <img src={resolveImageUrl(school.logo_path)} alt="Logo" style={{ maxWidth: '200px', height: 'auto' }} onError={(e) => { const raw = String(school.logo_path||'').trim(); const base = String(BACKEND_BASE||'').replace(/\/+$/, ''); const clean = raw.replace(/^\/+/, ''); let alt = null; if (/^images\//.test(clean)) alt = `${base}/storage/${clean}`; else if (/^storage\//.test(clean)) alt = `${base}/${clean.replace(/^storage\//,'')}`; if (alt){ e.currentTarget.onerror = null; e.currentTarget.src = alt; } }} />
                  ) : (
                    <span className="text-muted">Tidak ada</span>
                  )}
                </div>
              </div>
              <div className="mb-3">
                <strong>TTD Kepala Sekolah</strong>
                <div>
                  {school.signature_path ? (
                    <img src={resolveImageUrl(school.signature_path)} alt="TTD" style={{ maxWidth: '200px', height: 'auto' }} onError={(e) => { const raw = String(school.signature_path||'').trim(); const base = String(BACKEND_BASE||'').replace(/\/+$/, ''); const clean = raw.replace(/^\/+/, ''); let alt = null; if (/^images\//.test(clean)) alt = `${base}/storage/${clean}`; else if (/^storage\//.test(clean)) alt = `${base}/${clean.replace(/^storage\//,'')}`; if (alt){ e.currentTarget.onerror = null; e.currentTarget.src = alt; } }} />
                  ) : (
                    <span className="text-muted">Tidak ada</span>
                  )}
                </div>
              </div>
              <div className="mb-3">
                <strong>Cap/Stempel Sekolah</strong>
                <div>
                  {school.stamp_path ? (
                    <img src={resolveImageUrl(school.stamp_path)} alt="Cap" style={{ maxWidth: '200px', height: 'auto' }} onError={(e) => { const raw = String(school.stamp_path||'').trim(); const base = String(BACKEND_BASE||'').replace(/\/+$/, ''); const clean = raw.replace(/^\/+/, ''); let alt = null; if (/^images\//.test(clean)) alt = `${base}/storage/${clean}`; else if (/^storage\//.test(clean)) alt = `${base}/${clean.replace(/^storage\//,'')}`; if (alt){ e.currentTarget.onerror = null; e.currentTarget.src = alt; } }} />
                  ) : (
                    <span className="text-muted">Tidak ada</span>
                  )}
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}