import React, { useEffect, useState } from 'react';
import { FaTrash } from 'react-icons/fa';
import api from '../utils/axios';
import Swal from 'sweetalert2';
import DataTable from 'react-data-table-component';
import { useAuth } from '../context/AuthContext.jsx';
import { canAccess } from '../utils/permissionHelper';

export default function RolesList() {
  const { user } = useAuth();
  const canManageRoles = canAccess(user, 'manage-roles');
  const canViewRoles = canAccess(user, 'view-role') || canManageRoles;
  const [roles, setRoles] = useState([]);
  const [filteredRoles, setFilteredRoles] = useState([]);
  const [newRole, setNewRole] = useState('');
  const [loading, setLoading] = useState(true);
  const [showAddModal, setShowAddModal] = useState(false);
  const [searchText, setSearchText] = useState('');

  const fetchRoles = async () => {
    try {
      setLoading(true);
      const res = await api.get('/roles', {
        headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }
      });
      setRoles(res.data);
      setFilteredRoles(res.data);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk melihat daftar roles.',
        });
      } else {
        Swal.fire('Error', 'Gagal mengambil data roles!', 'error');
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => { if (canViewRoles) fetchRoles(); }, [canViewRoles]);

  // Filter roles berdasarkan search
  useEffect(() => {
    if (!searchText) {
      setFilteredRoles(roles);
    } else {
      const filtered = roles.filter(role =>
        role.name.toLowerCase().includes(searchText.toLowerCase())
      );
      setFilteredRoles(filtered);
    }
  }, [searchText, roles]);

  const handleAddRole = async () => {
    if (!newRole.trim()) return;

    if (!canManageRoles) {
      Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk menambahkan role.' });
      return;
    }

    try {
      await api.post('/roles', { name: newRole }, {
        headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }
      });
      setNewRole('');
      setShowAddModal(false);
      fetchRoles();
      Swal.fire('Berhasil', 'Role berhasil ditambahkan', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk menambahkan role.',
        });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Gagal menambahkan role', 'error');
      }
    }
  };

  const handleDeleteRole = async (id) => {
    const result = await Swal.fire({
      title: 'Apakah yakin ingin menghapus role ini?',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Ya, hapus!',
      cancelButtonText: 'Batal'
    });

    if (result.isConfirmed) {
      if (!canManageRoles) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk menghapus role.' });
        return;
      }
      try {
        await api.delete(`/roles/${id}`, {
          headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }
        });
        fetchRoles();
        Swal.fire('Berhasil', 'Role berhasil dihapus', 'success');
      } catch (err) {
        console.error(err);
        const status = err?.response?.status;
        if (status === 403) {
          Swal.fire({
            icon: 'warning',
            title: 'Akses Ditolak',
            text: 'Anda tidak memiliki izin untuk menghapus role.',
          });
        } else {
          Swal.fire('Gagal', err.response?.data?.message || 'Gagal menghapus role', 'error');
        }
      }
    }
  };

  if (loading) return <div className="text-center mt-5">Loading roles...</div>;
  if (!canViewRoles) return <div className="container mt-4"><h3>Akses Ditolak</h3><p>Anda tidak memiliki izin untuk mengakses daftar roles.</p></div>;

  const columns = [
    {
      name: '#',
      cell: (row, index) => index + 1,
      width: '60px'
    },
    {
      name: 'Nama Role',
      selector: row => row.name,
      sortable: true
    },
    ...(canManageRoles ? [{
      name: 'Aksi',
      cell: row => (
        <button
          className="btn btn-sm btn-danger"
          onClick={() => handleDeleteRole(row.id)}
          title="Hapus"
          aria-label="Hapus"
        >
          <FaTrash />
        </button>
      )
    }] : [])
  ];

  return (
    <div className="container mt-4">
      <h3>Daftar Roles</h3>

      <div className="mb-3 d-flex justify-content-between">
        {canManageRoles && (
          <button className="btn btn-primary" onClick={() => setShowAddModal(true)}>
            Tambah Role
          </button>
        )}

        <input
          type="text"
          placeholder="Cari role..."
          className="form-control w-25"
          value={searchText}
          onChange={e => setSearchText(e.target.value)}
        />
      </div>

      <DataTable
        columns={columns}
        data={filteredRoles}
        pagination
        highlightOnHover
        striped
        noHeader
      />

      {/* Modal Tambah Role */}
      {showAddModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Tambah Role</h5>
                <button type="button" className="btn-close" onClick={() => setShowAddModal(false)}></button>
              </div>
              <div className="modal-body">
                <input
                  type="text"
                  className="form-control"
                  placeholder="Nama role baru"
                  value={newRole}
                  onChange={e => setNewRole(e.target.value)}
                />
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowAddModal(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleAddRole}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}
      {showAddModal && <div className="modal-backdrop fade show"></div>}
    </div>
  );
}
