import React, { useEffect, useState, useMemo } from 'react';
import api from '../utils/axios';
import Swal from 'sweetalert2';
import CardCheckbox from '../components/CardCheckbox';
import PermissionCheckbox from '../components/PermissionCheckbox';

export default function RolePermissions() {
  const [roles, setRoles] = useState([]);
  const [permissions, setPermissions] = useState([]);
  const [selectedRoles, setSelectedRoles] = useState([]);
  const [assignedPermissions, setAssignedPermissions] = useState({});
  const [loading, setLoading] = useState(true);
  const [loadingRole, setLoadingRole] = useState(null);

  // Kategorisasi permissions agar tampilan lebih jelas
  const CATEGORY_ORDER = [
    'Dashboard',
    'Hak Akses',
    'Pengguna',
    // Akademik
    'Sekolah',
    'Grade',
    'Kelas',
    // Konten & Ujian
    'Mata Pelajaran',
    'Soal',
    'Ujian',
    'Peserta Ujian',
    // Laporan
    'Nilai',
    'Hasil',
    // Fallback
    'Lainnya'
  ];

  const getPermissionCategory = (name) => {
    const n = String(name || '').toLowerCase();
    // Spesifik terlebih dulu agar tidak tersapu oleh kategori yang lebih umum
    if (n.includes('dashboard')) return 'Dashboard';
    if (n.includes('role') || n.includes('permission')) return 'Hak Akses';
    if (n.includes('user')) return 'Pengguna';
    // Akademik
    if (n.includes('school')) return 'Sekolah';
    if (n.includes('grade')) return 'Grade';
    if (n.includes('class')) return 'Kelas';
    // Konten & Ujian
    if (n.includes('subject')) return 'Mata Pelajaran';
    if (n.includes('question')) return 'Soal';
    if (n.includes('exam-participant')) return 'Peserta Ujian';
    if (n.includes('exam')) return 'Ujian';
    // Laporan
    if (n.includes('score')) return 'Nilai';
    if (n.includes('result')) return 'Hasil';
    // Fallback
    return 'Lainnya';
  };

  const groupedPermissions = useMemo(() => {
    const map = {};
    (permissions || []).forEach(p => {
      const cat = getPermissionCategory(p?.name);
      if (!map[cat]) map[cat] = [];
      map[cat].push(p);
    });

    // Urutkan permissions dalam setiap kategori supaya tampilan rapi dan konsisten
    const ACTION_ORDER = ['view', 'create', 'edit', 'delete', 'manage', 'import', 'export', 'reset'];
    const actionKey = (n) => {
      const s = String(n || '').toLowerCase();
      if (s.startsWith('view-')) return 'view';
      if (s.startsWith('create-')) return 'create';
      if (s.startsWith('edit-')) return 'edit';
      if (s.startsWith('delete-')) return 'delete';
      if (s.startsWith('manage-')) return 'manage';
      if (s.startsWith('import-')) return 'import';
      if (s.startsWith('export-')) return 'export';
      if (s.includes('reset')) return 'reset';
      return 'zz';
    };

    Object.keys(map).forEach(cat => {
      map[cat] = map[cat]
        .slice()
        .sort((a, b) => {
          const ak = actionKey(a?.name);
          const bk = actionKey(b?.name);
          const ai = ACTION_ORDER.indexOf(ak);
          const bi = ACTION_ORDER.indexOf(bk);
          if (ai !== bi) return (ai === -1 ? 999 : ai) - (bi === -1 ? 999 : bi);
          // Jika sama jenis aksi, urutkan alfabetis
          return String(a?.name || '').localeCompare(String(b?.name || ''));
        });
    });

    return map;
  }, [permissions]);

  const fetchData = async () => {
    try {
      setLoading(true);
      const [rolesRes, permRes] = await Promise.all([
        api.get('/roles', { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } }),
        api.get('/permissions', { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } }),
      ]);

      setRoles(Array.isArray(rolesRes.data) ? rolesRes.data : []);
      setPermissions(Array.isArray(permRes.data) ? permRes.data : []);

      const assignObj = {};
      (rolesRes.data || []).forEach(role => {
        assignObj[role.id] = [];
      });
      setAssignedPermissions(assignObj);

    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk mengakses menu Role & Permissions.',
        });
      } else {
        Swal.fire('Error', 'Gagal mengambil data!', 'error');
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => { fetchData(); }, []);
  const toggleRole = async (roleId) => {
    if (selectedRoles.includes(roleId)) {
      // Hapus role dari selectedRoles
      setSelectedRoles(prev => prev.filter(r => r !== roleId));
    } else {
      // Tambahkan role
      setSelectedRoles(prev => [...prev, roleId]);
      setLoadingRole(roleId);
  
      try {
        // Ambil permissions role dari backend
        const res = await api.get(`/role/permissions/${roleId}`, {
          headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }
        });
  
        // Pastikan data selalu array
        let permArray = [];
        if (Array.isArray(res.data)) {
          permArray = res.data;
        } else if (res.data && typeof res.data === 'object') {
          // Jika backend mengembalikan object, ambil values
          permArray = Object.values(res.data);
        }
  
        // Ambil ID permission
        // Normalisasi tipe ID ke number untuk konsistensi toggle
        const permIds = permArray.map(p => Number(p.id));
  
        // Update assignedPermissions dengan role baru
        setAssignedPermissions(prev => ({
          ...prev,
          [roleId]: permIds
        }));
      } catch (err) {
        console.error(err);
        const status = err?.response?.status;
        if (status === 403) {
          Swal.fire({
            icon: 'warning',
            title: 'Akses Ditolak',
            text: 'Anda tidak memiliki izin untuk melihat permissions role ini.',
          });
        } else {
          Swal.fire('Error', 'Gagal mengambil permissions role ini', 'error');
        }
      } finally {
        setLoadingRole(null);
      }
    }
  };
  

  const togglePermission = (permissionId) => {
    const pid = Number(permissionId);
    setAssignedPermissions(prev => {
      const newAssign = { ...prev };
      selectedRoles.forEach(roleId => {
        if (!newAssign[roleId]) newAssign[roleId] = [];
        const has = newAssign[roleId].includes(pid);
        if (has) {
          newAssign[roleId] = newAssign[roleId].filter(p => p !== pid);
        } else {
          // gunakan spread agar tidak memutasi array lama
          newAssign[roleId] = [...newAssign[roleId], pid];
        }
      });
      return newAssign;
    });
  };

  const savePermissions = async () => {
    // Konfirmasi sebelum menyimpan perubahan
    const confirm = await Swal.fire({
      title: 'Simpan perubahan permissions?',
      text: 'Apakah Anda yakin ingin menerapkan perubahan ke role terpilih?',
      icon: 'question',
      showCancelButton: true,
      confirmButtonText: 'Ya, simpan',
      cancelButtonText: 'Batal'
    }).then(r => r.isConfirmed);

    if (!confirm) return;

    try {
      const payload = {
        roles: selectedRoles,
        permissions: Array.from(
          new Set(
            selectedRoles.flatMap(roleId => (assignedPermissions[roleId] || []).map(Number))
          )
        )
      };

      // POST ke backend untuk sinkronisasi permissions
      await api.post('/roles/permissions', payload, {
        headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }
      });

      Swal.fire({
        title: 'Berhasil',
        text: 'Permissions berhasil diupdate',
        icon: 'success',
      }).then(() => {
        window.location.reload(); // reload page
      });

    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk memperbarui permissions.',
          confirmButtonText: 'OK'
        });
      } else {
        Swal.fire({
          title: 'Gagal',
          text: err?.response?.data?.message || 'Gagal menyimpan permissions',
          icon: 'error',
          confirmButtonText: 'OK'
        });
      }
    }
  };
  

  if (loading) return <div className="text-center mt-5">Loading...</div>;

  return (
    <div className="container mt-4">
      <h3 className="mb-3 text-center">Role - Permissions Management</h3>

      <h5>Roles</h5>
      <div style={{ display: 'flex', flexWrap: 'wrap', gap: '15px', marginBottom: '30px', justifyContent: 'center' }}>
        {roles.map(role => (
          <CardCheckbox
            key={role.id}
            role={role}
            checked={selectedRoles.includes(role.id)}
            onChange={toggleRole}
            loading={loadingRole === role.id}
          />
        ))}
      </div>

      <h5>Permissions</h5>
      {CATEGORY_ORDER.filter(cat => groupedPermissions[cat]?.length).map(cat => (
        <div key={cat} className="mb-4">
          <div className="d-flex align-items-center justify-content-center mb-2">
            <span className="badge bg-secondary" style={{ fontSize: '0.95rem' }}>{cat}</span>
          </div>
          <div style={{ display: 'flex', flexWrap: 'wrap', gap: '15px', justifyContent: 'center' }}>
            {groupedPermissions[cat].map(perm => (
              <PermissionCheckbox
                key={perm.id}
                permission={perm}
                checked={selectedRoles.some(roleId => (assignedPermissions[roleId] || []).includes(Number(perm.id)))}
                onToggle={togglePermission}
                disabled={loadingRole !== null || selectedRoles.length === 0}
              />
            ))}
          </div>
        </div>
      ))}

      <div className="text-center">
      <button
        type="button"  // penting agar tidak submit form
        className="btn btn-success btn-lg"
        onClick={savePermissions}
        disabled={selectedRoles.length === 0}
      >
        Simpan Permissions untuk Role Terpilih
      </button>
    </div>


    </div>
  );
}
