import React, { useEffect, useMemo, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import Notification from '../components/Notification.jsx';
import ExamStartModal from '../components/ExamStartModal.jsx';
import Swal from 'sweetalert2';
import { useParticipantAuth } from '../context/ParticipantAuthContext.jsx';
import api from '../utils/axios';
import { requestFullscreenEnhanced } from '../utils/fullscreenHelper';

export default function ParticipantHome(){
  const navigate = useNavigate();
  const [list,setList] = useState([]);
  const [loading,setLoading] = useState(false);
  const [notif,setNotif] = useState(null);
  const { participant, logout } = useParticipantAuth();
  const participantKey = participant?.id || participant?.nisn || 'unknown';
  const [startingId, setStartingId] = useState(null);

  useEffect(() => {
    const fetchToday = async () => {
      setLoading(true);
      try {
        const res = await api.get('/exams/today');
        const data = res.data?.data || res.data || [];
        setList(Array.isArray(data) ? data : []);
      } catch(err){
        console.error(err);
        Swal.fire('Error', 'Gagal mengambil daftar ujian hari ini', 'error');
      } finally { setLoading(false); }
    };
    fetchToday();
  }, []);

  const statusLabel = (status) => {
    const map = {
      belum_mulai: 'Silahkan Kerjakan',
      mulai: 'Sedang Dikerjakan',
      expired: 'Expired',
      proses_hasil: 'Proses Hasil',
      success: 'Selesai',
      selesai: 'Selesai',
    };
    return map[status] || status;
  };

  const statusClass = (status) => {
    const map = {
      belum_mulai: 'text-bg-secondary',
      mulai: 'text-bg-primary',
      expired: 'text-bg-warning',
      proses_hasil: 'text-bg-info',
      success: 'text-bg-success',
      selesai: 'text-bg-success',
    };
    return map[status] || 'text-bg-secondary';
  };

  const getAttempts = (examId) => {
    try {
      const key = `examStartAttempts:${participantKey}:${examId}`;
      const raw = localStorage.getItem(key);
      const n = parseInt(raw || '0', 10);
      return Number.isFinite(n) && n >= 0 ? n : 0;
    } catch (_) { return 0; }
  };

  const incAttempts = (examId) => {
    try {
      const current = getAttempts(examId);
      const next = Math.min(3, current + 1);
      const key = `examStartAttempts:${participantKey}:${examId}`;
      localStorage.setItem(key, String(next));
      return next;
    } catch (_) { return 0; }
  };

  // Gabungkan status backend dengan aturan lokal (attempts/expired) agar sesuai alur
  const effectiveStatus = (exam) => {
    try {
      const now = new Date();
      const endAt = new Date(exam.end_at_iso || exam.end_at);
      const localUsed = getAttempts(exam.id);
      const backend = String(exam?.my_status || 'belum_mulai');
      const finishRequested = !!exam?.my_session?.finish_requested_at;
      const hasResult = backend === 'success' || backend === 'selesai';
      const entryAttempts = typeof exam?.my_session?.entry_attempts === 'number'
        ? exam.my_session.entry_attempts
        : localUsed;
      const hasSession = !!exam?.my_session?.id;

      if (hasResult) return 'success';
      if (finishRequested) return 'proses_hasil';
      if (entryAttempts > 3) return 'proses_hasil';
      if (!hasSession && (now > endAt)) return 'expired';
      return backend;
    } catch (_) {
      return String(exam?.my_status || 'belum_mulai');
    }
  };

  const sorted = useMemo(() => {
    const now = new Date();
    const canStart = (exam) => {
      const entryAttempts = typeof exam?.my_session?.entry_attempts === 'number' ? exam.my_session.entry_attempts : getAttempts(exam.id);
      const expired = now > new Date(exam.end_at_iso || exam.end_at);
      const eff = effectiveStatus(exam);
      const finished = ['success','selesai','proses_hasil'].includes(String(eff));
      return exam.can_start && entryAttempts < 3 && !expired && !finished;
    };
    const minutesUntilStart = (exam) => {
      const s = exam?.start_at_iso ? new Date(exam.start_at_iso) : (exam?.start_at ? new Date(exam.start_at) : null);
      if (!s) return Infinity;
      const diffMin = (s - now) / 60000;
      return diffMin;
    };
    const isStartingSoon = (exam) => {
      const entryAttempts = typeof exam?.my_session?.entry_attempts === 'number' ? exam.my_session.entry_attempts : getAttempts(exam.id);
      const expired = now > new Date(exam.end_at_iso || exam.end_at);
      const eff = effectiveStatus(exam);
      const finished = ['success','selesai','proses_hasil'].includes(String(eff));
      const m = minutesUntilStart(exam);
      return !expired && !finished && entryAttempts < 3 && m > 0 && m <= 5;
    };

    try {
      const sortedList = [...list].sort((a, b) => {
        const sa = minutesUntilStart(a);
        const sb = minutesUntilStart(b);
        const aSoon = isStartingSoon(a);
        const bSoon = isStartingSoon(b);
        if (aSoon && !bSoon) return -1;
        if (!aSoon && bSoon) return 1;
        if (sa !== sb) return sa - sb;
        const aCan = canStart(a);
        const bCan = canStart(b);
        if (aCan && !bCan) return -1;
        if (!aCan && bCan) return 1;
        return String(a.name).localeCompare(String(b.name));
      });
      return sortedList;
    } catch (_) {
      return list;
    }
  }, [list]);

  // Tambah: handler logout
  const handleLogout = async () => {
    try { await logout(); } catch (_) {}
    navigate('/exam/login');
  };

  const handleStart = async (exam) => {
    const used = getAttempts(exam.id);
    const now = new Date();
    const endAt = new Date(exam.end_at_iso || exam.end_at);
    const isExpired = now > endAt;
    if (!exam.can_start || isExpired || used >= 3) return;

    try {
      setStartingId(exam.id);
      const res = await api.post(`/exams/${exam.id}/start`);
      const queued = !!res?.data?.queued;
      const status = String(res?.data?.status || '');
      const sessionId = res?.data?.session_id;
      if (queued || status === 'proses_hasil') {
        try { await Swal.fire('Proses Hasil', 'Kesempatan sudah habis, hasil ujian Anda sedang diproses.', 'info'); } catch (_) {}
        // Reload daftar ujian
        try {
          const reload = await api.get('/exams/today');
          const data = reload.data?.data || reload.data || [];
          setList(Array.isArray(data) ? data : []);
        } catch (_) {}
        return;
      }
      if (!sessionId) throw new Error('Gagal memulai sesi ujian');
      incAttempts(exam.id);
      // Minta fullscreen segera setelah klik Mulai (masih dalam gesture pengguna)
      try {
        await requestFullscreenEnhanced(document.documentElement, {
          showWarning: false,
          fallbackToViewport: true,
          mobileOptimization: true
        });
      } catch (error) {
        console.warn('Fullscreen request failed:', error);
      }
      navigate(`/exam/session/${sessionId}`);
    } catch (err) {
      const statusCode = err?.response?.status;
      const msg = err?.response?.data?.message || err?.message || 'Gagal memulai ujian';
      if (statusCode === 403) {
        Swal.fire('Tidak Diizinkan', msg, 'warning');
      } else if (statusCode === 422) {
        Swal.fire('Waktu Tidak Sesuai', msg, 'info');
      } else {
        Swal.fire('Error', msg, 'error');
      }
    } finally {
      setStartingId(null);
    }
  };

  return (
    <div className="container py-3">
      {notif && (
        <Notification text={notif?.text} type={notif?.type || 'info'} onClose={() => setNotif(null)} />
      )}

      {/* Header identitas peserta + tombol logout */}
      <div className="card border-0 shadow-sm mb-3">
        <div className="card-body d-flex justify-content-between align-items-center gap-3 flex-wrap">
          <div className="d-flex flex-column">
            <div className="fw-bold mb-2">Assalamualaikum , Wr. Wb. <br />Periksa Identitas Anda!</div>
            <table className="table table-borderless table-sm mb-0" style={{ maxWidth: '520px' }}>
              <tbody>
                <tr>
                  <td className="fw-semibold" style={{ width: '120px' }}>Nama</td>
                  <td style={{ width: '10px' }}>:</td>
                  <td>{participant?.nama || '-'}</td>
                </tr>
                <tr>
                  <td className="fw-semibold" style={{ width: '120px' }}>NISN</td>
                  <td style={{ width: '10px' }}>:</td>
                  <td>{participant?.nisn || '-'}</td>
                </tr>
                <tr>
                  <td className="fw-semibold" style={{ width: '120px' }}>Kelas</td>
                  <td style={{ width: '10px' }}>:</td>
                  <td>{participant?.name_class || '-'}</td>
                </tr>
                <tr>
                  <td className="fw-semibold" style={{ width: '120px' }}>Sekolah</td>
                  <td style={{ width: '10px' }}>:</td>
                  <td>{participant?.school_name || '-'}</td>
                </tr>
              </tbody>
            </table>
          </div>
          <div className="d-flex align-items-center gap-2">
            <button className="btn btn-outline-danger btn-sm" onClick={handleLogout}>Logout</button>
          </div>
        </div>
      </div>

      <div className="card shadow-sm">
        <div className="card-body">
          <h5 className="card-title d-flex justify-content-between align-items-center">
            <span>Ujian Hari Ini</span>
            <span className="text-muted small">Jumlah: {sorted.length}</span>
          </h5>

          {loading ? (
            <div className="text-center py-4">Memuat...</div>
          ) : (
            <>
              <div className="table-responsive">
                <table className="table table-sm align-middle">
                  <thead>
                    <tr>
                      <th>Ujian</th>
                      <th>Subjek</th>
                      <th>Waktu</th>
                      <th>Durasi</th>
                      <th>Status</th>
                      <th>Aksi</th>
                    </tr>
                  </thead>
                  <tbody>
                    {sorted.length === 0 ? (
                      <tr>
                        <td colSpan="6" className="text-center text-muted">Tidak ada ujian hari ini</td>
                      </tr>
                    ) : (
                      sorted.map((exam) => (
                        <tr key={exam.id}>
                          <td>{exam.name}</td>
                          <td>{exam.subject_name || '-'}</td>
                          <td>{exam.start_at || exam.scheduled_at}</td>
                          <td>{exam.duration_minutes} menit</td>
                          <td>
                            {(() => {
                              const eff = effectiveStatus(exam);
                              return <span className={`badge ${statusClass(eff)}`}>{statusLabel(eff)}</span>;
                            })()}
                          </td>
                          <td>
                            {(() => {
                              const entryAttempts = typeof exam?.my_session?.entry_attempts === 'number' ? exam.my_session.entry_attempts : getAttempts(exam.id);
                              const now = new Date();
                              const startAt = new Date(exam.start_at_iso || exam.start_at);
                              const endAt = new Date(exam.end_at_iso || exam.end_at);
                              const isExpired = (now > endAt);
                              const eff = effectiveStatus(exam);
                              const finished = ['success','selesai','proses_hasil'].includes(String(eff));
                              const canStartNow = exam.can_start && !isExpired && entryAttempts < 3 && !finished;

                              if (entryAttempts >= 3) {
                                return <span className="badge bg-secondary">Maks percobaan</span>;
                              }
                              if (isExpired) {
                                return <span className="badge bg-danger">Expired</span>;
                              }
                              if (finished) {
                                return <span className="badge bg-success">Selesai</span>;
                              }

                              const diffMin = Math.ceil((startAt - now) / 60000);
                              if (diffMin > 0 && diffMin <= 5) {
                                return (
                                  <button className="btn btn-outline-primary btn-sm" disabled>
                                    Mulai {diffMin} mnt
                                  </button>
                                );
                              }
                              if (canStartNow) {
                                return (
                                  <button className="btn btn-primary btn-sm" disabled={startingId === exam.id} onClick={() => handleStart(exam)}>
                                    {startingId === exam.id ? 'Memulai...' : 'Mulai'}
                                  </button>
                                );
                              }
                              return <span className="text-muted">Belum mulai</span>;
                            })()}
                          </td>
                        </tr>
                      ))
                    )}
                  </tbody>
                </table>
              </div>

            </>
          )}
        </div>
      </div>

      <ExamStartModal onClose={() => {}} />
    </div>
  );
}
