import React, { useEffect, useState, useRef, useMemo } from 'react';
import { InlineMath, BlockMath } from 'react-katex';
import 'katex/dist/katex.min.css';
import renderMathInElement from 'katex/contrib/auto-render';
import { useParams, useNavigate } from 'react-router-dom';
import api from '../utils/axios';
import { useParticipantAuth } from '../context/ParticipantAuthContext.jsx';
import Swal from 'sweetalert2';
import { sanitizeLatex } from '../utils/latex';
import { requestFullscreenEnhanced } from '../utils/fullscreenHelper';

// Konsistensi gaya gambar untuk preview agar seragam dengan edit soal
const PREVIEW_IMG_STYLE = { maxWidth: '100%', maxHeight: '240px', objectFit: 'contain', borderRadius: 6, margin: '6px 0' };
const OPTION_PREVIEW_IMG_STYLE = { maxWidth: '100%', maxHeight: '180px', objectFit: 'contain', borderRadius: 6, margin: '6px 0' };

export default function ParticipantExam() {
  const examRef = useRef(null);

  const { sessionId } = useParams();
  const navigate = useNavigate();
  const { participant } = useParticipantAuth();
  const [loading, setLoading] = useState(true);
  const [exam, setExam] = useState(null);
  const [questions, setQuestions] = useState([]);
  const [current, setCurrent] = useState(0);
  const [answers, setAnswers] = useState({});
  const [showList, setShowList] = useState(false);
  const [flags, setFlags] = useState({});
  const [remainingMs, setRemainingMs] = useState(null);
  const lastFlagToggleAt = useRef(0);
  const [confirmFinishInfo, setConfirmFinishInfo] = useState(null);
  const warned5min = useRef(false);
  const [timeWarningVisible, setTimeWarningVisible] = useState(false);
  const [showFullscreenRequired, setShowFullscreenRequired] = useState(false);
  const switchAttempts = useRef(0);
  const autoSubmittedRef = useRef(false);
  const lastExitIncRef = useRef(0);
  const [attempts, setAttempts] = useState(0);
  const autosaveTimer = useRef(null);
  const lastSentDraft = useRef({});
  const autosaveInFlight = useRef(false);
  const autoSubmittedByStartAttemptsRef = useRef(false);
  const lastChanceWarnedRef = useRef(false);
  const autosaveBackoffRef = useRef(0);

  // Auto-render KaTeX pada area ujian saat konten berubah
  useEffect(() => {
    if (examRef.current) {
      renderMathInElement(examRef.current, {
        delimiters: [
          { left: "\\(", right: "\\)", display: false },
          { left: "\\[", right: "\\]", display: true },
          { left: "$", right: "$", display: false },
          { left: "$$", right: "$$", display: true }
        ],
        throwOnError: false
      });
    }
  }, [questions, current, showList, answers]);

  const currentCounts = useMemo(() => {
    try {
      const flaggedCount = Object.values(flags || {}).filter(Boolean).length;
      const unansweredCount = Array.isArray(questions) ? questions.filter(q => {
        const isEssay = isEssayQuestion(q);
        const val = answers[q.id];
        return isEssay ? !(typeof val === 'string' && val.trim() !== '') : !(typeof val === 'string' && /^[A-E]$/.test(val));
      }).length : 0;
      const hasLongTime = remainingMs != null && remainingMs > 5 * 60 * 1000;
      return { flaggedCount, unansweredCount, hasLongTime, remainingMs };
    } catch (_) {
      return { flaggedCount: 0, unansweredCount: 0, hasLongTime: false, remainingMs };
    }
  }, [questions, answers, flags, remainingMs]);

  const participantKey = useMemo(() => (participant?.id || participant?.nisn || 'anon'), [participant]);
  const getStartAttempts = (examId) => {
    try {
      const key = `examStartAttempts:${participantKey}:${examId}`;
      const raw = localStorage.getItem(key);
      const n = parseInt(raw || '0', 10);
      return Number.isFinite(n) && n >= 0 ? n : 0;
    } catch (_) { return 0; }
  };

  // Backend base untuk asset gambar (fallback ke 127.0.0.1:8000)
  const BACKEND_BASE = import.meta.env.VITE_BACKEND_URL || 'http://127.0.0.1:8000';
  const resolveImageUrl = (urlOrPath) => {
    if (!urlOrPath) return null;
    const s = String(urlOrPath).trim();
    const base = String(BACKEND_BASE || '').replace(/\/+$/, '');
    if (/^(data|blob):/.test(s)) return s;
    // Rebase absolut ke /storage/ atau /images/ agar origin mengikuti BACKEND_BASE
    if (/^https?:\/\/[^/]+\/(storage|images)\//.test(s)) {
      return s.replace(/^https?:\/\/[^/]+/, base);
    }
    if (/^https?:\/\//.test(s)) return s;
    const clean = s.replace(/^\/+/, '');
    if (/^storage\//.test(clean)) {
      return `${base}/storage/${clean.replace(/^storage\//,'')}`;
    }
    if (/^images\//.test(clean)) {
      return `${base}/${clean}`;
    }
    if (/^\/storage\//.test(s)) return `${base}${s}`;
    return `${base}/storage/${clean.replace(/^public\//,'')}`;
  };

  // Konversi style object ke atribut inline CSS string
  const styleObjectToString = (obj) => {
    if (!obj || typeof obj !== 'object') return '';
    return Object.entries(obj).map(([key, val]) => {
      const cssKey = key.replace(/[A-Z]/g, (m) => '-' + m.toLowerCase());
      const cssVal = typeof val === 'number' ? `${val}px` : String(val);
      return `${cssKey}: ${cssVal};`;
    }).join(' ');
  };

  // Util: anggap '-' sebagai kosong (untuk deteksi essay)
  const isDashOrBlank = (s) => {
    const t = String(s || '').trim();
    return t === '' || t === '-';
  };

  // Deteksi soal essay: opsi B–D kosong atau '-' (atau semua A–E kosong)
  const isEssayQuestion = (q) => {
    if (!q || typeof q !== 'object') return false;
    const blank = (v) => isDashOrBlank(v);
    const bdBlank = blank(q?.option_b) && blank(q?.option_c) && blank(q?.option_d);
    const allBlank = ['option_a','option_b','option_c','option_d','option_e'].every(k => blank(q[k]));
    return bdBlank || allBlank;
  };

  // Catatan: hindari escape HTML di sini agar LaTeX tetap utuh
  const escapeHtml = (str) => String(str || '');

  // Deteksi karakter Arab untuk dukungan RTL otomatis
  const containsArabic = (s) => {
    if (!s || typeof s !== 'string') return false;
    // Rentang Unicode Arab utama dan tambahan
    return /[\u0600-\u06FF\u0750-\u077F\u08A0-\u08FF\uFB50-\uFDFF\uFE70-\uFEFF]/.test(s);
  };

  // Deteksi sederhana apakah string tampak sebagai LaTeX
  const looksLatex = (s) => {
    if (!s || typeof s !== 'string') return false;
    return /\\(dfrac|frac|sqrt|cos|sin|tan|left|right|int|sum|prod|partial|nabla|pi|theta|alpha|beta|gamma|lim|bmatrix|pmatrix|begin|end|cdot|times|div|le|ge|neq|approx|equiv|infty|angle)/.test(s);
  };

  // Helper: konten murni LaTeX (hanya satu token latex + spasi)
  const isPureLatex = (s) => {
    const str = String(s || '').trim();
    if (!str) return false;
    return /^(\$\$[\s\S]+?\$\$|\$[^$]*\$|\\\[[\s\S]+?\\\]|\\\([\s\S]+?\\\)|\[tex\][\s\S]*?\[\/tex\]|\[math\][\s\S]*?\[\/math\]|<tex>[\s\S]*?<\/tex>|<katex>[\s\S]*?<\/katex>)$/m.test(str);
  };

  // Parser BBCode → HTML untuk [b], [i], [u], [s], [br], [img], align, table, video
  const bbcodeToHtml = (input, opts = {}) => {
    if (!input || typeof input !== 'string') return '';
    let s = input;
    // Teks format: jangan escape agar & pada LaTeX tidak jadi &amp;
    s = s.replace(/\[b\](.*?)\[\/b\]/gis, (m, p1) => `<strong>${p1}</strong>`);
    s = s.replace(/\[i\](.*?)\[\/i\]/gis, (m, p1) => `<em>${p1}</em>`);
    s = s.replace(/\[u\](.*?)\[\/u\]/gis, (m, p1) => `<u>${p1}</u>`);
    s = s.replace(/\[s\](.*?)\[\/s\]/gis, (m, p1) => `<s>${p1}</s>`);
    s = s.replace(/\[br\s*\/?\]/gi, '<br/>');

    // Alignment: [center]...[/center], [left]...[/left], [right]...[/right], [align=center]...[/align]
    s = s.replace(/\[center\](.*?)\[\/center\]/gis, (m, p1) => `<div style="text-align:center;">${p1}</div>`);
    s = s.replace(/\[left\](.*?)\[\/left\]/gis, (m, p1) => `<div style="text-align:left;">${p1}</div>`);
    s = s.replace(/\[right\](.*?)\[\/right\]/gis, (m, p1) => `<div style="text-align:right;">${p1}</div>`);
    s = s.replace(/\[align=(left|center|right)\](.*?)\[\/align\]/gis, (m, align, body) => `<div style="text-align:${align};">${body}</div>`);

    // Dukungan RTL eksplisit: [rtl]...[/rtl]
    s = s.replace(/\[rtl\](.*?)\[\/rtl\]/gis, (m, body) => `<div dir="rtl" style="text-align:right;">${body}</div>`);

    // Gambar: [img]path_or_url[/img]
    s = s.replace(/\[img\](.*?)\[\/img\]/gis, (m, p1) => {
      const raw = String(p1 || '').trim();
      const url = resolveImageUrl(raw);
      const style = styleObjectToString(opts.imgStyle || PREVIEW_IMG_STYLE);
      const safeUrl = String(url || '').replace(/\"/g, '&quot;');
      const base = String(BACKEND_BASE || '').replace(/\/+$/, '');
      const clean = raw.replace(/^\/+/, '');
      let alt = '';
      if (/^images\//.test(clean)) alt = `${base}/storage/${clean}`;
      else if (/^storage\//.test(clean)) alt = `${base}/${clean.replace(/^storage\//,'')}`;
      const onErrAttr = alt ? ` onerror="if(this._trFallback){return;}this._trFallback=1;this.src='${alt}'"` : '';
      return `<img src="${safeUrl}" alt="img" style="${style}"${onErrAttr} />`;
    });

    // YouTube: [youtube]ID atau URL[/youtube]
    const extractYouTubeId = (raw) => {
      const s2 = String(raw || '').trim();
      if (/^[A-Za-z0-9_-]{11}$/.test(s2)) return s2;
      // Dukung partial path: /watch?v=ID atau watch?v=ID
      const mWatch = s2.match(/^\/?watch\?v=([A-Za-z0-9_-]{11})/i);
      if (mWatch) return mWatch[1];
      // Dukung shorts/ dan embed/
      const mShorts = s2.match(/shorts\/([A-Za-z0-9_-]{11})/i);
      if (mShorts) return mShorts[1];
      const mEmbed = s2.match(/embed\/([A-Za-z0-9_-]{11})/i);
      if (mEmbed) return mEmbed[1];
      try {
        const u = new URL(s2);
        if (u.hostname.includes('youtu.be')) {
          const id = (u.pathname || '').replace(/^\//, '').split('/')[0];
          return id || null;
        }
        if (u.hostname.includes('youtube.com')) {
          const id = u.searchParams.get('v');
          if (id) return id;
          // Path lain: /shorts/ID
          const p = (u.pathname || '').toLowerCase();
          const m2 = p.match(/shorts\/([A-Za-z0-9_-]{11})/);
          if (m2) return m2[1];
        }
      } catch (_) { /* ignore */ }
      return null;
    };
    if (!opts.disableYoutube) {
      s = s.replace(/\[youtube\](.*?)\[\/youtube\]/gis, (m, p1) => {
        const id = extractYouTubeId(p1);
        if (!id) return p1;
        const src = `https://www.youtube.com/embed/${id}`;
        return `<div class="ratio ratio-16x9" style="max-width: 720px; margin: 6px auto;"><iframe src="${src}" title="YouTube" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" allowfullscreen style="width:100%; height:100%;"></iframe></div>`;
      });
    } else {
      s = s.replace(/\[youtube\](.*?)\[\/youtube\]/gis, '');
    }

    // Video umum: [video]URL[/video] — YouTube/Vimeo gunakan iframe, lainnya gunakan <video>
    const buildVideoEmbed = (raw) => {
      const style = 'max-width: 720px; width: 100%; margin: 6px auto;';
      try {
        const u = new URL(String(raw || '').trim());
        const host = u.hostname.toLowerCase();
        if (host.includes('youtu.be') || host.includes('youtube.com')) {
          const id = extractYouTubeId(raw);
          if (!id) return '';
          const src = `https://www.youtube.com/embed/${id}`;
          return `<div class="ratio ratio-16x9" style="${style}"><iframe src="${src}" title="YouTube" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" allowfullscreen style="width:100%; height:100%;"></iframe></div>`;
        }
        if (host.includes('vimeo.com')) {
          const pathParts = u.pathname.split('/').filter(Boolean);
          const id = pathParts[pathParts.length - 1];
          const src = id ? `https://player.vimeo.com/video/${id}` : u.toString();
          return `<div class="ratio ratio-16x9" style="${style}"><iframe src="${src}" title="Vimeo" frameborder="0" allow="autoplay; fullscreen; picture-in-picture" allowfullscreen style="width:100%; height:100%;"></iframe></div>`;
        }
        const safeUrl = u.toString().replace(/\"/g, '&quot;');
        const vStyle = 'max-width: 720px; width: 100%; max-height: 405px; border-radius: 6px;';
        return `<video src="${safeUrl}" controls style="${vStyle}"></video>`;
      } catch (_) {
        return '';
      }
    };
    s = s.replace(/\[video\](.*?)\[\/video\]/gis, (m, p1) => buildVideoEmbed(p1));

  // Tabel: [table] ... [tr][td]...[/td][/tr] ... [/table]
  s = s.replace(/\[table\](.*?)\[\/table\]/gis, (m, body) => {
    let inner = body || '';
    inner = inner
      .replace(/\[th\](.*?)\[\/th\]/gis, (mm, cell) => `<th style="border:1px solid #dee2e6; padding:6px;">${cell}</th>`)
      .replace(/\[td\](.*?)\[\/td\]/gis, (mm, cell) => `<td style="border:1px solid #dee2e6; padding:6px;">${cell}</td>`)
      .replace(/\[tr\](.*?)\[\/tr\]/gis, (mm, row) => `<tr>${row}</tr>`);
    const tblStyle = 'border-collapse: collapse; width: 100%; margin: 6px 0;';
    return `<table style="${tblStyle}"><tbody>${inner}</tbody></table>`;
  });
  // Konversi newline agar konten mengikuti enter/linebreak
  s = s.replace(/\r?\n/g, '<br/>');
  return s;
  };

  // Deteksi dan split token LaTeX (inline/block) dalam teks
  const splitLatexTokens = (s) => {
    if (!s || typeof s !== 'string') return [];
    const re = /(\$\$[\s\S]+?\$\$)|(\$[^$]*\$)|(\\\[[\s\S]+?\\\])|(\\\([\s\S]+?\\\))|(\[tex\][\s\S]*?\[\/tex\])|(\[math\][\s\S]*?\[\/math\])|(<tex>[\s\S]*?<\/tex>)|(<katex>[\s\S]*?<\/katex>)/g;
    const parts = [];
    let lastIndex = 0;
    let m;
    while ((m = re.exec(s))) {
      const before = s.slice(lastIndex, m.index);
      if (before) parts.push({ type: 'text', value: before });
      const token = m[0];
      const isBlock = token.startsWith('$$') || token.startsWith('\\[');
      parts.push({ type: isBlock ? 'latex_block' : 'latex_inline', value: token });
      lastIndex = m.index + token.length;
    }
    const rest = s.slice(lastIndex);
    if (rest) parts.push({ type: 'text', value: rest });
    return parts;
  };

  const stripLatexDelimiters = (token) => {
    if (!token) return '';
    if (token.startsWith('$$')) return token.replace(/^\$\$/, '').replace(/\$\$$/, '');
    if (token.startsWith('$')) return token.replace(/^\$/, '').replace(/\$$/, '');
    if (token.startsWith('\\[')) return token.replace(/^\\\[/, '').replace(/\\\]$/, '');
    if (token.startsWith('\\(')) return token.replace(/^\\\(/, '').replace(/\\\)$/, '');
    if (token.startsWith('[tex]')) return token.replace(/^\[tex\]/, '').replace(/\[\/tex\]$/, '');
    if (token.startsWith('[math]')) return token.replace(/^\[math\]/, '').replace(/\[\/math\]$/, '');
    if (token.startsWith('<tex>')) return token.replace(/^<tex>/, '').replace(/<\/tex>$/, '');
    if (token.startsWith('<katex>')) return token.replace(/^<katex>/, '').replace(/<\/katex>$/, '');
    return token;
  };

  // Tambahan: bungkus otomatis LaTeX jika tidak ada delimiter
  const autoWrapLatexIfMissing = (s) => {
    const str = String(s || '');
    const hasDelim = /(\$\$[\s\S]*?\$\$|\\\[[\s\S]*?\\\]|\\\([\s\S]*?\\\)|\$[^$]*\$|\[tex\][\s\S]*?\[\/tex\]|\[math\][\s\S]*?\[\/math\]|<tex>[\s\S]*?<\/tex>|<katex>[\s\S]*?<\/katex>)/.test(str);
    const looksLatex = /\\(dfrac|frac|sqrt|cos|sin|tan|left|right|int|sum|prod|partial|nabla|pi|theta|alpha|beta|gamma|lim|bmatrix|pmatrix|begin|end|cdot|times|div|le|ge|neq|approx|equiv|infty|angle)/.test(str);
    return (!hasDelim && looksLatex) ? `\\(${str}\\)` : str;
  };

  const renderTextOrLatex = (s, keyPrefix) => {
    const nodes = [];
    const input = autoWrapLatexIfMissing(s);
    const segments = splitLatexTokens(input);
    let idx = 0;

    // decode entity HTML khusus untuk token LaTeX
    const decodeHtmlEntities = (input) => {
      return String(input || '')
        .replace(/&amp;/g, '&')
        .replace(/&lt;/g, '<')
        .replace(/&gt;/g, '>')
        .replace(/&quot;/g, '"')
        .replace(/&#039;/g, "'")
        .replace(/&#39;/g, "'");
    };
    // bersihkan zero-width/combining + perbaiki aksen teks di math mode
    const cleanLatex = (input) => sanitizeLatex(input);

    for (const seg of segments) {
      if (seg.type === 'text') {
        const rtl = containsArabic(seg.value);
        nodes.push(
          <span
            key={`${keyPrefix}-t-${idx++}`}
            dir={rtl ? 'rtl' : 'auto'}
            style={rtl ? { textAlign: 'right', fontFamily: 'Noto Naskh Arabic, Amiri, serif' } : undefined}
            dangerouslySetInnerHTML={{ __html: seg.value }}
          />
        );
      } else if (seg.type === 'latex_inline') {
        const math = cleanLatex(decodeHtmlEntities(stripLatexDelimiters(seg.value)));
        nodes.push(
          <span key={`${keyPrefix}-m-${idx++}`} style={{ direction: 'ltr', unicodeBidi: 'isolate' }}>
            <InlineMath errorColor="#cc0000" throwOnError={false}>{math}</InlineMath>
          </span>
        );
      } else if (seg.type === 'latex_block') {
        const math = cleanLatex(decodeHtmlEntities(stripLatexDelimiters(seg.value)));
        nodes.push(
          <div key={`${keyPrefix}-b-${idx++}`} style={{ direction: 'ltr', unicodeBidi: 'isolate' }}>
            <BlockMath errorColor="#cc0000" throwOnError={false}>{math}</BlockMath>
          </div>
        );
      }
    }
    return <>{nodes}</>;
  };

  // Render campuran teks + gambar inline: konversi Markdown ![alt](url) → <img src="...">
  const renderMixed = (str, opts = {}) => {
    if (!str) return <span className="text-muted">-</span>;
    const imgStyle = opts.imgStyle || PREVIEW_IMG_STYLE;
    const processedBb = bbcodeToHtml(str, { ...opts, imgStyle });
    // Ganti token markdown gambar menjadi tag <img>; dukung lebar via alt: "img|w=80%"
    let html = processedBb.replace(/!\[([^\]]*)\]\(([^)]+)\)/g, (m, alt, url) => {
      const full = resolveImageUrl(String(url || '').trim());
      const altRaw = String(alt || '').trim();
      const altBase = altRaw.replace(/\|.*$/, '').trim() || 'img';
      // Ambil semua w=xx% dan gunakan yang terakhir untuk mengatasi duplikasi
      const widthMatches = altRaw.match(/\bw\s*=\s*(\d{1,3})\s*%/gi);
      let widthPct = null;
      if (widthMatches && widthMatches.length > 0) {
        const lastMatch = widthMatches[widthMatches.length - 1];
        const mW = lastMatch.match(/\bw\s*=\s*(\d{1,3})\s*%/i);
        widthPct = mW ? Math.max(5, Math.min(160, Number(mW[1]))) : null;
      }
      const baseStyleStr = styleObjectToString(imgStyle);
      const styleStr = widthPct ? `${baseStyleStr} width: ${widthPct}%;` : baseStyleStr;
      return `<img src="${full}" alt="${altBase}" style="${styleStr}">`;
    });
    const isRtl = containsArabic(html);
    const wrapperStyle = isRtl ? { textAlign: 'right', fontFamily: 'Noto Naskh Arabic, Amiri, serif' } : undefined;
    return <span dir={isRtl ? 'rtl' : 'auto'} style={wrapperStyle} dangerouslySetInnerHTML={{ __html: html }} />;
  };

  // Render hanya LaTeX (tanpa teks/gambar), paksa LTR agar konsisten
  const renderMathOnly = (s, key = 'math-only') => {
    if (!s || typeof s !== 'string') {
      return <span key={key} />;
    }
    const decodeHtmlEntitiesLocal = (input) => {
      return String(input || '')
        .replace(/&amp;/g, '&')
        .replace(/&lt;/g, '<')
        .replace(/&gt;/g, '>')
        .replace(/&quot;/g, '"')
        .replace(/&#039;/g, "'")
        .replace(/&#39;/g, "'");
    };
    const cleanLatexLocal = (input) => sanitizeLatex(input);
    const wrapped = autoWrapLatexIfMissing(s);
    const isBlock = /(\$\$[\s\S]*\$\$|\\\[[\s\S]*\\\])/m.test(wrapped);
    const inner = stripLatexDelimiters(wrapped);
    const math = cleanLatexLocal(decodeHtmlEntitiesLocal(inner));
    const wrapperStyle = { direction: 'ltr', unicodeBidi: 'isolate' };
    return isBlock
      ? <div key={`${key}-block`} style={wrapperStyle}><BlockMath errorColor="#cc0000" throwOnError={false}>{math}</BlockMath></div>
      : <span key={`${key}-inline`} style={wrapperStyle}><InlineMath errorColor="#cc0000" throwOnError={false}>{math}</InlineMath></span>;
  };

  // Opsi kebijakan lebih ketat
  const STRICT_OPTIONS = {
    // Berikan kesempatan hingga 3x sebelum auto submit
    autoFinishOnFirstSwitch: false,
    logViolationOnSwitch: true,
  };

  const postViolation = async (type, detail = {}) => {
    try {
      await api.post(`/exam-sessions/${sessionId}/violations`, { type, detail });
    } catch (_) {}
  };

  const incExitAttempt = (reason) => {
    const now = Date.now();
    if ((lastExitIncRef.current || 0) && (now - lastExitIncRef.current) < 3000) {
      return false; // masih dalam cooldown 3 detik
    }
    lastExitIncRef.current = now;

    const usedStart = exam ? getStartAttempts(exam.id) : 0;
    switchAttempts.current += 1;
    setAttempts(switchAttempts.current);

    const shouldAuto = (switchAttempts.current >= 3) && !autoSubmittedRef.current;
    if (shouldAuto) {
      autoSubmittedRef.current = true;
      submitAnswers('attempts_exhausted');
      return true;
    } else {
      setShowFullscreenRequired(true);
      return false;
    }
  };

  const autoFinishSession = async (reason) => {
    try {
      await api.put(`/exam-sessions/${sessionId}/finish`, { reason });
    } catch (_) {}
    navigate('/exam');
  };

  const handleFinish = () => {
    const flaggedCount = Object.values(flags).filter(Boolean).length;
    const unansweredCount = questions.filter(q => {
      const isEssay = isEssayQuestion(q);
      const val = answers[q.id];
      return isEssay ? !(typeof val === 'string' && val.trim() !== '') : !(typeof val === 'string' && /^[A-E]$/.test(val));
    }).length;
    const hasLongTime = remainingMs != null && remainingMs > 5 * 60 * 1000; // >= 5 menit dianggap masih lama
    setConfirmFinishInfo({ flaggedCount, unansweredCount, hasLongTime, remainingMs });
  };

  // Toggle penanda ragu-ragu pada soal saat ini
  const toggleFlag = () => {
    const q = questions[current];
    if (!q) return;
    const now = Date.now();
    // Cegah double-click dalam interval singkat agar warna tetap kuning
    if (now - (lastFlagToggleAt.current || 0) < 500) return;
    lastFlagToggleAt.current = now;
    // Toggle normal setelah jeda: klik pertama → kuning; klik berikutnya → non-kuning
    setFlags(prev => ({ ...prev, [q.id]: !prev[q.id] }));
  };

  // Konfirmasi yakin: hilangkan status ragu-ragu pada soal saat ini
  const confirmCurrent = () => {
    const q = questions[current];
    if (!q) return;
    setFlags(prev => ({ ...prev, [q.id]: false }));
  };

  const closeFinishConfirm = () => setConfirmFinishInfo(null);
  const submitAnswers = async (reason) => {
    // Kirim jawaban ke backend untuk disimpan dan dinilai
    try {
      setConfirmFinishInfo(null);
      const payload = reason ? { answers, reason } : { answers };
      await api.post(`/exam-sessions/${sessionId}/submit-answers`, payload);
      try {
        await api.put(`/exam-sessions/${sessionId}/finish`, reason ? { reason } : {});
      } catch (finishErr) {
        console.error('Gagal menandai sesi selesai:', finishErr);
      }
    } catch (e) {
      // tetap coba menandai sesi selesai walau submit gagal
      console.error('Gagal submit jawaban:', e);
      try {
        await api.put(`/exam-sessions/${sessionId}/finish`, reason ? { reason } : {});
      } catch (_) {}
    } finally {
      navigate('/exam');
    }
  };

  const fetchSession = async () => {
    setLoading(true);
    try {
      const res = await api.get(`/exam-sessions/${sessionId}`);
      setExam(res.data?.exam || null);
      const list = Array.isArray(res.data?.questions) ? res.data.questions : [];
      // Urutkan: pilihan ganda dulu, essay di akhir
      const mcqs = list.filter(q => !isEssayQuestion(q));
      const essays = list.filter(q => isEssayQuestion(q));
      setQuestions([...mcqs, ...essays]);
      // Prefill dari server draft_answers (pilihan ganda & essay)
      const serverDraft = (res.data && typeof res.data.draft_answers === 'object' && res.data.draft_answers) ? res.data.draft_answers : {};
      setAnswers(prev => ({ ...serverDraft, ...prev }));
      lastSentDraft.current = serverDraft;
    } catch (err) {
      console.error(err);
      navigate('/exam');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => { fetchSession(); }, [sessionId]);

  // Inisialisasi dan update sisa waktu berdasarkan scheduled_at + duration_minutes
  useEffect(() => {
    // Blokir interaksi: klik kanan, copy/cut/paste, seleksi massal, cetak, screenshot
    const preventDefault = (e) => { e.preventDefault(); e.stopPropagation(); };
    const onContextMenu = (e) => preventDefault(e);
    const onCopy = (e) => { preventDefault(e); postViolation('blocked_action', { action: 'copy' }); };
    const onCut = (e) => { preventDefault(e); postViolation('blocked_action', { action: 'cut' }); };
    const onPaste = (e) => { preventDefault(e); postViolation('blocked_action', { action: 'paste' }); };
    const onSelectStart = (e) => {
      const tag = (e.target?.tagName || '').toLowerCase();
      const editable = e.target?.isContentEditable;
      // Izinkan seleksi hanya di input/textarea
      if (!['input','textarea'].includes(tag) && !editable) preventDefault(e);
    };
    const onKeyDownStrict = (e) => {
      // Blokir kombinasi umum: Ctrl/Cmd + A/C/X/P/S dan PrintScreen
      const keyRaw = e.key || '';
      const key = keyRaw.toLowerCase();
      const combo = e.ctrlKey || e.metaKey;

      // Deteksi shortcut screenshot lintas platform
      const isPrintScreen = key === 'printscreen';
      const isWindowsSnip = e.metaKey && e.shiftKey && key === 's'; // Win + Shift + S
      const isMacScreenshot = e.metaKey && e.shiftKey && ['3','4','5','6'].includes(key); // Cmd + Shift + 3/4/5/6
      const isWindowsPrintFull = e.metaKey && isPrintScreen; // Win + PrtScn
      const isWindowsPrintActive = e.altKey && isPrintScreen; // Alt + PrtScn (window aktif)
      if (isPrintScreen || isWindowsSnip || isMacScreenshot || isWindowsPrintFull || isWindowsPrintActive) {
        preventDefault(e);
        try { if (navigator.clipboard) navigator.clipboard.writeText(''); } catch {}
        try { if (navigator.vibrate) navigator.vibrate([150, 75, 150]); } catch {}
        postViolation('blocked_action', {
          action: 'screenshot_attempt',
          combo: { meta: !!e.metaKey, ctrl: !!e.ctrlKey, shift: !!e.shiftKey, alt: !!e.altKey, key: keyRaw }
        });
        try {
          Swal.fire({
            title: 'Tindakan dilarang',
            text: 'Screenshot atau tangkapan layar tidak diizinkan saat ujian. Tetap fokus pada halaman ujian.',
            icon: 'warning',
            confirmButtonText: 'Saya mengerti',
            buttonsStyling: false,
            customClass: { confirmButton: 'btn btn-warning' },
          }).catch(() => {});
        } catch (_) {}
        // Perlakukan sebagai percobaan keluar dengan cooldown agar satu insiden tidak dihitung berulang
        try {
          incExitAttempt('screenshot_exit');
        } catch (_) {}
        return;
      }

      if ((combo && ['a','c','x','p','s'].includes(key))) {
        preventDefault(e);
        const action = `key_${key}`;
        postViolation('blocked_action', { action });
      }
    };

    // Peringatan saat dialog print dibuka via menu browser
    const onBeforePrint = () => {
      try {
        postViolation('blocked_action', { action: 'print_dialog' });
        incExitAttempt('print_dialog');
      } catch (_) {}
    };

    const onAfterPrint = () => {
      try {
        postViolation('blocked_action', { action: 'after_print' });
        setShowFullscreenRequired(true);
      } catch (_) {}
    };

    try { navigator.keyboard?.lock?.(['Escape','Tab','PrintScreen']); } catch {}

    // Minta fullscreen segera saat halaman sesi dimuat
    const initFullscreen = async () => {
      try {
        await requestFullscreenEnhanced(document.documentElement, {
          showWarning: false,
          fallbackToViewport: true,
          mobileOptimization: true
        });
      } catch (error) {
        console.warn('Initial fullscreen request failed:', error);
      }
    };
    
    // Delay sedikit untuk memastikan DOM sudah siap
    setTimeout(initFullscreen, 100);

    document.addEventListener('contextmenu', onContextMenu);
    document.addEventListener('copy', onCopy);
    document.addEventListener('cut', onCut);
    document.addEventListener('paste', onPaste);
    document.addEventListener('selectstart', onSelectStart, true);
    window.addEventListener('keydown', onKeyDownStrict, true);
    window.addEventListener('beforeprint', onBeforePrint);
    window.addEventListener('afterprint', onAfterPrint);

    // Tambahkan kelas untuk non-select ke root container saat ujian
    const root = document.documentElement;
    const prevUserSelect = root.style.userSelect;
    root.style.userSelect = 'none';

    return () => {
      try { navigator.keyboard?.unlock?.(); } catch {}
      document.removeEventListener('contextmenu', onContextMenu);
      document.removeEventListener('copy', onCopy);
      document.removeEventListener('cut', onCut);
      document.removeEventListener('paste', onPaste);
      document.removeEventListener('selectstart', onSelectStart, true);
      window.removeEventListener('keydown', onKeyDownStrict, true);
      window.removeEventListener('beforeprint', onBeforePrint);
      window.removeEventListener('afterprint', onAfterPrint);
      root.style.userSelect = prevUserSelect || '';
    };
  }, [sessionId]);

  // Autosave ke queue: debounce 1.5s, kirim hanya diff
  useEffect(() => {
    if (!questions || questions.length === 0) return;
    if (autosaveTimer.current) clearTimeout(autosaveTimer.current);
    const delay = Math.max(1500, autosaveBackoffRef.current || 0);
    autosaveTimer.current = setTimeout(async () => {
      try {
        // Hitung perbedaan antara jawaban saat ini dan yang sudah dikirim
        const diff = {};
        for (const [qidStr, opt] of Object.entries(answers || {})) {
          const qid = parseInt(qidStr, 10);
          if (!qid) continue;
          const optStr = typeof opt === 'string' ? opt : '';
          const isChoice = /^[A-E]$/.test(optStr);
          const isEssayFilled = !isChoice && optStr.trim() !== '';
          if (isChoice || isEssayFilled) {
            if (lastSentDraft.current[qid] !== optStr) diff[qid] = optStr;
          }
        }
        // Hapus jawaban yang sebelumnya ada tapi kini tidak dipilih (null)
        for (const q of questions) {
          const qid = q.id;
          const cur = answers[qid];
          const curStr = typeof cur === 'string' ? cur : '';
          const isChoice = /^[A-E]$/.test(curStr);
          const isEssayFilled = !isChoice && curStr.trim() !== '';
          if (lastSentDraft.current[qid] && !(isChoice || isEssayFilled)) diff[qid] = null;
        }
        const changeCount = Object.keys(diff).length;
        if (changeCount === 0) return;
        autosaveInFlight.current = true;
        await api.post(`/exam-sessions/${sessionId}/autosave-answers`, { answers: diff });
        // Reset backoff saat berhasil
        autosaveBackoffRef.current = 0;
        // Update lastSentDraft
        lastSentDraft.current = { ...lastSentDraft.current, ...diff };
      } catch (err) {
        // Penanganan khusus 429: tingkatkan backoff agar tidak menabrak rate limit
        if (err?.response?.status === 429) {
          const next = Math.min(Math.max(5000, (autosaveBackoffRef.current || 2000) * 2), 30000);
          autosaveBackoffRef.current = next;
          console.warn(`Autosave ditunda karena rate limit. Coba lagi dalam ~${Math.round(next/1000)}s`);
        } else {
          console.warn('Autosave gagal:', err?.response?.data || err.message);
        }
      } finally {
        autosaveInFlight.current = false;
      }
    }, delay);
    return () => { if (autosaveTimer.current) clearTimeout(autosaveTimer.current); };
  }, [answers, questions, sessionId]);
  // Hitung sisa waktu berdasarkan end_at (fallback ke start/scheduled + durasi)
  useEffect(() => {
    if (!exam) return;
    const endStr = exam.end_at_iso || exam.end_at;
    let endTs = endStr ? Date.parse(endStr) : NaN;
    if (!endTs || isNaN(endTs)) {
      const startStr = exam.start_at_iso || exam.start_at || exam.scheduled_at;
      const startTs = startStr ? Date.parse(startStr) : NaN;
      if (!startTs || isNaN(startTs) || !exam.duration_minutes) return;
      endTs = startTs + (Number(exam.duration_minutes) || 0) * 60 * 1000;
    }
    const tick = () => setRemainingMs(Math.max(0, endTs - Date.now()));
    tick();
    const interval = setInterval(tick, 1000);
    return () => clearInterval(interval);
  }, [sessionId, exam?.end_at_iso, exam?.end_at, exam?.start_at_iso, exam?.start_at, exam?.scheduled_at, exam?.duration_minutes]);

  // (Suara dihapus sesuai permintaan)

  // Peringatan 5 menit terakhir: bunyikan alarm dan tampilkan banner sekali
  useEffect(() => {
    const criticalThreshold = 5 * 60 * 1000;
    if (
      remainingMs != null &&
      remainingMs > 0 &&
      remainingMs <= criticalThreshold &&
      !warned5min.current
    ) {
      warned5min.current = true;
      // Tampilkan banner visual
      setTimeWarningVisible(true);
      // Getaran (hanya perangkat yang mendukung)
      try { if (navigator.vibrate) navigator.vibrate([200, 100, 200]); } catch {}
    }
  }, [remainingMs]);

  // Ketika waktu habis, tampilkan info non-blok dan kirim otomatis
  useEffect(() => {
    if (remainingMs === 0 && !autoSubmittedRef.current) {
      autoSubmittedRef.current = true;
      try {
        Swal.fire({
          title: 'Waktu anda habis',
          text: 'Jawaban sedang dikirim dan sesi ditutup.',
          icon: 'info',
          showConfirmButton: false,
          timer: 2000,
          timerProgressBar: true,
        });
      } catch (_) {}
      submitAnswers('time_up');
    }
  }, [remainingMs]);

  // Peringatan kesempatan terakhir saat masuk sesi pada attempt ke-3 (sekali saja)
  useEffect(() => {
    if (!exam) return;
    const used = getStartAttempts(exam.id);
    if (used >= 3 && !lastChanceWarnedRef.current) {
      (async () => {
        lastChanceWarnedRef.current = true;
        try {
          await Swal.fire({
            title: 'Kesempatan terakhir',
            text: 'Jika Anda keluar lagi saat mengerjakan, nilai ujian Anda akan apa adanya.',
            icon: 'warning',
            confirmButtonText: 'OK',
            buttonsStyling: false,
            customClass: { confirmButton: 'btn btn-warning' },
          });
        } catch (_) { /* ignore */ }
      })();
    }
  }, [exam]);

  // Pembatasan saat ujian: blokir tombol Esc/Tab, cegah back navigation,
  // dan tampilkan konfirmasi ketika mencoba keluar dari halaman.
  useEffect(() => {
    const requestFullscreen = async () => {
      try {
        await requestFullscreenEnhanced(document.documentElement, {
          showWarning: false,
          fallbackToViewport: true,
          mobileOptimization: true
        });
      } catch (error) {
        console.warn('Fullscreen request failed:', error);
      }
    };

    const onKeyDown = (e) => {
      // Blokir Esc agar tidak keluar fullscreen dan Tab agar tidak pindah fokus
      if (e.key === 'Escape') {
        e.preventDefault();
        e.stopPropagation();
        // Wajib fullscreen: tampilkan modal untuk kembali ke layar penuh
        setShowFullscreenRequired(true);
        return;
      }
      if (e.key === 'Tab') {
        e.preventDefault();
        e.stopPropagation();
      }
      // Cegah backspace memicu back navigation ketika bukan di input/textarea
      if (e.key === 'Backspace') {
        const tag = (document.activeElement?.tagName || '').toLowerCase();
        const editable = document.activeElement?.isContentEditable;
        if (!['input', 'textarea'].includes(tag) && !editable) {
          e.preventDefault();
          e.stopPropagation();
        }
      }
    };

    const onKeyUp = (e) => {
      if (e.key === 'Escape') {
        e.preventDefault();
        e.stopPropagation();
      }
    };

    // Dorong state awal untuk mencegah tombol back langsung keluar
    const pushBlockState = () => {
      try { window.history.pushState(null, '', window.location.href); } catch (_) {}
    };
    pushBlockState();

    const onPopState = (e) => {
      // Wajib fullscreen dan cegah keluar tanpa konfirmasi
      setShowFullscreenRequired(true);
      // Dorong kembali state agar tetap di halaman ujian bila dibatalkan
      pushBlockState();
      if (STRICT_OPTIONS.logViolationOnSwitch) {
        postViolation('force_exit', { event: 'popstate', attempts: switchAttempts.current });
      }
      incExitAttempt('popstate');
    };

    const onBeforeUnload = (e) => {
      // Tampilkan konfirmasi default browser saat mencoba reload/close tab
      e.preventDefault();
      e.returnValue = 'Apakah Anda yakin akan keluar ujian? Jika Anda keluar, ujian otomatis selesai.';
    };

    window.addEventListener('keydown', onKeyDown, true);
    window.addEventListener('keyup', onKeyUp, true);
    window.addEventListener('popstate', onPopState);
    window.addEventListener('beforeunload', onBeforeUnload);

    // Jika keluar fullscreen (mis. karena ESC), tampilkan modal wajib fullscreen
    const onFullscreenChange = () => {
      const isFs = !!(document.fullscreenElement || document.webkitFullscreenElement || document.msFullscreenElement);
      if (!isFs) {
        if (STRICT_OPTIONS.logViolationOnSwitch) {
          postViolation('exit_fullscreen', { event: 'fullscreenchange', attempts: switchAttempts.current });
        }
        incExitAttempt('fullscreen_exit');
      } else {
        setShowFullscreenRequired(false);
      }
    };
    document.addEventListener('fullscreenchange', onFullscreenChange);
    document.addEventListener('webkitfullscreenchange', onFullscreenChange);
    document.addEventListener('msfullscreenchange', onFullscreenChange);

    // Jika tab disembunyikan (switch app), ingatkan untuk kembali fokus
    const onVisibilityChange = () => {
      if (document.visibilityState === 'hidden') {
        if (STRICT_OPTIONS.logViolationOnSwitch) {
          postViolation('switch_app', { event: 'visibilitychange', attempts: switchAttempts.current });
        }
        incExitAttempt('visibility_hidden');
      }
    };
    const onBlur = () => {
      // Hanya tindak saat benar-benar kehilangan fokus jendela (bukan blur elemen)
      const lostWindowFocus = document.visibilityState === 'hidden' || !document.hasFocus();
      if (!lostWindowFocus) return;
      if (STRICT_OPTIONS.logViolationOnSwitch) {
        postViolation('switch_app', { event: 'blur', attempts: switchAttempts.current });
      }
      incExitAttempt('blur');
    };
    const onPageHide = () => {
      if (STRICT_OPTIONS.logViolationOnSwitch) {
        postViolation('switch_app', { event: 'pagehide', attempts: switchAttempts.current });
      }
      incExitAttempt('pagehide');
    };
    document.addEventListener('visibilitychange', onVisibilityChange);
    window.addEventListener('blur', onBlur, false);
    window.addEventListener('pagehide', onPageHide);

    return () => {
      window.removeEventListener('keydown', onKeyDown, true);
      window.removeEventListener('keyup', onKeyUp, true);
      window.removeEventListener('popstate', onPopState);
      window.removeEventListener('beforeunload', onBeforeUnload);
      document.removeEventListener('fullscreenchange', onFullscreenChange);
      document.removeEventListener('webkitfullscreenchange', onFullscreenChange);
      document.removeEventListener('msfullscreenchange', onFullscreenChange);
      document.removeEventListener('visibilitychange', onVisibilityChange);
    window.removeEventListener('blur', onBlur, false);
      window.removeEventListener('pagehide', onPageHide);
    };
  }, [navigate, sessionId, exam]);

  const confirmExit = () => {
    // Anggap tombol Keluar sebagai percobaan keluar (mengurangi kesempatan)
    try {
      try { postViolation('manual_exit', { attempts: switchAttempts.current }); } catch (_) {}
      const didAuto = incExitAttempt('manual_exit');
      if (!didAuto) {
        navigate('/exam');
      }
    } catch (_) {
      navigate('/exam');
    }
  };
  const cancelExit = () => { try { window.history.pushState(null, '', window.location.href); } catch (_) {} };

  // Muat jawaban & flag dari localStorage
  useEffect(() => {
    try {
      const savedAnswers = JSON.parse(localStorage.getItem(`exam_answers_${sessionId}`) || '{}');
      if (savedAnswers && typeof savedAnswers === 'object') setAnswers(prev => ({ ...prev, ...savedAnswers }));
    } catch {}
    try {
      const savedFlags = JSON.parse(localStorage.getItem(`exam_flags_${sessionId}`) || '{}');
      if (savedFlags && typeof savedFlags === 'object') setFlags(savedFlags);
    } catch {}
  }, [sessionId, questions.length]);

  // Muat flag ragu-ragu berbasis exam agar tetap kuning saat keluar & mulai lagi
  useEffect(() => {
    if (!exam || !exam.id) return;
    try {
      const key = `exam_flags_${participantKey}_${exam.id}`;
      const savedExamFlags = JSON.parse(localStorage.getItem(key) || '{}');
      if (savedExamFlags && typeof savedExamFlags === 'object') {
        // Gabungkan dengan flags saat ini (prioritas yang tersimpan)
        setFlags(prev => ({ ...prev, ...savedExamFlags }));
      }
    } catch {}
  }, [exam?.id, participantKey]);

  // Simpan otomatis ke localStorage
  useEffect(() => {
    localStorage.setItem(`exam_answers_${sessionId}`, JSON.stringify(answers));
  }, [answers, sessionId]);

  useEffect(() => {
    localStorage.setItem(`exam_flags_${sessionId}`, JSON.stringify(flags));
    // Simpan juga pada level exam agar persist saat sesi baru
    if (exam && exam.id) {
      try {
        localStorage.setItem(`exam_flags_${participantKey}_${exam.id}`, JSON.stringify(flags));
      } catch {}
    }
  }, [flags, sessionId, exam?.id, participantKey]);

  if (loading) return <div className="container mt-5 notranslate" translate="no">Memuat sesi ujian...</div>;
  if (!exam) return <div className="container mt-5 notranslate" translate="no">Sesi ujian tidak ditemukan.</div>;

  const formatTime = (ms) => {
    if (ms == null) return '-';
    const total = Math.floor(ms / 1000);
    const h = Math.floor(total / 3600);
    const m = Math.floor((total % 3600) / 60);
    const s = total % 60;
    if (h > 0) return `${String(h).padStart(2,'0')}:${String(m).padStart(2,'0')}:${String(s).padStart(2,'0')}`;
    return `${String(m).padStart(2,'0')}:${String(s).padStart(2,'0')}`;
  };

  // (Tidak ada fungsi audio)

  return (
    <div ref={examRef} className="container mt-4 notranslate" translate="no" style={{ maxWidth: 900 }}>
      <div className="d-flex justify-content-between align-items-center mb-3">
        <div>
          <h4 className="mb-1">{exam.name}</h4>
          <small className="text-muted">Kode: <code>{exam.code}</code> • Durasi: {exam.duration_minutes} menit</small>
        </div>
      </div>

      {/* Daftar Soal: tampil ketika tombol diklik, hijau untuk yang sudah dijawab */}
      <div className="mb-3">
        <button
          type="button"
          className="btn btn-sm btn-outline-primary"
          onClick={() => setShowList(prev => !prev)}
        >
          Daftar Soal {showList ? '▲' : '▼'}
        </button>

        {showList && (
          <div className="d-flex flex-wrap gap-2 align-items-center mt-2" style={{ maxWidth: '100%', overflowX: 'auto' }}>
            {questions.map((qq, i) => {
              const ans = answers[qq.id];
              const isEssay = isEssayQuestion(qq);
              const done = isEssay ? (typeof ans === 'string' && ans.trim() !== '') : (typeof ans === 'string' && /^[A-E]$/.test(ans));
              const isCurrent = i === current;
              const flagged = !!flags[qq.id];
              let btnClass = 'btn-outline-secondary';
              if (flagged) btnClass = 'btn-warning';
              else if (done) btnClass = 'btn-success';
              else if (isCurrent) btnClass = 'btn-primary';
              return (
                <button
                  type="button"
                  key={qq.id}
                  className={`btn btn-sm ${btnClass}`}
                  onClick={() => setCurrent(i)}
                >
                  {i + 1}
                </button>
              );
            })}
          </div>
        )}
      </div>

      {questions.length === 0 ? (
        <div className="alert alert-info">Belum ada soal untuk sesi ini.</div>
      ) : (
        (() => {
          const q = questions[current];
          const isEssay = isEssayQuestion(q);
          const letters = ['A', 'B', 'C', 'D'];
          if (!isEssay && !isDashOrBlank(q?.option_e)) {
            letters.push('E');
          }
          return (
            <>
              <div className="d-flex align-items-center justify-content-between mb-2">
                <div className="flex-shrink-0"><strong>Soal {current + 1}</strong> dari {questions.length}</div>
                <div className="flex-grow-1 text-center">
                  {timeWarningVisible && (
                    <div className="alert alert-danger py-1 px-2 mb-0 d-inline-block">
                      Waktu ujian tersisa 5 menit. Silakan cek jawaban Anda !
                    </div>
                  )}
                </div>
                <div className="flex-shrink-0 text-end">
                  <div className="d-flex flex-column align-items-end gap-2" style={{ width: 130 }}>
                    {attempts > 0 && (
                      <span className={`badge w-100 d-flex justify-content-between align-items-center py-2 px-3 ${attempts < 3 ? 'bg-warning text-dark' : 'bg-danger'}`}>
                        <span className="me-2" aria-hidden="true">🚪</span>
                        <span className="flex-grow-1 text-start">Keluar: {Math.min(attempts, 3)}/3</span>
                      </span>
                    )}
                    {exam && (
                      (() => {
                        const used = getStartAttempts(exam.id);
                        const limit = 3;
                        const badgeClass = used >= limit ? 'bg-danger' : (used === limit - 1 ? 'bg-warning text-dark' : 'bg-success');
                        return (
                          <span className={`badge w-100 d-flex justify-content-between align-items-center py-2 px-3 ${badgeClass}`}>
                            <span className="me-2" aria-hidden="true">🎫</span>
                            <span className="flex-grow-1 text-start">Masuk: {used}/{limit}</span>
                          </span>
                        );
                      })()
                    )}
                    <span className={`badge w-100 d-flex justify-content-between align-items-center py-2 px-3 ${remainingMs != null && remainingMs <= 5 * 60 * 1000 ? 'bg-danger' : 'bg-secondary'}`}>
                      <span className="me-2" aria-hidden="true">⏳</span>
                      <span className="flex-grow-1 text-start">Waktu: {formatTime(remainingMs)}</span>
                    </span>
                  </div>
                </div>
              </div>

              <div key={q.id} className="card mb-3">
                <div className="card-body">
                  {q.image_path && (
                    <div className="mb-2"><img src={resolveImageUrl(q.image_path)} alt="Gambar soal" style={PREVIEW_IMG_STYLE} /></div>
                  )}
                  <div className="preview-soal" style={{ whiteSpace: 'pre-wrap' }}>{renderMixed(q.text)}</div>
                  {!isEssay ? (
                    <div className="mt-2 exam-options">
                      {letters.map(letter => (
                        <div className="form-check" key={letter}>
                          <input
                            className="form-check-input"
                            type="radio"
                            name={`q_${q.id}`}
                            id={`q_${q.id}_${letter}`}
                            checked={answers[q.id] === letter}
                            onChange={() => setAnswers(prev => ({ ...prev, [q.id]: letter }))}
                          />
                          <label className="form-check-label" htmlFor={`q_${q.id}_${letter}`}>
                            <span className="me-2 fw-semibold">{letter}.</span>
                            <span className="preview-soal">{renderMixed(q[`option_${letter.toLowerCase()}`], { imgStyle: OPTION_PREVIEW_IMG_STYLE, disableYoutube: true })}</span>
                          </label>
                        </div>
                      ))}
                    </div>
                  ) : (
                    <div className="mt-3">
                      <label className="form-label fw-semibold">Jawaban Essay</label>
                      <textarea
                        className="form-control"
                        rows={6}
                        placeholder="Tuliskan jawaban Anda di sini..."
                        value={typeof answers[q.id] === 'string' ? answers[q.id] : ''}
                        onChange={e => setAnswers(prev => ({ ...prev, [q.id]: e.target.value }))}
                      />
                      <div className="mt-2">
                        <small className="text-muted d-block">Preview jawaban (LaTeX didukung):</small>
                        <div className="border rounded p-2 preview-soal" style={{ whiteSpace: 'pre-wrap' }}>{renderMixed(typeof answers[q.id] === 'string' ? answers[q.id] : '')}</div>
                      </div>
                    </div>
                  )}
                </div>
              </div>
              {/* Navigasi bawah: kiri (Sebelumnya), tengah (Ragu ragu), kanan (Berikutnya/Selesai) */}
              <div className="mt-3 d-flex align-items-center justify-content-between flex-column flex-md-row gap-2">
                <div className="flex-shrink-0" style={{ width: 140 }}>
                  <button
                    type="button"
                    className="btn btn-outline-primary btn-sm w-100 py-2"
                    onClick={() => setCurrent(i => Math.max(0, i - 1))}
                    disabled={current === 0}
                  >
                    <span aria-hidden="true" className="me-1">←</span>
                    Sebelumnya
                  </button>
                </div>
                <div className="d-flex justify-content-center w-100" style={{ maxWidth: 360 }}>
                  <div className="btn-group w-100" role="group" aria-label="Ragu atau Yakin">
                    <button
                      type="button"
                      className={`btn btn-warning btn-sm flex-grow-1 py-2 ${flags[q.id] ? 'active' : ''}`}
                      onClick={toggleFlag}
                      style={{ flex: '1 0 0', backgroundColor: '#fff3cd', color: '#664d03' }}
                    >
                      <span aria-hidden="true" className="me-1">❓</span>
                      Ragu Ragu
                    </button>
                    <button
                      type="button"
                      className="btn btn-success btn-sm flex-grow-1 py-2"
                      onClick={confirmCurrent}
                      style={{ flex: '1 0 0', backgroundColor: '#d1e7dd', color: '#0f5132' }}
                    >
                      Yakin <span aria-hidden="true" className="ms-1">✅</span>
                    </button>
                  </div>
                </div>
                <div className="flex-shrink-0" style={{ width: 140 }}>
                  {current < questions.length - 1 ? (
                    <button
                      type="button"
                      className="btn btn-primary btn-sm w-100 py-2"
                      onClick={() => setCurrent(i => Math.min(questions.length - 1, i + 1))}
                    >
                      Berikutnya
                      <span aria-hidden="true" className="ms-1">→</span>
                    </button>
                  ) : (
                    <button type="button" className="btn btn-success btn-sm w-100 py-2" onClick={handleFinish}>Selesai</button>
                  )}
                </div>
              </div>
            </>
          );
        })()
      )}

      {/* Notice 5 menit dipindahkan ke header; blok bawah dihapus */}

      {confirmFinishInfo && (
        <div
          className="position-fixed top-0 start-0 w-100 h-100"
          style={{ background: 'rgba(0,0,0,0.35)', backdropFilter: 'blur(2px)', zIndex: 1050 }}
        >
          <div className="d-flex align-items-center justify-content-center h-100 p-3">
            <div className="card border-0 shadow-lg rounded-3" style={{ maxWidth: 560, width: '100%' }}>
              <div className="card-header bg-light border-0 d-flex align-items-center justify-content-between">
                <div className="d-flex align-items-center gap-2">
                  <div
                    className="rounded-circle d-flex align-items-center justify-content-center"
                    style={{ width: 36, height: 36, background: '#fff3cd' }}
                  >
                    <span className="fw-bold" style={{ color: '#664d03' }}>⚠️</span>
                  </div>
                  <h5 className="mb-0">Konfirmasi Selesai Ujian</h5>
                </div>
                <button type="button" className="btn btn-sm btn-outline-secondary" onClick={closeFinishConfirm} aria-label="Tutup">✕</button>
              </div>
              <div className="card-body">
                {/* Ringkasan cepat */}
                <div className="d-flex flex-wrap align-items-center gap-2 mb-3">
                  <span className={`badge rounded-pill ${confirmFinishInfo.hasLongTime ? 'bg-info' : 'bg-secondary'}`}>
                    <span aria-hidden="true" className="me-1">⏳</span>
                    Sisa waktu: {formatTime(confirmFinishInfo.remainingMs)}
                  </span>
                  <span className="badge rounded-pill bg-warning text-dark">
                    Ragu-ragu: {confirmFinishInfo.flaggedCount}
                  </span>
                  <span className="badge rounded-pill bg-danger">
                    Belum dijawab: {confirmFinishInfo.unansweredCount}
                  </span>
                </div>

                {/* Detail nomor soal */}
                {(() => {
                  const flaggedNums = questions
                    .map((q, i) => (flags[q.id] ? i + 1 : null))
                    .filter(Boolean);
                  const unansweredNums = questions
                    .map((q, i) => (!answers[q.id] ? i + 1 : null))
                    .filter(Boolean);

                  return (
                    <>
                      {flaggedNums.length > 0 && (
                        <div className="mb-2">
                          <div className="fw-semibold mb-1">Soal ditandai ragu-ragu:</div>
                          <div className="d-flex flex-wrap gap-1">
                            {flaggedNums.slice(0, 25).map(n => (
                              <span key={`f_${n}`} className="badge rounded-pill bg-warning text-dark">{n}</span>
                            ))}
                            {flaggedNums.length > 25 && (
                              <span className="badge rounded-pill bg-warning text-dark">…</span>
                            )}
                          </div>
                        </div>
                      )}
                      {unansweredNums.length > 0 && (
                        <div className="mb-2">
                          <div className="fw-semibold mb-1">Soal belum dijawab:</div>
                          <div className="d-flex flex-wrap gap-1">
                            {unansweredNums.slice(0, 25).map(n => (
                              <span key={`u_${n}`} className="badge rounded-pill bg-danger">{n}</span>
                            ))}
                            {unansweredNums.length > 25 && (
                              <span className="badge rounded-pill bg-danger">…</span>
                            )}
                          </div>
                        </div>
                      )}
                      {flaggedNums.length === 0 && unansweredNums.length === 0 && (
                        <div className="alert alert-success mb-0">
                          Semua soal sudah dijawab dengan yakin. Anda dapat mengumpulkan jawaban.
                        </div>
                      )}
                    </>
                  );
                })()}

                {/* Tombol aksi */}
                <div className="d-flex justify-content-end gap-2 mt-3">
                  <button type="button" className="btn btn-outline-secondary" onClick={closeFinishConfirm}>✍️ Lanjutkan mengerjakan</button>
                  <button
                    type="button"
                    className="btn btn-success"
                    onClick={() => submitAnswers()}
                    disabled={confirmFinishInfo.unansweredCount > 0 || confirmFinishInfo.flaggedCount > 0}
                    style={(confirmFinishInfo.unansweredCount > 0 || confirmFinishInfo.flaggedCount > 0) ? { filter: 'blur(1px)', opacity: 0.6, cursor: 'not-allowed' } : {}}
                    title={(confirmFinishInfo.unansweredCount > 0 || confirmFinishInfo.flaggedCount > 0) ? 'Tidak bisa mengumpulkan: masih ada ragu-ragu atau belum dijawab' : 'Yakin kumpulkan jawaban'}
                  >
                    📤 Kumpulkan Jawaban
                  </button>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}

      {showFullscreenRequired && (
        <div
          className="position-fixed top-0 start-0 w-100 h-100"
          style={{ background: 'rgba(0,0,0,0.55)', backdropFilter: 'blur(3px)', zIndex: 1070 }}
        >
          <div className="d-flex align-items-center justify-content-center h-100 p-3">
            <div className="card border-0 shadow-lg rounded-3" style={{ maxWidth: 560, width: '100%' }}>
              <div className="card-header bg-light border-0 d-flex align-items-center justify-content-start">
                <div className="d-flex align-items-center gap-2">
                  <div
                    className="rounded-circle d-flex align-items-center justify-content-center"
                    style={{ width: 36, height: 36, background: '#e7f1ff' }}
                    aria-hidden="true"
                  >
                    <span className="fw-bold" style={{ color: '#084298' }}>🔒</span>
                  </div>
                  <h5 className="mb-0">Mode Layar Penuh Wajib</h5>
                </div>
              </div>
              <div className="card-body">
                <p className="mb-2" style={{ lineHeight: 1.6 }}>
                  Apakah Anda menekan tombol <code>Esc</code> atau keluar dari mode layar penuh?
                </p>
                <p className="mb-0" style={{ lineHeight: 1.6 }}>
                  Untuk keamanan ujian, mode layar penuh wajib diaktifkan. Pilih salah satu aksi di bawah ini.
                </p>
                <div className="d-flex justify-content-end gap-2 mt-3">
                  <button
                    className="btn btn-primary"
                    onClick={async () => {
                      try {
                        await requestFullscreenEnhanced(document.documentElement, {
                          showWarning: false,
                          fallbackToViewport: true,
                          mobileOptimization: true
                        });
                        setShowFullscreenRequired(false);
                      } catch (error) {
                        console.warn('Fullscreen request failed:', error);
                        setShowFullscreenRequired(false);
                      }
                    }}
                  >
                    Aktifkan Fullscreen
                  </button>

                </div>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal konfirmasi keluar ujian dihapus sesuai permintaan; gunakan hanya Mode Layar Penuh Wajib */}
    </div>
  );
}