import React, { useEffect, useMemo, useState } from 'react';
import DataTable from 'react-data-table-component';
import Swal from 'sweetalert2';
import { FaPlus, FaEdit, FaTrash, FaSearch, FaDownload, FaKey, FaCog, FaUpload } from 'react-icons/fa';
import api from '../utils/axios';
import { useAuth } from '../context/AuthContext.jsx';
import { canAccess } from '../utils/permissionHelper';

export default function ExamParticipants(){
  const { user, hasRole } = useAuth();
  const isSuperadmin = !!(hasRole && hasRole('superadmin'));
  const [participants, setParticipants] = useState([]);
  const [schools, setSchools] = useState([]);
  const [gradesFilter, setGradesFilter] = useState([]);
  const [classesFilter, setClassesFilter] = useState([]);
  const [classesAll, setClassesAll] = useState([]);
  const [todayExams, setTodayExams] = useState([]);
  const [gradesForm, setGradesForm] = useState([]);
  const [classesForm, setClassesForm] = useState([]);
  const [loading, setLoading] = useState(true);
  const [filterText, setFilterText] = useState('');
  const [selectedSchoolId, setSelectedSchoolId] = useState('');
  const [selectedGradeId, setSelectedGradeId] = useState('');
  const [selectedClassId, setSelectedClassId] = useState('');
  const [showAddModal, setShowAddModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [showImportModal, setShowImportModal] = useState(false);
  const [showDetailModal, setShowDetailModal] = useState(false);
  const [detail, setDetail] = useState(null);
  const [importClassId, setImportClassId] = useState('');
  const [importFile, setImportFile] = useState(null);
  const [form, setForm] = useState({ id_school: '', id_grade: '', id_kelas: '', name_class: '', nisn: '', nama: '', jurusan: '' });
  const [editing, setEditing] = useState(null);
  // Overlay loading ringan saat ekspor PDF
  const [exportingPdf, setExportingPdf] = useState(false);
  // Overlay loading ringan saat ekspor Token
  const [exportingTokens, setExportingTokens] = useState(false);

  const tokenHeader = { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } };

  const canManageParticipants = canAccess(user, 'manage-exam-participants');
  const canViewParticipants = canAccess(user, 'view-exam-participant') || canManageParticipants;
  const canCreateParticipant = canAccess(user, 'create-exam-participant') || canManageParticipants;
  const canEditParticipant = canAccess(user, 'edit-exam-participant') || canManageParticipants;
  const canDeleteParticipant = canAccess(user, 'delete-exam-participant') || canManageParticipants;
  const canResetParticipantPassword = canAccess(user, 'reset-exam-participant-password') || isSuperadmin;
  const canImportParticipants = canAccess(user, 'import-exam-participants');
  const canExportTokens = canAccess(user, 'export-exam-participants');
  const canExportCardsSeats = canAccess(user, 'export-exam-participants-cards-seats');
  const canManageUsers = canAccess(user, 'manage-users');
  // Izin tampilan untuk entitas filter terkait
  const canViewSchools = canAccess(user, 'view-school');
  const canViewGrades = canAccess(user, 'view-grade');
  const canViewClasses = canAccess(user, 'view-class');

  const fetchSchools = async () => {
    try {
      const res = await api.get('/schools', tokenHeader);
      setSchools(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk melihat daftar sekolah.' });
      } else {
        Swal.fire('Error', 'Gagal memuat daftar sekolah', 'error');
      }
    }
  };

  const fetchAllClassesForFilter = async (schoolId = '', gradeId = '') => {
    try {
      const params = [];
      if (schoolId) params.push(`id_school=${schoolId}`);
      if (gradeId) params.push(`id_grade=${gradeId}`);
      const url = params.length ? `/classes?${params.join('&')}` : '/classes';
      const res = await api.get(url, tokenHeader);
      setClassesFilter(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk melihat daftar kelas.' });
      } else {
        Swal.fire('Error', 'Gagal memuat daftar kelas', 'error');
      }
    }
  };

  // Simpan semua kelas untuk pemetaan id_kelas -> sekolah & grade
  const fetchAllClasses = async () => {
    try {
      const res = await api.get('/classes', tokenHeader);
      setClassesAll(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk melihat daftar kelas.' });
      }
    }
  };

  const fetchGradesForFilter = async (schoolId = '') => {
    try {
      const url = schoolId ? `/grades?id_school=${schoolId}` : '/grades';
      const res = await api.get(url, tokenHeader);
      setGradesFilter(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk melihat daftar grade.' });
      } else {
        Swal.fire('Error', 'Gagal memuat daftar grade', 'error');
      }
    }
  };

  const fetchGradesForForm = async (schoolId = '') => {
    try {
      if (!schoolId) { setGradesForm([]); return; }
      const url = `/grades?id_school=${schoolId}`;
      const res = await api.get(url, tokenHeader);
      setGradesForm(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk memuat grade untuk form.' });
      } else {
        Swal.fire('Error', 'Gagal memuat grade untuk form', 'error');
      }
    }
  };

  const fetchClassesForForm = async (schoolId = '', gradeId = '') => {
    try {
      if (!schoolId || !gradeId) { setClassesForm([]); return; }
      const url = `/classes?id_school=${schoolId}&id_grade=${gradeId}`;
      const res = await api.get(url, tokenHeader);
      setClassesForm(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk memuat kelas untuk form.' });
      } else {
        Swal.fire('Error', 'Gagal memuat kelas untuk form', 'error');
      }
    }
  };

  const fetchParticipants = async (classId = '') => {
    setLoading(true);
    try {
      const url = classId ? `/exam-participants?id_kelas=${classId}` : '/exam-participants';
      const res = await api.get(url, tokenHeader);
      setParticipants(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk melihat daftar peserta.' });
      } else {
        Swal.fire('Error', 'Gagal memuat daftar peserta', 'error');
      }
    } finally {
      setLoading(false);
    }
  };

  // Ambil daftar ujian hari ini (untuk admin: semua ujian hari ini)
  const fetchTodayExams = async () => {
    try {
      const res = await api.get('/exams/today', tokenHeader);
      setTodayExams(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk melihat ringkasan ujian hari ini.' });
      }
    }
  };

  useEffect(() => {
    if (canViewSchools) {
      fetchSchools();
    } else {
      setSchools([]);
    }
    if (canViewGrades) {
      fetchGradesForFilter('');
    } else {
      setGradesFilter([]);
    }
    if (canViewClasses) {
      fetchAllClassesForFilter('', '');
      fetchAllClasses();
    } else {
      setClassesFilter([]);
      setClassesAll([]);
    }
    if (canViewParticipants) {
      fetchParticipants('');
      fetchTodayExams();
    } else {
      setLoading(false);
    }
  }, [canViewParticipants, canViewSchools, canViewGrades, canViewClasses]);

  // Re-fetch grade and classes when school filter changes
  useEffect(() => {
    setSelectedGradeId('');
    setSelectedClassId('');
    if (canViewGrades) fetchGradesForFilter(selectedSchoolId || '');
    if (canViewClasses) fetchAllClassesForFilter(selectedSchoolId || '', '');
    if (canViewParticipants) fetchParticipants('');
  }, [selectedSchoolId, canViewParticipants, canViewGrades, canViewClasses]);

  // Re-fetch classes when grade filter changes
  useEffect(() => {
    setSelectedClassId('');
    if (canViewClasses) fetchAllClassesForFilter(selectedSchoolId || '', selectedGradeId || '');
    if (canViewParticipants) fetchParticipants('');
  }, [selectedGradeId, canViewParticipants, canViewClasses]);

  // Re-fetch participants when class filter changes
  useEffect(() => {
    if (canViewParticipants) fetchParticipants(selectedClassId || '');
  }, [selectedClassId, canViewParticipants]);

  // For modals: fetch grades and classes when form school/grade changes
  useEffect(() => {
    if (canViewGrades) {
      fetchGradesForForm(form.id_school || '');
    } else {
      setGradesForm([]);
    }
    if (!form.id_school) setForm(prev => ({ ...prev, id_grade: '' }));
  }, [form.id_school, canViewGrades]);

  useEffect(() => {
    if (canViewClasses) {
      fetchClassesForForm(form.id_school || '', form.id_grade || '');
    } else {
      setClassesForm([]);
    }
    if (!form.id_grade) setForm(prev => ({ ...prev, id_kelas: '', name_class: '' }));
  }, [form.id_grade, canViewClasses]);

  // When class selected in form, set name_class automatically
  useEffect(() => {
    if (!form.id_kelas) { setForm(prev => ({ ...prev, name_class: '' })); return; }
    const c = classesForm.find(x => x.id === Number(form.id_kelas));
    setForm(prev => ({ ...prev, name_class: c?.name || '' }));
  }, [form.id_kelas, classesForm]);

  const getSchoolNameByClassId = (classId) => {
    const c = classesAll.find(x => x.id === classId) || classesFilter.find(x => x.id === classId) || classesForm.find(x => x.id === classId);
    if (!c) return '-';
    const s = schools.find(ss => ss.id === c.id_school);
    return s?.nama || '-';
  };

  const getGradeNumberByClassId = (classId) => {
    const c = classesAll.find(x => x.id === classId) || classesFilter.find(x => x.id === classId) || classesForm.find(x => x.id === classId);
    if (!c) return '-';
    const g = gradesFilter.find(gg => gg.id === c.id_grade) || gradesForm.find(gg => gg.id === c.id_grade);
    return g?.grade ?? '-';
  };

  const filtered = useMemo(() => {
    let list = participants;
    // Filter berdasarkan pilihan sekolah/grade/kelas
    if (selectedSchoolId) {
      const sid = Number(selectedSchoolId);
      list = list.filter(p => {
        const c = classesAll.find(x => x.id === p.id_kelas);
        return c?.id_school === sid;
      });
    }
    if (selectedGradeId) {
      const gid = Number(selectedGradeId);
      list = list.filter(p => {
        const c = classesAll.find(x => x.id === p.id_kelas);
        return c?.id_grade === gid;
      });
    }
    if (selectedClassId) {
      const cid = Number(selectedClassId);
      list = list.filter(p => p.id_kelas === cid);
    }
    // Pencarian teks
    const q = String(filterText || '').toLowerCase();
    if (!q) return list;
    return list.filter(p => (
      String(p.nisn || '').toLowerCase().includes(q) ||
      String(p.nama || '').toLowerCase().includes(q) ||
      String(p.name_class || '').toLowerCase().includes(q) ||
      String(getSchoolNameByClassId(p.id_kelas)).toLowerCase().includes(q) ||
      String(getGradeNumberByClassId(p.id_kelas)).toLowerCase().includes(q) ||
      String(p.jurusan || '').toLowerCase().includes(q)
    ));
  }, [filterText, participants, selectedSchoolId, selectedGradeId, selectedClassId, classesAll, schools, gradesFilter, gradesForm]);

  // Ringkasan peserta berdasarkan ujian hari ini (kelompok per ujian/subject)
  const todaySummary = useMemo(() => {
    if (!Array.isArray(todayExams) || todayExams.length === 0) return [];
    return todayExams.map(ex => {
      const schoolName = schools.find(s => s.id === ex.subject_school_id)?.nama || '-';
      const gradeName = gradesFilter.find(g => g.id === ex.subject_grade_id)?.grade || '-';
      const matched = participants.filter(p => {
        const c = classesAll.find(x => x.id === p.id_kelas);
        if (!c) return false;
        const matchSchool = ex.subject_school_id ? c.id_school === ex.subject_school_id : true;
        const matchGrade = ex.subject_grade_id ? c.id_grade === ex.subject_grade_id : true;
        return matchSchool && matchGrade;
      });
      return {
        examId: ex.id,
        name: ex.name,
        code: ex.code,
        duration: ex.duration_minutes,
        scheduled_at: ex.scheduled_at,
        subjectName: ex.subject_name || '-',
        schoolName,
        gradeName,
        count: matched.length,
      };
    });
  }, [todayExams, participants, classesAll, schools, gradesFilter]);

  const openAdd = () => {
    setForm({ id_school: '', id_grade: '', id_kelas: '', name_class: '', nisn: '', nama: '', jurusan: '', password: '' });
    setGradesForm([]);
    setClassesForm([]);
    setShowAddModal(true);
  };

  const openEdit = (row) => {
    setEditing(row);
    // Derive school and grade from class id if possible
    const c = classesFilter.find(x => x.id === row.id_kelas) || classesForm.find(x => x.id === row.id_kelas);
    const id_school = c?.id_school ?? '';
    const id_grade = c?.id_grade ?? '';
    setForm({
      id_school: id_school,
      id_grade: id_grade,
      id_kelas: row.id_kelas ?? '',
      name_class: row.name_class ?? '',
      nisn: row.nisn ?? '',
      nama: row.nama ?? '',
      jurusan: row.jurusan ?? '',
      password: ''
    });
    setShowEditModal(true);
  };

  const openDetail = async (row) => {
    try {
      const res = await api.get(`/exam-participants/${row.id}`, tokenHeader);
      setDetail(res.data);
      setShowDetailModal(true);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk melihat detail peserta.' });
      } else {
        Swal.fire('Gagal', 'Gagal mengambil detail peserta', 'error');
      }
    }
  };

  const copyToken = async () => {
    if (!detail?.login_token) return;
    try {
      await navigator.clipboard.writeText(detail.login_token);
      Swal.fire('Disalin', 'Token ujian telah disalin ke clipboard', 'success');
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', 'Tidak bisa menyalin token', 'error');
    }
  };

  const downloadTemplate = async () => {
    try {
      const params = {};
      if (selectedClassId) params.id_kelas = selectedClassId;
      const res = await api.get('/exam-participants/template', {
        ...tokenHeader,
        params,
        responseType: 'blob'
      });
      const url = window.URL.createObjectURL(new Blob([res.data]));
      const a = document.createElement('a');
      a.href = url;
      a.download = 'template_peserta.xlsx';
      document.body.appendChild(a);
      a.click();
      a.remove();
      window.URL.revokeObjectURL(url);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk mengunduh template peserta.' });
      } else {
        Swal.fire('Error', 'Gagal mengunduh template', 'error');
      }
    }
  };

  const openImport = () => {
    setImportClassId(selectedClassId || '');
    setImportFile(null);
    setShowImportModal(true);
  };

  const validateForm = (payload) => {
    if (!String(payload.nisn).trim()) return 'NISN wajib diisi';
    if (!String(payload.nama).trim()) return 'Nama wajib diisi';
    if (!String(payload.id_kelas).trim()) return 'Kelas wajib dipilih';
    if (!String(payload.name_class).trim()) return 'Nama kelas otomatis tidak terbaca';
    if (String(form.password || '').trim() && String(form.password || '').trim().length < 6) return 'Password minimal 6 karakter';
    return null;
  };

  const handleAdd = async () => {
    const payload = {
      nisn: String(form.nisn || '').trim(),
      nama: String(form.nama || '').trim(),
      id_kelas: Number(form.id_kelas),
      name_class: String(form.name_class || '').trim(),
      jurusan: String(form.jurusan || '').trim() || null,
    };
    if (String(form.password || '').trim().length >= 6) {
      payload.password = String(form.password).trim();
    }
    const errMsg = validateForm(payload);
    if (errMsg) { Swal.fire('Validasi', errMsg, 'warning'); return; }
    try {
      await api.post('/exam-participants', payload, tokenHeader);
      setShowAddModal(false);
      await fetchParticipants(selectedClassId || '');
      Swal.fire('Berhasil', 'Peserta ditambahkan', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk menambah peserta.' });
      } else {
        const msg = err.response?.data?.message || err.response?.data?.errors?.nisn?.[0] || 'Gagal menambah peserta';
        Swal.fire('Gagal', msg, 'error');
      }
    }
  };

  const handleEditSave = async () => {
    if (!editing) return;
    const payload = {
      nisn: String(form.nisn || '').trim(),
      nama: String(form.nama || '').trim(),
      id_kelas: Number(form.id_kelas),
      name_class: String(form.name_class || '').trim(),
      jurusan: String(form.jurusan || '').trim() || null,
    };
    if (String(form.password || '').trim().length >= 6) {
      payload.password = String(form.password).trim();
    }
    const errMsg = validateForm(payload);
    if (errMsg) { Swal.fire('Validasi', errMsg, 'warning'); return; }
    try {
      await api.put(`/exam-participants/${editing.id}`, payload, tokenHeader);
      setShowEditModal(false);
      setEditing(null);
      await fetchParticipants(selectedClassId || '');
      Swal.fire('Berhasil', 'Peserta diperbarui', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk memperbarui peserta.' });
      } else {
        const msg = err.response?.data?.message || err.response?.data?.errors?.nisn?.[0] || 'Gagal memperbarui peserta';
        Swal.fire('Gagal', msg, 'error');
      }
    }
  };

  const handleDelete = async (row) => {
    const ok = await Swal.fire({
      title: 'Hapus peserta?',
      text: `${row.nama} (${row.nisn}) | Kelas: ${row.name_class}`,
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Ya, hapus',
      cancelButtonText: 'Batal'
    }).then(r => r.isConfirmed);
    if (!ok) return;
    try {
      await api.delete(`/exam-participants/${row.id}`, tokenHeader);
      await fetchParticipants(selectedClassId || '');
      Swal.fire('Berhasil', 'Peserta dihapus', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk menghapus peserta.' });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Gagal menghapus peserta', 'error');
      }
    }
  };

  const handleResetPassword = async (row) => {
    const ok = await Swal.fire({
      title: `Reset password ${row.nama}?`,
      text: 'Password akan direset ke 12345678',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Reset',
      cancelButtonText: 'Batal'
    }).then(r => r.isConfirmed);
    if (!ok) return;
    try {
      await api.put(`/exam-participants/${row.id}/reset-password`, {}, tokenHeader);
      Swal.fire('Berhasil', 'Password direset ke 12345678', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk reset password peserta.' });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Tidak bisa reset password', 'error');
      }
    }
  };

  const handleUnbindDevice = async (row) => {
    const ok = await Swal.fire({
      title: `Reset perangkat ${row.nama}?`,
      text: 'Peserta akan dapat login dari perangkat baru',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Reset',
      cancelButtonText: 'Batal'
    }).then(r => r.isConfirmed);
    if (!ok) return;
    try {
      await api.put(`/exam-participants/${row.id}/unbind-device`, {}, tokenHeader);
      Swal.fire('Berhasil', 'Perangkat peserta direset', 'success');
      setShowDetailModal(false);
      setDetail(null);
      await fetchParticipants(selectedClassId || '');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk reset perangkat peserta.' });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Tidak bisa reset perangkat', 'error');
      }
    }
  };

  const handleResetPasswordBulk = async () => {
    const scopeText = selectedClassId ? 'kelas terpilih' : 'SEMUA peserta';
    const ok = await Swal.fire({
      title: `Reset password ${scopeText}?`,
      text: 'Password akan direset ke 12345678',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Reset',
      cancelButtonText: 'Batal'
    }).then(r => r.isConfirmed);
    if (!ok) return;
    try {
      await api.put('/exam-participants/reset-password', { id_kelas: selectedClassId || null }, tokenHeader);
      await fetchParticipants(selectedClassId || '');
      Swal.fire('Berhasil', `Password ${scopeText} direset ke 12345678`, 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk mereset password massal.',
        });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Tidak bisa reset password massal', 'error');
      }
    }
  };

  const handleDeleteAllParticipants = async () => {
    const scopeText = selectedClassId ? 'SEMUA peserta di kelas terpilih' : 'SEMUA peserta';
    const ok = await Swal.fire({
      title: `Hapus ${scopeText}?`,
      text: 'Tindakan ini akan menghapus data peserta beserta sesi dan hasil ujian terkait. Tindakan tidak dapat dibatalkan.',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Ya, hapus',
      cancelButtonText: 'Batal'
    }).then(r => r.isConfirmed);
    if (!ok) return;
    try {
      const config = selectedClassId ? { ...tokenHeader, params: { id_kelas: selectedClassId } } : tokenHeader;
      await api.delete('/exam-participants', config);
      await fetchParticipants(selectedClassId || '');
      Swal.fire('Berhasil', selectedClassId ? 'Seluruh peserta di kelas terpilih dihapus' : 'Seluruh peserta dihapus', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk menghapus seluruh peserta.' });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Tidak bisa menghapus seluruh peserta', 'error');
      }
    }
  };

  const handleImport = async () => {
    if (!importFile) { Swal.fire('Validasi', 'Pilih file Excel/CSV terlebih dahulu', 'warning'); return; }
    const fd = new FormData();
    fd.append('file', importFile);
    if (importClassId) fd.append('id_kelas', importClassId);
    try {
      const res = await api.post('/exam-participants/import', fd, {
        headers: { 'Content-Type': 'multipart/form-data', Authorization: `Bearer ${localStorage.getItem('token')}` }
      });
      const { imported = 0, updated = 0, errors = [] } = res.data || {};
      setShowImportModal(false);
      await fetchParticipants(selectedClassId || '');
      Swal.fire('Selesai', `Import: ${imported}, Update: ${updated}, Error: ${errors.length}`, errors.length ? 'warning' : 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk melakukan import peserta.' });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Import gagal', 'error');
      }
    }
  };

  const handleExportTokens = async () => {
    if (!selectedClassId) {
      Swal.fire('Pilih Kelas', 'Silakan pilih kelas terlebih dahulu sebelum mengekspor token peserta', 'info');
      return;
    }
    try {
      setExportingTokens(true);
      const params = { id_kelas: selectedClassId };
      const res = await api.get('/exam-participants/export-tokens', {
        ...tokenHeader,
        params,
        responseType: 'blob'
      });
      const url = window.URL.createObjectURL(new Blob([res.data]));
      const a = document.createElement('a');
      a.href = url;
      a.download = 'tokens_peserta.xlsx';
      document.body.appendChild(a);
      a.click();
      a.remove();
      window.URL.revokeObjectURL(url);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk mengekspor token peserta.' });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Tidak bisa mengekspor token', 'error');
      }
    } finally {
      setExportingTokens(false);
    }
  };

  const handleExportCardsSeats = async () => {
    if (!selectedClassId) {
      Swal.fire('Pilih Kelas', 'Silakan pilih kelas terlebih dahulu untuk mengekspor PDF kartu ujian & nomor meja', 'info');
      return;
    }
    try {
      setExportingPdf(true);
      const res = await api.get('/exam-participants/export-cards-seats', {
        ...tokenHeader,
        params: { id_kelas: selectedClassId },
        responseType: 'blob'
      });
      const url = window.URL.createObjectURL(new Blob([res.data]));
      const a = document.createElement('a');
      a.href = url;
      a.download = 'kartu_nomor_meja.pdf';
      document.body.appendChild(a);
      a.click();
      a.remove();
      window.URL.revokeObjectURL(url);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk mengekspor PDF kartu & nomor meja.' });
      } else {
        let serverMessage = err?.response?.data?.message || 'Tidak bisa mengekspor PDF kartu & nomor meja';
        try {
          const data = err?.response?.data;
          if (data && typeof Blob !== 'undefined' && data instanceof Blob) {
            const text = await data.text();
            try {
              const json = JSON.parse(text);
              if (json?.message) serverMessage = json.message;
              if (json?.detail) serverMessage = `${serverMessage} (${json.detail})`;
            } catch (_) {
              if (text) serverMessage = text.substring(0, 500);
            }
          }
        } catch (_) {}
        Swal.fire('Gagal', serverMessage, 'error');
      }
    } finally {
      setExportingPdf(false);
    }
  };

  const handleGenerateTokens = async () => {
    const scopeText = selectedClassId ? 'kelas terpilih' : 'SEMUA peserta';
    const ok = await Swal.fire({
      title: `Generate token untuk ${scopeText}?`,
      text: 'Hanya peserta yang belum memiliki token akan dibuatkan',
      icon: 'question',
      showCancelButton: true,
      confirmButtonText: 'Ya, generate',
      cancelButtonText: 'Batal'
    }).then(r => r.isConfirmed);
    if (!ok) return;
    try {
      const payload = { id_kelas: selectedClassId || null };
      const res = await api.put('/exam-participants/generate-tokens', payload, tokenHeader);
      const generated = res.data?.generated || 0;
      await fetchParticipants(selectedClassId || '');
      const proceed = await Swal.fire('Selesai', `Token dibuat untuk ${generated} peserta`, 'success')
        .then(() => generated > 0 ? Swal.fire({
          title: 'Unduh token sekarang?',
          icon: 'info',
          showCancelButton: true,
          confirmButtonText: 'Unduh',
          cancelButtonText: 'Nanti'
        }).then(r => r.isConfirmed) : false);
      if (proceed) {
        await handleExportTokens();
      }
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk generate token massal.',
        });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Tidak bisa generate token', 'error');
      }
    }
  };

  const columns = [
    { name: '#', selector: (row, index) => index + 1, width: '60px' },
    { name: 'NISN', selector: row => row.nisn || '-', sortable: true },
    { name: 'Nama', selector: row => row.nama || '-', sortable: true, wrap: true },
    { name: 'Kelas', selector: row => row.name_class || '-', sortable: true, wrap: true },
    { name: 'Sekolah', selector: row => getSchoolNameByClassId(row.id_kelas), sortable: true, wrap: true },
    { name: 'Grade', selector: row => getGradeNumberByClassId(row.id_kelas), sortable: true },
    { name: 'Jurusan', selector: row => row.jurusan || '-', sortable: true, wrap: true },
    {
      name: 'Aksi',
      cell: row => (
        <div className="d-flex gap-2 flex-wrap">
          {canViewParticipants && (
            <button className="btn btn-sm btn-info" title="Detail" onClick={() => openDetail(row)}>
              <FaSearch />
            </button>
          )}
          {canEditParticipant && (
            <button className="btn btn-sm btn-primary" title="Edit" onClick={() => openEdit(row)}>
              <FaEdit />
            </button>
          )}
          {canResetParticipantPassword && (
            <button className="btn btn-sm btn-warning" title="Reset Password" onClick={() => handleResetPassword(row)}>
              <FaKey />
            </button>
          )}
          {canDeleteParticipant && (
            <button className="btn btn-sm btn-danger" title="Hapus" onClick={() => handleDelete(row)}>
              <FaTrash />
            </button>
          )}
        </div>
      ),
      width: '220px'
    }
  ];

  if (!canViewParticipants) {
    return (
      <div className="container mt-4">
        <h3>Akses Ditolak</h3>
        <p>Anda tidak memiliki izin untuk melihat daftar peserta ujian.</p>
      </div>
    );
  }

  if (loading) return <div className="text-center mt-5">Memuat peserta...</div>;

  return (
    <div className="container mt-4">
      <h3>Peserta Ujian</h3>

      {/* Ringkasan Ujian Hari Ini */}
      {todaySummary.length > 0 && (
        <div className="card mb-3">
          <div className="card-body">
            <h5 className="card-title">Ringkasan Peserta Berdasarkan Ujian Hari Ini</h5>
            <div className="table-responsive">
              <table className="table table-sm">
                <thead>
                  <tr>
                    <th>Nama Ujian</th>
                    <th>Kode</th>
                    <th>Sekolah</th>
                    <th>Grade</th>
                    <th>Mata Pelajaran</th>
                    <th>Jumlah Peserta</th>
                  </tr>
                </thead>
                <tbody>
                  {todaySummary.map(row => (
                    <tr key={row.examId}>
                      <td>{row.name}</td>
                      <td>{row.code}</td>
                      <td>{row.schoolName}</td>
                      <td>{row.gradeName}</td>
                      <td>{row.subjectName}</td>
                      <td>{row.count}</td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
            <small className="text-muted">Ringkasan otomatis dari ujian yang dijadwalkan hari ini.</small>
          </div>
        </div>
      )}

      {/* Tombol aksi utama */}
      <div className="d-flex align-items-center gap-2 mb-2 flex-wrap">
        {canCreateParticipant && (
          <button className="btn btn-primary btn-sm" onClick={openAdd}>
            <FaPlus className="me-1" /> Tambah
          </button>
        )}
        {canImportParticipants && (
          <button className="btn btn-success btn-sm" onClick={openImport}>
            <FaUpload className="me-1" /> Import Excel
          </button>
        )}
        {canImportParticipants && (
          <button className="btn btn-outline-secondary btn-sm" onClick={downloadTemplate}>
            <FaDownload className="me-1" /> Template
          </button>
        )}
        {canExportTokens && (
          <button className="btn btn-outline-success btn-sm" onClick={handleExportTokens} disabled={!selectedClassId || exportingTokens} title={!selectedClassId ? 'Pilih kelas terlebih dahulu' : (exportingTokens ? 'Sedang mengekspor...' : '')}>
            <FaDownload className="me-1" /> Export Token
          </button>
        )}
        {canExportCardsSeats && (
          <button
            className="btn btn-outline-dark btn-sm"
            onClick={handleExportCardsSeats}
            disabled={!selectedClassId || exportingPdf}
            title={!selectedClassId ? 'Pilih kelas terlebih dahulu' : (exportingPdf ? 'Sedang mengekspor...' : '')}
          >
            <FaDownload className="me-1" /> Export Kartu & Nomor Meja
          </button>
        )}
        {isSuperadmin && (
          <button className="btn btn-outline-primary btn-sm" onClick={handleGenerateTokens}>
            <FaCog className="me-1" /> Generate Token {selectedClassId ? 'Kelas' : 'Semua'}
          </button>
        )}
        {isSuperadmin && (
          <button className="btn btn-warning btn-sm" onClick={handleResetPasswordBulk}>
            <FaKey className="me-1" /> Reset Password {selectedClassId ? 'Kelas' : 'Semua'}
          </button>
        )}
        {isSuperadmin && (
          <button className="btn btn-outline-danger btn-sm" onClick={handleDeleteAllParticipants}>
            <FaTrash className="me-1" /> Peserta {selectedClassId ? 'Kelas' : 'Semua'}
          </button>
        )}
      </div>

      {/* Filter diletakkan di bawah tombol */}
      <div className="d-flex align-items-center gap-2 mb-3 flex-wrap">
        {canViewSchools && (
          <div className="input-group input-group-sm" style={{ maxWidth: '280px' }}>
            <span className="input-group-text">Sekolah</span>
            <select
              className="form-select"
              value={selectedSchoolId}
              onChange={e => setSelectedSchoolId(e.target.value)}
            >
              <option value="">Semua</option>
              {schools.map(s => (
                <option key={s.id} value={s.id}>{s.nama}</option>
              ))}
            </select>
          </div>
        )}
        {canViewGrades && (
          <div className="input-group input-group-sm" style={{ maxWidth: '220px' }}>
            <span className="input-group-text">Grade</span>
            <select
              className="form-select"
              value={selectedGradeId}
              onChange={e => setSelectedGradeId(e.target.value)}
            >
              <option value="">Semua</option>
              {gradesFilter.map(g => (
                <option key={g.id} value={g.id}>{g.grade}</option>
              ))}
            </select>
          </div>
        )}
        {canViewClasses && (
          <div className="input-group input-group-sm" style={{ maxWidth: '240px' }}>
            <span className="input-group-text">Kelas</span>
            <select
              className="form-select"
              value={selectedClassId}
              onChange={e => setSelectedClassId(e.target.value)}
            >
              <option value="">Semua</option>
              {classesFilter.map(c => (
                <option key={c.id} value={c.id}>{c.name}</option>
              ))}
            </select>
          </div>
        )}
        <div className="input-group input-group-sm" style={{ maxWidth: '300px' }}>
          <span className="input-group-text"><FaSearch /></span>
          <input
            type="text"
            placeholder="Cari NISN/Nama/Kelas/Jurusan..."
            className="form-control"
            value={filterText}
            onChange={e => setFilterText(e.target.value)}
          />
        </div>
      </div>

      <DataTable
        columns={columns}
        data={filtered}
        pagination
        highlightOnHover
        striped
        noHeader
      />

      {/* Modal Add */}
      {showAddModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Tambah Peserta</h5>
                <button className="btn-close" onClick={() => setShowAddModal(false)}></button>
              </div>
              <div className="modal-body">
                <select
                  className="form-select mb-2"
                  value={form.id_school}
                  onChange={e => setForm({ ...form, id_school: e.target.value })}
                >
                  <option value="">Pilih Sekolah</option>
                  {schools.map(s => (
                    <option key={s.id} value={s.id}>{s.nama}</option>
                  ))}
                </select>
                <select
                  className="form-select mb-2"
                  value={form.id_grade}
                  onChange={e => setForm({ ...form, id_grade: e.target.value })}
                  disabled={!form.id_school}
                >
                  <option value="">Pilih Grade</option>
                  {gradesForm.map(g => (
                    <option key={g.id} value={g.id}>{g.grade}</option>
                  ))}
                </select>
                <select
                  className="form-select mb-2"
                  value={form.id_kelas}
                  onChange={e => setForm({ ...form, id_kelas: e.target.value })}
                  disabled={!form.id_grade}
                >
                  <option value="">Pilih Kelas</option>
                  {classesForm.map(c => (
                    <option key={c.id} value={c.id}>{c.name}</option>
                  ))}
                </select>
                <input
                  type="text"
                  className="form-control mb-2"
                  placeholder="NISN"
                  value={form.nisn}
                  onChange={e => setForm({ ...form, nisn: e.target.value })}
                />
                <input
                  type="text"
                  className="form-control mb-2"
                  placeholder="Nama"
                  value={form.nama}
                  onChange={e => setForm({ ...form, nama: e.target.value })}
                />
                <input
                  type="password"
                  className="form-control mb-2"
                  placeholder="Password (opsional, min 6)"
                  value={form.password}
                  onChange={e => setForm({ ...form, password: e.target.value })}
                />
                <input
                  type="text"
                  className="form-control"
                  placeholder="Jurusan (opsional)"
                  value={form.jurusan}
                  onChange={e => setForm({ ...form, jurusan: e.target.value })}
                />
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowAddModal(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleAdd} disabled={!form.id_kelas || !form.nisn || !form.nama}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Edit */}
      {showEditModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Edit Peserta</h5>
                <button className="btn-close" onClick={() => setShowEditModal(false)}></button>
              </div>
              <div className="modal-body">
                <select
                  className="form-select mb-2"
                  value={form.id_school}
                  onChange={e => setForm({ ...form, id_school: e.target.value })}
                >
                  <option value="">Pilih Sekolah</option>
                  {schools.map(s => (
                    <option key={s.id} value={s.id}>{s.nama}</option>
                  ))}
                </select>
                <select
                  className="form-select mb-2"
                  value={form.id_grade}
                  onChange={e => setForm({ ...form, id_grade: e.target.value })}
                  disabled={!form.id_school}
                >
                  <option value="">Pilih Grade</option>
                  {gradesForm.map(g => (
                    <option key={g.id} value={g.id}>{g.grade}</option>
                  ))}
                </select>
                <select
                  className="form-select mb-2"
                  value={form.id_kelas}
                  onChange={e => setForm({ ...form, id_kelas: e.target.value })}
                  disabled={!form.id_grade}
                >
                  <option value="">Pilih Kelas</option>
                  {classesForm.map(c => (
                    <option key={c.id} value={c.id}>{c.name}</option>
                  ))}
                </select>
                <input
                  type="text"
                  className="form-control mb-2"
                  placeholder="NISN"
                  value={form.nisn}
                  onChange={e => setForm({ ...form, nisn: e.target.value })}
                />
                <input
                  type="text"
                  className="form-control mb-2"
                  placeholder="Nama"
                  value={form.nama}
                  onChange={e => setForm({ ...form, nama: e.target.value })}
                />
                <input
                  type="password"
                  className="form-control mb-2"
                  placeholder="Password (opsional, min 6; kosongkan jika tidak diubah)"
                  value={form.password}
                  onChange={e => setForm({ ...form, password: e.target.value })}
                />
                <input
                  type="text"
                  className="form-control"
                  placeholder="Jurusan (opsional)"
                  value={form.jurusan}
                  onChange={e => setForm({ ...form, jurusan: e.target.value })}
                />
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowEditModal(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleEditSave} disabled={!form.id_kelas || !form.nisn || !form.nama}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Import */}
      {showImportModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Import Peserta (Excel/CSV)</h5>
                <button className="btn-close" onClick={() => setShowImportModal(false)}></button>
              </div>
              <div className="modal-body">
                {canViewClasses && (
                  <select
                    className="form-select mb-2"
                    value={importClassId}
                    onChange={e => setImportClassId(e.target.value)}
                  >
                    <option value="">Pilih Kelas (opsional)</option>
                    {classesFilter.map(c => (
                      <option key={c.id} value={c.id}>{c.name}</option>
                    ))}
                  </select>
                )}
                <input
                  type="file"
                  accept=".xlsx,.csv,.txt"
                  className="form-control"
                  onChange={e => setImportFile(e.target.files?.[0] || null)}
                />
                <small className="text-muted d-block mt-2">
                  Format kolom minimal: <code>nisn</code>, <code>nama</code>. Jika tidak memilih kelas di atas, sertakan kolom <code>id_kelas</code> pada file.
                  {!canViewClasses && (
                    <span className="ms-1">Anda tidak memiliki izin melihat daftar kelas, pastikan kolom <code>id_kelas</code> tersedia.</span>
                  )}
                </small>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowImportModal(false)}>Batal</button>
                <button className="btn btn-success" onClick={handleImport} disabled={!importFile}>Import</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Detail Peserta */}
      {showDetailModal && detail && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Detail Peserta</h5>
                <button className="btn-close" onClick={() => { setShowDetailModal(false); setDetail(null); }}></button>
              </div>
              <div className="modal-body">
                <div className="mb-2"><strong>NISN:</strong> {detail.nisn || '-'}</div>
                <div className="mb-2"><strong>Nama:</strong> {detail.nama || '-'}</div>
                <div className="mb-2"><strong>Kelas:</strong> {detail.name_class || '-'}</div>
                <div className="mb-2"><strong>Sekolah:</strong> {getSchoolNameByClassId(detail.id_kelas) || '-'}</div>
                <div className="mb-2"><strong>Grade:</strong> {getGradeNumberByClassId(detail.id_kelas) || '-'}</div>
                <div className="mb-2"><strong>Jurusan:</strong> {detail.jurusan || '-'}</div>
                {detail.device_bound_at && (
                  <div className="mb-2"><strong>Perangkat Terikat:</strong> {new Date(detail.device_bound_at).toLocaleString()}</div>
                )}
                <hr />
                <div className="mb-2 d-flex align-items-center justify-content-between">
                  <div>
                    <strong>Token Ujian:</strong>
                    <div className="mt-1"><code>{detail.login_token || 'Belum dibuat'}</code></div>
                  </div>
                  <div>
                    <button className="btn btn-sm btn-outline-primary" onClick={copyToken} disabled={!detail.login_token}>Salin Token</button>
                    {detail?.device_bound_at && canManageUsers && (
                      <button className="btn btn-sm btn-outline-danger ms-2" onClick={() => handleUnbindDevice(detail)}>Reset Perangkat</button>
                    )}
                  </div>
                </div>
                <small className="text-muted">Token bersifat unik per siswa. Bagikan ke siswa untuk login cepat via token.</small>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => { setShowDetailModal(false); setDetail(null); }}>Tutup</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Overlay Loading saat ekspor PDF (ringan, tanpa ketergantungan berat) */}
      {exportingPdf && (
        <div
          className="position-fixed top-0 start-0 w-100 h-100"
          style={{ background: 'rgba(0,0,0,0.25)', backdropFilter: 'blur(2px)', zIndex: 1050 }}
          aria-live="polite"
          aria-busy="true"
        >
          <div className="d-flex align-items-center justify-content-center h-100 p-3">
            <div className="card shadow" style={{ minWidth: 260 }}>
              <div className="card-body d-flex align-items-center gap-3">
                <div className="spinner-border text-primary" role="status" aria-hidden="true"></div>
                <div>
                  <div className="fw-bold">Membuat PDF...</div>
                  <small className="text-muted">Mohon tunggu, proses sedang berjalan</small>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Overlay Loading saat ekspor Token */}
      {exportingTokens && (
        <div
          className="position-fixed top-0 start-0 w-100 h-100"
          style={{ background: 'rgba(0,0,0,0.25)', backdropFilter: 'blur(2px)', zIndex: 1050 }}
          aria-live="polite"
          aria-busy="true"
        >
          <div className="d-flex align-items-center justify-content-center h-100 p-3">
            <div className="card shadow" style={{ minWidth: 260 }}>
              <div className="card-body d-flex align-items-center gap-3">
                <div className="spinner-border text-success" role="status" aria-hidden="true"></div>
                <div>
                  <div className="fw-bold">Mengunduh Token...</div>
                  <small className="text-muted">Menyiapkan file Excel token peserta</small>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}