import React, { useEffect, useMemo, useRef, useState } from 'react';
import { useParams, useLocation } from 'react-router-dom';
import api from '../utils/axios';
import Swal from 'sweetalert2';
import { InlineMath, BlockMath } from 'react-katex';
import 'katex/dist/katex.min.css';
import renderMathInElement from 'katex/contrib/auto-render';
import RichTextEditor from '../components/RichTextEditor';
import { sanitizeLatex } from '../utils/latex';

// Konsistensi gaya gambar untuk preview agar seragam dengan edit soal
const PREVIEW_IMG_STYLE = { maxWidth: '100%', maxHeight: '240px', objectFit: 'contain', borderRadius: 6, margin: '6px 0' };


// Backend base untuk akses file di /storage (fallback dev)
const BACKEND_BASE = import.meta.env.VITE_BACKEND_URL || 'http://127.0.0.1:8000';
const resolveImageUrl = (urlOrPath) => {
  if (!urlOrPath) return '';
  const s = String(urlOrPath).trim();
  const base = String(BACKEND_BASE || '').replace(/\/+$/, '');
  if (/^(data|blob):/.test(s)) return s;
  // Rebase absolut ke /storage/ atau /images/ agar origin mengikuti BACKEND_BASE
  if (/^https?:\/\/[^/]+\/(storage|images)\//.test(s)) {
    return s.replace(/^https?:\/\/[^/]+/, base);
  }
  if (/^https?:\/\//.test(s)) return s;
  const clean = s.replace(/^\/+/, '');
  if (/^storage\//.test(clean)) {
    return `${base}/storage/${clean.replace(/^storage\//,'')}`;
  }
  if (/^images\//.test(clean)) {
    // layani langsung dari public/images
    return `${base}/${clean}`;
  }
  if (/^\/storage\//.test(s)) return `${base}${s}`;
  return `${base}/storage/${clean.replace(/^public\//,'')}`;
};

// Deteksi sederhana apakah string tampak sebagai LaTeX
const looksLatex = (s) => {
  if (!s || typeof s !== 'string') return false;
  return /\\(frac|sqrt|left|right|begin|end|bmatrix|pm|times|div|pi|theta)/.test(s);
};

// Util: konversi style object ke string inline
const styleObjectToString = (obj) => {
  if (!obj || typeof obj !== 'object') return '';
  return Object.entries(obj).map(([key, val]) => {
    const cssKey = key.replace(/[A-Z]/g, (m) => '-' + m.toLowerCase());
    const cssVal = typeof val === 'number' ? `${val}px` : String(val);
    return `${cssKey}: ${cssVal};`;
  }).join(' ');
};

// Parser BBCode → HTML untuk [b], [i], [u], [s], [br], [img], align, table, video
const bbcodeToHtml = (input, opts = {}) => {
  if (!input || typeof input !== 'string') return '';
  let s = input;
  // Teks format
  s = s.replace(/\[b\](.*?)\[\/b\]/gis, '<strong>$1</strong>');
  s = s.replace(/\[i\](.*?)\[\/i\]/gis, '<em>$1</em>');
  s = s.replace(/\[u\](.*?)\[\/u\]/gis, '<u>$1</u>');
  s = s.replace(/\[s\](.*?)\[\/s\]/gis, '<s>$1</s>');
  s = s.replace(/\[br\s*\/?\]/gi, '<br/>');
  // Alignment
  s = s.replace(/\[center\](.*?)\[\/center\]/gis, (m, p1) => `<div style="text-align:center;">${p1}</div>`);
  s = s.replace(/\[left\](.*?)\[\/left\]/gis, (m, p1) => `<div style="text-align:left;">${p1}</div>`);
  s = s.replace(/\[right\](.*?)\[\/right\]/gis, (m, p1) => `<div style="text-align:right;">${p1}</div>`);
  s = s.replace(/\[align=(left|center|right)\](.*?)\[\/align\]/gis, (m, align, body) => `<div style="text-align:${align};">${body}</div>`);
  // Gambar
  s = s.replace(/\[img\](.*?)\[\/img\]/gis, (m, p1) => {
    const raw = String(p1 || '').trim();
    const url = resolveImageUrl(raw);
    const style = styleObjectToString(opts.imgStyle || { maxWidth: '100%', maxHeight: '240px', objectFit: 'contain', borderRadius: 6, margin: '6px 0' });
    const safeUrl = String(url || '').replace(/\"/g, '&quot;');
    const base = String(BACKEND_BASE || '').replace(/\/+$/, '');
    const clean = raw.replace(/^\/+/, '');
    let alt = '';
    if (/^images\//.test(clean)) alt = `${base}/storage/${clean}`;
    else if (/^storage\//.test(clean)) alt = `${base}/${clean.replace(/^storage\//,'')}`;
    const onErrAttr = alt ? ` onerror="if(this._trFallback){return;}this._trFallback=1;this.src='${alt}'"` : '';
    return `<img src="${safeUrl}" alt="img" style="${style}"${onErrAttr} />`;
  });
  // YouTube (dapat dinonaktifkan via opts.disableYoutube)
  const extractYouTubeId = (raw) => {
    const s2 = String(raw || '').trim();
    if (/^[A-Za-z0-9_-]{11}$/.test(s2)) return s2;
    try {
      const u = new URL(s2);
      if (u.hostname.includes('youtu.be')) {
        const id = (u.pathname || '').replace(/^\//, '').split('/')[0];
        return id || null;
      }
      if (u.hostname.includes('youtube.com')) {
        const id = u.searchParams.get('v');
        return id || null;
      }
    } catch (_) {}
    return null;
  };
  if (!opts.disableYoutube) {
    s = s.replace(/\[youtube\](.*?)\[\/youtube\]/gis, (m, p1) => {
      const id = extractYouTubeId(p1);
      if (!id) return p1;
      const src = `https://www.youtube.com/embed/${id}`;
      return `<div class="ratio ratio-16x9" style="max-width: 720px; margin: 6px auto;"><iframe src="${src}" title="YouTube" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" allowfullscreen style="width:100%; height:100%;"></iframe></div>`;
    });
  } else {
    s = s.replace(/\[youtube\](.*?)\[\/youtube\]/gis, '');
  }
  // Video
  const buildVideoEmbed = (raw) => {
    const style = 'max-width: 720px; width: 100%; margin: 6px auto;';
    try {
      const u = new URL(String(raw || '').trim());
      const host = u.hostname.toLowerCase();
      if ((host.includes('youtu.be') || host.includes('youtube.com'))) {
        if (opts.disableYoutube) return '';
        const id = extractYouTubeId(raw);
        if (!id) return '';
        const src = `https://www.youtube.com/embed/${id}`;
        return `<div class="ratio ratio-16x9" style="${style}"><iframe src="${src}" title="YouTube" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" allowfullscreen style="width:100%; height:100%;"></iframe></div>`;
      }
      if (host.includes('vimeo.com')) {
        const pathParts = u.pathname.split('/').filter(Boolean);
        const id = pathParts[pathParts.length - 1];
        const src = id ? `https://player.vimeo.com/video/${id}` : u.toString();
        return `<div class="ratio ratio-16x9" style="${style}"><iframe src="${src}" title="Vimeo" frameborder="0" allow="autoplay; fullscreen; picture-in-picture" allowfullscreen style="width:100%; height:100%;"></iframe></div>`;
      }
      const safeUrl = u.toString().replace(/\"/g, '&quot;');
      const vStyle = 'max-width: 720px; width: 100%; max-height: 405px; border-radius: 6px;';
      return `<video src="${safeUrl}" controls style="${vStyle}"></video>`;
    } catch (_) {
      return '';
    }
  };
  s = s.replace(/\[video\](.*?)\[\/video\]/gis, (m, p1) => buildVideoEmbed(p1));
  // Tabel
  s = s.replace(/\[table\](.*?)\[\/table\]/gis, (m, body) => {
    let inner = body || '';
    inner = inner
      .replace(/\[th\](.*?)\[\/th\]/gis, (mm, cell) => `<th style="border:1px solid #dee2e6; padding:6px;">${cell}</th>`)
      .replace(/\[td\](.*?)\[\/td\]/gis, (mm, cell) => `<td style="border:1px solid #dee2e6; padding:6px;">${cell}</td>`)
      .replace(/\[tr\](.*?)\[\/tr\]/gis, (mm, row) => `<tr>${row}</tr>`);
    const tblStyle = 'border-collapse: collapse; width: 100%; margin: 6px 0;';
    return `<table style="${tblStyle}"><tbody>${inner}</tbody></table>`;
  });
  // Konversi newline agar preview mengikuti enter/linebreak
  s = s.replace(/\r?\n/g, '<br/>');
  return s;
};

// Render campuran teks + LaTeX + gambar inline (![alt](url)) — pramuat BBCode
function renderMixed(str, opts = {}){
  if (!str) return <span className="text-muted">Masukkan teks / LaTeX untuk preview</span>;
  const imgStyle = opts.imgStyle || { maxWidth: '100%', maxHeight: '240px', objectFit: 'contain', borderRadius: 6, margin: '6px 0' };
  const processed = bbcodeToHtml(str, { ...opts, imgStyle });
  const parts = [];
  const regex = /!\[[^\]]*\]\([^\)]+\)/g;
  let lastIndex = 0; let idx = 0; let m;
  while ((m = regex.exec(processed))){
    const before = processed.slice(lastIndex, m.index);
    if (before){
      parts.push(<React.Fragment key={`part-${idx}`}>{renderTextOrLatex(before, `part-${idx}`)}</React.Fragment>);
      idx++;
    }
    const token = m[0];
    const url = ((token.match(/\(([^\)]+)\)/) || [])[1] || '').trim();
    const altRaw = ((token.match(/!\[([^\]]*)\]/) || [])[1] || '').trim();
    // Ambil semua w=xx% dan gunakan yang terakhir untuk mengatasi duplikasi
    const widthMatches = altRaw.match(/\bw\s*=\s*(\d{1,3})\s*%/gi);
    let widthPct = null;
    if (widthMatches && widthMatches.length > 0) {
      const lastMatch = widthMatches[widthMatches.length - 1];
      const mW = lastMatch.match(/\bw\s*=\s*(\d{1,3})\s*%/i);
      widthPct = mW ? Math.max(5, Math.min(160, Number(mW[1]))) : null;
    }
    const altBase = String(altRaw.replace(/\|.*$/, '') || 'img');
    if (url){
      const full = resolveImageUrl(url);
      const style = widthPct ? { ...imgStyle, width: `${widthPct}%` } : imgStyle;
      parts.push(
        <img
          key={`part-${idx++}`}
          src={full}
          alt={altBase}
          style={style}
          onError={(e) => {
            const clean = String(url || '').trim().replace(/^\/+/, '');
            const base = String(BACKEND_BASE || '').replace(/\/+$/, '');
            let alt = null;
            if (/^images\//.test(clean)) alt = `${base}/storage/${clean}`;
            else if (/^storage\//.test(clean)) alt = `${base}/${clean.replace(/^storage\//,'')}`;
            if (alt) { e.currentTarget.onerror = null; e.currentTarget.src = alt; }
          }}
        />
      );
    }
    lastIndex = m.index + token.length;
  }
  const rest = processed.slice(lastIndex);
  if (rest){
    parts.push(<React.Fragment key={`part-${idx}`}>{renderTextOrLatex(rest, `part-${idx}`)}</React.Fragment>);
    idx++;
  }
  return <>{parts}</>;
}

function renderTextOrLatex(s, key){
  if (!s || typeof s !== 'string') {
    return <span key={key} />;
  }
  // Util lokal: segmentasi LaTeX dan bantuan render
  const splitLatexTokens = (str) => {
    if (!str || typeof str !== 'string') return [{ type: 'text', value: '' }];
    const regex = /(\$\$[\s\S]*?\$\$|\\\[[\s\S]*?\\\]|\\\([\s\S]*?\\\)|\$[^$]*\$)/g;
    const segments = []; let lastIndex = 0; let m;
    while ((m = regex.exec(str))) {
      const before = str.slice(lastIndex, m.index);
      if (before) segments.push({ type: 'text', value: before });
      const token = m[0];
      if (token.startsWith('$$') || token.startsWith('\\[')) segments.push({ type: 'latex_block', value: token });
      else segments.push({ type: 'latex_inline', value: token });
      lastIndex = m.index + token.length;
    }
    const tail = str.slice(lastIndex);
    if (tail) segments.push({ type: 'text', value: tail });
    return segments;
  };
  const stripLatexDelimiters = (token) => {
    if (!token) return '';
    if (token.startsWith('$$')) return token.replace(/^\$\$/, '').replace(/\$\$$/, '');
    if (token.startsWith('$')) return token.replace(/^\$/, '').replace(/\$$/, '');
    if (token.startsWith('\\[')) return token.replace(/^\\\[/, '').replace(/\\\]$/, '');
    if (token.startsWith('\\(')) return token.replace(/^\\\(/, '').replace(/\\\)$/, '');
    return token;
  };
  const decodeHtmlEntities = (input) => {
    return String(input || '')
      .replace(/&amp;/g, '&')
      .replace(/&lt;/g, '<')
      .replace(/&gt;/g, '>')
      .replace(/&quot;/g, '"')
      .replace(/&#039;/g, "'")
      .replace(/&#39;/g, "'");
  };
  const cleanLatex = (input) => sanitizeLatex(input);
  const autoWrapLatexIfMissing = (v) => {
    const str = String(v || '');
    const hasDelim = /(\$\$[\s\S]*?\$\$|\\\[[\s\S]*?\\\]|\\\([\s\S]*?\\\)|\$[^$]*\$)/.test(str);
    const looksLatex = /\\(dfrac|frac|sqrt|cos|sin|tan|left|right|int|sum|prod|partial|nabla|pi|theta|alpha|beta|gamma|lim|bmatrix|pmatrix|begin|end|cdot|times|div|le|ge|neq|approx|equiv|infty|angle)/.test(str);
    return (!hasDelim && looksLatex) ? `\\(${str}\\)` : str;
  };

  const parts = [];
  const input = autoWrapLatexIfMissing(s);
  const segments = splitLatexTokens(input);
  let idx = 0;
  for (const seg of segments) {
    if (seg.type === 'text') {
      parts.push(
        <span key={`${key}-text-${idx++}`} style={{ whiteSpace: 'inherit' }} dangerouslySetInnerHTML={{ __html: seg.value }} />
      );
    } else if (seg.type === 'latex_inline') {
      const math = cleanLatex(decodeHtmlEntities(stripLatexDelimiters(seg.value)));
      parts.push(
        <span key={`${key}-latex-${idx++}`} style={{ direction: 'ltr', unicodeBidi: 'isolate' }}>
          <InlineMath errorColor="#cc0000" throwOnError={false}>{math}</InlineMath>
        </span>
      );
    } else if (seg.type === 'latex_block') {
      const math = cleanLatex(decodeHtmlEntities(stripLatexDelimiters(seg.value)));
      parts.push(
        <div key={`${key}-block-${idx++}`} style={{ direction: 'ltr', unicodeBidi: 'isolate' }}>
          <BlockMath errorColor="#cc0000" throwOnError={false}>{math}</BlockMath>
        </div>
      );
    }
  }
  if (parts.length === 0) {
    return <span key={key} style={{ whiteSpace: 'inherit' }} dangerouslySetInnerHTML={{ __html: s }} />;
  }
  return <>{parts}</>;
}

function renderMathOnly(s, key = 'math-only'){
  if (!s || typeof s !== 'string') {
    return <span key={key} />;
  }
  const decodeHtmlEntitiesLocal = (input) => {
    return String(input || '')
      .replace(/&amp;/g, '&')
      .replace(/&lt;/g, '<')
      .replace(/&gt;/g, '>')
      .replace(/&quot;/g, '"')
      .replace(/&#039;/g, "'")
      .replace(/&#39;/g, "'");
  };
  const cleanLatexLocal = (input) => sanitizeLatex(input);
  const stripLatexDelimitersLocal = (token) => {
    const s = String(token || '');
    if (s.startsWith('$$') && s.endsWith('$$')) return s.slice(2, -2);
    if (s.startsWith('\\[') && s.endsWith('\\]')) return s.slice(2, -2);
    if (s.startsWith('\\(') && s.endsWith('\\)')) return s.slice(2, -2);
    if (s.startsWith('$') && s.endsWith('$')) return s.slice(1, -1);
    return s;
  };
  const autoWrapLatexIfMissingLocal = (v) => {
    const str = String(v || '');
    const hasDelim = /(\$\$[\s\S]*?\$\$|\\\[[\s\S]*?\\\]|\\\([\s\S]*?\\\)|\$[^$]*\$)/.test(str);
    const looksLatex = /\\(dfrac|frac|sqrt|cos|sin|tan|left|right|int|sum|prod|partial|nabla|pi|theta|alpha|beta|gamma|lim|bmatrix|pmatrix|begin|end|cdot|times|div|le|ge|neq|approx|equiv|infty|angle)/.test(str);
    return (!hasDelim && looksLatex) ? `\\(${str}\\)` : str;
  };
  const wrapped = autoWrapLatexIfMissingLocal(s);
  const isBlock = /(\$\$[\s\S]*\$\$|\\\[[\s\S]*\\\])/m.test(wrapped);
  const inner = stripLatexDelimitersLocal(wrapped);
  const math = cleanLatexLocal(decodeHtmlEntitiesLocal(inner));
  const wrapperStyle = { direction: 'ltr', unicodeBidi: 'isolate' };
  return isBlock
    ? <div key={`${key}-block`} style={wrapperStyle}><BlockMath errorColor="#cc0000" throwOnError={false}>{math}</BlockMath></div>
    : <span key={`${key}-inline`} style={wrapperStyle}><InlineMath errorColor="#cc0000" throwOnError={false}>{math}</InlineMath></span>;
}

// MathToolbar dihapus: Desmos/GeoGebra tidak didukung lagi

function FormattingToolbar({ onWrap, onInsert, allowTable = true, allowYoutube = true }){
  if (!onWrap) return null;
  const insert = (s) => onInsert && onInsert(s);
  const insertTable = async () => {
    const { value: formValues } = await Swal.fire({
      title: 'Sisipkan Tabel',
      html: `
        <div class="mb-2 text-start">
          <label class="form-label">Jumlah Kolom</label>
          <input id="tbl-cols" type="number" min="1" value="2" class="form-control" />
        </div>
        <div class="mb-2 text-start">
          <label class="form-label">Jumlah Baris</label>
          <input id="tbl-rows" type="number" min="1" value="2" class="form-control" />
        </div>
      `,
      focusConfirm: false,
      showCancelButton: true,
      confirmButtonText: 'Sisipkan',
      cancelButtonText: 'Batal',
      preConfirm: () => {
        const cols = parseInt(document.getElementById('tbl-cols').value, 10);
        const rows = parseInt(document.getElementById('tbl-rows').value, 10) || 2;
        if (!cols || cols < 1) {
          Swal.showValidationMessage('Kolom minimal 1');
          return false;
        }
        return { cols, rows };
      }
    });
    if (!formValues) return;
    const { cols, rows } = formValues;
    const header = Array.from({ length: cols }, (_, i) => `<th>Kolom ${i+1}</th>`).join('');
    const bodyRows = Array.from({ length: rows }, () => `<tr>${Array.from({ length: cols }, () => '<td>-</td>').join('')}</tr>`).join('');
    insert(`<table border="1" style="border-collapse:collapse"><tr>${header}</tr>${bodyRows}</table>`);
  };
  const insertYoutube = async () => {
    const { value: link } = await Swal.fire({
      title: 'Sisipkan Video YouTube',
      input: 'text',
      inputLabel: 'Masukkan URL atau ID video',
      inputPlaceholder: 'contoh: https://youtu.be/VIDEO_ID atau VIDEO_ID',
      showCancelButton: true,
      confirmButtonText: 'Sisipkan',
      cancelButtonText: 'Batal',
      preConfirm: (val) => {
        const s = String(val || '').trim();
        if (!s) { Swal.showValidationMessage('Tautan/ID tidak boleh kosong'); return false; }
        if (/^[A-Za-z0-9_-]{11}$/.test(s)) return s;
        try {
          const u = new URL(s);
          if (u.hostname.includes('youtu.be') || u.hostname.includes('youtube.com')) return s;
        } catch (_) {}
        Swal.showValidationMessage('Masukkan URL YouTube yang valid atau ID 11-karakter');
        return false;
      }
    });
    if (!link) return;
    insert(`[youtube]${String(link).trim()}[/youtube]`);
  };
  return (
    <div className="d-flex flex-wrap gap-1 align-items-center">
      <button type="button" className="btn btn-light btn-sm" onClick={() => onWrap('b')}>Bold</button>
      <button type="button" className="btn btn-light btn-sm" onClick={() => onWrap('i')}>Italic</button>
      <button type="button" className="btn btn-light btn-sm" onClick={() => onWrap('u')}>Underline</button>
      <span className="ms-2 text-muted">•</span>
      <button type="button" className="btn btn-light btn-sm" onClick={() => insert('[left][/left]')}>Rata Kiri</button>
      <button type="button" className="btn btn-light btn-sm" onClick={() => insert('[center][/center]')}>Rata Tengah</button>
      <button type="button" className="btn btn-light btn-sm" onClick={() => insert('[right][/right]')}>Rata Kanan</button>
      <span className="ms-2 text-muted">•</span>
      <button type="button" className="btn btn-light btn-sm" onClick={() => insert('[br/]')}>Baris Baru</button>
      <span className="ms-2 text-muted">•</span>
      {allowTable && (
        <button type="button" className="btn btn-light btn-sm" onClick={insertTable}>Sisipkan Tabel</button>
      )}
      {allowYoutube && (
        <button type="button" className="btn btn-light btn-sm" onClick={insertYoutube}>YouTube</button>
      )}
    </div>
  );
}

export default function EssayKeys() {
  const listRef = useRef(null);
  const previewRef = useRef(null);
  const answerPreviewRef = useRef(null);

  const { subjectId } = useParams();
  const location = useLocation();
  const [subjectName, setSubjectName] = useState('');
  const [list, setList] = useState([]);
  const [loading, setLoading] = useState(true);
  const [showForm, setShowForm] = useState(false);
  const [filterText, setFilterText] = useState('');
  const [editing, setEditing] = useState(null);
  const [form, setForm] = useState({ text: '', answer: '' });
  const [arabicText, setArabicText] = useState(false);
  const [arabicAnswer, setArabicAnswer] = useState(false);
  const textRef = useRef(null); // legacy
  const answerRef = useRef(null); // legacy
  const richTextRef = useRef(null);
  const richAnswerRef = useRef(null);

  // Auto-render KaTeX pada preview soal saat form.text berubah
  useEffect(() => {
    if (previewRef.current) {
      renderMathInElement(previewRef.current, {
        delimiters: [
          { left: "\\(", right: "\\)", display: false },
          { left: "\\[", right: "\\]", display: true },
          { left: "$", right: "$", display: false },
          { left: "$$", right: "$$", display: true }
        ],
        throwOnError: false
      });
    }
  }, [form.text]);

  // Auto-render KaTeX pada preview jawaban saat form.answer berubah
  useEffect(() => {
    if (answerPreviewRef?.current) {
      renderMathInElement(answerPreviewRef.current, {
        delimiters: [
          { left: "\\(", right: "\\)", display: false },
          { left: "\\[", right: "\\]", display: true },
          { left: "$", right: "$", display: false },
          { left: "$$", right: "$$", display: true }
        ],
        throwOnError: false
      });
    }
  }, [form.answer]);

  // Auto-buka form edit jika datang dari Questions.jsx dengan state editingEssay
  useEffect(() => {
    const st = location?.state;
    const row = st && st.editingEssay;
    if (row && row.id) {
      setEditing(row);
      setForm({ text: row.text || '', answer: row.key_answer || row.option_a || '' });
      setShowForm(true);
    }
  }, [location?.state]);

  // MathLive untuk Soal Essay
  const [showMathLiveText, setShowMathLiveText] = useState(false);
  const mathLiveTextContainerRef = useRef(null);
  const mathLiveTextFieldRef = useRef(null);
  const [mathLiveTextError, setMathLiveTextError] = useState('');

  // MathLive untuk Jawaban Essay
  const [showMathLiveAnswer, setShowMathLiveAnswer] = useState(false);
  const mathLiveAnswerContainerRef = useRef(null);
  const mathLiveAnswerFieldRef = useRef(null);
  const [mathLiveAnswerError, setMathLiveAnswerError] = useState('');

  // Simbol LaTeX cepat
  const LATEX_SYMBOLS = [
    { label: 'α', code: '\\alpha' },
    { label: 'β', code: '\\beta' },
    { label: 'γ', code: '\\gamma' },
    { label: 'θ', code: '\\theta' },
    { label: 'π', code: '\\pi' },
    { label: 'Δ', code: '\\Delta' },
    { label: 'Ω', code: '\\Omega' },
    { label: '√', code: '\\sqrt{x}' },
    { label: '∠', code: '\\angle' },
    { label: '∥', code: '\\parallel' },
    { label: '⟂', code: '\\perp' },
    { label: '≤', code: '\\le' },
    { label: '≥', code: '\\ge' },
    { label: '≠', code: '\\neq' },
    { label: '≈', code: '\\approx' },
    { label: '≡', code: '\\equiv' },
    { label: '∞', code: '\\infty' },
    { label: '×', code: '\\times' },
    { label: '÷', code: '\\div' },
    { label: '·', code: '\\cdot' },
    { label: '°', code: '^\\circ' },
    // Kalkulus
    { label: '∫', code: '\\int' },
    { label: '∮', code: '\\oint' },
    { label: '∑', code: '\\sum' },
    { label: '∏', code: '\\prod' },
    { label: '∂', code: '\\partial' },
    { label: '∇', code: '\\nabla' },
    { label: 'lim', code: '\\lim_{x\\to a}' },
    { label: 'd/dx', code: '\\frac{d}{dx}' },
    { label: '∂/∂x', code: '\\frac{\\partial}{\\partial x}' },
  ];

  const insertToText = (snippet) => {
    if (richTextRef.current && typeof richTextRef.current.insertSnippet === 'function') {
      richTextRef.current.insertSnippet(snippet);
    } else {
      insertAtCursor('text', snippet);
    }
  };
  const insertToAnswer = (snippet) => {
    if (richAnswerRef.current && typeof richAnswerRef.current.insertSnippet === 'function') {
      richAnswerRef.current.insertSnippet(snippet);
    } else {
      insertAtCursor('answer', snippet);
    }
  };

  const insertLatexSymbolText = (code) => insertToText(` \\(${code}\\) `);
  const insertLatexSymbolAnswer = (code) => insertToAnswer(` \\(${code}\\) `);

  const insertFromMathLiveText = () => {
    const mf = mathLiveTextFieldRef.current;
    let latex = '';
    try { latex = mf && typeof mf.getValue === 'function' ? mf.getValue('latex-expanded') : ''; } catch(_){}
    const s = String(latex || '').trim();
    if (!s) return;
    insertToText(` \\(${s}\\) `);
    setShowMathLiveText(false);
  };
  const insertFromMathLiveAnswer = () => {
    const mf = mathLiveAnswerFieldRef.current;
    let latex = '';
    try { latex = mf && typeof mf.getValue === 'function' ? mf.getValue('latex-expanded') : ''; } catch(_){}
    const s = String(latex || '').trim();
    if (!s) return;
    insertToAnswer(` \\(${s}\\) `);
    setShowMathLiveAnswer(false);
  };

  useEffect(() => {
    let disposed = false;
    const setup = async () => {
      if (!showMathLiveText) return;
      try {
        const mod = await import('mathlive');
        if (disposed) return;
        const el = new mod.MathfieldElement({
          // Hanya opsi yang masih valid
        });
        // Set opsi deprecated setelah constructor
        el.smartMode = true;
        // keypressSound: akses via el.mathVirtualKeyboard atau window.mathVirtualKeyboard
        try {
          if (el.mathVirtualKeyboard && typeof el.mathVirtualKeyboard === 'object') {
            if ('keypressSound' in el.mathVirtualKeyboard) {
              el.mathVirtualKeyboard.keypressSound = 'none';
            }
          } else if (typeof window !== 'undefined' && window.mathVirtualKeyboard && typeof window.mathVirtualKeyboard === 'object') {
            if ('keypressSound' in window.mathVirtualKeyboard) {
              window.mathVirtualKeyboard.keypressSound = 'none';
            }
          }
        } catch (_) { /* ignore */ }
        el.mathVirtualKeyboardPolicy = 'onfocus';
        el.style.minHeight = '40px';
        el.style.border = '1px solid #ced4da';
        el.style.borderRadius = '0.25rem';
        el.style.padding = '6px 8px';
        el.style.width = '100%';
        if (mathLiveTextContainerRef.current) {
          mathLiveTextContainerRef.current.innerHTML = '';
          mathLiveTextContainerRef.current.appendChild(el);
        }
        mathLiveTextFieldRef.current = el;
        setMathLiveTextError('');
      } catch (err) {
        setMathLiveTextError('Gagal memuat MathLive');
      }
    };
    setup();
    return () => { disposed = true; };
  }, [showForm, showMathLiveText]);

  useEffect(() => {
    let disposed = false;
    const setup = async () => {
      if (!showMathLiveAnswer) return;
      try {
        const mod = await import('mathlive');
        if (disposed) return;
        const el = new mod.MathfieldElement({
          // Hanya opsi yang masih valid
        });
        // Set opsi deprecated setelah constructor
        el.smartMode = true;
        // keypressSound: akses via el.mathVirtualKeyboard atau window.mathVirtualKeyboard
        try {
          if (el.mathVirtualKeyboard && typeof el.mathVirtualKeyboard === 'object') {
            if ('keypressSound' in el.mathVirtualKeyboard) {
              el.mathVirtualKeyboard.keypressSound = 'none';
            }
          } else if (typeof window !== 'undefined' && window.mathVirtualKeyboard && typeof window.mathVirtualKeyboard === 'object') {
            if ('keypressSound' in window.mathVirtualKeyboard) {
              window.mathVirtualKeyboard.keypressSound = 'none';
            }
          }
        } catch (_) { /* ignore */ }
        el.mathVirtualKeyboardPolicy = 'onfocus';
        el.style.minHeight = '40px';
        el.style.border = '1px solid #ced4da';
        el.style.borderRadius = '0.25rem';
        el.style.padding = '6px 8px';
        el.style.width = '100%';
        if (mathLiveAnswerContainerRef.current) {
          mathLiveAnswerContainerRef.current.innerHTML = '';
          mathLiveAnswerContainerRef.current.appendChild(el);
        }
        mathLiveAnswerFieldRef.current = el;
        setMathLiveAnswerError('');
      } catch (err) {
        setMathLiveAnswerError('Gagal memuat MathLive');
      }
    };
    setup();
    return () => { disposed = true; };
  }, [showForm, showMathLiveAnswer]);

  const tokenHeader = { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } };

  useEffect(() => {
    const fetchSubjectName = async () => {
      try {
        const res = await api.get(`/subjects/${subjectId}`, tokenHeader);
        setSubjectName(res.data?.name || '');
      } catch (e) {
        setSubjectName('');
      }
    };
    fetchSubjectName();
  }, [subjectId]);

  const fetchList = async () => {
    setLoading(true);
    try {
      const res = await api.get('/questions', { params: { subject_id: subjectId } });
      const rows = res.data?.data ?? res.data ?? [];
      // Helper: treat '-' as empty for essay detection
      const isBlankOrDash = (s) => {
        const t = String(s || '').trim();
        return t === '' || t === '-';
      };
      // Anggap 'essay' jika opsi B–D kosong atau berisi '-'
      const essays = rows.filter(q => (
        isBlankOrDash(q?.option_b) && isBlankOrDash(q?.option_c) && isBlankOrDash(q?.option_d)
      ));
      setList(essays);
    } catch (e) {
      console.error(e);
      Swal.fire('Error', 'Gagal memuat kunci jawaban essay', 'error');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => { fetchList(); }, [subjectId]);

  const filtered = useMemo(() => {
    if (!filterText) return list;
    const q = filterText.toLowerCase();
    return list.filter(item => (item?.text || '').toLowerCase().includes(q) || (item?.key_answer || item?.option_a || '').toLowerCase().includes(q));
  }, [filterText, list]);

  // Auto-render KaTeX pada daftar ketika hasil filter berubah
  useEffect(() => {
    if (listRef.current) {
      renderMathInElement(listRef.current, {
        delimiters: [
          { left: "\\(", right: "\\)", display: false },
          { left: "\\[", right: "\\]", display: true },
          { left: "$", right: "$", display: false },
          { left: "$$", right: "$$", display: true }
        ],
        throwOnError: false
      });
    }
  }, [filtered]);

  const openAdd = () => {
    setEditing(null);
    setForm({ text: '', answer: '' });
    setShowForm(true);
  };

  const openEdit = (row) => {
    setEditing(row);
    setForm({ text: row?.text || '', answer: row?.key_answer || row?.option_a || '' });
    setShowForm(true);
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    const text = form.text.trim();
    const answer = form.answer.trim();
    if (!text || !answer) return;
    try {
      const fd = new FormData();
      fd.append('text', text);
      // Simpan kunci essay di key_answer; isi '-' untuk opsi agar lolos validasi
      fd.append('option_a', '-');
      fd.append('option_b', '-');
      fd.append('option_c', '-');
      fd.append('option_d', '-');
      fd.append('option_e', '-');
      fd.append('key_answer', answer);
      fd.append('subject_id', String(subjectId));

      let res;
      if (editing) {
        res = await api.put(`/questions/${editing.id}`, fd, { headers: { 'Content-Type': 'multipart/form-data' } });
        setList(prev => prev.map(q => q.id === editing.id ? res.data : q));
        Swal.fire('Berhasil', 'Essay diperbarui', 'success');
      } else {
        res = await api.post('/questions', fd, { headers: { 'Content-Type': 'multipart/form-data' } });
        setList(prev => [res.data, ...prev]);
        Swal.fire('Berhasil', 'Essay ditambahkan', 'success');
      }
      setShowForm(false);
      setEditing(null);
      setForm({ text: '', answer: '' });
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', err.response?.data?.message || 'Gagal menyimpan essay', 'error');
    }
  };

  const wrapSelection = (which, tag) => {
    const el = which === 'text' ? textRef.current : answerRef.current;
    const open = `[${tag}]`; const close = `[/${tag}]`;
    const curr = which === 'text' ? form.text : form.answer;
    if (!el || typeof el.selectionStart !== 'number') {
      const next = (curr || '') + open + close;
      setForm(prev => ({ ...prev, [which]: next }));
      return;
    }
    const start = el.selectionStart ?? curr.length;
    const end = el.selectionEnd ?? curr.length;
    const sel = (curr || '').slice(start, end);
    const before = (curr || '').slice(0, start);
    const after = (curr || '').slice(end);
    const next = before + open + sel + close + after;
    setForm(prev => ({ ...prev, [which]: next }));
    setTimeout(() => {
      const target = which === 'text' ? textRef.current : answerRef.current;
      if (target) {
        const pos = start + open.length + sel.length + close.length;
        target.selectionStart = pos;
        target.selectionEnd = pos;
        target.focus();
      }
    }, 0);
  };

  // Sisipkan teks pada posisi kursor
  const insertAtCursor = (which, insertText) => {
    const el = which === 'text' ? textRef.current : answerRef.current;
    const curr = which === 'text' ? form.text : form.answer;
    if (!el || typeof el.selectionStart !== 'number') {
      const next = (curr || '') + insertText;
      setForm(prev => ({ ...prev, [which]: next }));
      return;
    }
    const start = el.selectionStart ?? curr.length;
    const end = el.selectionEnd ?? curr.length;
    const before = (curr || '').slice(0, start);
    const after = (curr || '').slice(end);
    const next = before + insertText + after;
    setForm(prev => ({ ...prev, [which]: next }));
    setTimeout(() => {
      const target = which === 'text' ? textRef.current : answerRef.current;
      if (target) {
        const pos = start + insertText.length;
        target.selectionStart = pos;
        target.selectionEnd = pos;
        target.focus();
      }
    }, 0);
  };

  const handleInlineImageUpload = async (which, file) => {
    if (!file) return;
    try {
      const fd = new FormData();
      fd.append('image', file);
      fd.append('subject_id', String(subjectId));
      const res = await api.post('/questions/upload-image', fd, {
        headers: { 'Content-Type': 'multipart/form-data', Authorization: `Bearer ${localStorage.getItem('token')}` }
      });
      const path = res?.data?.path || '';
      if (!path) {
        Swal.fire('Gagal', 'Upload gambar gagal', 'error');
        return;
      }
      const wRes = await Swal.fire({
        title: 'Ukuran gambar (%)',
        input: 'range',
        inputAttributes: { min: 5, max: 160, step: 1 },
        inputValue: 100,
        showCancelButton: true,
        confirmButtonText: 'OK',
        cancelButtonText: 'Batal'
      });
      const w = Math.max(5, Math.min(160, Number(wRes.isConfirmed ? wRes.value : 100)));
      const ref = which === 'text' ? richTextRef.current : richAnswerRef.current;
      if (ref && typeof ref.insertImageWithOptions === 'function') {
        ref.insertImageWithOptions(path, w);
      } else if (ref && typeof ref.insertImage === 'function') {
        ref.insertImage(path);
      } else {
        insertAtCursor(which, ` ![img|w=${w}%](${path}) `);
      }
    } catch (e) {
      console.error(e);
      Swal.fire('Gagal', 'Tidak bisa mengunggah gambar', 'error');
    }
  };

  // Wrapper: pilih file gambar dan panggil handleInlineImageUpload untuk Soal
  const choosePictureForEssayQuestion = () => {
    const input = document.createElement('input');
    input.type = 'file';
    input.accept = 'image/*';
    input.onchange = (ev) => {
      const file = ev.target && ev.target.files ? ev.target.files[0] : null;
      handleInlineImageUpload('text', file);
    };
    input.click();
  };

  // Wrapper: pilih file gambar dan panggil handleInlineImageUpload untuk Jawaban
  const choosePictureForEssayAnswer = () => {
    const input = document.createElement('input');
    input.type = 'file';
    input.accept = 'image/*';
    input.onchange = (ev) => {
      const file = ev.target && ev.target.files ? ev.target.files[0] : null;
      handleInlineImageUpload('answer', file);
    };
    input.click();
  };

  const handleDelete = async (row) => {
    const confirm = await Swal.fire({
      title: 'Hapus Essay?',
      text: (row?.text || '')?.slice(0, 80),
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Ya, hapus',
      cancelButtonText: 'Batal'
    });
    if (!confirm.isConfirmed) return;
    try {
      await api.delete(`/questions/${row.id}`, tokenHeader);
      setList(prev => prev.filter(q => q.id !== row.id));
      Swal.fire('Berhasil', 'Essay dihapus', 'success');
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', err.response?.data?.message || 'Gagal menghapus essay', 'error');
    }
  };

  if (loading) return <div className="text-center mt-5">Memuat kunci jawaban essay...</div>;

  return (
    <div className="container mt-4">
      <h3>Kunci Jawaban Essay — {subjectName || 'Mata Pelajaran'}</h3>

      <div className="d-flex justify-content-between align-items-center mb-3 gap-2 flex-wrap">
        <div className="d-flex gap-2 align-items-center">
          {!showForm ? (
            <button className="btn btn-primary" onClick={openAdd}>+ Essay</button>
          ) : (
            <button
              className="btn btn-warning"
              onClick={() => { setShowForm(false); setEditing(null); setForm({ text: '', answer: '' }); }}
            >Tutup Form</button>
          )}
        </div>

        <input
          type="text"
          placeholder="Cari essay..."
          className="form-control"
          style={{ maxWidth: '300px' }}
          value={filterText}
          onChange={e => setFilterText(e.target.value)}
        />
      </div>

      {showForm && (
        <form onSubmit={handleSubmit} className="card card-body mb-3">
          <div className="mb-3">
            <div className="d-flex justify-content-between align-items-center">
              <label className="form-label mb-0">Soal Essay</label>
              <div className="form-check">
                <input className="form-check-input" type="checkbox" id="arabicText" checked={arabicText} onChange={e => setArabicText(e.target.checked)} />
                <label className="form-check-label" htmlFor="arabicText">Mode Arab</label>
              </div>
            </div>
            <RichTextEditor
              ref={richTextRef}
              value={form.text}
              onChange={(v) => setForm(prev => ({ ...prev, text: v }))}
              className="form-control"
              style={arabicText ? { minHeight: 120, textAlign: 'right', fontFamily: 'Noto Naskh Arabic, Amiri, serif' } : { minHeight: 120 }}
              dir={arabicText ? 'rtl' : 'auto'}
              placeholder="Tulis soal essay. Pilih teks lalu klik Bold/Italic/Underline."
            />
            <div className="mt-2 d-flex flex-wrap gap-2 align-items-center">
              <span className="text-muted">Format:</span>
              <FormattingToolbar allowTable={true} allowYoutube={true} onWrap={(tag) => {
                if (richTextRef.current && typeof richTextRef.current.applyFormat === 'function') {
                  richTextRef.current.applyFormat(tag);
                } else {
                  wrapSelection('text', tag);
                }
              }} onInsert={(snippet) => {
                if (richTextRef.current && typeof richTextRef.current.insertSnippet === 'function') {
                  richTextRef.current.insertSnippet(snippet);
                } else {
                  insertAtCursor('text', snippet);
                }
              }} />
            </div>
            <div className="mt-2 p-2 border rounded bg-light">
              <div className="d-flex align-items-center gap-2 flex-wrap">
                <span className="text-muted">Insert:</span>
                <button type="button" className="btn btn-outline-primary btn-sm rounded-pill" onClick={() => setShowMathLiveText(prev => !prev)} title="Tampilkan panel Equation untuk Soal">∑ Equation</button>
                <button type="button" className="btn btn-outline-primary btn-sm rounded-pill" onClick={choosePictureForEssayQuestion} title="Pilih & sisipkan gambar">🖼️ Picture</button>
                <span className="text-muted ms-2">Symbols:</span>
                <div className="d-flex align-items-center flex-wrap gap-1" style={{ maxWidth:'100%', overflowX:'auto' }}>
                  {LATEX_SYMBOLS.map((s) => (
                    <button
                      key={`text-${s.code}`}
                      type="button"
                      className="btn btn-outline-secondary btn-sm rounded-pill"
                      title={s.code}
                      onClick={() => insertLatexSymbolText(s.code)}
                    >{s.label}</button>
                  ))}
                </div>
              </div>
              {showMathLiveText && (
                <div className="mt-2">
                  <small className="text-muted">Equation (MathLive) — Soal:</small>
                  <div ref={mathLiveTextContainerRef} className="mt-1" />
                  <div className="mt-2 d-flex gap-2 align-items-center">
                    <button type="button" className="btn btn-primary btn-sm" onClick={insertFromMathLiveText}>Sisipkan</button>
                    <button type="button" className="btn btn-outline-secondary btn-sm" onClick={() => setShowMathLiveText(false)}>Tutup</button>
                    {mathLiveTextError && (
                      <span className="text-danger small">{mathLiveTextError}</span>
                    )}
                  </div>
                </div>
              )}
            </div>

            <div className="mt-3">
              <small className="text-muted d-block">Preview Soal:</small>
              <div ref={previewRef} className="p-2 border rounded" style={{ minHeight: 60, whiteSpace: 'pre-wrap' }}>{renderMixed(form.text)}</div>
            </div>
          </div>

          <div className="mb-3">
            <div className="d-flex justify-content-between align-items-center">
              <label className="form-label mb-0">Kunci Essay</label>
              <div className="form-check">
                <input className="form-check-input" type="checkbox" id="arabicAnswer" checked={arabicAnswer} onChange={e => setArabicAnswer(e.target.checked)} />
                <label className="form-check-label" htmlFor="arabicAnswer">Mode Arab</label>
              </div>
            </div>
            <RichTextEditor
              ref={richAnswerRef}
              value={form.answer}
              onChange={(v) => setForm(prev => ({ ...prev, answer: v }))}
              className="form-control"
              style={arabicAnswer ? { minHeight: 100, textAlign: 'right', fontFamily: 'Noto Naskh Arabic, Amiri, serif' } : { minHeight: 100 }}
              dir={arabicAnswer ? 'rtl' : 'auto'}
              placeholder="Tulis kunci jawaban essay di sini."
            />
            <div className="mt-2 d-flex flex-wrap gap-2 align-items-center">
              <span className="text-muted">Format:</span>
              <FormattingToolbar allowTable={false} allowYoutube={false} onWrap={(tag) => {
                if (richAnswerRef.current && typeof richAnswerRef.current.applyFormat === 'function') {
                  richAnswerRef.current.applyFormat(tag);
                } else {
                  wrapSelection('answer', tag);
                }
              }} onInsert={(snippet) => {
                if (richAnswerRef.current && typeof richAnswerRef.current.insertSnippet === 'function') {
                  richAnswerRef.current.insertSnippet(snippet);
                } else {
                  insertAtCursor('answer', snippet);
                }
              }} />
            </div>
            <div className="mt-2 p-2 border rounded bg-light">
              <div className="d-flex align-items-center gap-2 flex-wrap">
                <span className="text-muted">Insert:</span>
                <button type="button" className="btn btn-outline-primary btn-sm rounded-pill" onClick={() => setShowMathLiveAnswer(prev => !prev)} title="Tampilkan panel Equation untuk Jawaban">∑ Equation</button>
                <button type="button" className="btn btn-outline-primary btn-sm rounded-pill" onClick={() => {
                  const input = document.createElement('input');
                  input.type = 'file';
                  input.accept = 'image/*';
                  input.onchange = (ev) => {
                    const file = ev.target && ev.target.files ? ev.target.files[0] : null;
                    handleInlineImageUpload('answer', file);
                  };
                  input.click();
                }} title="Pilih & sisipkan gambar">🖼️ Picture</button>
                <span className="text-muted ms-2">Symbols:</span>
                <div className="d-flex align-items-center flex-wrap gap-1" style={{ maxWidth:'100%', overflowX:'auto' }}>
                  {LATEX_SYMBOLS.map((s) => (
                    <button
                      key={`ans-${s.code}`}
                      type="button"
                      className="btn btn-outline-secondary btn-sm rounded-pill"
                      title={s.code}
                      onClick={() => insertLatexSymbolAnswer(s.code)}
                    >{s.label}</button>
                  ))}
                </div>
              </div>
              {showMathLiveAnswer && (
                <div className="mt-2">
                  <small className="text-muted">Equation (MathLive) — Jawaban:</small>
                  <div ref={mathLiveAnswerContainerRef} className="mt-1" />
                  <div className="mt-2 d-flex gap-2 align-items-center">
                    <button type="button" className="btn btn-primary btn-sm" onClick={insertFromMathLiveAnswer}>Sisipkan</button>
                    <button type="button" className="btn btn-outline-secondary btn-sm" onClick={() => setShowMathLiveAnswer(false)}>Tutup</button>
                    {mathLiveAnswerError && (
                      <span className="text-danger small">{mathLiveAnswerError}</span>
                    )}
                  </div>
                </div>
              )}
            </div>
            <div className="mt-3">
              <small className="text-muted d-block">Preview Jawaban:</small>
              <div ref={answerPreviewRef} className="p-2 border rounded" style={{ minHeight: 60, whiteSpace: 'pre-wrap' }}>{renderMixed(form.answer)}</div>
            </div>
          </div>

          <div className="d-flex gap-2">
            <button type="submit" className="btn btn-success">Simpan</button>
            <button type="button" className="btn btn-secondary" onClick={() => { setShowForm(false); setEditing(null); setForm({ text: '', answer: '' }); }}>Tutup Form</button>
          </div>
        </form>
      )}

      {!showForm && (
        <div className="table-responsive" ref={listRef}>
          <table className="table table-striped">
            <thead>
              <tr>
                <th style={{width:'60px'}}>#</th>
                <th>Soal</th>
                <th>Kunci Jawaban</th>
                <th style={{width:'160px'}}>Aksi</th>
              </tr>
            </thead>
            <tbody>
              {filtered.map((row, idx) => (
                <tr key={row.id}>
                  <td>{idx + 1}</td>
                  <td style={{ whiteSpace: 'pre-wrap', verticalAlign: 'top' }}>
                      <div style={{ fontSize: '0.95rem', lineHeight: '1.4' }}>
                      <div>{renderMixed(row.text)}</div>
                      </div>
                  </td>
                  <td style={{ whiteSpace: 'pre-wrap', verticalAlign: 'top' }}>
                    <div style={{ fontSize: '0.95rem', lineHeight: '1.4' }}>
                      <div>{renderMixed(row.key_answer || row.option_a)}</div>
                    </div>
                  </td>
                  <td>
                    <div className="d-flex gap-2">
                      <button className="btn btn-sm btn-outline-primary" onClick={() => openEdit(row)}>Edit</button>
                      <button className="btn btn-sm btn-outline-danger" onClick={() => handleDelete(row)}>Hapus</button>
                    </div>
                  </td>
                </tr>
              ))}
              {filtered.length === 0 && (
                <tr>
                  <td colSpan={4} className="text-center text-muted">Belum ada essay</td>
                </tr>
              )}
            </tbody>
          </table>
        </div>
      )}
    </div>
  );
}

// Helper: konversi File ke data URL base64 untuk inline tanpa API
const readFileAsDataUrl = (file) => new Promise((resolve, reject) => {
  try {
    const reader = new FileReader();
    reader.onload = () => resolve(reader.result);
    reader.onerror = () => reject(new Error('FileReader failed'));
    reader.readAsDataURL(file);
  } catch (err) {
    reject(err);
  }
});