import React, { createContext, useContext, useEffect, useState } from 'react';
import api from '../utils/axios';

export const ParticipantAuthContext = createContext();

export const ParticipantAuthProvider = ({ children }) => {
  const [participant, setParticipant] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const stored = localStorage.getItem('participant');
    if (stored) setParticipant(JSON.parse(stored));
    setLoading(false);
  }, []);

  // Heartbeat ringan tiap 2 menit
  useEffect(() => {
    let id;
    if (participant) {
      const payload = participant?.id ? { id: participant.id } : { nisn: participant?.nisn };
      // Ping segera saat participant tersedia (tidak perlu menunggu 2 menit)
      api.post('/participant/update-activity', payload).catch(() => {});
      // Lanjutkan heartbeat periodik
      id = setInterval(() => {
        api.post('/participant/update-activity', payload).catch(() => {});
      }, 120000);
    }
    return () => { if (id) clearInterval(id); };
  }, [participant]);

  // Peringatan saat ingin menutup atau keluar halaman ketika peserta login
  useEffect(() => {
    const handleBeforeUnload = (e) => {
      e.preventDefault();
      e.returnValue = '';
    };
    if (participant) {
      window.addEventListener('beforeunload', handleBeforeUnload);
    }
    return () => {
      window.removeEventListener('beforeunload', handleBeforeUnload);
    };
  }, [participant]);

  const login = async (nisn, password, device_fp) => {
    try {
      const res = await api.post('/participant/login', { nisn, password, device_fp });
      const p = res.data?.participant;
      const token = res.data?.token;
      if (!p || !token) return { success: false, message: 'Login gagal' };
      localStorage.setItem('participantToken', token);
      localStorage.setItem('participant', JSON.stringify(p));
      setParticipant(p);
      return { success: true };
    } catch (e) {
      const msg = e.response?.data?.message || 'Login gagal, periksa NISN/password';
      return { success: false, message: msg };
    }
  };

  const loginToken = async (token, device_fp) => {
    try {
      const res = await api.post('/participant/login-token', { token, device_fp });
      const p = res.data?.participant;
      const t = res.data?.token;
      if (!p || !t) return { success: false, message: 'Login gagal' };
      localStorage.setItem('participantToken', t);
      localStorage.setItem('participant', JSON.stringify(p));
      setParticipant(p);
      return { success: true };
    } catch (e) {
      const msg = e.response?.data?.message || 'Login dengan token gagal';
      return { success: false, message: msg };
    }
  };

  const logout = async () => {
    try { await api.post('/participant/logout'); } catch (_) {}
    localStorage.removeItem('participantToken');
    localStorage.removeItem('participant');
    setParticipant(null);
  };

  return (
    <ParticipantAuthContext.Provider value={{ participant, login, loginToken, logout, loading }}>
      {children}
    </ParticipantAuthContext.Provider>
  );
};

export const useParticipantAuth = () => useContext(ParticipantAuthContext);