import React, { createContext, useState, useEffect, useContext } from "react";
import api from "../utils/axios";

export const AuthContext = createContext(); // ← tambahkan export di sini

export const AuthProvider = ({ children }) => {
  const [user, setUser] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const storedUser = localStorage.getItem("user");
    if (storedUser) setUser(JSON.parse(storedUser));
    setLoading(false);
  }, []);

  // Heartbeat: perbarui aktivitas tiap 1 menit bila login
  useEffect(() => {
    let intervalId;
    if (user) {
      intervalId = setInterval(() => {
        api.post('/update-activity').catch(() => {});
      }, 60000);
    }
    return () => {
      if (intervalId) clearInterval(intervalId);
    };
  }, [user]);

  const login = async (email, password) => {
    try {
      const response = await api.post("/login", { email, password });
      const { user: rawUser, token, permissions } = response.data;
      const user = { ...rawUser, permissions: permissions || [] };
      localStorage.setItem("token", token);
      localStorage.setItem("user", JSON.stringify(user));
      setUser(user);
      return { success: true };
    } catch (err) {
      console.error("Login failed", err);
      const errorMessage = err.response?.data?.message || "Login gagal, silakan coba lagi";
      return { success: false, message: errorMessage };
    }
  };

  const logout = () => {
    localStorage.removeItem("token");
    localStorage.removeItem("user");
    setUser(null);
  };

  // Memperbarui user di local state dan localStorage
  const updateUser = (partial) => {
    if (!user) return;
    const updated = { ...user, ...partial };
    localStorage.setItem("user", JSON.stringify(updated));
    setUser(updated);
  };

  // Mengambil ulang data user dari API /me untuk sinkronisasi roles & permissions
  const refreshUser = async () => {
    try {
      const res = await api.get('/me');
      const { user: rawUser, permissions } = res.data;
      const updated = { ...rawUser, permissions: permissions || [] };
      localStorage.setItem("user", JSON.stringify(updated));
      setUser(updated);
      return updated;
    } catch (e) {
      // Abaikan error refresh, tidak kritis
      return null;
    }
  };

  const hasRole = (roles) => {
    if (!user || !user.roles || !user.roles.length) return false;
    
    // Mengambil nama role dari user
    const userRoleNames = user.roles.map(role => role.name);
    
    return Array.isArray(roles)
      ? roles.some(role => userRoleNames.includes(role))
      : userRoleNames.includes(roles);
  };

  return (
    <AuthContext.Provider value={{ user, login, logout, loading, hasRole, updateUser, refreshUser }}>
      {children}
    </AuthContext.Provider>
  );
};

export const useAuth = () => useContext(AuthContext);
