import React, { useCallback, useMemo, useRef, useState } from 'react';
import { Rnd } from 'react-rnd';

/**
 * RndImageDropper
 * - Drag & Drop gambar (file atau URL) ke area ini
 * - Resize/drag menggunakan react-rnd
 * - Klik "Sisipkan ke teks" untuk memasukkan <img> ke editor sebagai HTML
 * - Lebar akan di-encode ke BBCode via alt: img|w=NN%
 */
export default function RndImageDropper({ onInsert, height = 260 }) {
  const containerRef = useRef(null);
  const [src, setSrc] = useState('');
  const [box, setBox] = useState({ width: 180, height: 120, x: 10, y: 10 });

  const onDrop = useCallback((e) => {
    e.preventDefault();
    e.stopPropagation();
    try {
      const dt = e.dataTransfer;
      if (dt && dt.files && dt.files.length > 0) {
        const f = dt.files[0];
        if (f && f.type && f.type.startsWith('image/')) {
          const reader = new FileReader();
          reader.onload = () => {
            setSrc(String(reader.result || ''));
          };
          reader.readAsDataURL(f);
          return;
        }
      }
      const text = dt.getData('text/uri-list') || dt.getData('text/plain') || '';
      const s = String(text || '').trim();
      if (s && /^(https?:|data:)/.test(s)) {
        setSrc(s);
        return;
      }
    } catch (_) {}
  }, []);

  const onDragOver = useCallback((e) => {
    e.preventDefault();
    e.stopPropagation();
  }, []);

  const reset = useCallback(() => {
    setSrc('');
    setBox({ width: 180, height: 120, x: 10, y: 10 });
  }, []);

  const insert = useCallback(() => {
    if (!src || typeof onInsert !== 'function') return;
    const wrap = containerRef.current;
    const containerWidth = wrap ? wrap.getBoundingClientRect().width : 1;
    const widthPct = Math.max(5, Math.min(160, Math.round((box.width / containerWidth) * 100)));
    const style = `width:${widthPct}%;height:auto;border-radius:6px;margin:6px 0`;
    const html = `<img src="${src}" alt="img|w=${widthPct}%" style="${style}" />`;
    onInsert(html);
    reset();
  }, [src, box.width, onInsert, reset]);

  const hint = useMemo(() => {
    if (!src) return 'Tarik & lepas file gambar atau URL (https/data:).';
    return 'Resize/drag gambar lalu klik "Sisipkan ke teks".';
  }, [src]);

  return (
    <div ref={containerRef}
         onDrop={onDrop}
         onDragOver={onDragOver}
         className="p-2 border rounded position-relative bg-light"
         style={{ minHeight: height }}>
      <div className="text-muted small mb-2">{hint}</div>
      {!src && (
        <div className="d-flex align-items-center justify-content-center text-muted" style={{minHeight: height-40}}>
          <span>Drop area</span>
        </div>
      )}
      {src && (
        <Rnd
          bounds="parent"
          size={{ width: box.width, height: box.height }}
          position={{ x: box.x, y: box.y }}
          onDragStop={(e, d) => setBox(prev => ({ ...prev, x: d.x, y: d.y }))}
          onResizeStop={(e, dir, ref, delta, pos) => {
            setBox({
              width: ref.offsetWidth,
              height: ref.offsetHeight,
              x: pos.x,
              y: pos.y
            });
          }}
          style={{ border: '1px dashed #9aa0a6', background: '#fff' }}
        >
          <img src={src} alt="preview" style={{ width: '100%', height: '100%', objectFit: 'contain', borderRadius: 6 }} />
        </Rnd>
      )}
      <div className="mt-2 d-flex gap-2">
        <button type="button" className="btn btn-primary btn-sm" disabled={!src} onClick={insert}>Sisipkan ke teks</button>
        <button type="button" className="btn btn-outline-secondary btn-sm" onClick={reset}>Reset</button>
      </div>
    </div>
  );
}