import React, { useMemo, useState, useEffect } from "react";
import { NavLink, Link } from "react-router-dom";
import { useAuth } from "../context/AuthContext.jsx";
import { canAccess } from "../utils/permissionHelper";
import api from "../utils/axios";
import Swal from "sweetalert2";
import { FaHome, FaUsers, FaUserShield, FaKey, FaListAlt, FaQuestionCircle, FaBook, FaSchool, FaLayerGroup, FaChalkboard, FaUserGraduate, FaCalendarAlt, FaSave, FaTimes, FaUserEdit } from "react-icons/fa";

const AdminLayout = ({ children }) => {
  const { user, logout, refreshUser } = useAuth();
  const initials = (user?.name || "").split(" ").map(w => w[0]).slice(0,2).join("").toUpperCase();

  // State modal edit profil
  const [showEdit, setShowEdit] = useState(false);
  const [saving, setSaving] = useState(false);
  const [profileForm, setProfileForm] = useState({ name: user?.name || "", email: user?.email || "" });
  const [passwordForm, setPasswordForm] = useState({ current_password: "", new_password: "", new_password_confirmation: "" });

  useEffect(() => {
    setProfileForm({ name: user?.name || "", email: user?.email || "" });
  }, [user]);

  // Visibility guards untuk dropdown Akademik
  const akademikVisibility = useMemo(() => {
    const showSchools = canAccess(user, 'view-school') || canAccess(user, 'manage-schools');
    const showGrades = canAccess(user, 'view-grade') || canAccess(user, 'manage-grades');
    const showClasses = canAccess(user, 'view-class') || canAccess(user, 'manage-classes');
    const showExamParticipants = canAccess(user, 'view-exam-participant') || canAccess(user, 'manage-exam-participants');
    const hasAny = showSchools || showGrades || showClasses || showExamParticipants;
    return { showSchools, showGrades, showClasses, showExamParticipants, hasAny };
  }, [user]);

  // Visibility untuk dropdown Hak Akses (Roles/Permissions/Role Permissions)
  const canManageRoles = useMemo(() => canAccess(user, 'manage-roles'), [user]);
  const canViewRoles = useMemo(() => canAccess(user, 'view-role') || canManageRoles, [user, canManageRoles]);
  const canViewPermissions = useMemo(() => canAccess(user, 'view-permission') || canManageRoles, [user, canManageRoles]);
  const hakAksesHasAny = useMemo(() => canViewRoles || canViewPermissions || canManageRoles, [canViewRoles, canViewPermissions, canManageRoles]);

  // Visibility untuk menu utama berdasarkan izin view/manage
  const showDashboard = useMemo(() => canAccess(user, 'view-dashboard'), [user]);
  const showUsers = useMemo(() => canAccess(user, 'view-user') || canAccess(user, 'manage-users'), [user]);
  const showSubjects = useMemo(() => canAccess(user, 'view-subject') || canAccess(user, 'manage-subjects'), [user]);
  const showExams = useMemo(() => canAccess(user, 'view-exam') || canAccess(user, 'manage-exams'), [user]);
  const canViewScoreMenu = useMemo(() => (
    canAccess(user, 'view-score') ||
    canAccess(user, 'manage-score') ||
    canAccess(user, 'export-score') ||
    canAccess(user, 'edit-score') ||
    canAccess(user, 'input-score') ||
    canAccess(user, 'input score')
  ), [user]);
  const showMonitor = useMemo(() => canAccess(user, 'view-exam-participant') || canAccess(user, 'manage-exam-participants'), [user]);
  const canEditSelf = useMemo(() => canAccess(user, 'edit-user'), [user]);

  const closeSidebar = () => {
    try {
      const el = document.getElementById('adminSidebar');
      if (!el) return;
      const oc = window.bootstrap?.Offcanvas?.getOrCreateInstance(el);
      if (oc && typeof oc.hide === 'function') oc.hide();
    } catch (e) {
      // ignore
    }
  };

  const openEditProfile = () => {
    setProfileForm({ name: user?.name || "", email: user?.email || "" });
    setPasswordForm({ current_password: "", new_password: "", new_password_confirmation: "" });
    setShowEdit(true);
  };

  const closeEditProfile = () => setShowEdit(false);

  const saveEditProfile = async () => {
    if (!user?.id) return;
    setSaving(true);
    try {
      const payload = { name: profileForm.name, email: profileForm.email };
      await api.put(`/users/${user.id}`, payload);

      // Jika user mengisi form password, jalankan perubahan password
      if (passwordForm.current_password || passwordForm.new_password || passwordForm.new_password_confirmation) {
        await api.put('/me/password', passwordForm);
      }
      await refreshUser();
      setShowEdit(false);
      Swal.fire('Berhasil', (passwordForm.new_password ? 'Profil & password berhasil diperbarui.' : 'Profil berhasil diperbarui.'), 'success');
    } catch (err) {
      const status = err?.response?.status;
      const data = err?.response?.data || {};
      if (status === 422) {
        const msg = data.message || 'Validasi gagal';
        const details = data.errors
          ? (Array.isArray(data.errors)
              ? data.errors.join('\n')
              : Object.values(data.errors).flat().join('\n'))
          : '';
        Swal.fire('Validasi gagal', [msg, details].filter(Boolean).join('\n'), 'warning');
      } else if (status === 403) {
        Swal.fire('Akses ditolak', 'Anda tidak memiliki izin untuk mengubah profil.', 'error');
      } else {
        const msg = data.message || 'Gagal memperbarui profil.';
        Swal.fire('Gagal', msg, 'error');
      }
    } finally {
      setSaving(false);
    }
  };

  return (
    <div>
      <nav
        className="navbar navbar-expand-lg navbar-dark sticky-top shadow-lg border-bottom"
        style={{ 
          background: "linear-gradient(135deg, #2e7d32 0%, #4caf50 50%, #81c784 100%)",
          minHeight: "70px"
        }}
      >
        <div className="container-fluid px-2 px-md-4">
          {/* Sidebar toggler on the left */}
          <button
            className="btn btn-outline-light btn-sm me-2 d-inline-flex align-items-center"
            type="button"
            data-bs-toggle="offcanvas"
            data-bs-target="#adminSidebar"
            aria-controls="adminSidebar"
            style={{ background: 'rgba(255,255,255,0.1)', borderRadius: '8px' }}
          >
            <span className="navbar-toggler-icon"></span>
          </button>
          <Link 
            to="/dashboard" 
            className="navbar-brand fw-bold d-flex align-items-center"
            style={{ fontSize: "1.5rem", letterSpacing: "0.5px" }}
          >
            <FaUserShield className="me-2" style={{ color: "#e8f5e9" }} />
            {user?.name || 'Admin'}
          </Link>

          {/* Tombol Logout di samping brand */}
          <button
            className="btn btn-outline-light btn-sm d-inline-flex align-items-center ms-2"
            onClick={logout}
            style={{ background: 'rgba(255,255,255,0.15)', borderRadius: '25px', padding: '8px 14px' }}
            aria-label="Logout"
          >
            <FaTimes className="me-2" /> Logout
          </button>

          {/* Navigasi dipindahkan ke sidebar offcanvas */}
          <div className="d-none">
            <ul className="navbar-nav me-auto mb-2 mb-lg-0">
               {showDashboard && (
                 <li className="nav-item mx-1">
                   <NavLink 
                     className={({isActive}) => `nav-link px-3 py-2 rounded-3 transition-all${isActive ? ' active' : ''}`} 
                     to="/dashboard"
                     style={({isActive}) => ({
                       background: isActive ? 'rgba(0, 212, 255, 0.2)' : 'transparent',
                       color: isActive ? '#00d4ff' : '#ffffff',
                       fontWeight: isActive ? '600' : '500',
                       transition: 'all 0.3s ease'
                     })}
                   >
                     <FaHome className="me-2" /> Dashboard
                   </NavLink>
                 </li>
               )}
               {showUsers && (
                 <li className="nav-item mx-1">
                   <NavLink 
                     className={({isActive}) => `nav-link px-3 py-2 rounded-3 transition-all${isActive ? ' active' : ''}`} 
                     to="/users"
                     style={({isActive}) => ({
                       background: isActive ? 'rgba(0, 212, 255, 0.2)' : 'transparent',
                       color: isActive ? '#00d4ff' : '#ffffff',
                       fontWeight: isActive ? '600' : '500',
                       transition: 'all 0.3s ease'
                     })}
                   >
                     <FaUsers className="me-2" /> Users
                   </NavLink>
                 </li>
               )}
               {/* Dropdown Hak Akses: tampil jika ada akses view-only atau manage */}
               {hakAksesHasAny && (
                <li className="nav-item dropdown position-static mx-1">
                  <a
                    className="nav-link dropdown-toggle px-3 py-2 rounded-3"
                    href="#"
                    id="hakAksesDropdown"
                    role="button"
                    data-bs-toggle="dropdown"
                    aria-expanded="false"
                    style={{
                      color: '#ffffff',
                      fontWeight: '500',
                      transition: 'all 0.3s ease',
                      background: 'transparent'
                    }}
                    onMouseEnter={(e) => {
                      e.target.style.background = 'rgba(255, 255, 255, 0.1)';
                      e.target.style.color = '#00d4ff';
                    }}
                    onMouseLeave={(e) => {
                      e.target.style.background = 'transparent';
                      e.target.style.color = '#ffffff';
                    }}
                  >
                    <FaUserShield className="me-2" /> Hak Akses
                  </a>
                  <ul 
                    className="dropdown-menu shadow-lg border-0" 
                    aria-labelledby="hakAksesDropdown" 
                    style={{ 
                      maxHeight: '50vh', 
                      overflowY: 'auto',
                      background: 'linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%)',
                      borderRadius: '12px',
                      padding: '8px'
                    }}
                  >
                     {canViewRoles && (
                       <li>
                         <NavLink className={({isActive}) => `dropdown-item${isActive ? ' active' : ''}`} to="/roles" onClick={closeSidebar}>
                           <FaUserShield className="me-2" /> Roles
                         </NavLink>
                       </li>
                     )}
                     {canViewPermissions && (
                       <li>
                         <NavLink className={({isActive}) => `dropdown-item${isActive ? ' active' : ''}`} to="/permissions" onClick={closeSidebar}>
                           <FaKey className="me-2" /> Permissions
                         </NavLink>
                       </li>
                     )}
                     {canManageRoles && (
                       <li>
                         <NavLink className={({isActive}) => `dropdown-item${isActive ? ' active' : ''}`} to="/role-permissions" onClick={closeSidebar}>
                           <FaListAlt className="me-2" /> Role Permissions
                         </NavLink>
                       </li>
                     )}
                   </ul>
                 </li>
               )}
               {showSubjects && (
                 <li className="nav-item">
                   <NavLink className={({isActive}) => `nav-link${isActive ? ' active' : ''}`} to="/subjects">
                     <FaBook className="me-1" /> Mata Pelajaran
                   </NavLink>
                 </li>
               )}
               {showExams && (
                 <li className="nav-item">
                   <NavLink className={({isActive}) => `nav-link${isActive ? ' active' : ''}`} to="/exams">
                     <FaQuestionCircle className="me-1" /> Ujian
                   </NavLink>
                 </li>
               )}
               {showMonitor && (
                 <li className="nav-item">
                   <NavLink className={({isActive}) => `nav-link${isActive ? ' active' : ''}`} to="/monitor-peserta">
                     <FaUserGraduate className="me-1" /> Monitor Peserta
                   </NavLink>
                 </li>
               )}
               <li className="nav-item">
                 <NavLink className={({isActive}) => `nav-link${isActive ? ' active' : ''}`} to="/today-public">
                   <FaCalendarAlt className="me-1" /> Mapel Hari Ini (Publik)
                 </NavLink>
               </li>
               {/* Dropdown Kurikulum: Nilai */}
               {canViewScoreMenu && (
                <li className="nav-item dropdown position-static">
                  <a
                    className="nav-link dropdown-toggle"
                    href="#"
                    id="kurikulumDropdown"
                    role="button"
                    data-bs-toggle="dropdown"
                    aria-expanded="false"
                  >
                    <FaBook className="me-1" /> Kurikulum
                  </a>
                  <ul className="dropdown-menu" aria-labelledby="kurikulumDropdown" style={{ maxHeight: '50vh', overflowY: 'auto' }}>
                     <li>
                       <NavLink className={({isActive}) => `dropdown-item${isActive ? ' active' : ''}`} to="/scores" onClick={closeSidebar}>
                         <FaListAlt className="me-2" /> Nilai
                       </NavLink>
                     </li>
                   </ul>
                 </li>
               )}

               {/* Dropdown Akademik: Schools, Grades, Classes, Exam Participants */}
               {akademikVisibility.hasAny && (
                <li className="nav-item dropdown position-static">
                  <a
                    className="nav-link dropdown-toggle"
                    href="#"
                    id="akademikDropdown"
                    role="button"
                    data-bs-toggle="dropdown"
                    aria-expanded="false"
                  >
                    <FaSchool className="me-1" /> Akademik
                  </a>
                  <ul className="dropdown-menu" aria-labelledby="akademikDropdown" style={{ maxHeight: '50vh', overflowY: 'auto' }}>
                     {akademikVisibility.showSchools && (
                       <li>
                         <NavLink className={({isActive}) => `dropdown-item${isActive ? ' active' : ''}`} to="/schools" onClick={closeSidebar}>
                           <FaSchool className="me-2" /> Sekolah
                         </NavLink>
                       </li>
                     )}
                     {akademikVisibility.showGrades && (
                       <li>
                         <NavLink className={({isActive}) => `dropdown-item${isActive ? ' active' : ''}`} to="/grades" onClick={closeSidebar}>
                           <FaLayerGroup className="me-2" /> Grade
                         </NavLink>
                       </li>
                     )}
                     {akademikVisibility.showClasses && (
                       <li>
                         <NavLink className={({isActive}) => `dropdown-item${isActive ? ' active' : ''}`} to="/classes" onClick={closeSidebar}>
                           <FaChalkboard className="me-2" /> Kelas
                         </NavLink>
                       </li>
                     )}
                     {akademikVisibility.showExamParticipants && (
                       <li>
                         <NavLink className={({isActive}) => `dropdown-item${isActive ? ' active' : ''}`} to="/exam-participants" onClick={closeSidebar}>
                           <FaUserGraduate className="me-2" /> Peserta Ujian
                         </NavLink>
                       </li>
                     )}
                   </ul>
                 </li>
               )}
             </ul>

             {/* user dropdown dihapus, logout dipindah ke samping brand */}
           </div>
         </div>
       </nav>

      {/* Offcanvas left sidebar */}
      <div className="offcanvas offcanvas-start text-bg-dark" tabIndex="-1" id="adminSidebar" aria-labelledby="adminSidebarLabel" style={{ width: '280px' }}>
        <div className="offcanvas-header border-bottom">
          <h5 className="offcanvas-title d-flex align-items-center" id="adminSidebarLabel">
            <FaUserShield className="me-2" /> Menu Admin
          </h5>
          <button type="button" className="btn-close btn-close-white" data-bs-dismiss="offcanvas" aria-label="Close"></button>
        </div>
        <div className="offcanvas-body">
          <ul className="list-unstyled">
            {/* Main links */}
            {showDashboard && (
              <li className="mb-1">
                <NavLink className={({isActive}) => `d-flex align-items-center px-3 py-2 rounded-3 text-decoration-none ${isActive ? 'bg-primary text-white' : 'text-white-50'}`} to="/dashboard" onClick={closeSidebar}>
                  <FaHome className="me-2" /> Dashboard
                </NavLink>
              </li>
            )}
            {showUsers && (
              <li className="mb-1">
                <NavLink className={({isActive}) => `d-flex align-items-center px-3 py-2 rounded-3 text-decoration-none ${isActive ? 'bg-primary text-white' : 'text-white-50'}`} to="/users" onClick={closeSidebar}>
                  <FaUsers className="me-2" /> Users
                </NavLink>
              </li>
            )}
            {showSubjects && (
              <li className="mb-1">
                <NavLink className={({isActive}) => `d-flex align-items-center px-3 py-2 rounded-3 text-decoration-none ${isActive ? 'bg-primary text-white' : 'text-white-50'}`} to="/subjects" onClick={closeSidebar}>
                  <FaBook className="me-2" /> Mata Pelajaran
                </NavLink>
              </li>
            )}
            {showExams && (
              <li className="mb-1">
                <NavLink className={({isActive}) => `d-flex align-items-center px-3 py-2 rounded-3 text-decoration-none ${isActive ? 'bg-primary text-white' : 'text-white-50'}`} to="/exams" onClick={closeSidebar}>
                  <FaQuestionCircle className="me-2" /> Ujian
                </NavLink>
              </li>
            )}
            {showMonitor && (
              <li className="mb-1">
                <NavLink className={({isActive}) => `d-flex align-items-center px-3 py-2 rounded-3 text-decoration-none ${isActive ? 'bg-primary text-white' : 'text-white-50'}`} to="/monitor-peserta" onClick={closeSidebar}>
                  <FaUserGraduate className="me-2" /> Monitor Peserta
                </NavLink>
              </li>
            )}
            <li className="mb-1">
              <NavLink className={({isActive}) => `d-flex align-items-center px-3 py-2 rounded-3 text-decoration-none ${isActive ? 'bg-primary text-white' : 'text-white-50'}`} to="/today-public" onClick={closeSidebar}>
                <FaCalendarAlt className="me-2" /> Mapel Hari Ini (Publik)
              </NavLink>
            </li>

            {/* Hak Akses dropdown mirror */}
            {hakAksesHasAny && (
              <li className="mt-3">
                <div className="text-uppercase text-white-50 small px-3 mb-2">Hak Akses</div>
                <ul className="list-unstyled ms-2">
                  {canViewRoles && (
                    <li className="mb-1">
                      <NavLink className={({isActive}) => `d-flex align-items-center px-3 py-2 rounded-3 text-decoration-none ${isActive ? 'bg-primary text-white' : 'text-white-50'}`} to="/roles" onClick={closeSidebar}>
                        <FaUserShield className="me-2" /> Roles
                      </NavLink>
                    </li>
                  )}
                  {canViewPermissions && (
                    <li className="mb-1">
                      <NavLink className={({isActive}) => `d-flex align-items-center px-3 py-2 rounded-3 text-decoration-none ${isActive ? 'bg-primary text-white' : 'text-white-50'}`} to="/permissions" onClick={closeSidebar}>
                        <FaKey className="me-2" /> Permissions
                      </NavLink>
                    </li>
                  )}
                  {canManageRoles && (
                    <li className="mb-1">
                      <NavLink className={({isActive}) => `d-flex align-items-center px-3 py-2 rounded-3 text-decoration-none ${isActive ? 'bg-primary text-white' : 'text-white-50'}`} to="/role-permissions" onClick={closeSidebar}>
                        <FaListAlt className="me-2" /> Role Permissions
                      </NavLink>
                    </li>
                  )}
                </ul>
              </li>
            )}

            {/* Kurikulum */}
            {canViewScoreMenu && (
              <li className="mt-3">
                <div className="text-uppercase text-white-50 small px-3 mb-2">Kurikulum</div>
                <ul className="list-unstyled ms-2">
                  <li className="mb-1">
                    <NavLink className={({isActive}) => `d-flex align-items-center px-3 py-2 rounded-3 text-decoration-none ${isActive ? 'bg-primary text-white' : 'text-white-50'}`} to="/scores" onClick={closeSidebar}>
                      <FaListAlt className="me-2" /> Nilai
                    </NavLink>
                  </li>
                </ul>
              </li>
            )}

            {/* Akademik */}
            {akademikVisibility.hasAny && (
              <li className="mt-3">
                <div className="text-uppercase text-white-50 small px-3 mb-2">Akademik</div>
                <ul className="list-unstyled ms-2">
                  {akademikVisibility.showSchools && (
                    <li className="mb-1">
                      <NavLink className={({isActive}) => `d-flex align-items-center px-3 py-2 rounded-3 text-decoration-none ${isActive ? 'bg-primary text-white' : 'text-white-50'}`} to="/schools" onClick={closeSidebar}>
                        <FaSchool className="me-2" /> Sekolah
                      </NavLink>
                    </li>
                  )}
                  {akademikVisibility.showGrades && (
                    <li className="mb-1">
                      <NavLink className={({isActive}) => `d-flex align-items-center px-3 py-2 rounded-3 text-decoration-none ${isActive ? 'bg-primary text-white' : 'text-white-50'}`} to="/grades" onClick={closeSidebar}>
                        <FaLayerGroup className="me-2" /> Grade
                      </NavLink>
                    </li>
                  )}
                  {akademikVisibility.showClasses && (
                    <li className="mb-1">
                      <NavLink className={({isActive}) => `d-flex align-items-center px-3 py-2 rounded-3 text-decoration-none ${isActive ? 'bg-primary text-white' : 'text-white-50'}`} to="/classes" onClick={closeSidebar}>
                        <FaChalkboard className="me-2" /> Kelas
                      </NavLink>
                    </li>
                  )}
                  {akademikVisibility.showExamParticipants && (
                    <li className="mb-1">
                      <NavLink className={({isActive}) => `d-flex align-items-center px-3 py-2 rounded-3 text-decoration-none ${isActive ? 'bg-primary text-white' : 'text-white-50'}`} to="/exam-participants" onClick={closeSidebar}>
                        <FaUserGraduate className="me-2" /> Peserta Ujian
                      </NavLink>
                    </li>
                  )}
                </ul>
              </li>
            )}
          </ul>
        </div>
      </div>
      <div className="container-fluid mt-4 px-2 px-md-3 px-lg-4">{children}</div>


      {/* Modal Edit Profil */}
      {showEdit && (
        <>
          <div className="modal fade show d-block" tabIndex="-1" role="dialog" aria-modal="true">
            <div className="modal-dialog">
              <div className="modal-content">
                <div className="modal-header">
                  <h5 className="modal-title">Edit Profil</h5>
                  <button type="button" className="btn-close" onClick={closeEditProfile} aria-label="Close"></button>
                </div>
                <div className="modal-body">
                  <div className="mb-3">
                    <label className="form-label">Nama</label>
                    <input
                      type="text"
                      className="form-control"
                      value={profileForm.name}
                      onChange={(e) => setProfileForm(f => ({ ...f, name: e.target.value }))}
                    />
                  </div>
                  <div className="mb-3">
                    <label className="form-label">Email</label>
                    <input
                      type="email"
                      className="form-control"
                      value={profileForm.email}
                      onChange={(e) => setProfileForm(f => ({ ...f, email: e.target.value }))}
                    />
                  </div>

                  <hr />
                  <div className="mb-2 d-flex align-items-center gap-2">
                    <FaKey />
                    <strong>Ubah Password</strong>
                    <small className="text-muted">(opsional)</small>
                  </div>
                  <div className="mb-3">
                    <label className="form-label">Password Saat Ini</label>
                    <input
                      type="password"
                      className="form-control"
                      value={passwordForm.current_password}
                      onChange={(e) => setPasswordForm(f => ({ ...f, current_password: e.target.value }))}
                      placeholder="Masukkan password saat ini"
                    />
                  </div>
                  <div className="mb-3">
                    <label className="form-label">Password Baru</label>
                    <input
                      type="password"
                      className="form-control"
                      value={passwordForm.new_password}
                      onChange={(e) => setPasswordForm(f => ({ ...f, new_password: e.target.value }))}
                      placeholder="Minimal 6 karakter"
                    />
                  </div>
                  <div className="mb-3">
                    <label className="form-label">Konfirmasi Password Baru</label>
                    <input
                      type="password"
                      className="form-control"
                      value={passwordForm.new_password_confirmation}
                      onChange={(e) => setPasswordForm(f => ({ ...f, new_password_confirmation: e.target.value }))}
                      placeholder="Ulangi password baru"
                    />
                  </div>
                </div>
                <div className="modal-footer">
                  <button type="button" className="btn btn-outline-secondary d-flex align-items-center gap-2" onClick={closeEditProfile} disabled={saving}>
                    <FaTimes /> <span>Batal</span>
                  </button>
                  <button type="button" className="btn btn-primary d-flex align-items-center gap-2" onClick={saveEditProfile} disabled={saving}>
                    <FaSave /> <span>{saving ? 'Menyimpan...' : 'Simpan'}</span>
                  </button>
                </div>
              </div>
            </div>
          </div>
          <div className="modal-backdrop fade show"></div>
        </>
      )}
    </div>
  );
};

export default AdminLayout;
