/* 0.29.1 */
import type { Rational } from '../numerics/types';
import { NumericValue } from '../numeric-value/types';
import type { BoxedExpression, ComputeEngine } from '../global-types';
/**
 * Group terms in a product by common term.
 *
 * All the terms should be canonical.
 * - the arguments should have been flattened for `Multiply`
 *
 * - any argument of power been distributed, i.e.
 *      (ab)^2 ->  a^2 b^2
 * *
 * 3 + √5 + √(x+1) + x^2 + (a+b)^2 + d
 *  -> [ [[3, "d"], [1, 1]],
 *       [[5, "x+1"], [1, 2]],
 *       [[1, "a+b"], [2, 1]]
 *      ]
 *
 */
export declare class Product {
    readonly options?: {
        canonical?: boolean;
    };
    engine: ComputeEngine;
    coefficient: NumericValue;
    terms: {
        term: BoxedExpression;
        exponent: Rational;
    }[];
    private _isCanonical;
    static from(expr: BoxedExpression): Product;
    constructor(ce: ComputeEngine, xs?: ReadonlyArray<BoxedExpression>, options?: {
        canonical?: boolean;
    });
    /**
     * Add a term to the product.
     *
     * If `this._isCanonical` a running product of exact terms is kept.
     * Otherwise, terms and their exponent are tallied.
     */
    mul(term: BoxedExpression, exp?: Rational): void;
    /** Divide the product by a term of coefficient */
    div(term: NumericValue | BoxedExpression): void;
    /** The terms of the product, grouped by degrees.
     *
     * If `mode` is `rational`, rationals are split into separate numerator and
     * denominator, so that a rational expression can be created later
     * If `mode` is `expression`, a boxed expression is returned, without
     * splitting rationals
     * If `mode` is `numeric`, the literals are combined into one expression
     *
     */
    groupedByDegrees(options?: {
        mode?: 'rational' | 'expression' | 'numeric';
    }): {
        exponent: Rational;
        terms: BoxedExpression[];
    }[] | null;
    asExpression(options?: {
        numericApproximation: boolean;
    }): BoxedExpression;
    /** The product, expressed as a numerator and denominator */
    asNumeratorDenominator(): [BoxedExpression, BoxedExpression];
    asRationalExpression(): BoxedExpression;
}
export declare function commonTerms(lhs: Product, rhs: Product): [NumericValue, BoxedExpression];
