/* 0.29.1 */
import type { BoxedExpression } from '../global-types';
export type Order = 'lex' | 'dexlex' | 'grevlex' | 'elim';
export declare const DEFAULT_COMPLEXITY = 100000;
/**
 * The sorting order of arguments of the Add function uses a modified degrevlex:
 * - Sort by total degree (sum of degree)
 * - Sort by max degree.
 * - Sort reverse lexicographically
 * - Sort by rank
 *
 *
 * E.g.
 * - 2x^2 + 3x + 1
 * - 2x^2y^3 + 5x^3y
 */
export declare function addOrder(a: BoxedExpression, b: BoxedExpression): number;
export declare function equalOrder(a: BoxedExpression, b: BoxedExpression): number;
declare const RANKS: readonly ["integer", "rational", "radical", "real", "complex", "constant", "symbol", "multiply", "divide", "add", "trig", "fn", "power", "string", "other"];
export type Rank = (typeof RANKS)[number];
/**
 * Given two expressions `a` and `b`, return:
 * - `-1` if `a` should be ordered before `b`
 * - `+1` if `b` should be ordered before `a`
 * - `0` if they have the same order (they are structurally equal)
 *
 * The default order is as follow:
 *
 * 1/ Literal numeric values (rational,  machine numbers and Decimal numbers),
 *  ordered by their numeric value (smaller numbers before larger numbers)
 *
 * 2/ Literal complex numbers, ordered by their real parts. In case of a tie,
 * ordered by the absolute value of their imaginary parts. In case of a tie,
 * ordered by the value of their imaginary parts.
 *
 * 3/ Symbols, ordered by their name as strings
 *
 * 4/ Addition, ordered as a polynom, with higher degree terms first
 *
 * 5/ Other functions, ordered by their `complexity` property. In case
 * of a tie, ordered by the operator of the expression as a string. In case of a
 * tie, by the leaf count of each expression. In case of a tie, by the order
 * of each argument, left to right.
 *
 * 6/ Strings, ordered by comparing their Unicode code point values. While this
 * sort order is quick to calculate, it can produce unexpected results, for
 * example "E" < "e" < "È" and "11" < "2". This ordering is not suitable to
 * collate natural language strings.
 *
 * See https://reference.wolfram.com/language/ref/Sort.html for a
 * description of the ordering of expressions in Mathematica.
 *
 */
export declare function order(a: BoxedExpression, b: BoxedExpression): number;
/** Return a version of the expression with its arguments sorted in
 * canonical order
 */
export declare function canonicalOrder(expr: BoxedExpression, { recursive }: {
    recursive?: boolean;
}): BoxedExpression;
export declare function sortOperands(operator: string, xs: ReadonlyArray<BoxedExpression>): ReadonlyArray<BoxedExpression>;
/**
 * Sort the terms of a polynomial expression (`Add` expression) according
 * to the deglex polynomial ordering
 *
 */
export declare function polynomialOrder(expr: BoxedExpression): BoxedExpression;
export declare function lexicographicOrder(expr: BoxedExpression, vars?: ReadonlyArray<string>): BoxedExpression;
export declare function degreeLexicographicOrder(expr: BoxedExpression, vars?: ReadonlyArray<string>): BoxedExpression;
export declare function degreeReverseLexicographicOrder(expr: BoxedExpression, vars?: ReadonlyArray<string>): BoxedExpression;
export declare function eliminationOrder(expr: BoxedExpression, vars?: ReadonlyArray<string>): BoxedExpression;
export {};
