<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\URL;
use Illuminate\Cache\RateLimiting\Limit;
use Illuminate\Http\Request;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        // Paksa skema HTTPS jika diaktifkan melalui env (production)
        if (filter_var(env('FORCE_HTTPS', false), FILTER_VALIDATE_BOOLEAN)) {
            URL::forceScheme('https');
        }

        // Definisikan rate limiter agar middleware `throttle:login` dan `throttle:api` dikenali
        RateLimiter::for('api', function (Request $request) {
            $max = (int) env('API_RATE_LIMIT', 60);
            return Limit::perMinute($max)->by(optional($request->user())->id ?: $request->ip());
        });

        RateLimiter::for('login', function (Request $request) {
            $max = (int) env('LOGIN_RATE_LIMIT', 5);
            // Per-identifier: email atau nisn; fallback token; sertakan device_fp + IP untuk edge limiting
            $identifier = strtolower((string) (
                $request->input('email') ??
                $request->input('nisn') ??
                $request->input('token') ??
                'guest'
            ));
            $device = (string) ($request->input('device_fp') ?? 'no-device');
            return Limit::perMinute($max)->by($identifier . ':' . $device . ':' . $request->ip());
        });

        // Rate limiter untuk submit jawaban ujian (default 2/menit per sesi+user+IP)
        RateLimiter::for('submit-exam', function (Request $request) {
            $max = (int) env('SUBMIT_EXAM_RATE_LIMIT', 2);
            $sessionId = optional($request->route('session'))->id ?? (string)$request->route('session');
            $by = ($sessionId ?: 'no-session') . ':' . (optional($request->user())->id ?: 'guest') . ':' . $request->ip();
            return Limit::perMinute($max)->by($by);
        });

        // Rate limiter untuk autosave jawaban (default 30/menit per sesi+user+IP)
        RateLimiter::for('autosave-exam', function (Request $request) {
            $max = (int) env('AUTOSAVE_EXAM_RATE_LIMIT', 30);
            $sessionId = optional($request->route('session'))->id ?? (string)$request->route('session');
            $by = ($sessionId ?: 'no-session') . ':' . (optional($request->user())->id ?: 'guest') . ':' . $request->ip();
            return Limit::perMinute($max)->by($by);
        });

        // Rate limiter untuk upload gambar soal (default 10/menit per user)
        RateLimiter::for('upload', function (Request $request) {
            $max = (int) env('UPLOAD_RATE_LIMIT', 10);
            return Limit::perMinute($max)->by((optional($request->user())->id ?: 'guest') . ':' . $request->ip());
        });

        // Rate limiter untuk export (PDF/Excel) default 2/menit per user
        RateLimiter::for('export', function (Request $request) {
            $max = (int) env('EXPORT_RATE_LIMIT', 2);
            return Limit::perMinute($max)->by((optional($request->user())->id ?: 'guest') . ':' . $request->ip());
        });

        // Rate limiter untuk import peserta (default 2/menit per user)
        RateLimiter::for('import', function (Request $request) {
            $max = (int) env('IMPORT_RATE_LIMIT', 2);
            return Limit::perMinute($max)->by((optional($request->user())->id ?: 'guest') . ':' . $request->ip());
        });

        // Rate limiter untuk laporan pelanggaran (default 30/menit per sesi+user+IP)
        RateLimiter::for('report', function (Request $request) {
            $max = (int) env('REPORT_RATE_LIMIT', 30);
            $sessionId = optional($request->route('session'))->id ?? (string)$request->route('session');
            $by = ($sessionId ?: 'no-session') . ':' . (optional($request->user())->id ?: 'guest') . ':' . $request->ip();
            return Limit::perMinute($max)->by($by);
        });

        // Rate limiter khusus registrasi publik (default 5/menit per email+IP)
        RateLimiter::for('register', function (Request $request) {
            $max = (int) env('REGISTER_RATE_LIMIT', 5);
            $identifier = strtolower((string) ($request->input('email') ?? 'guest'));
            return Limit::perMinute($max)->by($identifier . ':' . $request->ip());
        });
    }
}
