<?php

namespace App\Jobs;

use App\Models\ExamSession;
use App\Models\Question;
use App\Models\ExamResult;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\Middleware\WithoutOverlapping;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ProcessFinalExamSubmission implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $sessionId;
    public int $participantId;
    public array $answers;
    public ?string $reason;

    // Job reliability settings
    public $tries = 3;
    public $timeout = 90;
    public function backoff(): array { return [10, 30, 120]; }
    public function middleware(): array { return [(new \Illuminate\Queue\Middleware\WithoutOverlapping('finalize-session-'.$this->sessionId))->expireAfter(300)]; }

    public function __construct(int $sessionId, int $participantId, array $answers, ?string $reason = null)
    {
        $this->sessionId = $sessionId;
        $this->participantId = $participantId;
        $this->answers = $answers;
        $this->reason = $reason;
        $this->onQueue('finalize');
    }

    public function handle(): void
    {
        $session = ExamSession::find($this->sessionId);
        if (!$session) { return; }
        if ((int) $session->exam_participant_id !== (int) $this->participantId) { return; }

        $exam = $session->exam()->first();
        if (!$exam) { return; }
        $participant = $session->participant()->first();
        if (!$participant) { return; }

        $subjectId = (int) $exam->id_subject;
        $subject = $exam->subject()->first();

        $questions = Question::where('subject_id', $subjectId)
            ->orderBy('created_at', 'asc')
            ->orderBy('id', 'asc')
            ->get(['id', 'key_answer', 'option_b', 'option_c', 'option_d']);
        if ($questions->isEmpty()) {
            Log::warning('Finalize skipped: no questions for subject', [
                'session_id' => $session->id,
                'subject_id' => $subjectId,
            ]);
            return;
        }

        // Split answers into MCQ and essay
        $answersChoice = [];
        $answersEssay = [];
        foreach ($this->answers as $qid => $opt) {
            $qidInt = (int) $qid;
            if ($qidInt <= 0) continue;
            $optStr = strtoupper((string) $opt);
            if (in_array($optStr, ['A','B','C','D','E'], true)) {
                $answersChoice[$qidInt] = $optStr;
            } elseif (is_string($opt) && trim((string)$opt) !== '') {
                $answersEssay[$qidInt] = (string) $opt;
            }
        }

        $isBlankOrDash = function ($s) { $t = trim((string)$s); return $t === '' || $t === '-'; };

        $correctIndexes = [];
        $wrongIndexes = [];
        $total = 0;
        $correct = 0;
        $essayCorrect = 0;
        $essayWrong = 0;
        $essayDetails = [];
        $mcqCorrect = 0;
        $mcqWrong = 0;
        $totalChoice = 0;
        $correctEssayOrders = [];
        $wrongEssayOrders = [];

        $positionByQid = [];
        $order = is_array($session->question_order) ? $session->question_order : [];
        if (!empty($order)) {
            foreach ($order as $i => $qid) { $positionByQid[(int)$qid] = $i + 1; }
        }

        foreach ($questions as $idx => $q) {
            $position = $positionByQid[$q->id] ?? ($idx + 1);
            $isEssay = ($isBlankOrDash($q->option_b) && $isBlankOrDash($q->option_c) && $isBlankOrDash($q->option_d));
            if ($isEssay) {
                $rawKey = trim((string)$q->key_answer);
                $ans = isset($answersEssay[$q->id]) ? trim((string)$answersEssay[$q->id]) : '';
                $detail = [
                    'question_id' => $q->id,
                    'answer' => $ans,
                    'has_key' => $rawKey !== '',
                    'best_match_key' => null,
                    'cosine' => 0.0,
                    'char_percent' => 0.0,
                    'jaccard' => 0.0,
                    'rabin_karp_found' => false,
                    'is_correct' => false,
                    'reason' => null,
                    'keyword_coverage' => 0.0,
                    'levenshtein_hits' => 0,
                    'synonym_hits' => 0,
                    'total_keywords' => 0,
                    'decision_basis' => null,
                ];
                if ($rawKey === '' || $ans === '') {
                    $total++;
                    $wrongIndexes[] = $position;
                    $essayWrong++;
                    $wrongEssayOrders[] = $position;
                    $detail['reason'] = $rawKey === '' ? 'no_key' : 'empty_answer';
                    $essayDetails[$q->id] = $detail;
                    continue;
                }
                $keys = preg_split('/[|;\/\n]+/', $rawKey, -1, PREG_SPLIT_NO_EMPTY) ?: [$rawKey];
                $isCorrectEssay = false;
                $bestScore = -1.0;
                $best = ['key' => null, 'cosine' => 0.0, 'char_percent' => 0.0, 'jaccard' => 0.0, 'rabin_karp_found' => false, 'keyword_coverage' => 0.0, 'levenshtein_hits' => 0, 'synonym_hits' => 0, 'total_keywords' => 0];
                foreach ($keys as $keyAlt) {
                    $keyAlt = trim($keyAlt);
                    $cos = $this->cosineSimilarity($ans, $keyAlt);
                    $rk = $this->rabinKarpContains($ans, $keyAlt);
                    $metrics = $this->similarityMetrics($ans, $keyAlt);
                    $cov = $this->keywordCoverageWithFuzzy($ans, $keyAlt);
                    $mix = ($cos * 0.7) + ($metrics['jaccard'] * 0.2) + (($metrics['char_percent'] / 100.0) * 0.1);
                    if ($mix >= $bestScore) {
                        $bestScore = $mix;
                        $best = [
                            'key' => $keyAlt,
                            'cosine' => $cos,
                            'char_percent' => $metrics['char_percent'],
                            'jaccard' => $metrics['jaccard'],
                            'rabin_karp_found' => $rk,
                            'keyword_coverage' => $cov['coverage'],
                            'levenshtein_hits' => $cov['levenshtein_hits'],
                            'synonym_hits' => $cov['synonym_hits'],
                            'total_keywords' => $cov['total_keywords'],
                        ];
                    }
                }
                // Decide correctness using best metrics and coverage
                $thr = $this->essayThresholds();
                $isShort = count($this->wordSet($ans)) <= $thr['short_tokens'];
                $basis = null;
                if ($best['rabin_karp_found']) { $isCorrectEssay = true; $basis = 'rabin_karp'; }
                elseif ($isShort && ($best['char_percent'] >= $thr['char_percent_short'] || $best['cosine'] >= $thr['cosine_short'])) { $isCorrectEssay = true; $basis = $best['char_percent'] >= $thr['char_percent_short'] ? 'char_percent_short' : 'cosine_short'; }
                elseif ($best['keyword_coverage'] >= $thr['mandatory_coverage']) { $isCorrectEssay = true; $basis = 'keyword_coverage'; }
                elseif ($best['cosine'] >= $thr['cosine_long']) { $isCorrectEssay = true; $basis = 'cosine_long'; }
                $total++;
                $detail['best_match_key'] = $best['key'];
                $detail['cosine'] = $best['cosine'];
                $detail['char_percent'] = $best['char_percent'];
                $detail['jaccard'] = $best['jaccard'];
                $detail['rabin_karp_found'] = $best['rabin_karp_found'];
                $detail['keyword_coverage'] = $best['keyword_coverage'];
                $detail['levenshtein_hits'] = $best['levenshtein_hits'];
                $detail['synonym_hits'] = $best['synonym_hits'];
                $detail['total_keywords'] = $best['total_keywords'];
                $detail['is_correct'] = $isCorrectEssay;
                $detail['decision_basis'] = $basis;
                if ($isCorrectEssay) {
                    $correct++;
                    $essayCorrect++;
                    $correctIndexes[] = $position;
                    $correctEssayOrders[] = $position;
                } else {
                    $essayWrong++;
                    $wrongIndexes[] = $position;
                    $wrongEssayOrders[] = $position;
                }
                $essayDetails[$q->id] = $detail;
                continue;
            }
            // MCQ
            $total++;
            $totalChoice++;
            $selected = $answersChoice[$q->id] ?? null;
            if ($selected !== null && $selected === $q->key_answer) {
                $correct++;
                $mcqCorrect++;
                $correctIndexes[] = $position;
            } else {
                $mcqWrong++;
                $wrongIndexes[] = $position;
            }
        }

        $wrong = max(0, $total - $correct);
        $mcqScore = $totalChoice > 0 ? round(($mcqCorrect / $totalChoice) * 100, 2) : 0.0;
        $score = $total > 0 ? round(($correct / $total) * 100, 2) : 0.0;

        $payload = [
            'exam_session_id' => $session->id,
            'exam_id' => $exam->id,
            'subject_id' => $subjectId,
            'participant_id' => $participant->id,
            'participant_nisn' => (string) $participant->nisn,
            'participant_nama' => (string) $participant->nama,
            'subject_name' => (string) ($subject?->name ?? ''),
            'correct_count' => $mcqCorrect,
            'wrong_count' => $mcqWrong,
            'essay_correct_count' => $essayCorrect,
            'essay_wrong_count' => $essayWrong,
            'total_count' => $total,
            // Maintain current behavior: score reflects MCQ only
            'score' => $mcqScore,
            'answers' => array_merge($answersChoice, $answersEssay),
            'essay_details' => $essayDetails,
            'correct_order_indexes' => $correctIndexes,
            'wrong_order_indexes' => $wrongIndexes,
            'correct_orders_essays' => $correctEssayOrders,
            'wrong_orders_essay' => $wrongEssayOrders,
        ];

        $existing = ExamResult::where('exam_session_id', $session->id)->first();
        if ($existing) {
            $existing->fill($payload);
            $existing->save();
        } else {
            ExamResult::create($payload);
        }

        // Increment jumlah submit (attempt) pada sesi bila kolom tersedia
        if (\Illuminate\Support\Facades\Schema::hasColumn('exam_sessions','submissions_count')) {
            $session->submissions_count = (int) $session->submissions_count + 1;
        }
        // Set finished_at saat job memproses finalisasi
        if (!$session->finished_at) {
            $session->finished_at = Carbon::now();
        }
        $session->save();

        if ($this->reason) {
            Log::info('Exam answers queued submit processed', [
                'session_id' => $session->id,
                'participant_id' => $this->participantId,
                'reason' => $this->reason,
                'finished_at' => optional($session->finished_at)->toDateTimeString(),
                'submissions_count' => (int) $session->submissions_count,
            ]);
        }
    }

    private function normalizeText(string $text): string
    {
        $t = trim($text);
        $t = @iconv('UTF-8', 'ASCII//TRANSLIT', $t) ?: $t;
        $t = mb_strtolower($t, 'UTF-8');
        $t = preg_replace('/[\p{P}\p{S}]+/u', ' ', $t);
        $t = preg_replace('/\s+/u', ' ', $t);
        return trim((string)$t);
    }

    private function wordTokens(string $text): array
    {
        $normalized = $this->normalizeText($text);
        $parts = preg_split('/\s+/u', $normalized, -1, PREG_SPLIT_NO_EMPTY) ?: [];
        $stop = [
            'dan','yang','di','ke','dari','untuk','dengan','atau','pada','sebagai','adalah','itu','ini','kami','kita','saya','aku','anda','para','serta','hingga','agar','karena','namun','bahwa','jadi','dalam','akan','atau','tidak','ya','bukan','oleh','sebuah','suatu','sebuah','terhadap','lebih','kurang','lain','juga'
        ];
        $stopSet = array_fill_keys($stop, true);
        $freq = [];
        foreach ($parts as $p) {
            if (mb_strlen($p, 'UTF-8') < 2) continue;
            if (isset($stopSet[$p])) continue;
            $freq[$p] = ($freq[$p] ?? 0) + 1;
        }
        return $freq;
    }

    private function cosineSimilarity(string $a, string $b): float
    {
        $fa = $this->wordTokens($a);
        $fb = $this->wordTokens($b);
        if (empty($fa) || empty($fb)) return 0.0;
        $dot = 0.0; $normA = 0.0; $normB = 0.0;
        foreach ($fa as $t => $ca) {
            $normA += $ca * $ca;
            if (isset($fb[$t])) { $dot += $ca * $fb[$t]; }
        }
        foreach ($fb as $cb) { $normB += $cb * $cb; }
        $den = sqrt($normA) * sqrt($normB);
        if ($den <= 0.0) return 0.0;
        return $dot / $den;
    }

    private function rabinKarpContains(string $text, string $pattern): bool
    {
        $t = $this->normalizeText($text);
        $p = $this->normalizeText($pattern);
        $n = strlen($t);
        $m = strlen($p);
        if ($m === 0) return true;
        if ($m > $n) return false;

        $d = 256; $q = 101;
        $h = 1;
        for ($i = 0; $i < $m - 1; $i++) { $h = ($h * $d) % $q; }
        $pHash = 0; $tHash = 0;
        for ($i = 0; $i < $m; $i++) {
            $pHash = ($d * $pHash + ord($p[$i])) % $q;
            $tHash = ($d * $tHash + ord($t[$i])) % $q;
        }
        for ($i = 0; $i <= $n - $m; $i++) {
            if ($pHash === $tHash) {
                $substr = substr($t, $i, $m);
                if ($substr === $p) return true;
            }
            if ($i < $n - $m) {
                $tHash = ($d * ($tHash - ord($t[$i]) * $h) + ord($t[$i + $m])) % $q;
                if ($tHash < 0) $tHash += $q;
            }
        }
        return false;
    }

    private function wordSet(string $text): array
    {
        $normalized = $this->normalizeText($text);
        $parts = preg_split('/\s+/u', $normalized, -1, PREG_SPLIT_NO_EMPTY) ?: [];
        $stop = [
            'dan','yang','di','ke','dari','untuk','dengan','atau','pada','sebagai','adalah','itu','ini','kami','kita','saya','aku','anda','para','serta','hingga','agar','karena','namun','bahwa','jadi','dalam','akan','atau','tidak','ya','bukan','oleh','sebuah','suatu','sebuah','terhadap','lebih','kurang','lain','juga'
        ];
        $stopSet = array_fill_keys($stop, true);
        $tokens = [];
        foreach ($parts as $p) {
            if (mb_strlen($p, 'UTF-8') < 2) continue;
            if (isset($stopSet[$p])) continue;
            $tokens[$p] = true;
        }
        return array_keys($tokens);
    }

    private function similarityMetrics(string $a, string $b): array
    {
        $an = $this->normalizeText($a);
        $bn = $this->normalizeText($b);

        $percent = 0.0;
        similar_text($an, $bn, $percent);

        $wa = $this->wordSet($a);
        $wb = $this->wordSet($b);
        $setA = array_fill_keys($wa, true);
        $setB = array_fill_keys($wb, true);
        $inter = 0; $union = 0;
        $seen = [];
        foreach ($setA as $k => $_) { $seen[$k] = true; if (isset($setB[$k])) $inter++; $union++; }
        foreach ($setB as $k => $_) { if (!isset($seen[$k])) $union++; }
        $jaccard = $union > 0 ? ($inter / $union) : 0.0;

        return [
            'char_percent' => $percent,
            'jaccard' => $jaccard,
        ];
    }

    private function essayThresholds(): array
    {
        return [
            'mandatory_coverage' => (float) env('ESSAY_MANDATORY_COVERAGE', 0.70),
            'weighted_coverage' => (float) env('ESSAY_WEIGHTED_COVERAGE', 0.75),
            'cosine_short' => (float) env('ESSAY_COSINE_SHORT', 0.80),
            'cosine_long' => (float) env('ESSAY_COSINE_LONG', 0.75),
            'char_percent_short' => (float) env('ESSAY_CHAR_SHORT', 90.0),
            'levenshtein_dist' => (int) env('ESSAY_LEVENSHTEIN', 2),
            'short_tokens' => (int) env('ESSAY_SHORT_TOKENS', 10),
        ];
    }

    private function synonymCanonicalMap(): array
    {
        $groups = [
            'tujuan' => ['maksud','sasaran','niat'],
            'awal' => ['mula','permulaan','awalan'],
            'fungsi' => ['peran','kegunaan','manfaat'],
            'akibat' => ['dampak','efek','konsekuensi'],
            'penyebab' => ['sebab','alasan'],
            'tujuan' => ['maksud','sasaran','niat'],
            'cara' => ['metode','langkah','prosedur'],
            'contoh' => ['misal','ilustrasi'],
            'kegiatan' => ['aktivitas','aktifitas'],
            'masalah' => ['problem','isu'],
        ];
        $map = [];
        foreach ($groups as $canon => $syns) {
            $canon = $this->normalizeText($canon);
            $map[$canon] = $canon;
            foreach ($syns as $s) {
                $sn = $this->normalizeText($s);
                $map[$sn] = $canon;
            }
        }
        return $map;
    }

    private function canonicalToken(string $t): string
    {
        $n = $this->normalizeText($t);
        $map = $this->synonymCanonicalMap();
        return $map[$n] ?? $n;
    }

    private function keywordCoverageWithFuzzy(string $answer, string $key): array
    {
        $thr = $this->essayThresholds();
        $lev = (int) $thr['levenshtein_dist'];
        $ansTokensRaw = $this->wordSet($answer);
        $keyTokensRaw = $this->wordSet($key);
        if (empty($keyTokensRaw)) {
            return ['coverage' => 0.0, 'levenshtein_hits' => 0, 'synonym_hits' => 0, 'total_keywords' => 0];
        }
        $ansTokens = array_map(fn($x) => $this->canonicalToken($x), $ansTokensRaw);
        $keyTokens = array_map(fn($x) => $this->canonicalToken($x), $keyTokensRaw);
        $matched = 0; $levHits = 0; $synHits = 0;
        foreach ($keyTokens as $i => $kw) {
            $kwLen = mb_strlen($kw, 'UTF-8');
            $maxDist = $kwLen <= 4 ? max(1, min($lev, 1)) : $lev;
            $found = false; $viaSyn = false; $viaLev = false;
            foreach ($ansTokens as $atIdx => $at) {
                if ($at === $kw) { $found = true; $viaSyn = ($keyTokensRaw[$i] !== $ansTokensRaw[$atIdx]); break; }
                $d = levenshtein($kw, $at);
                if ($d <= $maxDist) { $found = true; $viaLev = true; break; }
            }
            if ($found) { $matched++; if ($viaSyn) $synHits++; if ($viaLev) $levHits++; }
        }
        $totalKw = count($keyTokens);
        $coverage = $totalKw > 0 ? ($matched / $totalKw) : 0.0;
        return [
            'coverage' => $coverage,
            'levenshtein_hits' => $levHits,
            'synonym_hits' => $synHits,
            'total_keywords' => $totalKw,
        ];
    }

}