<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Subject;
use App\Models\Question;
use App\Models\Exam;
use App\Models\ExamSession;
use App\Models\ExamResult;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class SubjectController extends Controller
{
    // Middleware permission dipindahkan ke routes/api.php untuk menghindari error pada base Controller

    /**
     * Ekstrak path storage relatif (disk 'public') dari token gambar markdown
     * yang disisipkan pada teks atau opsi. Hanya memproses path yang berada di
     * bawah folder `questions/...`.
     */
    private function extractInlineImageStoragePaths(?string $content): array
    {
        if (!$content || !is_string($content)) return [];
        $paths = [];
        if (preg_match_all('/!\[[^\]]*\]\(([^\)]+)\)/', $content, $matches)) {
            foreach ($matches[1] as $rawUrl) {
                $url = trim((string)$rawUrl);
                $candidate = '';
                if ($url === '') continue;
                if (str_starts_with($url, 'http://') || str_starts_with($url, 'https://')) {
                    $path = parse_url($url, PHP_URL_PATH) ?: '';
                    $candidate = ltrim((string)$path, '/');
                } else {
                    $candidate = ltrim($url, '/');
                }
                $candidate = urldecode($candidate);
                if (str_starts_with($candidate, 'storage/')) {
                    $candidate = substr($candidate, strlen('storage/'));
                }
                if (str_starts_with($candidate, 'questions/')) {
                    $paths[] = $candidate;
                }
            }
        }
        return array_values(array_unique($paths));
    }

    public function index(Request $request)
    {
        $query = Subject::withCount('questions')->orderBy('name');
        if ($request->filled('id_school')) {
            $query->where('id_school', (int)$request->input('id_school'));
        }
        if ($request->filled('id_grade')) {
            $query->where('id_grade', (int)$request->input('id_grade'));
        }
        // Filter khusus: guru hanya melihat mapel yang dibuat oleh dirinya
        $user = $request->user();
        if ($user && $user->hasUserRole('guru')) {
            // Jika guru tidak memiliki izin view-all-subjects, batasi ke created_by
            if (!$user->hasPermission('view-all-subjects')) {
                $query->where('created_by', $user->id);
            }
        }
        $subjects = $query->get();
        return response()->json($subjects);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'nullable|string|max:50',
            'description' => 'nullable|string',
            'id_school' => 'required|exists:schools,id',
            'id_grade' => 'required|exists:grades,id',
        ]);

        $data = $validated;
        // Tetapkan pemilik mapel sebagai user yang login
        if ($request->user()) {
            $data['created_by'] = $request->user()->id;
        }
        $subject = Subject::create($data);
        return response()->json($subject, 201);
    }

    public function show(Subject $subject)
    {
        return response()->json($subject);
    }

    public function update(Request $request, Subject $subject)
    {
        // Guru hanya boleh mengubah mapel miliknya
        $user = $request->user();
        if ($user && $user->hasUserRole('guru')) {
            if ($subject->created_by !== $user->id) {
                return response()->json(['message' => 'Anda tidak berhak mengubah mata pelajaran milik orang lain'], 403);
            }
        }
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'nullable|string|max:50',
            'description' => 'nullable|string',
            'id_school' => 'required|exists:schools,id',
            'id_grade' => 'required|exists:grades,id',
        ]);
        $subject->update($validated);
        return response()->json($subject);
    }

    public function destroy(Subject $subject)
    {
        // Guru hanya boleh menghapus mapel miliknya
        $user = request()->user();
        if ($user && $user->hasUserRole('guru')) {
            if ($subject->created_by !== $user->id) {
                return response()->json(['message' => 'Anda tidak berhak menghapus mata pelajaran milik orang lain'], 403);
            }
        }
        // Hapus semua file gambar soal yang terkait dan folder mapel
        try {
            $subjectSlug = Str::slug((string)($subject?->name ?? 'mapel'));
            $baseDir = 'questions/' . $subjectSlug;

            // Hapus file image pada setiap soal di subject ini (utama + inline)
            $questions = Question::where('subject_id', $subject->id)
                ->get(['id','image_path','text','option_a','option_b','option_c','option_d','option_e']);
            foreach ($questions as $q) {
                // Hapus gambar utama
                if ($q->image_path) {
                    Storage::disk('public')->delete($q->image_path);
                }
                // Hapus gambar inline dari teks dan opsi
                $all = [$q->text, $q->option_a, $q->option_b, $q->option_c, $q->option_d, $q->option_e];
                $toDelete = [];
                foreach ($all as $s) {
                    foreach ($this->extractInlineImageStoragePaths($s) as $p) {
                        $toDelete[$p] = true;
                    }
                }
                foreach (array_keys($toDelete) as $p) {
                    Storage::disk('public')->delete($p);
                }
            }
            // Hapus folder subject jika ada (ini juga menghapus sisa inline images baru)
            if (Storage::disk('public')->exists($baseDir)) {
                Storage::disk('public')->deleteDirectory($baseDir);
            }
        } catch (\Throwable $e) {
            // Lanjutkan penghapusan subject meskipun terjadi error pada file system
        }

        $subject->delete();
        return response()->json(['message' => 'Deleted']);
    }

    /**
     * Hapus banyak mata pelajaran (opsional: filter berdasarkan sekolah/grade) beserta seluruh
     * soal dan gambar terkait, serta ujian yang terkait dengan subject tersebut.
     * Endpoint ini untuk superadmin.
     */
    public function destroyAll(Request $request)
    {
        $idSchool = $request->input('id_school');
        $idGrade = $request->input('id_grade');

        return DB::transaction(function () use ($idSchool, $idGrade) {
            $subjectQuery = Subject::query();
            if (!empty($idSchool)) {
                $subjectQuery->where('id_school', (int)$idSchool);
            }
            if (!empty($idGrade)) {
                $subjectQuery->where('id_grade', (int)$idGrade);
            }

            $subjects = $subjectQuery->get();
            if ($subjects->isEmpty()) {
                return response()->json([
                    'deleted_subjects' => 0,
                    'deleted_questions' => 0,
                    'deleted_exams' => 0,
                    'deleted_sessions' => 0,
                    'deleted_results' => 0,
                    'message' => 'Tidak ada mata pelajaran yang cocok untuk dihapus'
                ]);
            }

            $subjectIds = $subjects->pluck('id')->all();

            // Hapus file gambar untuk setiap subject terlebih dahulu
            foreach ($subjects as $subject) {
                try {
                    $subjectSlug = Str::slug((string)($subject?->name ?? 'mapel'));
                    $baseDir = 'questions/' . $subjectSlug;

                    $questions = Question::where('subject_id', $subject->id)
                        ->get(['id','image_path','text','option_a','option_b','option_c','option_d','option_e']);
                    foreach ($questions as $q) {
                        if ($q->image_path) {
                            Storage::disk('public')->delete($q->image_path);
                        }
                        $all = [$q->text, $q->option_a, $q->option_b, $q->option_c, $q->option_d, $q->option_e];
                        $toDelete = [];
                        foreach ($all as $s) {
                            foreach ($this->extractInlineImageStoragePaths($s) as $p) {
                                $toDelete[$p] = true;
                            }
                        }
                        foreach (array_keys($toDelete) as $p) {
                            Storage::disk('public')->delete($p);
                        }
                    }
                    if (Storage::disk('public')->exists($baseDir)) {
                        Storage::disk('public')->deleteDirectory($baseDir);
                    }
                } catch (\Throwable $e) {
                    // Abaikan error filesystem agar proses DB tetap berjalan
                }
            }

            // Hapus ujian terkait subject ini (beserta sesi & hasilnya)
            $examIds = Exam::whereIn('id_subject', $subjectIds)->pluck('id')->all();
            $deletedResults = 0;
            $deletedSessions = 0;
            $deletedExams = 0;
            if (!empty($examIds)) {
                $sessionIds = ExamSession::whereIn('exam_id', $examIds)->pluck('id')->all();
                if (!empty($sessionIds)) {
                    $deletedResults = ExamResult::whereIn('exam_session_id', $sessionIds)->delete();
                    $deletedSessions = ExamSession::whereIn('id', $sessionIds)->delete();
                }
                $deletedExams = Exam::whereIn('id', $examIds)->delete();
            }

            // Hapus soal di semua subject tersebut
            $deletedQuestions = Question::whereIn('subject_id', $subjectIds)->delete();

            // Terakhir, hapus subject
            $deletedSubjects = Subject::whereIn('id', $subjectIds)->delete();

            return response()->json([
                'deleted_subjects' => $deletedSubjects,
                'deleted_questions' => $deletedQuestions,
                'deleted_exams' => $deletedExams,
                'deleted_sessions' => $deletedSessions,
                'deleted_results' => $deletedResults,
                'message' => 'Berhasil menghapus mata pelajaran beserta soal, gambar, dan ujian terkait'
            ]);
        });
    }
}