<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Question;
use App\Models\Subject;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class QuestionController extends Controller
{
    /**
     * Ekstrak path storage relatif (disk 'public') dari token gambar markdown
     * yang disisipkan pada teks atau opsi. Hanya memproses path yang berada di
     * bawah folder `questions/...` agar tidak menghapus file lain.
     */
    private function extractInlineImageStoragePaths(?string $content): array
    {
        if (!$content || !is_string($content)) return [];
        $paths = [];
        // Cocokkan token markdown: ![alt](url)
        if (preg_match_all('/!\[[^\]]*\]\(([^\)]+)\)/', $content, $matches)) {
            foreach ($matches[1] as $rawUrl) {
                $url = trim((string)$rawUrl);
                $candidate = '';
                if ($url === '') continue;
                // Jika absolute URL, ambil path-nya saja
                if (str_starts_with($url, 'http://') || str_starts_with($url, 'https://')) {
                    $path = parse_url($url, PHP_URL_PATH) ?: '';
                    $candidate = ltrim((string)$path, '/');
                } else {
                    $candidate = ltrim($url, '/');
                }
                $candidate = urldecode($candidate);
                // Normalisasi: hapus prefix 'storage/' → menjadi path relatif di disk public
                if (str_starts_with($candidate, 'storage/')) {
                    $candidate = substr($candidate, strlen('storage/'));
                }
                // Hanya terima path di bawah folder questions/
                if (str_starts_with($candidate, 'questions/')) {
                    $paths[] = $candidate;
                }
            }
        }
        // Unikkan agar tidak delete berulang
        return array_values(array_unique($paths));
    }
    private function sanitizeText(string $value, bool $allowBasicTags = true): string
    {
        // Hilangkan tag <script> dan atribut event handler sederhana
        $clean = preg_replace('/<script\b[^>]*>(.*?)<\/script>/is', '', $value);
        $clean = preg_replace('/on[a-z]+\s*=\s*"[^"]*"/i', '', $clean);
        $clean = preg_replace("/on[a-z]+\s*=\s*'[^']*'/i", '', $clean);
        if ($allowBasicTags) {
            // Izinkan tag dasar termasuk strong/em agar tetap dirender sebagai bold/italic
            $clean = strip_tags($clean, '<b><strong><i><em><u><br><p><ul><ol><li><sup><sub>');
        } else {
            $clean = strip_tags($clean);
        }
        return trim($clean ?? '');
    }
    public function index(Request $request)
    {
        // Urutkan soal berdasarkan waktu pembuatan paling awal terlebih dahulu
        $query = Question::query()->orderBy('created_at', 'asc')->orderBy('id', 'asc');
        if ($request->filled('subject_id')) {
            $query->where('subject_id', (int) $request->input('subject_id'));
        }
        return $query->paginate(20);
    }

    public function store(Request $request)
    {
        // Deteksi mode essay: key_answer berisi teks (bukan A–E) atau opsi B–D kosong/'-'
        $valKey = strtoupper((string)$request->input('key_answer', ''));
        $optB = (string)$request->input('option_b', '');
        $optC = (string)$request->input('option_c', '');
        $optD = (string)$request->input('option_d', '');
        $isBlankOrDash = function ($s) { $t = trim((string)$s); return $t === '' || $t === '-'; };
        $isEssay = (!in_array($valKey, ['A','B','C','D','E'], true)) || ($isBlankOrDash($optB) && $isBlankOrDash($optC) && $isBlankOrDash($optD));
    
        if ($isEssay) {
            $rules = [
                'text' => ['required','string'],
                'option_a' => ['nullable','string','max:2000'],
                'option_b' => ['nullable','string','max:2000'],
                'option_c' => ['nullable','string','max:2000'],
                'option_d' => ['nullable','string','max:2000'],
                'option_e' => ['nullable','string','max:2000'],
                'key_answer' => ['required','string','max:5000'],
                'image' => ['nullable','image','mimes:jpeg,jpg,png,webp','max:2048'],
                'subject_id' => ['required','exists:subjects,id'],
            ];
        } else {
            $rules = [
                'text' => ['required','string'],
                'option_a' => ['required','string','max:2000'],
                'option_b' => ['required','string','max:2000'],
                'option_c' => ['required','string','max:2000'],
                'option_d' => ['required','string','max:2000'],
                'option_e' => ['nullable','string','max:2000'],
                'key_answer' => ['required','in:A,B,C,D,E'],
                'image' => ['nullable','image','mimes:jpeg,jpg,png,webp','max:2048'],
                'subject_id' => ['required','exists:subjects,id'],
            ];
        }
    
        $validated = $request->validate($rules);

        $path = null;
        // Tentukan folder berdasarkan nama mapel
        $subject = Subject::find((int)$validated['subject_id']);
        $subjectSlug = Str::slug((string)($subject?->name ?? 'mapel'));
        $baseDir = 'questions/'.$subjectSlug;
        Storage::disk('public')->makeDirectory($baseDir);
        if ($request->hasFile('image')) {
            $ext = strtolower($request->file('image')->getClientOriginalExtension());
            $filename = Str::uuid().'.'.$ext;
            $path = $request->file('image')->storeAs($baseDir, $filename, 'public');
        }

        $question = Question::create([
            'text' => $this->sanitizeText($validated['text'], true),
            // Izinkan tag dasar pada opsi agar bold/italic/underline tampil
            'option_a' => $this->sanitizeText($validated['option_a'], true),
            'option_b' => isset($validated['option_b']) && $validated['option_b'] !== '' ? $this->sanitizeText($validated['option_b'], true) : '',
            'option_c' => isset($validated['option_c']) && $validated['option_c'] !== '' ? $this->sanitizeText($validated['option_c'], true) : '',
            'option_d' => isset($validated['option_d']) && $validated['option_d'] !== '' ? $this->sanitizeText($validated['option_d'], true) : '',
            'option_e' => isset($validated['option_e']) && $validated['option_e'] !== '' ? $this->sanitizeText($validated['option_e'], true) : '',
            'key_answer' => $validated['key_answer'],
            'image_path' => $path,
            'created_by' => $request->user()?->id,
            'subject_id' => $validated['subject_id'],
        ]);

        return response()->json($question, 201);
    }

    // Upload image untuk disisipkan di tengah teks (inline)
    public function uploadImage(Request $request)
    {
        $request->validate([
            'image' => ['required','image','mimes:jpeg,jpg,png,webp','max:2048'],
            'subject_id' => ['required','exists:subjects,id'],
        ]);

        $ext = strtolower($request->file('image')->getClientOriginalExtension());
        $filename = Str::uuid().'.'.$ext;
        // Simpan ke folder berdasarkan nama mapel
        $subject = Subject::find((int)$request->input('subject_id'));
        $subjectSlug = Str::slug((string)($subject?->name ?? 'mapel'));
        $baseDir = 'questions/'.$subjectSlug;
        Storage::disk('public')->makeDirectory($baseDir);
        $path = $request->file('image')->storeAs($baseDir, $filename, 'public');
        $url = Storage::url($path); // menghasilkan /storage/questions/...

        return response()->json([
            'path' => $path,
            'url' => $url,
        ], 201);
    }

    public function update(Request $request, Question $question)
    {
        // Deteksi mode essay: key_answer berisi teks (bukan A–E) atau opsi B–D kosong/'-'
        $valKey = strtoupper((string)$request->input('key_answer', ''));
        $optB = (string)$request->input('option_b', '');
        $optC = (string)$request->input('option_c', '');
        $optD = (string)$request->input('option_d', '');
        $isBlankOrDash = function ($s) { $t = trim((string)$s); return $t === '' || $t === '-'; };
        $isEssay = (!in_array($valKey, ['A','B','C','D','E'], true)) || ($isBlankOrDash($optB) && $isBlankOrDash($optC) && $isBlankOrDash($optD));
    
        if ($isEssay) {
            $rules = [
                'text' => ['required','string'],
                'option_a' => ['nullable','string','max:2000'],
                'option_b' => ['nullable','string','max:2000'],
                'option_c' => ['nullable','string','max:2000'],
                'option_d' => ['nullable','string','max:2000'],
                'option_e' => ['nullable','string','max:2000'],
                'key_answer' => ['required','string','max:5000'],
                'image' => ['nullable','image','mimes:jpeg,jpg,png,webp','max:2048'],
                'subject_id' => ['required','exists:subjects,id'],
            ];
        } else {
            $rules = [
                'text' => ['required','string'],
                'option_a' => ['required','string','max:2000'],
                'option_b' => ['required','string','max:2000'],
                'option_c' => ['required','string','max:2000'],
                'option_d' => ['required','string','max:2000'],
                'option_e' => ['nullable','string','max:2000'],
                'key_answer' => ['required','in:A,B,C,D,E'],
                'image' => ['nullable','image','mimes:jpeg,jpg,png,webp','max:2048'],
                'subject_id' => ['required','exists:subjects,id'],
            ];
        }
    
        $validated = $request->validate($rules);

        // Update optional image if provided
        if ($request->hasFile('image')) {
            // Hapus file lama jika ada
            if ($question->image_path) {
                Storage::disk('public')->delete($question->image_path);
            }
            $ext = strtolower($request->file('image')->getClientOriginalExtension());
            $filename = Str::uuid().'.'.$ext;
            // Simpan ke folder berdasarkan nama mapel
            $subject = Subject::find((int)$validated['subject_id']);
            $subjectSlug = Str::slug((string)($subject?->name ?? 'mapel'));
            $baseDir = 'questions/'.$subjectSlug;
            Storage::disk('public')->makeDirectory($baseDir);
            $path = $request->file('image')->storeAs($baseDir, $filename, 'public');
            $question->image_path = $path;
        }

        $question->update([
            'text' => $this->sanitizeText($validated['text'], true),
            // Izinkan tag dasar pada opsi agar bold/italic/underline tampil
            'option_a' => $this->sanitizeText($validated['option_a'], true),
            'option_b' => isset($validated['option_b']) && $validated['option_b'] !== '' ? $this->sanitizeText($validated['option_b'], true) : '',
            'option_c' => isset($validated['option_c']) && $validated['option_c'] !== '' ? $this->sanitizeText($validated['option_c'], true) : '',
            'option_d' => isset($validated['option_d']) && $validated['option_d'] !== '' ? $this->sanitizeText($validated['option_d'], true) : '',
            'option_e' => isset($validated['option_e']) && $validated['option_e'] !== '' ? $this->sanitizeText($validated['option_e'], true) : '',
            'key_answer' => $validated['key_answer'],
            'subject_id' => $validated['subject_id'],
        ]);

        return response()->json($question);
    }

    public function destroy(Question $question)
    {
        // Hapus file image utama jika ada
        if ($question->image_path) {
            Storage::disk('public')->delete($question->image_path);
        }
        // Hapus file gambar inline yang disisipkan pada teks dan opsi
        try {
            $all = [
                $question->text,
                $question->option_a,
                $question->option_b,
                $question->option_c,
                $question->option_d,
                $question->option_e,
            ];
            $toDelete = [];
            foreach ($all as $s) {
                foreach ($this->extractInlineImageStoragePaths($s) as $p) {
                    $toDelete[$p] = true;
                }
            }
            foreach (array_keys($toDelete) as $p) {
                Storage::disk('public')->delete($p);
            }
        } catch (\Throwable $e) {
            // Abaikan error filesystem agar proses penghapusan data tetap lanjut
        }
        $question->delete();
        return response()->json(['message' => 'Question deleted']);
    }
}