<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\Models\ExamParticipant;
use Illuminate\Support\Str;

class ParticipantAuthController extends Controller
{
    private function generateNumericToken(int $digits = 5): string
    {
        $min = (int) pow(10, $digits - 1);
        $max = (int) pow(10, $digits) - 1;
        $tries = 0;
        do {
            $candidate = (string) random_int($min, $max);
            $exists = ExamParticipant::where('login_token', $candidate)->exists();
            $tries++;
            if (!$exists) return $candidate;
        } while ($tries < 100);
        // Fallback: gunakan token acak panjang jika kehabisan kombinasi
        return Str::random(12);
    }
    public function login(Request $request)
    {
        $request->validate([
            'nisn' => 'required|string',
            'password' => 'required|string',
            'device_fp' => 'required|string|max:255',
        ]);

        $acquired = false;
        $lock = null;
        try {
            // Coba gunakan Redis lock; jika store tidak mendukung locking, fallback ke transaksi saja
            $lock = \Illuminate\Support\Facades\Cache::lock('participant-login:nisn:' . strtolower($request->nisn), 5);
            $lock->block(3);
            $acquired = true;
        } catch (\Throwable $ignored) {
            // Fallback: lanjut tanpa lock Redis
        }

        try {
            return \Illuminate\Support\Facades\DB::transaction(function() use ($request) {
                $participant = ExamParticipant::where('nisn', $request->nisn)->lockForUpdate()->first();
                if (!$participant) {
                    return response()->json(['message' => 'NISN tidak terdaftar'], 404);
                }

                $hashed = $participant->password;
                $usesBcrypt = is_string($hashed) && (
                    Str::startsWith($hashed, '$2y$') || Str::startsWith($hashed, '$2a$') || Str::startsWith($hashed, '$2b$')
                );

                if ($usesBcrypt) {
                    if (!Hash::check($request->password, $hashed)) {
                        return response()->json(['message' => 'NISN atau password salah'], 401);
                    }
                } else {
                    if ($request->password !== $hashed) {
                        return response()->json(['message' => 'NISN atau password salah'], 401);
                    }
                    $participant->password = Hash::make($request->password);
                }

                if ($participant->bound_device_hash) {
                    if (!Hash::check($request->device_fp, $participant->bound_device_hash)) {
                        return response()->json(['message' => 'Akun sudah terikat dengan perangkat lain'], 403);
                    }
                } else {
                    $participant->bound_device_hash = Hash::make($request->device_fp);
                    $participant->device_bound_at = now();
                }

                if (empty($participant->login_token)) {
                    $participant->login_token = $this->generateNumericToken(5);
                }

                $participant->last_activity = now();
                $participant->save();

                $token = $participant->createToken('participant_token')->plainTextToken;

                // Tambah: ambil nama sekolah dari kelas peserta
                $class = \App\Models\SchoolClass::find($participant->id_kelas);
                $schoolName = $class?->school?->nama;

                return response()->json([
                    'participant' => [
                        'id' => $participant->id,
                        'nisn' => $participant->nisn,
                        'nama' => $participant->nama,
                        'id_kelas' => $participant->id_kelas,
                        'name_class' => $participant->name_class,
                        'jurusan' => $participant->jurusan,
                        'school_name' => $schoolName,
                    ],
                    'token' => $token,
                ]);
            });
        } finally {
            if ($acquired && $lock) {
                $lock->release();
            }
        }
    }

    public function loginByToken(Request $request)
    {
        $request->validate([
            'token' => 'required|string',
            'device_fp' => 'required|string|max:255',
        ]);

        $acquired = false;
        $lock = null;
        try {
            // Coba gunakan Redis lock; jika store tidak mendukung locking, fallback ke transaksi saja
            $lock = \Illuminate\Support\Facades\Cache::lock('participant-login:token:' . strtolower($request->token), 5);
            $lock->block(3);
            $acquired = true;
        } catch (\Throwable $ignored) {
            // Fallback: lanjut tanpa lock Redis
        }

        try {
            return \Illuminate\Support\Facades\DB::transaction(function() use ($request) {
                $participant = ExamParticipant::where('login_token', $request->token)->lockForUpdate()->first();
                if (!$participant) {
                    return response()->json(['message' => 'Token tidak valid'], 404);
                }

                if ($participant->bound_device_hash) {
                    if (!Hash::check($request->device_fp, $participant->bound_device_hash)) {
                        return response()->json(['message' => 'Token ini sudah terikat dengan perangkat lain'], 403);
                    }
                } else {
                    $participant->bound_device_hash = Hash::make($request->device_fp);
                    $participant->device_bound_at = now();
                }

                $participant->last_activity = now();
                $participant->save();

                $token = $participant->createToken('participant_token')->plainTextToken;

                // Tambah: ambil nama sekolah dari kelas peserta
                $class = \App\Models\SchoolClass::find($participant->id_kelas);
                $schoolName = $class?->school?->nama;

                return response()->json([
                    'participant' => [
                        'id' => $participant->id,
                        'nisn' => $participant->nisn,
                        'nama' => $participant->nama,
                        'id_kelas' => $participant->id_kelas,
                        'name_class' => $participant->name_class,
                        'jurusan' => $participant->jurusan,
                        'school_name' => $schoolName,
                    ],
                    'token' => $token,
                ]);
            });
        } finally {
            if ($acquired && $lock) {
                $lock->release();
            }
        }
    }

    public function logout(Request $request)
    {
        $user = $request->user();
        if ($user) {
            $user->tokens()->delete();
            $user->update(['last_activity' => null]);
        }
        return response()->json(['message' => 'Logout berhasil']);
    }

    public function me(Request $request)
    {
        $p = $request->user();
        if (!$p) return response()->json(['message' => 'Unauthenticated'], 401);
        $isOnline = $p->last_activity && now()->diffInMinutes($p->last_activity) <= 15;

        // Tambah: ambil nama sekolah dari kelas peserta
        $class = \App\Models\SchoolClass::find($p->id_kelas);
        $schoolName = $class?->school?->nama;

        return response()->json([
            'participant' => [
                'id' => $p->id,
                'nisn' => $p->nisn,
                'nama' => $p->nama,
                'id_kelas' => $p->id_kelas,
                'name_class' => $p->name_class,
                'jurusan' => $p->jurusan,
                'school_name' => $schoolName,
            ],
            'is_online' => $isOnline,
        ]);
    }

    // Heartbeat untuk peserta: update last_activity milik user peserta saat ini
    public function updateActivity(Request $request)
    {
        $p = $request->user();
        if (!$p || !($p instanceof \App\Models\ExamParticipant)) {
            return response()->json(['message' => 'Unauthenticated'], 401);
        }
        $p->last_activity = now();
        $p->save();
        return response()->json(['message' => 'Aktivitas peserta diperbarui']);
    }
}