<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ExamParticipant;
use App\Models\ExamSession;
use App\Models\ExamResult;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Dompdf\Dompdf;
use Dompdf\Options;
use Carbon\Carbon;
 

class ExamParticipantController extends Controller
{
    private function generateNumericToken(int $digits = 5): string
    {
        $min = (int) pow(10, $digits - 1);
        $max = (int) pow(10, $digits) - 1;
        $tries = 0;
        do {
            $candidate = (string) random_int($min, $max);
            $exists = ExamParticipant::where('login_token', $candidate)->exists();
            $tries++;
            if (!$exists) return $candidate;
        } while ($tries < 100);
        // Fallback bila penuh kombinasi: gunakan token acak panjang
        return Str::random(12);
    }
    public function index(Request $request)
    {
        $query = ExamParticipant::query()->orderByDesc('id');
        if ($request->filled('id_kelas')) {
            $query->where('id_kelas', (int)$request->input('id_kelas'));
        }
        // Jangan expose field sensitif (login_token, bound_device_hash)
        return response()->json($query->get()->makeHidden(['login_token','bound_device_hash']));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nisn' => 'required|string|unique:exam_participants,nisn',
            'nama' => 'required|string|max:255',
            'id_kelas' => 'required|exists:classes,id',
            'name_class' => 'required|string|max:255',
            'jurusan' => 'nullable|string|max:255',
            'password' => 'nullable|string|min:6',
        ]);

        $data = $validated;
        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->input('password'));
        } else {
            $data['password'] = Hash::make('12345678');
        }

        // Generate token login unik 5 digit saat pembuatan peserta
        $data['login_token'] = $this->generateNumericToken(5);

        $participant = ExamParticipant::create($data);
        return response()->json($participant, 201);
    }

    public function show(Request $request, ExamParticipant $examParticipant)
    {
        // Batasi akses: peserta ujian tidak boleh melihat detail peserta (termasuk token)
        $user = $request->user();
        if ($user instanceof \App\Models\ExamParticipant) {
            return response()->json(['message' => 'Hanya admin dapat melihat detail peserta'], 403);
        }
        // Tampilkan login_token untuk admin, tetap sembunyikan bound_device_hash
        return response()->json($examParticipant->makeVisible(['login_token']));
    }

    public function update(Request $request, ExamParticipant $examParticipant)
    {
        $validated = $request->validate([
            'nisn' => 'required|string|unique:exam_participants,nisn,' . $examParticipant->id,
            'nama' => 'required|string|max:255',
            'id_kelas' => 'required|exists:classes,id',
            'name_class' => 'required|string|max:255',
            'jurusan' => 'nullable|string|max:255',
            'password' => 'nullable|string|min:6',
        ]);
        $data = $validated;
        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->input('password'));
        } else {
            unset($data['password']);
        }

        $examParticipant->update($data);
        return response()->json($examParticipant);
    }

    public function destroy(ExamParticipant $examParticipant)
    {
        $examParticipant->delete();
        return response()->json(['message' => 'Deleted']);
    }

    public function resetPassword(ExamParticipant $examParticipant)
    {
        $examParticipant->update(['password' => Hash::make('12345678')]);
        return response()->json(['message' => 'Password direset ke 12345678']);
    }

    /**
     * Hapus semua peserta ujian (opsional berdasarkan kelas) beserta sesi & hasilnya
     */
    public function destroyAll(Request $request)
    {
        $classId = $request->input('id_kelas');

        return DB::transaction(function () use ($classId) {
            $participantsQuery = ExamParticipant::query();
            if (!empty($classId)) {
                $participantsQuery->where('id_kelas', (int)$classId);
            }

            $participantIds = $participantsQuery->pluck('id')->all();
            if (empty($participantIds)) {
                return response()->json(['message' => 'Tidak ada peserta untuk dihapus', 'deleted' => ['participants' => 0, 'sessions' => 0, 'results' => 0]]);
            }

            // Ambil semua sesi yang terkait peserta-peserta ini
            $sessionIds = ExamSession::whereIn('exam_participant_id', $participantIds)->pluck('id')->all();

            // Hapus hasil ujian untuk sesi-sesi tersebut
            $deletedResults = 0;
            if (!empty($sessionIds)) {
                $deletedResults = ExamResult::whereIn('exam_session_id', $sessionIds)->delete();
            }

            // Hapus sesi ujian
            $deletedSessions = 0;
            if (!empty($sessionIds)) {
                $deletedSessions = ExamSession::whereIn('id', $sessionIds)->delete();
            }

            // Hapus peserta
            $deletedParticipants = ExamParticipant::whereIn('id', $participantIds)->delete();

            return response()->json([
                'message' => 'Peserta, sesi, dan hasil dihapus',
                'deleted' => [
                    'participants' => $deletedParticipants,
                    'sessions' => $deletedSessions,
                    'results' => $deletedResults,
                ],
            ]);
        });
    }

    // Reset binding perangkat: kosongkan fingerprint terikat sehingga peserta bisa login dari perangkat baru
    public function unbindDevice(Request $request, ExamParticipant $examParticipant)
    {
        // Batasi agar hanya admin (User dengan permission) yang boleh reset perangkat
        $actor = $request->user();
        if ($actor instanceof \App\Models\ExamParticipant) {
            return response()->json(['message' => 'Hanya admin dapat mereset perangkat peserta'], 403);
        }
        $examParticipant->update([
            'bound_device_hash' => null,
            'device_bound_at' => null,
        ]);
        return response()->json(['message' => 'Perangkat peserta berhasil direset']);
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,csv,txt|max:5120',
            'id_kelas' => 'nullable|exists:classes,id',
        ]);

        $idKelasDefault = $request->input('id_kelas');
        $nameClassDefault = null;
        if ($idKelasDefault) {
            $class = \App\Models\SchoolClass::find($idKelasDefault);
            if (!$class) return response()->json(['message' => 'Kelas tidak valid'], 422);
            $nameClassDefault = $class->name;
        }

        $file = $request->file('file');
        $path = $file->getRealPath();
        $ext = strtolower($file->getClientOriginalExtension());

        $rows = [];
        if (in_array($ext, ['csv', 'txt'])) {
            $handle = fopen($path, 'r');
            $header = null;
            while (($data = fgetcsv($handle, 0, ',')) !== false) {
                if (!$header) { $header = array_map('trim', $data); continue; }
                $row = [];
                foreach ($header as $i => $col) { $row[$col] = $data[$i] ?? null; }
                $rows[] = $row;
            }
            fclose($handle);
        } else {
            $spreadsheet = IOFactory::load($path);
            $sheet = $spreadsheet->getActiveSheet()->toArray();
            $header = array_map(fn($v) => trim((string) $v), $sheet[0] ?? []);
            for ($i = 1; $i < count($sheet); $i++) {
                $line = $sheet[$i];
                if (count(array_filter($line, fn($v) => $v !== null && $v !== '')) === 0) continue;
                $row = [];
                foreach ($header as $idx => $col) { $row[$col] = $line[$idx] ?? null; }
                $rows[] = $row;
            }
        }

        $imported = 0; $updated = 0; $errors = [];
        foreach ($rows as $row) {
            $nisn = trim((string)($row['nisn'] ?? ''));
            $nama = trim((string)($row['nama'] ?? ''));
            $jurusan = isset($row['jurusan']) ? trim((string)$row['jurusan']) : null;
            $passwordPlain = isset($row['password']) ? trim((string)$row['password']) : '';
            $id_kelas = $idKelasDefault ?: (isset($row['id_kelas']) ? (int)$row['id_kelas'] : null);

            if (!$nisn || !$nama || !$id_kelas) {
                $errors[] = ['row' => $row, 'message' => 'Kolom wajib (nisn, nama, id_kelas) tidak lengkap'];
                continue;
            }

            $name_class = $nameClassDefault;
            if (!$name_class) {
                $kelas = \App\Models\SchoolClass::find($id_kelas);
                if (!$kelas) { $errors[] = ['row' => $row, 'message' => 'id_kelas tidak ditemukan']; continue; }
                $name_class = $kelas->name;
            }

            $existing = ExamParticipant::where('nisn', $nisn)->first();
            if ($existing) {
                $updateData = [
                    'nama' => $nama,
                    'id_kelas' => $id_kelas,
                    'name_class' => $name_class,
                    'jurusan' => $jurusan,
                ];
                if ($passwordPlain !== '') {
                    $updateData['password'] = Hash::make($passwordPlain);
                }
                // Jika belum memiliki token, buatkan token 5 digit
                if (empty($existing->login_token)) {
                    $updateData['login_token'] = $this->generateNumericToken(5);
                }
                $existing->update($updateData);
                $updated++;
            } else {
                ExamParticipant::create([
                    'nisn' => $nisn,
                    'nama' => $nama,
                    'id_kelas' => $id_kelas,
                    'name_class' => $name_class,
                    'jurusan' => $jurusan,
                    'password' => $passwordPlain !== '' ? Hash::make($passwordPlain) : Hash::make('12345678'),
                    'login_token' => $this->generateNumericToken(5),
                ]);
                $imported++;
            }
        }

        return response()->json(['imported' => $imported, 'updated' => $updated, 'errors' => $errors]);
    }

    // Export token peserta (nisn, nama, kelas, token) ke Excel
    public function exportTokens(Request $request)
    {
        $query = ExamParticipant::query()->orderBy('id');
        if ($request->filled('id_kelas')) {
            $query->where('id_kelas', (int)$request->input('id_kelas'));
        }
        $rows = $query->get(['nisn','nama','name_class','login_token']);

        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $headers = ['nisn', 'nama', 'kelas', 'token'];
        foreach ($headers as $i => $h) {
            $colLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($i + 1);
            $sheet->setCellValue($colLetter . '1', $h);
            $sheet->getColumnDimension($colLetter)->setAutoSize(true);
        }
        $rowIdx = 2;
        foreach ($rows as $r) {
            $sheet->setCellValue('A'.$rowIdx, $r->nisn);
            $sheet->setCellValue('B'.$rowIdx, $r->nama);
            $sheet->setCellValue('C'.$rowIdx, $r->name_class);
            $sheet->setCellValue('D'.$rowIdx, $r->login_token ?? '');
            $rowIdx++;
        }

        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        ob_start();
        $writer->save('php://output');
        $excelOutput = ob_get_clean();

        $filename = 'tokens_peserta.xlsx';
        return response($excelOutput, 200, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Content-Disposition' => 'attachment; filename="'.$filename.'"',
        ]);
    }

    // Generate token massal untuk peserta yang belum memiliki token
    public function generateTokens(Request $request)
    {
        $validated = $request->validate([
            'id_kelas' => 'nullable|exists:classes,id',
        ]);

        $query = ExamParticipant::query();
        if (!empty($validated['id_kelas'])) {
            $query->where('id_kelas', $validated['id_kelas']);
        }
        $query->where(function($q){
            $q->whereNull('login_token')->orWhere('login_token', '');
        });

        $toGenerate = $query->get();
        $count = 0;
        foreach ($toGenerate as $p) {
            $p->login_token = $this->generateNumericToken(5);
            $p->save();
            $count++;
        }

        return response()->json(['generated' => $count]);
    }

    public function template(Request $request)
    {
        $idKelas = $request->query('id_kelas');
        $includeIdKelas = empty($idKelas);

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headers = $includeIdKelas ? ['nisn', 'nama', 'password', 'id_kelas'] : ['nisn', 'nama', 'password'];
        foreach ($headers as $i => $h) {
            $colLetter = Coordinate::stringFromColumnIndex($i + 1);
            $sheet->setCellValue($colLetter . '1', $h);
            $sheet->getColumnDimension($colLetter)->setAutoSize(true);
        }

        $writer = new Xlsx($spreadsheet);
        ob_start();
        $writer->save('php://output');
        $excelOutput = ob_get_clean();

        return response($excelOutput, 200, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Content-Disposition' => 'attachment; filename="template_peserta.xlsx"',
        ]);
    }

    // Bulk reset password to default (12345678) for all or per class
    public function resetPasswordBulk(Request $request)
    {
        $validated = $request->validate([
            'id_kelas' => 'nullable|exists:classes,id',
        ]);

        $query = ExamParticipant::query();
        if (!empty($validated['id_kelas'])) {
            $query->where('id_kelas', $validated['id_kelas']);
        }

        $updated = $query->update(['password' => Hash::make('12345678')]);
        return response()->json(['updated' => $updated, 'message' => 'Password peserta berhasil direset']);
    }

    // Export PDF: gabungkan Nomor Meja (kiri) dan Kartu Ujian (kanan), 5 baris per halaman
    public function exportCardsSeats(Request $request)
{
    $validated = $request->validate([
        'id_kelas' => 'required|exists:classes,id',
    ]);

    try {
        $class = \App\Models\SchoolClass::with(['school','grade'])->find((int)$validated['id_kelas']);
        if (!$class) {
            return response()->json(['message' => 'Kelas tidak ditemukan'], 404);
        }

        $school = $class->school;
        $grade = $class->grade;
        $participants = ExamParticipant::where('id_kelas', $class->id)
            ->orderBy('nama', 'asc')
            ->get(['id','nisn','nama','id_kelas','name_class','login_token']);

        if ($participants->isEmpty()) {
            return response()->json(['message' => 'Tidak ada peserta pada kelas ini'], 422);
        }

        $schoolName = htmlspecialchars((string)($school?->nama ?? ''));
        $className = htmlspecialchars((string)$class->name);
        $gradeName = htmlspecialchars((string)($grade?->grade ?? ''));
        $defaultPassword = '12345678';
        $address = htmlspecialchars((string)($school?->alamat ?? ''));
        $dateStr = \Carbon\Carbon::now()->locale('id')->isoFormat('D MMMM YYYY');
        $yearNow = \Carbon\Carbon::now()->format('Y');
        $cityName = htmlspecialchars((string)($school?->kota ?? ''));
        $schoolCode = htmlspecialchars((string)($school?->kode ?? ''));

        // --- fungsi helper gambar ---
        $imgToDataUri = function (?string $path): ?string {
            if (!$path) return null;
            try {
                if (preg_match('#^https?://#', $path)) {
                    $content = @file_get_contents($path);
                } elseif (Storage::disk('public')->exists($path)) {
                    $content = Storage::disk('public')->get($path);
                } elseif (file_exists(public_path($path))) {
                    $content = file_get_contents(public_path($path));
                } else {
                    return null;
                }
                if (!$content) return null;
                $ext = pathinfo($path, PATHINFO_EXTENSION) ?: 'png';
                $mime = 'image/' . ($ext === 'jpg' ? 'jpeg' : $ext);
                return 'data:' . $mime . ';base64,' . base64_encode($content);
            } catch (\Throwable $e) {
                return null;
            }
        };

        $logoSrc = $imgToDataUri((string)($school?->logo_path));
        $signSrc = $imgToDataUri((string)($school?->signature_path));
        $stampSrc = $imgToDataUri((string)($school?->stamp_path));
        $eduLogoSrc = $imgToDataUri('data/pendidikan.png');

        // ========================
        // SUSUN HTML
        // ========================
        $rowsHtml = '';
        $idx = 0;
        foreach ($participants as $p) {
            $idx++;
            $seatNo = $idx;
            $nama = htmlspecialchars((string)$p->nama);
            $nisn = htmlspecialchars((string)$p->nisn);
            $token = htmlspecialchars((string)($p->login_token ?? ''));
            $participantNo = htmlspecialchars((string)($yearNow . '-' . $gradeName . '-' . $p->id));

            $rowsHtml .= "
<table style='width:100%; border-collapse:collapse; margin-bottom:5px;'>
<tr>
<td style='width:50%; vertical-align:top; border:1px solid #000; padding:6px; height:160px;'>
    <div class=\"card-content\">
        <div class=\"kop-wrap\">
            ".($logoSrc ? "<img class=\"logo-left\" src=\"$logoSrc\"/>" : "")."
            ".($eduLogoSrc ? "<img class=\"logo-right\" src=\"$eduLogoSrc\"/>" : "")."
            <div class=\"kop-center\">
                <div class=\"kop-title\" style='font-size:10pt; font-weight:bold;'>NOMOR MEJA</div>
                <div class=\"kop-school\" style='font-size:8pt;'>$schoolName</div>
                <div class=\"kop-address\" style='font-size:7pt;'>$address</div>
            </div>
        </div>
        <div class=\"line\" style='border-top:1px solid #000; margin:2px 0;'></div>
        <div style='text-align:center; margin-top:2mm;'>
            <div style='font-weight:bold; font-size:9pt;'>Nomor Meja</div>
            <div style='font-size:22pt; line-height:1;'>$seatNo</div>
            <div style='font-weight:bold; font-size:8pt;'>No Peserta: $participantNo</div>
        </div>
    </div>
</td>

<td style='width:50%; vertical-align:top; border:1px solid #000; padding:6px; height:160px;'>
    <div class=\"card-content\">
        <!-- ✅ KOP DIPINDAH KE ATAS -->
        <div class=\"kop-wrap\">
            ".($logoSrc ? "<img class=\"logo-left\" src=\"$logoSrc\"/>" : "")."
            ".($eduLogoSrc ? "<img class=\"logo-right\" src=\"$eduLogoSrc\"/>" : "")."
            <div class=\"kop-center\">
                <div class=\"kop-title\" style='font-size:10pt; font-weight:bold;'>KARTU PESERTA</div>
                <div class=\"kop-school\" style='font-size:8pt;'>$schoolName</div>
                <div class=\"kop-address\" style='font-size:7pt;'>$address</div>
            </div>
        </div>

        <div class=\"line\" style='border-top:1px solid #000; margin:2px 0;'></div>
        <table style='width:100%; font-size:7pt; border-collapse:collapse;'>
            <tr><td style='width:90px;'>No Peserta</td><td>:</td><td>$participantNo</td></tr>
            <tr><td>Nama</td><td>:</td><td>$nama</td></tr>
            <tr><td>NISN</td><td>:</td><td>$nisn</td></tr>
            <tr><td>Jenjang</td><td>:</td><td>$gradeName</td></tr>
            <tr><td>Kelas</td><td>:</td><td>$className</td></tr>
            <tr><td>Password</td><td>:</td><td>$defaultPassword</td></tr>
            <tr><td>Token</td><td>:</td><td>$token</td></tr>
        </table>
        

        <div class=\"signature\" style='margin-top:2mm; font-size:7pt;'>
            ".($stampSrc ? "<img class=\"stamp\" src=\"$stampSrc\"/>" : "")."
            ".($signSrc ? "<img class=\"sign\" src=\"$signSrc\"/>" : "")."
            <div class=\"sign-info\">
                <div class=\"addr\">$cityName, $dateStr</div>
                ".(!$signSrc ? "<div class=\"ttd-label\">TTD Kepala Sekolah</div>" : "")."
                <div class=\"kepsek-name\">" . htmlspecialchars((string)($school?->kepsek ?? '')) . "</div>
                ".($schoolCode ? "<div class=\"school-code\">$schoolCode</div>" : "")."
            </div>
            <div class=\"kepsek-info\">
            </div>
        </div>
    </div>
</td>
</tr>
</table>
";

            // 5 baris per halaman
            if ($idx % 5 === 0) {
                $rowsHtml .= "<div class='page-break'></div>";
            }
        }

        $html = "<!DOCTYPE html><html><head><meta charset='utf-8'>
        <style>
        @page { size: A4; margin: 10mm; }
        body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 10pt; color: #111; }
        table { width: 100%; border-collapse: collapse; }
        .page-break { page-break-after: always; }
        .kop-wrap { position: relative; height: 21mm; overflow: hidden; }
        .logo-left { position: absolute; left: 2mm; top: 3mm; width: 18mm; opacity: 0.15; z-index: 1; }
        .logo-right { position: absolute; right: 2mm; top: 3mm; width: 18mm; opacity: 0.3; z-index: 1; }
        .kop-center { position: absolute; left: 0; right: 0; top: 5mm; text-align: center; z-index: 2; }
        .kop-title { font-weight: 700; font-size: 10pt; }
        .kop-school { font-size: 9pt; }
        .kop-address { font-size: 7pt; color: #444; }
        .signature { position: relative; min-height: 18mm; }
        .signature .sign { position: absolute; right: 10mm; top: 2mm; width: 25mm; opacity: 0.9; z-index: 2; }
        .signature .stamp { position: absolute; left: 22mm; top: 10mm; width: 24mm; opacity: 0.7; z-index: 1; }
        .sign-info { position: absolute; bottom: -3mm; right: 2mm; top: 2mm; text-align: right; font-size: 7pt; }
        .sign-info .kepsek-name { font-weight: bold; display:inline-block; border-bottom:1px solid #000; padding-bottom:1px; }
        </style></head><body>$rowsHtml</body></html>";

        // render PDF
        $dompdf = new \Dompdf\Dompdf();
        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();
        $filename = 'kartu_ujian_nomor_meja_kelas_'.$class->id.'.pdf';
        return response($dompdf->output(), 200, [
            'Content-Type' => 'application/pdf',
            'Content-Disposition' => 'attachment; filename="'.$filename.'"'
        ]);
    } catch (\Throwable $e) {
        return response()->json(['message' => $e->getMessage()], 500);
    }
}


    // Update activity ping for exam participants (by id or nisn)
    public function updateActivity(Request $request)
    {
        $request->validate([
            'id' => 'nullable|exists:exam_participants,id',
            'nisn' => 'nullable|string',
        ]);

        $participant = null;
        if ($request->filled('id')) {
            $participant = ExamParticipant::find($request->input('id'));
        } elseif ($request->filled('nisn')) {
            $participant = ExamParticipant::where('nisn', $request->input('nisn'))->first();
        }

        if (!$participant) {
            return response()->json(['message' => 'Peserta tidak ditemukan'], 404);
        }

        $participant->last_activity = now();
        $participant->save();

        return response()->json(['message' => 'Aktivitas peserta diperbarui']);
    }

    // [REMOVED] exportTokensJson: endpoint khusus load test dihapus sesuai permintaan
}