# Security Hardening Guide

This project includes baseline hardening for Laravel API + React frontend. Use this guide to deploy securely in production.

## Application Settings
- Set `APP_ENV=production` and `APP_DEBUG=false`.
- Enforce HTTPS by setting `FORCE_HTTPS=true` (enables `URL::forceScheme('https')`).
- Rate limits via env:
  - `API_RATE_LIMIT=60`
  - `LOGIN_RATE_LIMIT=5`
  - `SUBMIT_EXAM_RATE_LIMIT=2`
  - `UPLOAD_RATE_LIMIT=10`
  - `IMPORT_RATE_LIMIT=2`
  - `REPORT_RATE_LIMIT=30`
- Session cookie hardening:
  - `SESSION_SECURE_COOKIE=true`
  - `SESSION_HTTP_ONLY=true`
  - `SESSION_SAME_SITE=lax` (use `none` only with HTTPS + cross-site cookies)

## Laravel Middleware
- CSRF enabled for `web` group via `VerifyCsrfToken`.
- Security headers via `App\Http\Middleware\SecureHeaders`:
  - `X-Frame-Options: DENY`, `X-Content-Type-Options: nosniff`, etc.
- Rate limiters defined in `AppServiceProvider` and applied to critical routes:
  - login, submit answers, upload question images, import participants, report violations.

## File Uploads
- Question image upload: MIME restricted to `jpeg,jpg,png,webp`; size max 2–4 MB depending on endpoint.
- Import participants: only `xlsx,csv,txt`; size max 5 MB.
- Octane/RoadRunner static server forbids access to `.env`, `.env.*`, `.git`, `.php`, `.htaccess` via `.rr.yaml`.

## Reverse Proxy (Nginx) Example
```
server {
  listen 443 ssl http2;
  server_name example.com;

  ssl_certificate     /etc/ssl/example.crt;
  ssl_certificate_key /etc/ssl/example.key;

  # HSTS (preload optional)
  add_header Strict-Transport-Security "max-age=31536000; includeSubDomains" always;
  add_header X-Frame-Options DENY always;
  add_header X-Content-Type-Options nosniff always;
  add_header Referrer-Policy no-referrer-when-downgrade always;

  # Deny sensitive files
  location ~* /(\.env|\.env\..*|\.git|composer\.json|composer\.lock|package\.json|yarn\.lock|vendor|storage|tests) {
    deny all;
  }

  # Laravel public directory
  root /var/www/html/public;
  index index.php;

  location / {
    try_files $uri $uri/ /index.php?$query_string;
  }

  location ~ \.php$ {
    include fastcgi_params;
    fastcgi_pass 127.0.0.1:9000; # or php-fpm socket
    fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
  }
}

server {
  listen 80;
  server_name example.com;
  return 301 https://$host$request_uri;
}
```

## Apache Example
- Use `RewriteRule` to force HTTPS and `RedirectMatch` to deny access to `.env`, `.git`, `storage`, and `vendor`.

```
# Force HTTPS
RewriteEngine On
RewriteCond %{HTTPS} !=on
RewriteRule ^ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]

# Deny sensitive files
RedirectMatch 403 ^/(\.env|\.env\..*|\.git|composer\.json|composer\.lock|vendor|storage|tests)
```

## Additional Recommendations
- Place backend behind a firewall/WAF (Cloudflare, AWS WAF) and only expose required ports.
- Rotate tokens and review `login` attempts in logs.
- Keep dependencies updated and run `composer audit` during CI.
- Disable `expose_php` in php.ini, configure `max_upload_size` appropriately.
- Backups of database and storage (encrypted at rest).

This document provides operational guidance to complement code-level protections implemented in the repository.