# Scaling Guide: 5,000+ Concurrent Exam Users

This guide outlines backend, frontend, and infrastructure steps to reliably support 5,000+ simultaneous students taking exams.

## Backend (Laravel Octane + RoadRunner)

- Set Octane to RoadRunner and disable debug in production:
  - `.env`: `APP_ENV=production`, `APP_DEBUG=false`, `OCTANE_SERVER=roadrunner`, `OCTANE_HTTPS=true`
  - Use persistent workers to avoid PHP boot cost per request.
- Tune RoadRunner worker counts per CPU core:
  - Start with `workers = cores * 2` and benchmark.
  - Limit memory via Octane GC: `OCTANE_MAX_REQUESTS=500` (restart worker after N requests).
- Use Redis for cache/session/queue:
  - `.env`: `CACHE_DRIVER=redis`, `SESSION_DRIVER=redis`, `QUEUE_CONNECTION=redis`
  - Provision dedicated Redis for queue to avoid contention with cache/session.
- Queue workers for autosave and background tasks:
  - Run `php artisan queue:work --queue=autosave,default --sleep=1 --tries=3 --timeout=30` across multiple nodes.
  - Scale workers horizontally (e.g., 10–50 workers depending on throughput).
- Database optimization:
  - Enable read/write splitting if possible.
  - Add indexes on `exam_sessions.exam_id`, `exam_sessions.exam_participant_id`, and frequently filtered columns.
  - Use connection pool settings tuned for concurrency and DB server capacity.
- HTTP layer:
  - Gzip/Brotli compression and keep-alive enabled on the reverse proxy.
  - Set strict timeouts to protect from slow clients; serve static assets via CDN if available.

## Horizontal Load Balancing

- Put multiple app instances behind a reverse proxy (Nginx/HAProxy/Envoy):
  - Round-robin or least-connections balancing.
  - Health checks on `/api/exams/today` or a dedicated `/health` endpoint.
- Sticky session NOT required when using Redis for session storage.
- Terminate TLS at the proxy and forward `X-Forwarded-Proto` for HTTPS enforcement.

## Autosave Strategy

- Frontend sends debounced diffs to `/api/exam-sessions/{id}/autosave-answers`.
- Backend enqueues `SaveExamDraftAnswers` job on the `autosave` queue to avoid blocking the request.
- Use idempotent merge: draft is a map of `{question_id: answer}`. `null` removes a previously selected answer.
- Submit final answers via `/submit-answers` which performs evaluation and marks the session finished.

## Security & Anti-Cheating

- Participant login requires `device_fp` and binds the first login to a specific device.
- Token login mode (`/participant/login-token`) uses a unique per-student token, also bound to device.
- Frontend blocks right-click, copy/paste, mass selection, and logs screenshot attempts; impossible to fully block OS-level screenshots.
- Violations are posted to `/exam-sessions/{id}/violations` with throttling.

## Observability

- Centralized logging (e.g., Loki/ELK) with correlation IDs.
- Metrics: request throughput, worker busy %, queue depth, DB slow queries.
- Alerts on queue backlog and memory usage spikes.

## Deployment Checklist

- Use `docker-compose.prod.yml` or orchestrate via Kubernetes.
- Set resource limits and requests; pin CPU/memory for workers.
- Preload application classes (OPcache) and warm Octane.
- Run smoke tests before opening gates.

## Quick Benchmarking

- Use `wrk` or `k6` against `/api/exams/today` and `/api/exam-sessions/{id}`.
- Observe p95 latency under 5,000 concurrent users; scale workers horizontally if p95 > 300ms.