#!/usr/bin/env bash
set -euo pipefail

# Setup cron + flock for hosting and local development
# Usage:
#   ./scripts/setup-cron-flock.sh hosting
#   ./scripts/setup-cron-flock.sh local
# Notes:
# - This script does NOT edit your crontab automatically; it prints recommended lines.
# - For hosting (cPanel), copy the printed lines into cPanel Cron Jobs.

ENVIRONMENT="${1:-hosting}"

# Hosting defaults (adjust if your paths differ)
HOSTING_BACKEND_DIR="/home/learningsqdfsch/backend"
HOSTING_PHP_BIN="/opt/cpanel/ea-php83/root/usr/bin/php"
HOSTING_LOCK_DIR="$HOSTING_BACKEND_DIR/storage/locks"
HOSTING_LOG_DIR="$HOSTING_BACKEND_DIR/storage/logs"
HOSTING_FLOCK_BIN="$(command -v flock || echo /usr/bin/flock)"

# Local defaults (auto-detected relative to repo)
REPO_ROOT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")/.." && pwd)"
LOCAL_BACKEND_DIR="$REPO_ROOT_DIR/backend"
LOCAL_PHP_BIN="$(command -v php || true)"
LOCAL_LOCK_DIR="$LOCAL_BACKEND_DIR/storage/locks"
LOCAL_LOG_DIR="$LOCAL_BACKEND_DIR/storage/logs"
LOCAL_FLOCK_BIN="$(command -v flock || echo /usr/bin/flock)"

ensure_dir() {
  local dir="$1"
  mkdir -p "$dir"
  chmod 775 "$dir" || true
}

print_hosting_cron() {
  echo "# Add these lines to cPanel Cron Jobs (per minute):"
  echo "* * * * * cd $HOSTING_BACKEND_DIR && $HOSTING_PHP_BIN artisan schedule:run >> $HOSTING_LOG_DIR/cron-schedule.log 2>&1"
  echo "* * * * * cd $HOSTING_BACKEND_DIR && $HOSTING_FLOCK_BIN -n $HOSTING_LOCK_DIR/queue.lock $HOSTING_PHP_BIN artisan queue:work --queue=finalize,autosave --sleep=2 --timeout=120 --tries=3 --stop-when-empty >> $HOSTING_LOG_DIR/cron-queue.log 2>&1"
}

print_local_help() {
  echo "# Local development equivalents (run manually in separate terminals):"
  echo "cd $LOCAL_BACKEND_DIR && $LOCAL_PHP_BIN artisan schedule:work"
  echo "cd $LOCAL_BACKEND_DIR && $LOCAL_PHP_BIN artisan queue:work --queue=finalize,autosave --sleep=2 --timeout=120 --tries=3"
  echo "# To simulate flock locally (optional):"
  echo "$LOCAL_FLOCK_BIN -n $LOCAL_LOCK_DIR/queue.lock $LOCAL_PHP_BIN artisan queue:work --queue=finalize,autosave --sleep=2 --timeout=120 --tries=3"
}

case "$ENVIRONMENT" in
  hosting)
    if [ ! -x "$HOSTING_PHP_BIN" ]; then
      echo "[WARN] Hosting PHP binary not found at: $HOSTING_PHP_BIN" >&2
    fi
    if [ ! -x "$HOSTING_FLOCK_BIN" ]; then
      echo "[WARN] flock not found; expected at: $HOSTING_FLOCK_BIN" >&2
    fi
    ensure_dir "$HOSTING_LOCK_DIR"
    ensure_dir "$HOSTING_LOG_DIR"
    echo "[OK] Created/verified: $HOSTING_LOCK_DIR and $HOSTING_LOG_DIR"
    print_hosting_cron
    ;;
  local)
    if [ -z "$LOCAL_PHP_BIN" ]; then
      echo "[ERROR] PHP not found locally. Please install PHP." >&2
      exit 1
    fi
    ensure_dir "$LOCAL_LOCK_DIR"
    ensure_dir "$LOCAL_LOG_DIR"
    echo "[OK] Created/verified: $LOCAL_LOCK_DIR and $LOCAL_LOG_DIR"
    print_local_help
    ;;
  *)
    echo "Usage: $0 [hosting|local]" >&2
    exit 1
    ;;
esac