/**
 * Utility functions for handling fullscreen functionality across different devices and browsers
 */

/**
 * Request fullscreen with enhanced mobile support
 * @param {HTMLElement} element - Element to make fullscreen (defaults to document.documentElement)
 * @returns {Promise<boolean>} - Returns true if successful, false otherwise
 */
export const requestFullscreen = async (element = document.documentElement) => {
  try {
    // Check if fullscreen is already active
    if (document.fullscreenElement || 
        document.webkitFullscreenElement || 
        document.mozFullScreenElement || 
        document.msFullscreenElement) {
      return true;
    }

    // Try different fullscreen methods
    if (element.requestFullscreen) {
      await element.requestFullscreen();
    } else if (element.webkitRequestFullscreen) {
      // Safari and older Chrome
      await element.webkitRequestFullscreen();
    } else if (element.webkitRequestFullScreen) {
      // Older Safari
      await element.webkitRequestFullScreen();
    } else if (element.mozRequestFullScreen) {
      // Firefox
      await element.mozRequestFullScreen();
    } else if (element.msRequestFullscreen) {
      // IE/Edge
      await element.msRequestFullscreen();
    } else {
      console.warn('Fullscreen API not supported');
      return false;
    }

    return true;
  } catch (error) {
    console.warn('Failed to enter fullscreen:', error);
    return false;
  }
};

/**
 * Exit fullscreen
 * @returns {Promise<boolean>} - Returns true if successful, false otherwise
 */
export const exitFullscreen = async () => {
  try {
    if (document.exitFullscreen) {
      await document.exitFullscreen();
    } else if (document.webkitExitFullscreen) {
      await document.webkitExitFullscreen();
    } else if (document.webkitCancelFullScreen) {
      await document.webkitCancelFullScreen();
    } else if (document.mozCancelFullScreen) {
      await document.mozCancelFullScreen();
    } else if (document.msExitFullscreen) {
      await document.msExitFullscreen();
    } else {
      return false;
    }
    // Always cleanup CSS-based fallback after exiting fullscreen
    try { disableFullscreenFallback(); } catch (_) {}
    return true;
  } catch (error) {
    console.warn('Failed to exit fullscreen:', error);
    return false;
  }
};

/**
 * Check if currently in fullscreen mode
 * @returns {boolean}
 */
export const isFullscreen = () => {
  return !!(
    document.fullscreenElement ||
    document.webkitFullscreenElement ||
    document.mozFullScreenElement ||
    document.msFullscreenElement
  );
};

/**
 * Check if fullscreen is supported
 * @returns {boolean}
 */
export const isFullscreenSupported = () => {
  const element = document.documentElement;
  return !!(
    element.requestFullscreen ||
    element.webkitRequestFullscreen ||
    element.webkitRequestFullScreen ||
    element.mozRequestFullScreen ||
    element.msRequestFullscreen
  );
};

/**
 * Enhanced fullscreen request with mobile-specific handling
 * This function should be called within a user gesture (click, touch, etc.)
 * @param {HTMLElement} element - Element to make fullscreen
 * @param {Object} options - Additional options
 * @returns {Promise<boolean>}
 */
export const requestFullscreenEnhanced = async (element = document.documentElement, options = {}) => {
  const { 
    showWarning = true, 
    fallbackToViewport = true,
    mobileOptimization = true 
  } = options;

  // Check if fullscreen is supported
  if (!isFullscreenSupported()) {
    if (showWarning) {
      console.warn('Fullscreen not supported on this device');
    }
    // If unsupported, optionally apply CSS fallback on mobile
    if (fallbackToViewport && isMobileDevice()) {
      enableFullscreenFallback();
      if (mobileOptimization && isIOSBrowser()) {
        applyIOSViewportFix();
      }
      return true;
    }
    return false;
  }

  // Mobile-specific optimizations
  if (mobileOptimization && isMobileDevice()) {
    // Hide address bar on mobile browsers
    try {
      window.scrollTo(0, 1);
      setTimeout(() => window.scrollTo(0, 0), 100);
    } catch (e) {
      // Ignore errors
    }

    // Set viewport meta tag for better mobile experience
    let viewport = document.querySelector('meta[name="viewport"]');
    if (!viewport) {
      viewport = document.createElement('meta');
      viewport.name = 'viewport';
      document.head.appendChild(viewport);
    }
    viewport.content = 'width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no';

    // iOS visualViewport fix using 100dvh
    if (isIOSBrowser()) {
      applyIOSViewportFix();
    }
  }

  // Request fullscreen
  const success = await requestFullscreen(element);
  
  if (!success && fallbackToViewport) {
    // Fallback: maximize viewport on mobile
    if (isMobileDevice()) {
      // Apply a CSS-based fullscreen fallback that works better on iOS Safari
      enableFullscreenFallback();
      if (mobileOptimization && isIOSBrowser()) {
        applyIOSViewportFix();
      }
      return true;
    }
  }

  return success;
};

/**
 * Detect if device is mobile
 * @returns {boolean}
 */
export const isMobileDevice = () => {
  return /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent) ||
         (navigator.maxTouchPoints && navigator.maxTouchPoints > 2 && /MacIntel/.test(navigator.platform));
};

/**
 * Add fullscreen change event listener
 * @param {Function} callback - Callback function to execute on fullscreen change
 * @returns {Function} - Cleanup function to remove listeners
 */
export const onFullscreenChange = (callback) => {
  const events = [
    'fullscreenchange',
    'webkitfullscreenchange',
    'mozfullscreenchange',
    'MSFullscreenChange'
  ];

  const handler = () => callback(isFullscreen());
  
  events.forEach(event => {
    document.addEventListener(event, handler);
  });

  // Return cleanup function
  return () => {
    events.forEach(event => {
      document.removeEventListener(event, handler);
    });
  };
};

/**
 * Detect iOS Safari reliably
 */
export const isIOSSafari = () => {
  const ua = navigator.userAgent;
  const platform = navigator.platform || '';
  const isIOSDevice = /iPad|iPhone|iPod/.test(ua) || (platform === 'MacIntel' && navigator.maxTouchPoints > 1);
  const isSafari = /^((?!chrome|android).)*safari/i.test(ua);
  return isIOSDevice && isSafari;
};

/**
 * Detect any iOS browser (Safari, Chrome iOS, Firefox iOS all use WebKit)
 */
export const isIOSBrowser = () => {
  const ua = navigator.userAgent;
  const platform = navigator.platform || '';
  return /iPad|iPhone|iPod/.test(ua) || (platform === 'MacIntel' && navigator.maxTouchPoints > 1);
};

/**
 * Detect Android browser
 */
export const isAndroidBrowser = () => {
  const ua = navigator.userAgent.toLowerCase();
  return ua.includes('android');
};

/**
 * Heuristic: detect if iOS viewport is near full screen (address/tool bars likely hidden)
 * Returns true when viewport height is close to screen height (>= 97%).
 */
export const isIOSViewportNearFullScreen = () => {
  if (!isIOSBrowser()) return true;
  try {
    const vh = window.visualViewport ? window.visualViewport.height : window.innerHeight;
    const sh = window.screen?.height || vh;
    if (!vh || !sh) return false;
    const ratio = vh / sh;
    return ratio >= 0.97; // allow slight differences due to safe-area insets
  } catch (_) { return false; }
};

/**
 * Apply viewport fix for iOS Safari to simulate fullscreen area
 */
export const applyIOSViewportFix = () => {
  try {
    const updateVh = () => {
      const vh = window.visualViewport ? window.visualViewport.height : window.innerHeight;
      document.documentElement.style.setProperty('--app-vh', `${vh}px`);
    };
    updateVh();
    window.addEventListener('resize', updateVh);
    window.addEventListener('orientationchange', updateVh);
  } catch (_) { /* ignore */ }
};

/**
 * Enable CSS-based fullscreen fallback suitable for iOS Safari and other mobiles
 */
export const enableFullscreenFallback = () => {
  // Inject style once
  if (!document.getElementById('fs-fallback-style')) {
    const style = document.createElement('style');
    style.id = 'fs-fallback-style';
    style.textContent = `
      html.fs-fallback, body.fs-fallback { 
        overflow: hidden !important; 
        height: 100dvh !important; 
        width: 100vw !important; 
        background-color: #fff !important;
      }
      body.fs-fallback { 
        position: fixed !important; 
        inset: 0 !important; 
        padding-bottom: env(safe-area-inset-bottom);
        background-color: #fff !important; 
      }
      /* Container root (Vite default id may vary) */
      #app.fs-fallback-container, #root.fs-fallback-container { 
        min-height: 100dvh; 
        height: 100dvh; 
        overflow: auto; 
        -webkit-overflow-scrolling: touch; 
      }
      /* visualViewport-based variable as extra guard */
      html.fs-fallback { height: var(--app-vh, 100dvh); }
      /* iOS older fallback */
      html.fs-fallback, body.fs-fallback { height: -webkit-fill-available; }
    `;
    document.head.appendChild(style);
  }

  // Apply classes
  document.documentElement.classList.add('fs-fallback');
  document.body.classList.add('fs-fallback');
  const root = document.getElementById('app') || document.getElementById('root');
  if (root) root.classList.add('fs-fallback-container');

  // iOS fix
  if (isIOSBrowser()) applyIOSViewportFix();
};

/**
 * Disable CSS-based fullscreen fallback and clean up
 */
export const disableFullscreenFallback = () => {
  document.documentElement.classList.remove('fs-fallback');
  document.body.classList.remove('fs-fallback');
  const root = document.getElementById('app') || document.getElementById('root');
  if (root) root.classList.remove('fs-fallback-container');
};

/**
 * Check if CSS-based fullscreen fallback is active
 */
export const isFullscreenFallbackActive = () => {
  return document.documentElement.classList.contains('fs-fallback') && document.body.classList.contains('fs-fallback');
};

/**
 * Detect if app is running in standalone (PWA) display mode
 */
export const isStandaloneDisplayMode = () => {
  const mm = typeof window !== 'undefined' && window.matchMedia ? window.matchMedia('(display-mode: standalone)') : null;
  const byMedia = !!(mm && mm.matches);
  const byNavigator = typeof navigator !== 'undefined' && navigator.standalone === true;
  return byMedia || byNavigator;
};