// Stable device ID generator using Web Crypto if available; cached in localStorage
export function getStableDeviceId() {
  try {
    const existing = localStorage.getItem('DEVICE_ID');
    if (existing && typeof existing === 'string' && existing.length >= 16) {
      return existing;
    }
  } catch (_) {}

  const generate = async () => {
    try {
      const components = [
        navigator.userAgent || '',
        navigator.language || '',
        navigator.platform || '',
        String(screen.width || ''),
        String(screen.height || ''),
        String(screen.colorDepth || ''),
        String(navigator.hardwareConcurrency || ''),
        Intl.DateTimeFormat().resolvedOptions().timeZone || ''
      ].join('|');
      const enc = new TextEncoder();
      const buf = await crypto.subtle.digest('SHA-256', enc.encode(components));
      const hex = Array.from(new Uint8Array(buf)).map(b => b.toString(16).padStart(2, '0')).join('');
      return hex;
    } catch (_) {
      // Fallback: pseudo-random UUID v4-like
      const rand = () => Math.floor((1 + Math.random()) * 0x10000).toString(16).substring(1);
      return `${rand()}${rand()}-${rand()}-${rand()}-${rand()}-${rand()}${rand()}${rand()}`;
    }
  };

  // Create synchronously-friendly promise accessor
  // We compute once and store; subsequent calls read from localStorage
  const createAndStore = async () => {
    const id = await generate();
    try { localStorage.setItem('DEVICE_ID', id); } catch (_) {}
    return id;
  };

  // Since some callers expect sync, try best-effort: if crypto available, kick off async and temporarily return a placeholder
  // But to keep it reliable, if we can't read existing, we synchronously generate a fallback and then overwrite when async finishes
  const fallbackId = (() => {
    const rand = () => Math.floor((1 + Math.random()) * 0x10000).toString(16).substring(1);
    return `${rand()}${rand()}-${rand()}-${rand()}-${rand()}-${rand()}${rand()}${rand()}`;
  })();
  try { localStorage.setItem('DEVICE_ID', fallbackId); } catch (_) {}
  // Fire and forget async upgrade
  createAndStore();
  return fallbackId;
}