import React, { useEffect, useMemo, useState } from 'react';
import { FaEye, FaEdit, FaKey, FaTrash, FaPlus, FaToggleOn, FaToggleOff } from 'react-icons/fa';
import api from '../utils/axios';
import { useAuth } from "../context/AuthContext.jsx";
import DataTable from 'react-data-table-component';
import { canAccess } from '../utils/permissionHelper';
import Swal from 'sweetalert2';

export default function UsersList(){
  const { user: currentUser } = useAuth();
  const [users, setUsers] = useState([]);
  const [usersAll, setUsersAll] = useState([]);
  const [loading, setLoading] = useState(true);
  const [searchText, setSearchText] = useState('');
  const [perPage, setPerPage] = useState(10);
  const [roles, setRoles] = useState([]);
  const [selectedUser, setSelectedUser] = useState(null);
  const [showDetail, setShowDetail] = useState(false);
  const [showEdit, setShowEdit] = useState(false);
  const [form, setForm] = useState({ name: '', email: '', status: 'aktif', role_id: null });
  const [showAdd, setShowAdd] = useState(false);
  const [newForm, setNewForm] = useState({ name: '', email: '', password: '', status: 'aktif', role_id: null });

  const fetchAllUsers = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('token');
      const acc = [];
      let page = 1; let lastPage = 1;
      const pageSize = 50; // ambil per halaman dan gabungkan (client-side pagination seperti Subjects)
      do {
        const res = await api.get('/users', {
          headers: { Authorization: `Bearer ${token}` },
          params: { page, per_page: pageSize }
        });
        const data = res.data;
        if (Array.isArray(data)) {
          acc.push(...data);
          lastPage = page;
        } else {
          const list = Array.isArray(data?.data) ? data.data : [];
          acc.push(...list);
          lastPage = Number(data?.last_page || 1);
        }
        page += 1;
      } while (page <= lastPage && page < 200);
      setUsersAll(acc);
      setUsers(acc);
    } catch(e){
      console.error(e);
      const status = e?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk melihat daftar users.',
        });
      } else {
        Swal.fire('Error', 'Gagal mengambil data users!', 'error');
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(()=>{ fetchAllUsers(); }, []);

  const fetchRoles = async () => {
    try {
      const token = localStorage.getItem('token');
      const res = await api.get('/roles', { headers:{ Authorization:`Bearer ${token}` }});
      setRoles(Array.isArray(res.data) ? res.data : []);
    } catch(e){
      const status = e?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk melihat daftar roles.',
        });
      }
    }
  };
  useEffect(()=>{ fetchRoles(); }, []);

  // Pencarian kini client-side seperti Subjects
  const filtered = useMemo(() => {
    const term = String(searchText || '').trim().toLowerCase();
    if (!term) return usersAll;
    return usersAll.filter(u => (
      String(u?.name || '').toLowerCase().includes(term) ||
      String(u?.email || '').toLowerCase().includes(term) ||
      String(u?.roles?.[0]?.name || '').toLowerCase().includes(term)
    ));
  }, [usersAll, searchText]);

  const handleDelete = async (u) => {
    if(!canAccess(currentUser,'delete-user')) {
      Swal.fire({
        icon: 'warning',
        title: 'Akses Ditolak',
        text: 'Anda tidak memiliki izin untuk menghapus user.',
      });
      return;
    }
    const ok = await Swal.fire({title:`Hapus ${u.name}?`, showCancelButton:true, confirmButtonText:'Hapus'}); 
    if(ok.isConfirmed){
      try {
        const token = localStorage.getItem('token');
        await api.delete(`/users/${u.id}`, { headers:{ Authorization:`Bearer ${token}` }});
        Swal.fire('Dihapus','Sukses','success');
        fetchAllUsers();
      } catch(e){
        const status = e?.response?.status;
        if (status === 403) {
          Swal.fire({
            icon: 'warning',
            title: 'Akses Ditolak',
            text: 'Anda tidak memiliki izin untuk menghapus user.',
          });
        } else {
          Swal.fire('Gagal', e?.response?.data?.message || 'Tidak bisa hapus', 'error');
        }
      }
    }
  };

  const openDetail = (u) => { setSelectedUser(u); setShowDetail(true); };
  const openEdit = (u) => {
    setSelectedUser(u);
    setForm({
      name: u.name || '',
      email: u.email || '',
      status: (u.status || 'aktif'),
      role_id: u.roles?.[0]?.id || null,
    });
    setShowEdit(true);
  };

  // Helper: cek status online dalam 15 menit terakhir
  const isOnlineUser = (u) => !!(u?.last_activity && (new Date() - new Date(u.last_activity) < 15*60*1000));

  const handleResetPassword = async (u) => {
    if(!canAccess(currentUser,'edit-user')) {
      Swal.fire({
        icon: 'warning',
        title: 'Akses Ditolak',
        text: 'Anda tidak memiliki izin untuk mereset password user.',
      });
      return;
    }
    const ok = await Swal.fire({title:`Reset password ${u.name}?`, showCancelButton:true, confirmButtonText:'Reset'});
    if(ok.isConfirmed){
      try{
        const token = localStorage.getItem('token');
        await api.put(`/users/${u.id}/reset-password`, {}, { headers:{ Authorization:`Bearer ${token}` }});
        Swal.fire('Berhasil','Password direset ke 12345678','success');
      } catch(e){
        const status = e?.response?.status;
        if (status === 403) {
          Swal.fire({
            icon: 'warning',
            title: 'Akses Ditolak',
            text: 'Anda tidak memiliki izin untuk mereset password user.',
          });
        } else {
          Swal.fire('Gagal', e?.response?.data?.message || 'Tidak bisa reset', 'error');
        }
      }
    }
  };

  const handleUpdate = async () => {
    if(!selectedUser) return;
    try{
      const token = localStorage.getItem('token');
      const payload = { name: form.name, email: form.email, status: form.status, role_id: form.role_id };
      const res = await api.put(`/users/${selectedUser.id}`, payload, { headers:{ Authorization:`Bearer ${token}` }});
      Swal.fire('Tersimpan','User diperbarui','success');
      setShowEdit(false);
      setSelectedUser(null);
      fetchAllUsers();
    } catch(e){
      const status = e?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk memperbarui user.',
        });
      } else {
        Swal.fire('Gagal', e.response?.data?.message || 'Tidak bisa update', 'error');
      }
    }
  };

  const isActive = (u) => {
    const s = String(u?.status || '').toLowerCase();
    return s === 'aktif';
  };

  const toggleActive = async (u) => {
    if (!canAccess(currentUser,'edit-user')) {
      Swal.fire({
        icon: 'warning',
        title: 'Akses Ditolak',
        text: 'Anda tidak memiliki izin untuk mengubah status user.',
      });
      return;
    }
    try {
      const token = localStorage.getItem('token');
      const nextStatus = isActive(u) ? 'Tidak Aktif' : 'aktif';
      // Backend kini menerima update parsial: cukup kirim status saja
      await api.put(`/users/${u.id}`, { status: nextStatus }, { headers:{ Authorization:`Bearer ${token}` } });
      Swal.fire('Berhasil', `Status diubah menjadi ${nextStatus}`, 'success');
      fetchAllUsers();
    } catch (e) {
      const status = e?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk mengubah status user.',
        });
      } else {
        Swal.fire('Gagal', e?.response?.data?.message || 'Tidak bisa mengubah status', 'error');
      }
    }
  };

  const columns = [
    { name: '#', selector: (_, index) => index + 1, width: '90px' },
    { name: 'Nama', selector: row => row.name || '-', sortable: true },
    { name: 'Email', selector: row => row.email || '-', sortable: true },
    { name: 'Role', selector: row => (row.roles?.[0]?.name ?? '-'), sortable: true },
    { name: 'Online', selector: row => (row.last_activity && (new Date() - new Date(row.last_activity) < 15*60*1000)) ? 'online' : 'offline', sortable: true },
    {
      name: 'Aktif',
      cell: (row) => (
        <button
          className="btn btn-link p-0"
          title={isActive(row) ? 'Aktif' : 'Tidak Aktif'}
          aria-label={isActive(row) ? 'Aktif' : 'Tidak Aktif'}
          onClick={() => toggleActive(row)}
          disabled={!canAccess(currentUser,'edit-user')}
        >
          {isActive(row) ? (
            <FaToggleOn size={22} className="text-success" />
          ) : (
            <FaToggleOff size={22} className="text-secondary" />
          )}
        </button>
      ),
      sortable: false,
      width: '100px',
    },
    {
      name: 'Aksi',
      cell: (row) => (
        <div>
          {canAccess(currentUser,'view-user') && (
            <button className="btn btn-sm btn-info me-1" onClick={() => openDetail(row)} title="Detail" aria-label="Detail">
              <FaEye />
            </button>
          )}
          {canAccess(currentUser,'edit-user') && (
            <>
              <button className="btn btn-sm btn-primary me-1" onClick={() => openEdit(row)} title="Edit" aria-label="Edit">
                <FaEdit />
              </button>
              <button className="btn btn-sm btn-warning me-1" onClick={() => handleResetPassword(row)} title="Reset Password" aria-label="Reset Password">
                <FaKey />
              </button>
            </>
          )}
          {canAccess(currentUser,'delete-user') && (
            <button className="btn btn-sm btn-danger" onClick={() => handleDelete(row)} title="Delete" aria-label="Delete">
              <FaTrash />
            </button>
          )}
        </div>
      )
    }
  ];

  return (
    <div className="container mt-4">
      <h3>Daftar Users</h3>

      <div className="mb-3 d-flex justify-content-between align-items-center">
        {canAccess(currentUser,'create-user') && (
          <button className="btn btn-primary" onClick={() => setShowAdd(true)}>
            <FaPlus className="me-1" /> Tambah Users
          </button>
        )}
        <div className="d-flex align-items-center gap-2">
          <input
            type="text"
            placeholder="Cari nama atau email..."
            className="form-control flex-grow-1"
            style={{ minWidth: '320px', maxWidth: '600px' }}
            value={searchText}
            onChange={e => setSearchText(e.target.value)}
          />
          <button className="btn btn-outline-secondary" onClick={() => setSearchText('')}>Reset</button>
        </div>
      </div>

      {loading ? (
        <div className="text-center mt-5">Loading users...</div>
      ) : (
        <DataTable
          columns={columns}
          data={filtered}
          pagination
          paginationPerPage={perPage}
          paginationRowsPerPageOptions={[10, 20, 50, 100, Math.max(1, filtered.length)]}
          highlightOnHover
          striped
          noHeader
        />
      )}

      {/* Modal Detail */}
      {showDetail && selectedUser && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Detail User</h5>
                <button type="button" className="btn-close" onClick={() => setShowDetail(false)}></button>
              </div>
              <div className="modal-body">
                <div className="d-flex align-items-center gap-3 mb-3">
                  <div className="rounded-circle bg-primary text-white d-flex align-items-center justify-content-center" style={{ width: 48, height: 48, fontWeight: 600 }}>
                    {(selectedUser?.name?.trim()?.charAt(0)?.toUpperCase()) || '?'}
                  </div>
                  <div className="flex-grow-1">
                    <div className="fw-semibold">{selectedUser.name || '-'}</div>
                    <div className="d-flex flex-wrap gap-2 mt-1">
                      <span className="badge bg-primary">{selectedUser.roles?.[0]?.name ?? '-'}</span>
                      <span className={`badge ${isOnlineUser(selectedUser) ? 'bg-success' : 'bg-secondary'}`}>{isOnlineUser(selectedUser) ? 'Online' : 'Offline'}</span>
                      <span className="badge bg-light text-dark">{selectedUser.status ?? '-'}</span>
                    </div>
                  </div>
                </div>

                <div className="list-group mb-3">
                  <div className="list-group-item">
                    <div className="small text-muted">Email</div>
                    {selectedUser.email ? (
                      <a href={`mailto:${selectedUser.email}`} className="text-decoration-none">{selectedUser.email}</a>
                    ) : (
                      <span>-</span>
                    )}
                  </div>
                  <div className="list-group-item">
                    <div className="small text-muted">Aktivitas Terakhir</div>
                    <div>{selectedUser.last_activity ? new Date(selectedUser.last_activity).toLocaleString() : '-'}</div>
                  </div>
                </div>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowDetail(false)}>Tutup</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Edit */}
      {showEdit && selectedUser && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Edit User</h5>
                <button type="button" className="btn-close" onClick={() => setShowEdit(false)}></button>
              </div>
              <div className="modal-body">
                <div className="mb-3">
                  <label>Nama</label>
                  <input className="form-control" value={form.name} onChange={e=>setForm({...form, name:e.target.value})} />
                </div>
                <div className="mb-3">
                  <label>Email</label>
                  <input className="form-control" value={form.email} onChange={e=>setForm({...form, email:e.target.value})} />
                </div>
                <div className="mb-3">
                  <label>Status</label>
                  <select className="form-select" value={form.status} onChange={e=>setForm({...form, status:e.target.value})}>
                    <option value="aktif">Aktif</option>
                    <option value="Tidak Aktif">Tidak Aktif</option>
                  </select>
                </div>
                <div className="mb-3">
                  <label>Role</label>
                  <select className="form-select" value={form.role_id ?? ''} onChange={e=>setForm({...form, role_id: e.target.value ? Number(e.target.value) : null})}>
                    <option value="">-</option>
                    {roles.map(r => (
                      <option key={r.id} value={r.id}>{r.name}</option>
                    ))}
                  </select>
                </div>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowEdit(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleUpdate}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Tambah Users */}
      {showAdd && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Tambah Users</h5>
                <button type="button" className="btn-close" onClick={() => setShowAdd(false)}></button>
              </div>
              <div className="modal-body">
                <div className="mb-3">
                  <label>Nama</label>
                  <input className="form-control" value={newForm.name} onChange={e=>setNewForm({...newForm, name:e.target.value})} />
                </div>
                <div className="mb-3">
                  <label>Email</label>
                  <input className="form-control" value={newForm.email} onChange={e=>setNewForm({...newForm, email:e.target.value})} />
                </div>
                <div className="mb-3">
                  <label>Password (opsional)</label>
                  <input type="password" className="form-control" value={newForm.password} onChange={e=>setNewForm({...newForm, password:e.target.value})} placeholder="Kosongkan untuk default 12345678" />
                </div>
                <div className="mb-3">
                  <label>Status</label>
                  <select className="form-select" value={newForm.status} onChange={e=>setNewForm({...newForm, status:e.target.value})}>
                    <option value="aktif">Aktif</option>
                    <option value="Tidak Aktif">Tidak Aktif</option>
                  </select>
                </div>
                <div className="mb-3">
                  <label>Role</label>
                  <select className="form-select" value={newForm.role_id ?? ''} onChange={e=>setNewForm({...newForm, role_id: e.target.value ? Number(e.target.value) : null})}>
                    <option value="">-</option>
                    {roles.map(r => (
                      <option key={r.id} value={r.id}>{r.name}</option>
                    ))}
                  </select>
                </div>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowAdd(false)}>Batal</button>
                <button className="btn btn-primary" onClick={async () => {
                  try {
                    const token = localStorage.getItem('token');
                    const payload = { name: newForm.name, email: newForm.email, status: newForm.status, role_id: newForm.role_id };
                    if (newForm.password && newForm.password.trim().length >= 6) {
                      payload.password = newForm.password.trim();
                    }
                    const res = await api.post('/users', payload, { headers:{ Authorization:`Bearer ${token}` }});
                    Swal.fire('Tersimpan','User ditambahkan','success');
                    setShowAdd(false);
                    setNewForm({ name:'', email:'', password:'', status:'aktif', role_id:null });
                    fetchAllUsers();
                  } catch(e){
                    const status = e?.response?.status;
                    if (status === 403) {
                      Swal.fire({
                        icon: 'warning',
                        title: 'Akses Ditolak',
                        text: 'Anda tidak memiliki izin untuk menambahkan user.',
                      });
                    } else {
                      Swal.fire('Gagal', e.response?.data?.message || 'Tidak bisa menambah user', 'error');
                    }
                  }
                }}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
