import React, { useEffect, useState } from 'react';
import { useParams } from 'react-router-dom';
import api from '../utils/axios';

export default function TodayPublicDetails() {
  const { examId } = useParams();
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [exam, setExam] = useState(null);
  const [inProgress, setInProgress] = useState([]);
  const [finished, setFinished] = useState([]);
  const [notStarted, setNotStarted] = useState([]);
  const [inactive, setInactive] = useState([]);
  const [counts, setCounts] = useState({ in_progress: 0, finished: 0, not_started: 0, inactive: 0 });
  const [search, setSearch] = useState('');
  const [classFilter, setClassFilter] = useState(''); // holds class ID
  const [schoolFilter, setSchoolFilter] = useState(''); // holds school ID
  const [statusFilter, setStatusFilter] = useState('all');
  const [classesAll, setClassesAll] = useState([]);
  const [schoolsAll, setSchoolsAll] = useState([]);
  // Pagination states per status
  const [perPage, setPerPage] = useState(20);
  const [pageInProgress, setPageInProgress] = useState(1);
  const [pageFinished, setPageFinished] = useState(1);
  const [pageNotStarted, setPageNotStarted] = useState(1);
  const [pageInactive, setPageInactive] = useState(1);

  useEffect(() => {
    const loadMeta = async () => {
      setLoading(true); setError('');
      try {
        const res = await api.get(`/exams/${examId}/participants-status`, { params: { only_meta: true } });
        setExam(res.data.exam || null);
        setClassesAll(Array.isArray(res.data.classes) ? res.data.classes : []);
        setSchoolsAll(Array.isArray(res.data.schools) ? res.data.schools : []);
        // kosongkan data berat di awal
        setInProgress([]); setFinished([]); setNotStarted([]); setInactive([]);
        setCounts({ in_progress: 0, finished: 0, not_started: 0, inactive: 0 });
      } catch (e) {
        console.error('Gagal memuat metadata peserta ujian', e);
        setError('Gagal memuat metadata. Pastikan Anda login.');
      } finally {
        setLoading(false);
      }
    };
    loadMeta();
  }, [examId]);

  // Ketika sekolah dipilih, muat ulang daftar kelas untuk sekolah tsb (metadata ringan)
  useEffect(() => {
    const loadClassesForSchool = async () => {
      if (!schoolFilter) { setClassesAll([]); return; }
      try {
        const res = await api.get(`/exams/${examId}/participants-status`, { params: { only_meta: true, school_id: schoolFilter } });
        setClassesAll(Array.isArray(res.data.classes) ? res.data.classes : []);
      } catch (e) {
        console.error('Gagal memuat daftar kelas untuk sekolah', e);
      }
    };
    loadClassesForSchool();
  }, [examId, schoolFilter]);

  // Setelah sekolah & kelas dipilih, baru muat data berat
  useEffect(() => {
    const loadData = async () => {
      if (!schoolFilter || !classFilter) return;
      setLoading(true); setError('');
      try {
        const res = await api.get(`/exams/${examId}/participants-status`, { params: { school_id: schoolFilter, class_id: classFilter } });
        setInProgress(Array.isArray(res.data.in_progress) ? res.data.in_progress : []);
        setFinished(Array.isArray(res.data.finished) ? res.data.finished : []);
        setNotStarted(Array.isArray(res.data.not_started) ? res.data.not_started : []);
        setInactive(Array.isArray(res.data.inactive) ? res.data.inactive : []);
        setCounts(res.data.counts || { in_progress: 0, finished: 0, not_started: 0, inactive: 0 });
      } catch (e) {
        console.error('Gagal memuat data peserta ujian (setelah filter)', e);
        setError('Gagal memuat data setelah filter. Coba ulangi.');
      } finally {
        setLoading(false);
      }
    };
    loadData();
  }, [examId, schoolFilter, classFilter]);

  const normalize = (s) => (s || '').toString().toLowerCase();
  const applyFilters = (rows) => {
    const q = normalize(search);
    return rows.filter((row) => {
      const name = normalize(row.participant?.nama);
      const nisn = normalize(row.participant?.nisn);
      const classId = String(row.class?.id || '');
      const schoolId = String(row.school?.id || row.class?.id_school || '');
      const matchQ = !q || name.includes(q) || nisn.includes(q);
      const matchClass = !classFilter || classId === String(classFilter);
      const matchSchool = !schoolFilter || schoolId === String(schoolFilter);
      return matchQ && matchClass && matchSchool;
    });
  };

  const allRows = [...inProgress, ...finished, ...notStarted, ...inactive];
  // Daftar sekolah & kelas dari server (metadata)
  const uniqueSchools = Array.isArray(schoolsAll) ? schoolsAll.map(s => ({ id: s.id, nama: s.nama })) : [];
  const uniqueClasses = Array.isArray(classesAll) ? classesAll.map(c => ({ id: c.id, name: c.name })) : [];

  const filteredInProgress = applyFilters(inProgress);
  const filteredFinished = applyFilters(finished);
  const filteredNotStarted = applyFilters(notStarted);
  const filteredInactive = applyFilters(inactive);

  // Reset pages when filters change
  useEffect(() => {
    setPageInProgress(1);
    setPageFinished(1);
    setPageNotStarted(1);
    setPageInactive(1);
  }, [search, classFilter, schoolFilter, statusFilter]);

  // Helpers: pagination slicing
  const slicePaginated = (rows, page) => {
    const size = perPage || 20;
    const start = (page - 1) * size;
    return rows.slice(start, start + size);
  };
  const lastPage = (rows) => Math.max(1, Math.ceil((rows?.length || 0) / (perPage || 20)));

  const resetFilters = () => {
    setSearch('');
    setClassFilter('');
    setSchoolFilter('');
    setStatusFilter('all');
  };

  return (
    <div className="container mt-4">
      <h3>Detail Peserta Ujian</h3>
      {exam && (
        <div className="mb-3">
          <div><strong>Ujian:</strong> {exam.name} ({exam.code || '-'})</div>
          <div><strong>Mata Pelajaran:</strong> {exam.subject_name || '-'} ({exam.subject_code || '-'})</div>
          <div><strong>Jadwal:</strong> {String(exam.scheduled_at || '-')}</div>
          {(schoolFilter && classFilter) ? (
            <>
              <div><strong>Ringkasan:</strong> Sedang: {counts.in_progress} • Selesai: {counts.finished} • Belum Mulai: {counts.not_started} • Tidak Aktif: {counts.inactive}</div>
              <div className="text-muted small">Ringkasan dan tabel muncul setelah memilih sekolah & kelas.</div>
            </>
          ) : (
            <div className="text-muted">Silakan pilih sekolah terlebih dahulu, kemudian pilih kelas untuk menampilkan data.</div>
          )}
        </div>
      )}

      {/* Filter Bar */}
      <div className="card mb-3">
        <div className="card-body">
          <div className="row g-2 align-items-end">
            <div className="col-md-4">
              <label className="form-label">Cari (Nama/NISN)</label>
              <input
                type="text"
                className="form-control"
                placeholder="mis. Budi / 1234567890"
                value={search}
                onChange={(e) => setSearch(e.target.value)}
              />
            </div>
            <div className="col-md-3">
              <label className="form-label">Sekolah</label>
              <select className="form-select" value={schoolFilter} onChange={(e) => { setSchoolFilter(e.target.value); setClassFilter(''); }}>
                <option value="">Pilih Sekolah</option>
                {uniqueSchools.map((s) => (
                  <option key={s.id} value={s.id}>{s.nama}</option>
                ))}
              </select>
            </div>
            <div className="col-md-3">
              <label className="form-label">Kelas</label>
              <select className="form-select" value={classFilter} onChange={(e) => setClassFilter(e.target.value)} disabled={!schoolFilter}>
                <option value="">{schoolFilter ? 'Pilih Kelas' : 'Pilih Sekolah dulu'}</option>
                {uniqueClasses.map((c) => (
                  <option key={c.id} value={c.id}>{c.name}</option>
                ))}
              </select>
            </div>
            <div className="col-md-2">
              <label className="form-label">Status</label>
              <select className="form-select" value={statusFilter} onChange={(e) => setStatusFilter(e.target.value)}>
                <option value="all">Semua</option>
                <option value="in_progress">Sedang</option>
                <option value="finished">Selesai</option>
                <option value="not_started">Belum Mulai</option>
                <option value="inactive">Tidak Aktif</option>
              </select>
            </div>
            <div className="col-12 mt-2">
              <button className="btn btn-sm btn-outline-secondary" onClick={resetFilters}>Reset Filter</button>
            </div>
          </div>
        </div>
      </div>

      {loading ? (
        <div className="text-muted">Memuat data...</div>
      ) : error ? (
        <div className="alert alert-danger">{error}</div>
      ) : (
        <div className="row">
          {(!schoolFilter || !classFilter) && (
            <div className="col-12">
              <div className="alert alert-info">Data tabel akan tampil setelah Anda memilih sekolah dan kelas.</div>
            </div>
          )}
          {(statusFilter === 'all' || statusFilter === 'in_progress') && (
            <div className="col-md-6">
              <div className="card">
                <div className="card-header d-flex justify-content-between align-items-center">
                  <span>Sedang Mengerjakan ({filteredInProgress.length})</span>
                  <div className="d-flex align-items-center gap-2">
                    <label className="form-label mb-0 small">Per Halaman</label>
                    <select
                      className="form-select form-select-sm"
                      style={{ maxWidth: '100px' }}
                      value={perPage}
                      onChange={(e) => { setPerPage(Number(e.target.value)); setPageInProgress(1); }}
                    >
                      {[20,50,100].map(n => (<option key={n} value={n}>{n}</option>))}
                    </select>
                  </div>
                </div>
                <div className="card-body p-0">
                  <div className="table-responsive">
                    <table className="table table-sm mb-0">
                      <thead>
                        <tr>
                          <th>NISN</th>
                          <th>Nama</th>
                          <th>Kelas</th>
                          <th>Sekolah</th>
                          <th>Mulai</th>
                          <th>Aktivitas</th>
                        </tr>
                      </thead>
                      <tbody>
                        {slicePaginated(filteredInProgress, pageInProgress).map(row => (
                          <tr key={row.session_id}>
                            <td>{row.participant?.nisn || '-'}</td>
                            <td>{row.participant?.nama || '-'}</td>
                            <td>{row.class?.name || '-'}</td>
                            <td>{row.school?.nama || '-'}</td>
                            <td>{row.started_at || '-'}</td>
                            <td>{row.participant?.last_activity || '-'}</td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                  <div className="d-flex justify-content-between align-items-center p-2">
                    <div className="btn-group btn-group-sm">
                      <button className="btn btn-outline-secondary" disabled={pageInProgress <= 1} onClick={() => setPageInProgress(p => Math.max(1, p-1))}>Prev</button>
                      <span className="btn btn-outline-secondary disabled">{pageInProgress} / {lastPage(filteredInProgress)}</span>
                      <button className="btn btn-outline-secondary" disabled={pageInProgress >= lastPage(filteredInProgress)} onClick={() => setPageInProgress(p => Math.min(lastPage(filteredInProgress), p+1))}>Next</button>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          )}
          {(statusFilter === 'all' || statusFilter === 'finished') && (
            <div className="col-md-6">
              <div className="card">
                <div className="card-header d-flex justify-content-between align-items-center">
                  <span>Sudah Selesai ({filteredFinished.length})</span>
                  <div className="d-flex align-items-center gap-2">
                    <label className="form-label mb-0 small">Per Halaman</label>
                    <select
                      className="form-select form-select-sm"
                      style={{ maxWidth: '100px' }}
                      value={perPage}
                      onChange={(e) => { setPerPage(Number(e.target.value)); setPageFinished(1); }}
                    >
                      {[20,50,100].map(n => (<option key={n} value={n}>{n}</option>))}
                    </select>
                  </div>
                </div>
                <div className="card-body p-0">
                  <div className="table-responsive">
                    <table className="table table-sm mb-0">
                      <thead>
                        <tr>
                          <th>NISN</th>
                          <th>Nama</th>
                          <th>Kelas</th>
                          <th>Sekolah</th>
                          <th>Mulai</th>
                          <th>Selesai</th>
                        </tr>
                      </thead>
                      <tbody>
                        {slicePaginated(filteredFinished, pageFinished).map(row => (
                          <tr key={row.session_id}>
                            <td>{row.participant?.nisn || '-'}</td>
                            <td>{row.participant?.nama || '-'}</td>
                            <td>{row.class?.name || '-'}</td>
                            <td>{row.school?.nama || '-'}</td>
                            <td>{row.started_at || '-'}</td>
                            <td>{row.finished_at || '-'}</td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                  <div className="d-flex justify-content-between align-items-center p-2">
                    <div className="btn-group btn-group-sm">
                      <button className="btn btn-outline-secondary" disabled={pageFinished <= 1} onClick={() => setPageFinished(p => Math.max(1, p-1))}>Prev</button>
                      <span className="btn btn-outline-secondary disabled">{pageFinished} / {lastPage(filteredFinished)}</span>
                      <button className="btn btn-outline-secondary" disabled={pageFinished >= lastPage(filteredFinished)} onClick={() => setPageFinished(p => Math.min(lastPage(filteredFinished), p+1))}>Next</button>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          )}
          {(statusFilter === 'all' || statusFilter === 'not_started') && (
            <div className="col-md-6 mt-3">
              <div className="card">
                <div className="card-header d-flex justify-content-between align-items-center">
                  <span>Belum Mulai ({filteredNotStarted.length})</span>
                  <div className="d-flex align-items-center gap-2">
                    <label className="form-label mb-0 small">Per Halaman</label>
                    <select
                      className="form-select form-select-sm"
                      style={{ maxWidth: '100px' }}
                      value={perPage}
                      onChange={(e) => { setPerPage(Number(e.target.value)); setPageNotStarted(1); }}
                    >
                      {[20,50,100].map(n => (<option key={n} value={n}>{n}</option>))}
                    </select>
                  </div>
                </div>
                <div className="card-body p-0">
                  <div className="table-responsive">
                    <table className="table table-sm mb-0">
                      <thead>
                        <tr>
                          <th>NISN</th>
                          <th>Nama</th>
                          <th>Kelas</th>
                          <th>Sekolah</th>
                          <th>Aktivitas Terakhir</th>
                        </tr>
                      </thead>
                      <tbody>
                        {slicePaginated(filteredNotStarted, pageNotStarted).map(row => (
                          <tr key={row.participant?.id || `${row.participant?.nisn}-ns`}>
                            <td>{row.participant?.nisn || '-'}</td>
                            <td>{row.participant?.nama || '-'}</td>
                            <td>{row.class?.name || '-'}</td>
                            <td>{row.school?.nama || '-'}</td>
                            <td>{row.participant?.last_activity || '-'}</td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                  <div className="d-flex justify-content-between align-items-center p-2">
                    <div className="btn-group btn-group-sm">
                      <button className="btn btn-outline-secondary" disabled={pageNotStarted <= 1} onClick={() => setPageNotStarted(p => Math.max(1, p-1))}>Prev</button>
                      <span className="btn btn-outline-secondary disabled">{pageNotStarted} / {lastPage(filteredNotStarted)}</span>
                      <button className="btn btn-outline-secondary" disabled={pageNotStarted >= lastPage(filteredNotStarted)} onClick={() => setPageNotStarted(p => Math.min(lastPage(filteredNotStarted), p+1))}>Next</button>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          )}
          {(statusFilter === 'all' || statusFilter === 'inactive') && (
            <div className="col-md-6 mt-3">
              <div className="card">
                <div className="card-header d-flex justify-content-between align-items-center">
                  <span>Tidak Aktif ({filteredInactive.length})</span>
                  <div className="d-flex align-items-center gap-2">
                    <label className="form-label mb-0 small">Per Halaman</label>
                    <select
                      className="form-select form-select-sm"
                      style={{ maxWidth: '100px' }}
                      value={perPage}
                      onChange={(e) => { setPerPage(Number(e.target.value)); setPageInactive(1); }}
                    >
                      {[20,50,100].map(n => (<option key={n} value={n}>{n}</option>))}
                    </select>
                  </div>
                </div>
                <div className="card-body p-0">
                  <div className="table-responsive">
                    <table className="table table-sm mb-0">
                      <thead>
                        <tr>
                          <th>NISN</th>
                          <th>Nama</th>
                          <th>Kelas</th>
                          <th>Sekolah</th>
                          <th>Aktivitas Terakhir</th>
                        </tr>
                      </thead>
                      <tbody>
                        {slicePaginated(filteredInactive, pageInactive).map(row => (
                          <tr key={row.participant?.id || `${row.participant?.nisn}-inactive`}>
                            <td>{row.participant?.nisn || '-'}</td>
                            <td>{row.participant?.nama || '-'}</td>
                            <td>{row.class?.name || '-'}</td>
                            <td>{row.school?.nama || '-'}</td>
                            <td>{row.participant?.last_activity || '-'}</td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                  <div className="d-flex justify-content-between align-items-center p-2">
                    <div className="btn-group btn-group-sm">
                      <button className="btn btn-outline-secondary" disabled={pageInactive <= 1} onClick={() => setPageInactive(p => Math.max(1, p-1))}>Prev</button>
                      <span className="btn btn-outline-secondary disabled">{pageInactive} / {lastPage(filteredInactive)}</span>
                      <button className="btn btn-outline-secondary" disabled={pageInactive >= lastPage(filteredInactive)} onClick={() => setPageInactive(p => Math.min(lastPage(filteredInactive), p+1))}>Next</button>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          )}
        </div>
      )}
    </div>
  );
}