import React, { useEffect, useMemo, useState } from 'react';
import { Link } from 'react-router-dom';
import api from '../utils/axios';

export default function TodayPublic() {
  const [loading, setLoading] = useState(true);
  const [exams, setExams] = useState([]);
  const [error, setError] = useState('');
  const [page, setPage] = useState(1);
  const [perPage, setPerPage] = useState(10);

  useEffect(() => {
    const load = async () => {
      setLoading(true); setError('');
      try {
        const res = await api.get('/exams/today');
        setExams(Array.isArray(res.data) ? res.data : []);
      } catch (e) {
        console.error('Gagal memuat ujian hari ini (publik)', e);
        setError('Gagal memuat data ujian hari ini.');
      } finally {
        setLoading(false);
      }
    };
    load();
  }, []);

  const fmt = (iso) => {
    if (!iso) return '-';
    const d = new Date(iso);
    return d.toLocaleString();
  };

  const lastPage = useMemo(() => {
    const total = Array.isArray(exams) ? exams.length : 0;
    return Math.max(1, Math.ceil(total / (perPage || 10)));
  }, [exams, perPage]);

  const paginatedExams = useMemo(() => {
    const start = (page - 1) * (perPage || 10);
    return exams.slice(start, start + (perPage || 10));
  }, [exams, page, perPage]);

  return (
    <div className="container mt-4">
      <h3>Mata Pelajaran Hari Ini (Publik)</h3>
      <p className="text-muted">Halaman ini menggunakan endpoint publik dan dapat diakses tanpa login.</p>

      {loading ? (
        <div className="text-muted">Memuat data...</div>
      ) : error ? (
        <div className="alert alert-danger">{error}</div>
      ) : exams.length === 0 ? (
        <div className="text-muted">Tidak ada ujian yang dijadwalkan hari ini.</div>
      ) : (
        <div className="card">
          <div className="card-body">
            <div className="d-flex align-items-center justify-content-between mb-2">
              <div className="text-muted">Total: {exams.length}</div>
              <div className="d-flex align-items-center gap-2">
                <label className="form-label mb-0">Per Halaman</label>
                <select
                  className="form-select form-select-sm"
                  style={{ maxWidth: '120px' }}
                  value={perPage}
                  onChange={(e) => { const v = Number(e.target.value); setPerPage(v); setPage(1); }}
                >
                  {[10,20,50,100].map(n => (<option key={n} value={n}>{n}</option>))}
                  {exams.length > 0 && ![10,20,50,100].includes(exams.length) && (
                    <option value={exams.length}>Semua</option>
                  )}
                </select>
              </div>
            </div>
            <div className="table-responsive">
              <table className="table table-sm">
                <thead>
                  <tr>
                    <th>Mata Pelajaran</th>
                    <th>Kode Mapel</th>
                    <th>Nama Ujian</th>
                    <th>Kode Ujian</th>
                   <th>Jadwal</th>
                   <th>Sedang</th>
                   <th>Selesai</th>
                    <th>Aksi</th>
                  </tr>
                </thead>
                <tbody>
                  {paginatedExams.map(ex => (
                    <tr key={ex.id}>
                      <td>{ex.subject_name || '-'}</td>
                      <td>{ex.subject_code || '-'}</td>
                      <td>{ex.name}</td>
                      <td>{ex.code || '-'}</td>
                      <td>{fmt(ex.start_at_iso)} - {fmt(ex.end_at_iso)}</td>
                      <td>{ex.in_progress_count ?? 0}</td>
                      <td>{ex.finished_count ?? 0}</td>
                      <td>
                        <Link to={`/today-public/${ex.id}`} className="btn btn-sm btn-outline-primary">Detail</Link>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
            <small className="text-muted">Data diambil dari /exams/today.</small>
            <div className="d-flex justify-content-between align-items-center mt-2">
              <div className="btn-group">
                <button className="btn btn-outline-secondary btn-sm" disabled={page <= 1} onClick={() => setPage(p => Math.max(1, p-1))}>Prev</button>
                <span className="btn btn-outline-secondary btn-sm disabled">{page} / {lastPage}</span>
                <button className="btn btn-outline-secondary btn-sm" disabled={page >= lastPage} onClick={() => setPage(p => Math.min(lastPage, p+1))}>Next</button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}