import React, { useEffect, useMemo, useState } from 'react';
import api from '../utils/axios';
import DataTable from 'react-data-table-component';
import Swal from 'sweetalert2';
import { FaPlus, FaEdit, FaTrash, FaInfoCircle, FaDownload } from 'react-icons/fa';
import { useAuth } from '../context/AuthContext.jsx';
import { canAccess } from '../utils/permissionHelper';

export default function Scores() {
  const { user } = useAuth();
  const [scores, setScores] = useState([]);
  const [loading, setLoading] = useState(true);
  const [filterText, setFilterText] = useState('');
  const [showAddModal, setShowAddModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [showDetailModal, setShowDetailModal] = useState(false);
  const [editing, setEditing] = useState(null);
  const [detailItem, setDetailItem] = useState(null);
  const [subjects, setSubjects] = useState([]);
  const [form, setForm] = useState({
    nisn: '',
    nama: '',
    id_pelajaran: '',
    name_pelajaran: '',
    sts_ganjil: '',
    sas: '',
    sts_genap: '',
    sat: '',
    us: '',
    harian: '',
    keterampilan: '',
    keterangan: '',
    teacher: '',
    tahun: '',
  });

  const tokenHeader = { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } };

  const canViewScore = canAccess(user, 'view-score') || canAccess(user, 'manage-score') || canAccess(user, 'input-score') || canAccess(user, 'input score') || canAccess(user, 'edit-score');
  const canCreateScore = canAccess(user, 'input-score') || canAccess(user, 'input score') || canAccess(user, 'manage-score');
  const canEditScore = canAccess(user, 'edit-score') || canAccess(user, 'manage-score');
  const canDeleteScore = canAccess(user, 'manage-score');
  const canExportScore = canAccess(user, 'export-score') || canAccess(user, 'manage-score');

  const fetchScores = async () => {
    setLoading(true);
    try {
      const res = await api.get('/scores', tokenHeader);
      setScores(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      Swal.fire('Error', 'Gagal memuat data nilai', 'error');
    } finally {
      setLoading(false);
    }
  };

  const fetchSubjects = async () => {
    try {
      const res = await api.get('/subjects', tokenHeader);
      setSubjects(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
    }
  };

  useEffect(() => { if (canViewScore) fetchScores(); }, [canViewScore]);
  useEffect(() => { fetchSubjects(); }, []);

  const openAdd = () => {
    setForm({
      nisn: '', nama: '', id_pelajaran: '', name_pelajaran: '', sts_ganjil: '', sas: '', sts_genap: '', sat: '', us: '', harian: '', keterampilan: '', keterangan: '', teacher: '', tahun: ''
    });
    setShowAddModal(true);
  };

  const openEdit = (item) => {
    setEditing(item);
    setForm({
      nisn: item.nisn ?? '',
      nama: item.nama ?? '',
      id_pelajaran: item.id_pelajaran ?? '',
      name_pelajaran: item.name_pelajaran ?? '',
      sts_ganjil: item.sts_ganjil ?? '',
      sas: item.sas ?? '',
      sts_genap: item.sts_genap ?? '',
      sat: item.sat ?? '',
      us: item.us ?? '',
      harian: item.harian ?? '',
      keterampilan: item.keterampilan ?? '',
      keterangan: item.keterangan ?? '',
      teacher: item.teacher ?? '',
      tahun: item.tahun ?? '',
    });
    setShowEditModal(true);
  };

  const openDetail = (item) => {
    setDetailItem(item);
    setShowDetailModal(true);
  };

  const saveAdd = async () => {
    try {
      const payload = { ...form };
      if (!payload.name_pelajaran && payload.id_pelajaran) {
        const subj = subjects.find(s => s.id === Number(payload.id_pelajaran));
        payload.name_pelajaran = subj ? subj.name : '';
      }
      const res = await api.post('/scores', payload, tokenHeader);
      setShowAddModal(false);
      Swal.fire('Berhasil', 'Data nilai ditambahkan', 'success');
      await fetchScores();
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', 'Tidak dapat menambah data nilai', 'error');
    }
  };

  const saveEdit = async () => {
    try {
      const payload = { ...form };
      if (!payload.name_pelajaran && payload.id_pelajaran) {
        const subj = subjects.find(s => s.id === Number(payload.id_pelajaran));
        payload.name_pelajaran = subj ? subj.name : '';
      }
      const res = await api.put(`/scores/${editing.id}`, payload, tokenHeader);
      setShowEditModal(false);
      setEditing(null);
      Swal.fire('Berhasil', 'Data nilai diperbarui', 'success');
      await fetchScores();
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', 'Tidak dapat memperbarui data nilai', 'error');
    }
  };

  const removeItem = async (item) => {
    const confirm = await Swal.fire({ title: 'Hapus nilai?', text: 'Tindakan ini tidak dapat dibatalkan.', icon: 'warning', showCancelButton: true, confirmButtonText: 'Ya, hapus', cancelButtonText: 'Batal' });
    if (!confirm.isConfirmed) return;
    try {
      await api.delete(`/scores/${item.id}`, tokenHeader);
      Swal.fire('Berhasil', 'Data nilai dihapus', 'success');
      await fetchScores();
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', 'Tidak dapat menghapus data nilai', 'error');
    }
  };

  const exportData = async () => {
    try {
      const res = await api.get('/scores/export', { ...tokenHeader, responseType: 'blob' });
      const blob = new Blob([res.data], { type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' });
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `scores_export_${new Date().toISOString().slice(0,19).replace(/[:T]/g,'-')}.xlsx`;
      document.body.appendChild(a);
      a.click();
      a.remove();
      window.URL.revokeObjectURL(url);
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', 'Tidak dapat mengekspor data nilai', 'error');
    }
  };

  const columns = useMemo(() => [
    { name: 'NISN', selector: row => row.nisn || '-', sortable: true },
    { name: 'Nama', selector: row => row.nama || '-', sortable: true },
    { name: 'Pelajaran', selector: row => row.name_pelajaran || '-', sortable: true },
    { name: 'Tahun', selector: row => row.tahun || '-', sortable: true },
    { name: 'Guru', selector: row => row.teacher || '-', sortable: true },
    { name: 'STS Ganjil', selector: row => row.sts_ganjil ?? '-', sortable: true, right: true },
    { name: 'SAS', selector: row => row.sas ?? '-', sortable: true, right: true },
    { name: 'STS Genap', selector: row => row.sts_genap ?? '-', sortable: true, right: true },
    { name: 'SAT', selector: row => row.sat ?? '-', sortable: true, right: true },
    { name: 'US', selector: row => row.us ?? '-', sortable: true, right: true },
    { name: 'Harian', selector: row => row.harian ?? '-', sortable: true, right: true },
    { name: 'Keterampilan', selector: row => row.keterampilan ?? '-', sortable: true, right: true },
    {
      name: 'Aksi',
      cell: row => (
        <div className="d-flex gap-2">
          <button className="btn btn-sm btn-outline-secondary" title="Detail" onClick={() => openDetail(row)}>
            <FaInfoCircle />
          </button>
          {canEditScore && (
            <button className="btn btn-sm btn-outline-primary" title="Edit" onClick={() => openEdit(row)}>
              <FaEdit />
            </button>
          )}
          {canDeleteScore && (
            <button className="btn btn-sm btn-outline-danger" title="Hapus" onClick={() => removeItem(row)}>
              <FaTrash />
            </button>
          )}
        </div>
      ),
      ignoreRowClick: true,
      allowOverflow: true,
      button: true,
    }
  ], [canEditScore, canDeleteScore]);

  const filtered = useMemo(() => {
    const txt = (filterText || '').toLowerCase();
    return (scores || []).filter(s =>
      ((s.nisn || '').toLowerCase().includes(txt)) ||
      ((s.nama || '').toLowerCase().includes(txt)) ||
      ((s.name_pelajaran || '').toLowerCase().includes(txt)) ||
      ((s.tahun || '').toLowerCase().includes(txt)) ||
      ((s.teacher || '').toLowerCase().includes(txt))
    );
  }, [scores, filterText]);

  if (loading) return <div className="text-center mt-5">Memuat data nilai...</div>;

  return (
    <div className="container mt-4">
      <h3>Nilai (Scores)</h3>

      <div className="d-flex justify-content-between align-items-center mb-3 gap-2 flex-wrap">
        <div className="d-flex gap-2 align-items-center">
          {canCreateScore && (
            <button className="btn btn-sm btn-primary d-flex align-items-center" onClick={openAdd}>
              <FaPlus className="me-1" /> Tambah
            </button>
          )}
          {canExportScore && (
            <button className="btn btn-sm btn-success d-flex align-items-center" onClick={exportData}>
              <FaDownload className="me-1" /> Export
            </button>
          )}
        </div>
        <input
          type="text"
          placeholder="Cari NISN/Nama/Pelajaran/Tahun/Guru..."
          className="form-control"
          style={{ maxWidth: '320px' }}
          value={filterText}
          onChange={e => setFilterText(e.target.value)}
        />
      </div>

      <DataTable
        columns={columns}
        data={filtered}
        pagination
        highlightOnHover
        striped
        noHeader
      />

      {/* Modal Add */}
      {showAddModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog modal-lg">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Tambah Nilai</h5>
                <button type="button" className="btn-close" onClick={() => setShowAddModal(false)}></button>
              </div>
              <div className="modal-body">
                <div className="row g-2">
                  <div className="col-md-4">
                    <input className="form-control" placeholder="NISN" value={form.nisn} onChange={e => setForm({ ...form, nisn: e.target.value })} />
                  </div>
                  <div className="col-md-4">
                    <input className="form-control" placeholder="Nama" value={form.nama} onChange={e => setForm({ ...form, nama: e.target.value })} />
                  </div>
                  <div className="col-md-4">
                    <input className="form-control" placeholder="Guru" value={form.teacher} onChange={e => setForm({ ...form, teacher: e.target.value })} />
                  </div>
                  <div className="col-md-6">
                    <select className="form-select" value={form.id_pelajaran} onChange={e => setForm({ ...form, id_pelajaran: e.target.value })}>
                      <option value="">Pilih Pelajaran (opsional)</option>
                      {subjects.map(s => (
                        <option key={s.id} value={s.id}>{s.name}</option>
                      ))}
                    </select>
                  </div>
                  <div className="col-md-6">
                    <input className="form-control" placeholder="Nama Pelajaran (opsional)" value={form.name_pelajaran} onChange={e => setForm({ ...form, name_pelajaran: e.target.value })} />
                  </div>
                  <div className="col-md-3">
                    <input type="number" step="0.01" className="form-control" placeholder="STS Ganjil" value={form.sts_ganjil} onChange={e => setForm({ ...form, sts_ganjil: e.target.value })} />
                  </div>
                  <div className="col-md-3">
                    <input type="number" step="0.01" className="form-control" placeholder="SAS" value={form.sas} onChange={e => setForm({ ...form, sas: e.target.value })} />
                  </div>
                  <div className="col-md-3">
                    <input type="number" step="0.01" className="form-control" placeholder="STS Genap" value={form.sts_genap} onChange={e => setForm({ ...form, sts_genap: e.target.value })} />
                  </div>
                  <div className="col-md-3">
                    <input type="number" step="0.01" className="form-control" placeholder="SAT" value={form.sat} onChange={e => setForm({ ...form, sat: e.target.value })} />
                  </div>
                  <div className="col-md-4">
                    <input type="number" step="0.01" className="form-control" placeholder="US" value={form.us} onChange={e => setForm({ ...form, us: e.target.value })} />
                  </div>
                  <div className="col-md-4">
                    <input type="number" step="0.01" className="form-control" placeholder="Harian" value={form.harian} onChange={e => setForm({ ...form, harian: e.target.value })} />
                  </div>
                  <div className="col-md-4">
                    <input type="number" step="0.01" className="form-control" placeholder="Keterampilan" value={form.keterampilan} onChange={e => setForm({ ...form, keterampilan: e.target.value })} />
                  </div>
                  <div className="col-md-6">
                    <input className="form-control" placeholder="Tahun" value={form.tahun} onChange={e => setForm({ ...form, tahun: e.target.value })} />
                  </div>
                  <div className="col-md-6">
                    <textarea className="form-control" placeholder="Keterangan" rows={2} value={form.keterangan} onChange={e => setForm({ ...form, keterangan: e.target.value })} />
                  </div>
                </div>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowAddModal(false)}>Tutup</button>
                <button className="btn btn-primary" onClick={saveAdd}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Edit */}
      {showEditModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog modal-lg">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Edit Nilai</h5>
                <button type="button" className="btn-close" onClick={() => setShowEditModal(false)}></button>
              </div>
              <div className="modal-body">
                <div className="row g-2">
                  <div className="col-md-4">
                    <input className="form-control" placeholder="NISN" value={form.nisn} onChange={e => setForm({ ...form, nisn: e.target.value })} />
                  </div>
                  <div className="col-md-4">
                    <input className="form-control" placeholder="Nama" value={form.nama} onChange={e => setForm({ ...form, nama: e.target.value })} />
                  </div>
                  <div className="col-md-4">
                    <input className="form-control" placeholder="Guru" value={form.teacher} onChange={e => setForm({ ...form, teacher: e.target.value })} />
                  </div>
                  <div className="col-md-6">
                    <select className="form-select" value={form.id_pelajaran} onChange={e => setForm({ ...form, id_pelajaran: e.target.value })}>
                      <option value="">Pilih Pelajaran (opsional)</option>
                      {subjects.map(s => (
                        <option key={s.id} value={s.id}>{s.name}</option>
                      ))}
                    </select>
                  </div>
                  <div className="col-md-6">
                    <input className="form-control" placeholder="Nama Pelajaran (opsional)" value={form.name_pelajaran} onChange={e => setForm({ ...form, name_pelajaran: e.target.value })} />
                  </div>
                  <div className="col-md-3">
                    <input type="number" step="0.01" className="form-control" placeholder="STS Ganjil" value={form.sts_ganjil} onChange={e => setForm({ ...form, sts_ganjil: e.target.value })} />
                  </div>
                  <div className="col-md-3">
                    <input type="number" step="0.01" className="form-control" placeholder="SAS" value={form.sas} onChange={e => setForm({ ...form, sas: e.target.value })} />
                  </div>
                  <div className="col-md-3">
                    <input type="number" step="0.01" className="form-control" placeholder="STS Genap" value={form.sts_genap} onChange={e => setForm({ ...form, sts_genap: e.target.value })} />
                  </div>
                  <div className="col-md-3">
                    <input type="number" step="0.01" className="form-control" placeholder="SAT" value={form.sat} onChange={e => setForm({ ...form, sat: e.target.value })} />
                  </div>
                  <div className="col-md-4">
                    <input type="number" step="0.01" className="form-control" placeholder="US" value={form.us} onChange={e => setForm({ ...form, us: e.target.value })} />
                  </div>
                  <div className="col-md-4">
                    <input type="number" step="0.01" className="form-control" placeholder="Harian" value={form.harian} onChange={e => setForm({ ...form, harian: e.target.value })} />
                  </div>
                  <div className="col-md-4">
                    <input type="number" step="0.01" className="form-control" placeholder="Keterampilan" value={form.keterampilan} onChange={e => setForm({ ...form, keterampilan: e.target.value })} />
                  </div>
                  <div className="col-md-6">
                    <input className="form-control" placeholder="Tahun" value={form.tahun} onChange={e => setForm({ ...form, tahun: e.target.value })} />
                  </div>
                  <div className="col-md-6">
                    <textarea className="form-control" placeholder="Keterangan" rows={2} value={form.keterangan} onChange={e => setForm({ ...form, keterangan: e.target.value })} />
                  </div>
                </div>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowEditModal(false)}>Tutup</button>
                <button className="btn btn-primary" onClick={saveEdit}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Detail */}
      {showDetailModal && detailItem && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Detail Nilai</h5>
                <button type="button" className="btn-close" onClick={() => setShowDetailModal(false)}></button>
              </div>
              <div className="modal-body">
                <ul className="list-group list-group-flush">
                  <li className="list-group-item"><strong>NISN:</strong> {detailItem.nisn || '-'}</li>
                  <li className="list-group-item"><strong>Nama:</strong> {detailItem.nama || '-'}</li>
                  <li className="list-group-item"><strong>Pelajaran:</strong> {detailItem.name_pelajaran || '-'}</li>
                  <li className="list-group-item"><strong>Tahun:</strong> {detailItem.tahun || '-'}</li>
                  <li className="list-group-item"><strong>Guru:</strong> {detailItem.teacher || '-'}</li>
                  <li className="list-group-item"><strong>STS Ganjil:</strong> {detailItem.sts_ganjil ?? '-'}</li>
                  <li className="list-group-item"><strong>SAS:</strong> {detailItem.sas ?? '-'}</li>
                  <li className="list-group-item"><strong>STS Genap:</strong> {detailItem.sts_genap ?? '-'}</li>
                  <li className="list-group-item"><strong>SAT:</strong> {detailItem.sat ?? '-'}</li>
                  <li className="list-group-item"><strong>US:</strong> {detailItem.us ?? '-'}</li>
                  <li className="list-group-item"><strong>Harian:</strong> {detailItem.harian ?? '-'}</li>
                  <li className="list-group-item"><strong>Keterampilan:</strong> {detailItem.keterampilan ?? '-'}</li>
                  <li className="list-group-item"><strong>Keterangan:</strong> {detailItem.keterangan || '-'}</li>
                </ul>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowDetailModal(false)}>Tutup</button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}