import React, { useState, useEffect, useRef } from "react";
import { useParams, useLocation, useNavigate } from "react-router-dom";
import api from "../utils/axios";
import { InlineMath, BlockMath } from "react-katex";
import "katex/dist/katex.min.css";
import katex from "katex";
import { renderMathInElement } from "mathlive";
import RichTextEditor from "../components/RichTextEditor";
import { sanitizeLatex } from '../utils/latex';
import { FaArrowLeft, FaPrint, FaFileWord } from 'react-icons/fa';

import Swal from 'sweetalert2';
// Backend base untuk akses file di /storage (fallback dev)
const BACKEND_BASE = import.meta.env.VITE_BACKEND_URL || 'http://127.0.0.1:8000';
const SCHOOL_NAME_DEFAULT = import.meta.env.VITE_SCHOOL_NAME || 'Nama Sekolah';
const SCHOOL_ADDRESS_DEFAULT = import.meta.env.VITE_SCHOOL_ADDRESS || 'Alamat Sekolah';
const resolveImageUrl = (urlOrPath) => {
  if (!urlOrPath) return '';
  const s = String(urlOrPath).trim();
  const base = String(BACKEND_BASE || '').replace(/\/+$/, '');
  if (/^(data|blob):/.test(s)) return s;
  // Rebase absolut ke /storage/ atau /images/ agar origin mengikuti BACKEND_BASE
  if (/^https?:\/\/[^/]+\/(storage|images)\//.test(s)) {
    return s.replace(/^https?:\/\/[^/]+/, base);
  }
  if (/^https?:\/\//.test(s)) return s;
  const clean = s.replace(/^\/+/, '');
  if (/^storage\//.test(clean)) {
    return `${base}/storage/${clean.replace(/^storage\//,'')}`;
  }
  if (/^images\//.test(clean)) {
    return `${base}/${clean}`;
  }
  if (/^\/storage\//.test(s)) return `${base}${s}`;
  return `${base}/storage/${clean.replace(/^public\//,'')}`;
};

// Deteksi sederhana apakah string tampak sebagai LaTeX (global scope)
const looksLatex = (s) => {
  if (!s || typeof s !== 'string') return false;
  // Lebih konservatif: hanya deteksi jika ada perintah LaTeX yang umum
  return /\\(frac|sqrt|left|right|begin|end|bmatrix|pm|times|div)/.test(s);
};

// Tambahkan helper untuk normalisasi konten soal (sanitize lebih awal)
const normalizeQuestion = (q) => {
  if (!q || typeof q !== 'object') return q;
  const copy = { ...q };
  ['text','option_a','option_b','option_c','option_d','option_e'].forEach(k => {
    const v = copy[k];
    if (typeof v === 'string' && looksLatex(v)) {
      copy[k] = sanitizeLatex(v);
    }
  });
  // Pastikan key_answer terset huruf besar dan tidak kosong
  const keyRaw = String(copy.key_answer || '').trim().toUpperCase();
  copy.key_answer = keyRaw || 'A';
  return copy;
};

// Masking tautan gambar di input agar tidak tampil mentah
const IMAGE_TOKEN_REGEX = /!\[[^\]]*\]\([^\)]+\)/g;
const PLACEHOLDER_REGEX = /\[Gambar(?:\s*\d+)?\]/g;
const maskImages = (s) => {
  if (!s) return s;
  return String(s).replace(IMAGE_TOKEN_REGEX, () => '[Gambar]');
};
const unmaskWithTokens = (prevRaw, newMasked) => {
  if (!newMasked) return '';
  const tokens = (String(prevRaw).match(IMAGE_TOKEN_REGEX) || []);
  let idx = 0;
  // Ganti placeholder dengan token gambar sesuai urutan lama; placeholder ekstra dihapus
  return String(newMasked).replace(PLACEHOLDER_REGEX, () => tokens[idx++] || '');
};

class ErrorBoundaryQuestions extends React.Component {
  constructor(props){
    super(props);
    this.state = { error: null };
  }
  static getDerivedStateFromError(error){
    return { error };
  }
  componentDidCatch(error, info){
    // Log ke console agar mudah dilacak saat dev
    // eslint-disable-next-line no-console
    console.error('Questions page crashed:', error, info);
  }
  render(){
    const { error } = this.state;
    if (error){
      return (
        <div className="container mt-4">
          <div className="alert alert-danger" role="alert">
            Terjadi error saat memuat halaman Questions. Detail: {String(error?.message || error)}
          </div>
        </div>
      );
    }
    return this.props.children;
  }
}

export default function Questions() {
  const { subjectId } = useParams();
  const location = useLocation();
  const navigate = useNavigate();
  const addMode = new URLSearchParams(location.search).get('add') === '1';
  const [subjectName, setSubjectName] = useState('');
  const [subjectSchoolId, setSubjectSchoolId] = useState(null);
  const [schoolInfo, setSchoolInfo] = useState(null);
  const [text, setText] = useState("");
  const [opts, setOpts] = useState({ A: "", B: "", C: "", D: "", E: "" });
  const [key, setKey] = useState("A");
  const [essayAnswer, setEssayAnswer] = useState("");
  
  const [submitting, setSubmitting] = useState(false);
  const [list, setList] = useState([]);
  const [showForm, setShowForm] = useState(false);
  const [editingQuestion, setEditingQuestion] = useState(null);
  const [formErrors, setFormErrors] = useState({});
  const [errorMessage, setErrorMessage] = useState("");
  
  // Nonaktifkan fitur rumus sesuai permintaan
  const [arabicQuestion, setArabicQuestion] = useState(false);
  const [arabicOption, setArabicOption] = useState({ A:false, B:false, C:false, D:false, E:false });
  // Tambahkan kontrol lebar gambar

  const textRef = useRef(null); // legacy ref untuk textarea (fallback)
  const richTextRef = useRef(null); // ref untuk editor WYSIWYG soal
  const formTopRef = useRef(null); // anchor untuk scroll ke atas form saat edit
  const inlineImageInputRef = useRef(null);
  const optRefs = {
    A: useRef(null),
    B: useRef(null),
    C: useRef(null),
    D: useRef(null),
    E: useRef(null),
  };
  const richOptRefs = {
    A: useRef(null),
    B: useRef(null),
    C: useRef(null),
    D: useRef(null),
    E: useRef(null),
  };
  const inlineImageOptionInputRefs = {
    A: useRef(null),
    B: useRef(null),
    C: useRef(null),
    D: useRef(null),
    E: useRef(null),
  };
  const listRef = useRef(null);
  const optionColors = { A: '#0d6efd', B: '#198754', C: '#fd7e14', D: '#6f42c1', E: '#20c997' };
  const optionBadgeClasses = { A: 'bg-primary', B: 'bg-success', C: 'bg-warning text-dark', D: 'bg-info text-dark', E: 'bg-danger' };

  // Util: deteksi essay (opsi kosong atau '-' pada B,C,D atau semua kosong)
  const isEssayType = (q) => {
    if (!q || typeof q !== 'object') return false;
    const blank = (v) => { const t = String(v || '').trim(); return t === '' || t === '-'; };
    const bdBlank = blank(q?.option_b) && blank(q?.option_c) && blank(q?.option_d);
    const allBlank = ['option_a','option_b','option_c','option_d','option_e'].every(k => blank(q[k]));
    return bdBlank || allBlank;
  };

  // Bangun HTML untuk Export/Print
  const buildExportHtml = (opts = {}) => {
    const mathAsImage = !!opts.mathAsImage;
    const mathAsWord = !!opts.mathAsWord;
    const forWord = !!opts.forWord;
    const safe = (s) => String(s || '');
    const imgStyleStr = forWord
      ? styleObjectToString({ width: '7cm', height: '4cm', objectFit: 'contain', borderRadius: 6, margin: '4px 0' })
      : styleObjectToString({ maxWidth: '75%', maxHeight: '240px', objectFit: 'contain', borderRadius: 6, margin: '4px 0' });
    const decodeHtmlEntities = (input) => {
      return String(input || '')
        .replace(/&amp;/g, '&')
        .replace(/&lt;/g, '<')
        .replace(/&gt;/g, '>')
        .replace(/&quot;/g, '"')
        .replace(/&#039;/g, "'")
        .replace(/&#39;/g, "'");
    };
    const processText = (raw) => {
      // Split LaTeX tokens, render KaTeX for math, BBCode→HTML for text
      const input = autoWrapLatexIfMissing(safe(raw));
      const segments = splitLatexTokens(input);
      let out = '';
      for (const seg of segments) {
        if (seg.type === 'text') {
          let html = bbcodeToHtml(seg.value, { imgStyle: forWord ? { width: '7cm', height: '4cm', objectFit: 'contain', borderRadius: 6, margin: '4px 0' } : { maxWidth: '75%', maxHeight: '240px', objectFit: 'contain', borderRadius: 6, margin: '4px 0' }, disableYoutube: true });
          // markdown image → img
          html = html.replace(/!\[([^\]]*)\]\(([^)]+)\)/g, (m, alt, url) => {
            const full = resolveImageUrl(String(url || '').trim());
            const altRaw = String(alt || '').trim();
            const altBase = altRaw.replace(/\|.*$/, '').trim() || 'img';
            const widthMatches = altRaw.match(/\bw\s*=\s*(\d{1,3})\s*%/gi);
            let widthPct = null;
            if (widthMatches && widthMatches.length > 0) {
              const lastMatch = widthMatches[widthMatches.length - 1];
              const mW = lastMatch.match(/\bw\s*=\s*(\d{1,3})\s*%/i);
              widthPct = mW ? Math.max(5, Math.min(160, Number(mW[1]))) : null;
            }
            if (forWord) {
              const styleStr = `${imgStyleStr}`;
              // 7cm x 4cm in pixels approximated at 96dpi: 265 x 151
              return `<img src="${full}" alt="${altBase}" width="265" height="151" style="${styleStr}">`;
            } else {
              const styleStr = widthPct ? `${imgStyleStr} width: ${widthPct}%;` : imgStyleStr;
              return `<img src="${full}" alt="${altBase}" style="${styleStr}">`;
            }
          });
          out += html;
        } else if (seg.type === 'latex_inline' || seg.type === 'latex_block') {
          const display = seg.type === 'latex_block';
          const math = sanitizeLatex(decodeHtmlEntities(stripLatexDelimiters(seg.value)));
          // Untuk Word, coba render MathML agar terlihat sebagai equation
          if (forWord && typeof mathAsMathML !== 'undefined' && mathAsMathML) {
            let rendered = '';
            try {
              rendered = katex.renderToString(math, { output: 'htmlAndMathml', throwOnError: false, displayMode: display });
            } catch (_) {
              rendered = `<span>${math}</span>`;
            }
            out += rendered;
            continue;
          }
          if (forWord && mathAsWord) {
            // Render sebagai Equation Word (OMML) sederhana agar LaTeX tampil di kontainer equation
            const omml = `<!--[if gte mso 9]><m:oMathPara><m:oMath><m:r><w:rPr/><m:t>${math.replace(/</g,'&lt;').replace(/>/g,'&gt;')}</m:t></m:r></m:oMath></m:oMathPara><![endif]-->`;
            const wrapper = display ? `<div style="text-align:center; margin:6px 0;">${omml}</div>` : omml;
            out += wrapper;
          } else if (mathAsImage) {
            // Konversi LaTeX menjadi gambar PNG (bukan SVG) agar kompatibel dengan Word
            const base = 'https://latex.codecogs.com/png.image?';
            const url = base + encodeURIComponent('\\dpi{150} ' + math);
            const style = display
              ? 'display:block;margin:8px auto;max-width:75%;height:auto;vertical-align:middle;'
              : 'display:inline-block;margin:0 2px;max-width:75%;height:auto;vertical-align:middle;';
            // Tandai gambar math untuk di-embed sebagai data URI saat ekspor Word
            out += `<img src="${url}" data-img-math="1" alt="latex" style="${style}">`;
          } else {
            let rendered = '';
            try {
              rendered = katex.renderToString(math, { throwOnError: false, displayMode: display });
            } catch (_) {
              rendered = `<span>${math}</span>`;
            }
            out += rendered;
          }
        }
      }
      return out;
    };

    const logoUrl = schoolInfo?.logo_path ? resolveImageUrl(schoolInfo.logo_path) : '';
    const kop = `
      <div style="position: relative; margin-bottom: 10px; min-height: 64px;">
        ${logoUrl ? (forWord
          ? `<img src="${logoUrl}" alt="Logo" width="60" height="60" style="position:absolute; left:0; top:0; width:1.6cm; height:1.6cm; object-fit:contain; z-index:2; pointer-events:none;"/>`
          : `<img src="${logoUrl}" alt="Logo" style="position:absolute; left:0; top:0; height:64px; width:auto; object-fit:contain;"/>`
        ) : ''}
        <div style="text-align:center;">
          <div style="font-size: 18px; font-weight: 700;">${schoolInfo?.nama || SCHOOL_NAME_DEFAULT}</div>
          <div style="font-size: 12px; margin-bottom: 30px;"">${schoolInfo?.alamat || SCHOOL_ADDRESS_DEFAULT}</div>
          <div style="display:block; width:100%; clear:both; border-bottom:2px solid #000; margin:8px 0 24px;"></div>
          <div style="font-size: 18px; font-weight: 700;">Bank Soal</div>
          <div style="font-size: 14px;">${subjectName ? ('Mata Pelajaran: ' + subjectName) : 'Mata Pelajaran'}</div>
        </div>
      </div>
    `;

    const items = (Array.isArray(list) ? list : []).map((q, idx) => {
      const qNum = idx + 1;
      const qText = processText(q.text);
      const isEssay = isEssayType(q);
      const keyAns = String(q.key_answer || '').trim();
      const optHtml = isEssay ? '' : ['A','B','C','D','E'].map(letter => {
        const raw = q[`option_${letter.toLowerCase()}`];
        if (!raw || String(raw).trim() === '') {
          if (letter === 'E') return '';
        }
        const body = raw ? processText(raw) : '<span class="text-muted">-</span>';
        return `
          <div style="margin: 2px 0;">
            <span style="display:inline-block; min-width:22px; font-weight:600;">${letter}.</span>
            <span>${body}</span>
          </div>
        `;
      }).join('');

      const keyBlock = isEssay
        ? `<div style="margin-top:6px"><strong>Kunci Essay:</strong> ${processText(q.key_answer)}</div>`
        : `<div style="margin-top:6px"><strong>Kunci:</strong> ${keyAns}</div>`;

      return `
        <div style="padding: 8px 10px; margin-bottom:10px;">
          <div style="font-weight:700; margin-bottom:6px;">Soal #${qNum}</div>
          <div style="white-space:pre-wrap;">${qText}</div>
          ${!isEssay ? `<div style="margin-top:6px;">${optHtml}</div>` : ''}
          ${keyBlock}
        </div>
      `;
    }).join('');

    const style = `
      <style>
        body { font-family: Arial, sans-serif; font-size: 12pt; }
        .text-muted { color: #6c757d; }
        @media print {
          .no-print { display: none !important; }
        }
        ${forWord
          ? 'img { page-break-inside: avoid; }'
          : 'img { page-break-inside: avoid; max-width: 75%; height: auto; object-fit: contain; }'
        }
        /* Minimal KaTeX fallback to keep math readable in Word */
        .katex { font-size: 1.05em; }
        .katex-display { text-align: center; margin: 8px 0; }
        .katex .mord, .katex .mop, .katex .mbin, .katex .mrel, .katex .mopen, .katex .mclose, .katex .mpunct, .katex .mspace { font-size: 1em; }
        .katex .fontsize-ensurer { display:inline-block; }
        .katex .strut { height: .8em; }
      </style>
    `;

    const htmlDoc = `
      <!doctype html>
      <html>
        <head>
          <meta charset="utf-8" />
          <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/katex@0.16.9/dist/katex.min.css" />
          ${style}
          <title>Export Soal — ${subjectName || ''}</title>
        </head>
        <body>
          ${kop}
          ${items}
        </body>
      </html>
    `;
    return htmlDoc;
  };

  const handlePrint = () => {
    const html = buildExportHtml({ mathAsImage: false, forWord: false });
    const w = window.open('', '_blank');
    if (!w) return;
    w.document.open('text/html');
    w.document.write(html);
    w.document.close();
    setTimeout(() => { try { w.focus(); w.print(); } catch(_) {} }, 300);
  };

  const handleExportWord = async () => {
    // Gunakan HTML khusus Word: logo kecil dan menimpa, gambar 7x4 cm, LaTeX → gambar PNG
    let html = buildExportHtml({ mathAsImage: true, mathAsWord: false, mathAsMathML: false, forWord: true });

    // Parse HTML dan embed gambar LaTeX (data-img-math) sebagai data:image/png;base64
    const parser = new DOMParser();
    const doc = parser.parseFromString(html, 'text/html');
    const imgs = Array.from(doc.querySelectorAll('img[data-img-math="1"]'));

    const fetchToBase64 = async (src) => {
      try {
        const res = await fetch(src);
        if (!res.ok) throw new Error('HTTP ' + res.status);
        const blob = await res.blob();
        const b64 = await new Promise((resolve, reject) => {
          const reader = new FileReader();
          reader.onload = () => {
            const dataUrl = reader.result || '';
            const base64 = String(dataUrl).split(',')[1] || '';
            resolve(base64);
          };
          reader.onerror = reject;
          reader.readAsDataURL(blob);
        });
        return b64;
      } catch (err) {
        console.warn('Gagal konversi gambar LaTeX ke base64:', err);
        return null;
      }
    };

    for (const img of imgs) {
      const src = img.getAttribute('src');
      if (!src) continue;
      const b64 = await fetchToBase64(src);
      if (b64) {
        img.setAttribute('src', 'data:image/png;base64,' + b64);
      }
    }

    // Tambahkan namespace Word pada tag <html>
    doc.documentElement.setAttribute('xmlns:o', 'urn:schemas-microsoft-com:office:office');
    doc.documentElement.setAttribute('xmlns:w', 'urn:schemas-microsoft-com:office:word');
    doc.documentElement.setAttribute('xmlns:m', 'http://schemas.microsoft.com/office/2007/omml');
    doc.documentElement.setAttribute('xmlns:mml', 'http://www.w3.org/1998/Math/MathML');
    doc.documentElement.setAttribute('xmlns', 'http://www.w3.org/TR/REC-html40');

    // Sisipkan Office settings di <head>
    const head = doc.querySelector('head');
    if (head) {
      const officeMarker = doc.createElement('meta');
      officeMarker.setAttribute('data-office', 'mso9-settings');
      head.insertAdjacentHTML('afterbegin', '<!--[if gte mso 9]><xml><w:WordDocument><w:View>Print</w:View><w:Zoom>100</w:Zoom><w:DoNotOptimizeForBrowser/></w:WordDocument></xml><![endif]-->');
    }

    let finalHtml = '<!doctype html>\n' + doc.documentElement.outerHTML;

    // Tambahkan BOM agar Word menangkap encoding UTF-8 dengan benar
    const blob = new Blob(["\ufeff", finalHtml], { type: 'application/msword;charset=utf-8' });
    const a = document.createElement('a');
    const url = URL.createObjectURL(blob);
    a.href = url;
    const safeSubject = sanitizeFileName(subjectName ? subjectName : 'Mata_Pelajaran');
    a.download = `Soal_${safeSubject}.doc`;
    document.body.appendChild(a);
    a.click();
    setTimeout(() => { URL.revokeObjectURL(url); document.body.removeChild(a); }, 0);
  };

  // Auto-render KaTeX pada container daftar saat list berubah
  useEffect(() => {
    if (listRef.current) {
      renderMathInElement(listRef.current, {
        delimiters: [
          { left: "\\(", right: "\\)", display: false },
          { left: "\\[", right: "\\]", display: true }
        ],
        throwOnError: false
      });
    }
  }, [list]);

  // Deteksi apakah sedang mengedit soal bertipe Essay (selaras dengan backend, berbasis state saat ini)
  const blankOrDash = (v) => { const t = String(v || '').trim(); return t === '' || t === '-'; };
  const upperKeyNow = String(key || '').trim().toUpperCase();
  const isEditingEssay = !!editingQuestion && (
    !['A','B','C','D','E'].includes(upperKeyNow) ||
    (blankOrDash(opts.B) && blankOrDash(opts.C) && blankOrDash(opts.D)) ||
    (blankOrDash(editingQuestion?.option_b) && blankOrDash(editingQuestion?.option_c) && blankOrDash(editingQuestion?.option_d))
  );

  // MathLive panel state & refs
  const [showMathLive, setShowMathLive] = useState(false);
  const mathLiveContainerRef = useRef(null);
  const mathLiveFieldRef = useRef(null);
  const [mathLiveError, setMathLiveError] = useState('');
  // Flag: sedang edit rumus aktif dari editor (klik rumus)
  const [editingActiveLatex, setEditingActiveLatex] = useState(false);

  // MathLive untuk Opsi (panel per huruf)
  const [mathLiveOptTarget, setMathLiveOptTarget] = useState(null); // 'A'|'B'|'C'|'D'|'E'|null
  const mathLiveOptionContainerRefs = {
    A: useRef(null),
    B: useRef(null),
    C: useRef(null),
    D: useRef(null),
    E: useRef(null),
  };
  const mathLiveOptionFieldRef = useRef(null);
  const [mathLiveOptError, setMathLiveOptError] = useState('');
  // Flag per opsi: sedang edit rumus aktif
  const [editingActiveOptionLatex, setEditingActiveOptionLatex] = useState({ A:false, B:false, C:false, D:false, E:false });

  // Style gambar untuk berbagai konteks
  const PREVIEW_IMG_STYLE = { maxWidth: '100%', maxHeight: '240px', objectFit: 'contain', borderRadius: 6, margin: '6px 0' };
  const LIST_INLINE_IMG_STYLE = { maxWidth: '100%', maxHeight: '140px', objectFit: 'contain', borderRadius: 6, margin: '4px 0' };
  const OPTION_PREVIEW_IMG_STYLE = { maxWidth: '100%', maxHeight: '180px', objectFit: 'contain', borderRadius: 6, margin: '6px 0' };

  // looksLatex tersedia di scope global

  useEffect(() => {
    const fetchList = async () => {
      try {
        // Ambil semua halaman pertanyaan untuk subject ini (paginate 20 di backend)
        if (subjectId) {
          const acc = [];
          let page = 1; let lastPage = 1;
          do {
            const res = await api.get('/questions', { params: { subject_id: subjectId, page } });
            const data = res.data;
            if (Array.isArray(data)) {
              acc.push(...data.map(normalizeQuestion));
              lastPage = page; // non-paginated fallback
            } else {
              const list = Array.isArray(data?.data) ? data.data : [];
              acc.push(...list.map(normalizeQuestion));
              lastPage = Number(data?.last_page || 1);
            }
            page += 1;
          } while (page <= lastPage && page < 200);
          setList(acc);
        } else {
          // Tanpa subjectId, tetap ambil sesuai respons (fallback)
          const res = await api.get('/questions');
          const raw = res.data?.data ?? res.data ?? [];
          const cleaned = Array.isArray(raw) ? raw.map(normalizeQuestion) : raw;
          setList(cleaned);
        }
      } catch(e) { /* ignore */ }
    };
    fetchList();
  }, [subjectId]);

  // Penomoran form: saat menambah, nomor = jumlah soal + 1; saat edit, nomor sesuai indeks + 1
  const currentNumber = (() => {
    if (editingQuestion) {
      const idx = Array.isArray(list) ? list.findIndex(q => q.id === editingQuestion.id) : -1;
      return idx >= 0 ? (idx + 1) : 1;
    }
    const len = Array.isArray(list) ? list.length : 0;
    return Math.max(1, len + 1);
  })();

  // Ambil nama mata pelajaran untuk ditampilkan di header
  useEffect(() => {
    const fetchSubjectAndSchool = async () => {
      if (!subjectId) { setSubjectName(''); setSubjectSchoolId(null); setSchoolInfo(null); return; }
      try {
        const res = await api.get(`/subjects/${subjectId}`, { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } });
        setSubjectName(res.data?.name || '');
        const sid = res.data?.id_school || null;
        setSubjectSchoolId(sid);
        try {
          const schoolsRes = await api.get('/schools', { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } });
          const arr = Array.isArray(schoolsRes.data) ? schoolsRes.data : [];
          let found = null;
          if (sid) { found = arr.find(s => Number(s.id) === Number(sid)) || null; }
          if (!found) { found = arr.find(s => String(s.status).toLowerCase() === 'aktif') || arr[0] || null; }
          setSchoolInfo(found);
        } catch (_) {
          setSchoolInfo(null);
        }
      } catch (e) {
        setSubjectName('');
        setSubjectSchoolId(null);
        setSchoolInfo(null);
      }
    };
    fetchSubjectAndSchool();
  }, [subjectId]);

  useEffect(() => {
    if (addMode) setShowForm(true);
  }, [addMode]);

  // Auto scroll + fokus ke atas form saat mulai edit
  useEffect(() => {
    if (showForm && editingQuestion) {
      setTimeout(() => {
        try {
          formTopRef.current?.scrollIntoView({ behavior: 'smooth', block: 'start' });
        } catch (_) {}
        try {
          if (richTextRef.current && typeof richTextRef.current.focus === 'function') {
            richTextRef.current.focus();
          }
        } catch (_) {}
      }, 0);
    }
  }, [showForm, editingQuestion]);

  const insertAtCursor = (insertText) => {
    const el = textRef.current;
    if (!el) { setText((prev) => (prev + insertText)); return; }
    const start = el.selectionStart ?? el.value.length;
    const end = el.selectionEnd ?? el.value.length;
    const before = text.slice(0, start);
    const after = text.slice(end);
    const next = before + insertText + after;
    setText(next);
    // restore cursor after update (best-effort)
    setTimeout(() => {
      if (textRef.current) {
        const pos = start + insertText.length;
        textRef.current.selectionStart = pos;
        textRef.current.selectionEnd = pos;
        textRef.current.focus();
      }
    }, 0);
  };

  const wrapTextSelection = (tag) => {
    const el = textRef.current;
    const open = `[${tag}]`;
    const close = `[/${tag}]`;
    if (!el || typeof el.selectionStart !== 'number') {
      insertAtCursor(`${open}${close}`);
      return;
    }
    const start = el.selectionStart ?? el.value.length;
    const end = el.selectionEnd ?? el.value.length;
    const sel = text.slice(start, end);
    const before = text.slice(0, start);
    const after = text.slice(end);
    const next = before + open + sel + close + after;
    setText(next);
    setTimeout(() => {
      if (textRef.current) {
        const pos = start + open.length + sel.length + close.length;
        textRef.current.selectionStart = pos;
        textRef.current.selectionEnd = pos;
        textRef.current.focus();
      }
    }, 0);
  };

  // Handler untuk WYSIWYG soal: gunakan execCommand melalui RichTextEditor
  const formatSoal = (tag) => {
    if (richTextRef.current && typeof richTextRef.current.applyFormat === 'function') {
      richTextRef.current.applyFormat(tag);
    } else {
      // fallback ke BBCode
      wrapTextSelection(tag);
    }
  };
  const insertToSoal = (snippet) => {
    if (richTextRef.current && typeof richTextRef.current.insertSnippet === 'function') {
      richTextRef.current.insertSnippet(snippet);
    } else {
      insertAtCursor(snippet);
    }
  };

  // Insert → Equation sekarang menampilkan panel MathLive
  const insertEquation = () => {
    setShowMathLive(prev => !prev);
  };

  const insertFromMathLive = () => {
    const mf = mathLiveFieldRef.current;
    let latex = '';
    try {
      latex = mf && typeof mf.getValue === 'function' ? mf.getValue('latex-expanded') : '';
    } catch(e) { /* ignore */ }
    const s = String(latex || '').trim();
    if (!s) return;
    // Jika sedang mengedit rumus aktif (klik rumus), ganti; jika tidak, sisipkan baru sebagai KaTeX span
    try {
      if (editingActiveLatex && richTextRef.current && typeof richTextRef.current.replaceActiveLatex === 'function') {
        richTextRef.current.replaceActiveLatex(s);
      } else if (richTextRef.current && typeof richTextRef.current.insertLatex === 'function') {
        richTextRef.current.insertLatex(s);
      } else {
        insertToSoal(` \\(${s}\\) `);
      }
    } catch(_) {
      insertToSoal(` \\(${s}\\) `);
    }
    setEditingActiveLatex(false);
    setShowMathLive(false);
  };

  useEffect(() => {
    let disposed = false;
    const setup = async () => {
      try {
        const mod = await import('mathlive');
        if (disposed) return;
        const el = new mod.MathfieldElement({
          // Hanya opsi yang masih valid
        });
        // Set opsi deprecated setelah constructor
        el.smartMode = true;
        // keypressSound: akses via el.mathVirtualKeyboard atau window.mathVirtualKeyboard
        try {
          if (el.mathVirtualKeyboard && typeof el.mathVirtualKeyboard === 'object') {
            if ('keypressSound' in el.mathVirtualKeyboard) {
              el.mathVirtualKeyboard.keypressSound = 'none';
            }
          } else if (typeof window !== 'undefined' && window.mathVirtualKeyboard && typeof window.mathVirtualKeyboard === 'object') {
            if ('keypressSound' in window.mathVirtualKeyboard) {
              window.mathVirtualKeyboard.keypressSound = 'none';
            }
          }
        } catch (_) { /* ignore */ }
        el.mathVirtualKeyboardPolicy = 'onfocus';
        // Styling agar menyatu dengan Bootstrap
        el.style.minHeight = '40px';
        el.style.border = '1px solid #ced4da';
        el.style.borderRadius = '0.25rem';
        el.style.padding = '6px 8px';
        el.style.width = '100%';
        if (mathLiveContainerRef.current) {
          mathLiveContainerRef.current.innerHTML = '';
          mathLiveContainerRef.current.appendChild(el);
        }
        mathLiveFieldRef.current = el;
        setMathLiveError('');
      } catch (err) {
        setMathLiveError('Gagal memuat MathLive');
      }
    };
    if (showMathLive) setup();
    return () => { disposed = true; };
  }, [showMathLive]);

  // Panel simbol LaTeX ringan untuk sisipkan cepat
  const LATEX_SYMBOLS = [
    { label: 'α', code: '\\alpha' },
    { label: 'β', code: '\\beta' },
    { label: 'γ', code: '\\gamma' },
    { label: 'θ', code: '\\theta' },
    { label: 'π', code: '\\pi' },
    { label: 'Δ', code: '\\Delta' },
    { label: 'Ω', code: '\\Omega' },
    { label: '√', code: '\\sqrt{x}' },
    { label: '∠', code: '\\angle' },
    { label: '∥', code: '\\parallel' },
    { label: '⟂', code: '\\perp' },
    { label: '≤', code: '\\le' },
    { label: '≥', code: '\\ge' },
    { label: '≠', code: '\\neq' },
    { label: '≈', code: '\\approx' },
    { label: '≡', code: '\\equiv' },
    { label: '∞', code: '\\infty' },
    { label: '×', code: '\\times' },
    { label: '÷', code: '\\div' },
    { label: '·', code: '\\cdot' },
    { label: '°', code: '^\\circ' },
    // Kalkulus
    { label: '∫', code: '\\int' },
    { label: '∮', code: '\\oint' },
    { label: '∑', code: '\\sum' },
    { label: '∏', code: '\\prod' },
    { label: '∂', code: '\\partial' },
    { label: '∇', code: '\\nabla' },
    { label: 'lim', code: '\\lim_{x\\to a}' },
    { label: 'd/dx', code: '\\frac{d}{dx}' },
    { label: '∂/∂x', code: '\\frac{\\partial}{\\partial x}' },
  ];
  const insertLatexSymbol = (code) => {
    insertToSoal(` \\(${code}\\) `);
  };

  const insertToOptionLetter = (letter, snippet) => {
    const ref = richOptRefs[letter]?.current;
    if (ref && typeof ref.insertSnippet === 'function') ref.insertSnippet(snippet);
    else insertAtOptionCursor(letter, snippet);
  };

  const insertOptionLatexSymbol = (letter, code) => {
    insertToOptionLetter(letter, ` \\(${code}\\) `);
  };

  const toggleOptionEquation = (letter) => {
    setMathLiveOptTarget(prev => (prev === letter ? null : letter));
  };

  const insertFromMathLiveOption = () => {
    if (!mathLiveOptTarget) return;
    const mf = mathLiveOptionFieldRef.current;
    let latex = '';
    try {
      latex = mf && typeof mf.getValue === 'function' ? mf.getValue('latex-expanded') : '';
    } catch (_) {}
    const s = String(latex || '').trim();
    if (!s) return;
    const ref = richOptRefs[mathLiveOptTarget]?.current;
    try {
      if (editingActiveOptionLatex[mathLiveOptTarget] && ref && typeof ref.replaceActiveLatex === 'function') {
        ref.replaceActiveLatex(s);
      } else if (ref && typeof ref.insertLatex === 'function') {
        ref.insertLatex(s);
      } else {
        insertToOptionLetter(mathLiveOptTarget, ` \\(${s}\\) `);
      }
    } catch(_) {
      insertToOptionLetter(mathLiveOptTarget, ` \\(${s}\\) `);
    }
    setEditingActiveOptionLatex(prev => ({ ...prev, [mathLiveOptTarget]: false }));
    setMathLiveOptTarget(null);
  };

  useEffect(() => {
    let disposed = false;
    const setupForOption = async () => {
      if (!mathLiveOptTarget) return;
      try {
        const mod = await import('mathlive');
        if (disposed) return;
        const el = new mod.MathfieldElement({
          // Hanya opsi yang masih valid
        });
        // Set opsi deprecated setelah constructor
        el.smartMode = true;
        // keypressSound: akses via el.mathVirtualKeyboard atau window.mathVirtualKeyboard
        try {
          if (el.mathVirtualKeyboard && typeof el.mathVirtualKeyboard === 'object') {
            if ('keypressSound' in el.mathVirtualKeyboard) {
              el.mathVirtualKeyboard.keypressSound = 'none';
            }
          } else if (typeof window !== 'undefined' && window.mathVirtualKeyboard && typeof window.mathVirtualKeyboard === 'object') {
            if ('keypressSound' in window.mathVirtualKeyboard) {
              window.mathVirtualKeyboard.keypressSound = 'none';
            }
          }
        } catch (_) { /* ignore */ }
        el.mathVirtualKeyboardPolicy = 'onfocus';
        el.style.minHeight = '40px';
        el.style.border = '1px solid #ced4da';
        el.style.borderRadius = '0.25rem';
        el.style.padding = '6px 8px';
        el.style.width = '100%';
        const container = mathLiveOptionContainerRefs[mathLiveOptTarget]?.current;
        if (container) {
          container.innerHTML = '';
          container.appendChild(el);
        }
        mathLiveOptionFieldRef.current = el;
        setMathLiveOptError('');
      } catch (err) {
        setMathLiveOptError('Gagal memuat MathLive');
      }
    };
    setupForOption();
    return () => { disposed = true; };
  }, [mathLiveOptTarget]);

  const insertAtOptionCursor = (letter, insertText) => {
    const el = optRefs[letter]?.current;
    const curr = opts[letter] ?? '';
    if (!el || typeof el.selectionStart !== 'number') {
      setOpts(prev => ({ ...prev, [letter]: curr + insertText }));
      return;
    }
    const start = el.selectionStart ?? curr.length;
    const end = el.selectionEnd ?? curr.length;
    const before = curr.slice(0, start);
    const after = curr.slice(end);
    const next = before + insertText + after;
    setOpts(prev => ({ ...prev, [letter]: next }));
    setTimeout(() => {
      const target = optRefs[letter]?.current;
      if (target) {
        const pos = start + insertText.length;
        target.selectionStart = pos;
        target.selectionEnd = pos;
        target.focus();
      }
    }, 0);
  };

  const handleInlineImageUpload = async (file) => {
    if (!file) return;
    try {
      const fd = new FormData();
      const finalSubjectId = subjectId || editingQuestion?.subject_id;
      fd.append('image', file);
      fd.append('subject_id', String(finalSubjectId));
      const res = await api.post('/questions/upload-image', fd, {
        headers: { 'Content-Type': 'multipart/form-data', Authorization: `Bearer ${localStorage.getItem('token')}` }
      });
      const url = res?.data?.url || res?.data?.path || '';
      if (!url) return;
      const wRes = await Swal.fire({
        title: 'Ukuran gambar (%)',
        input: 'range',
        inputAttributes: { min: 5, max: 160, step: 1 },
        inputValue: 100,
        showCancelButton: true,
        confirmButtonText: 'OK',
        cancelButtonText: 'Batal'
      });
      const w = Math.max(5, Math.min(160, Number(wRes.isConfirmed ? wRes.value : 100)));
      if (richTextRef.current && typeof richTextRef.current.insertImageWithOptions === 'function') {
        richTextRef.current.insertImageWithOptions(url, w);
      } else if (richTextRef.current && typeof richTextRef.current.insertImage === 'function') {
        richTextRef.current.insertImage(url);
      } else {
        insertAtCursor(` ![img|w=${w}%](${url}) `);
      }
    } catch (e) {
      console.error(e);
      alert('Gagal memproses gambar');
    }
  };

  const choosePictureForQuestion = async () => {
    const res = await Swal.fire({
      title: 'Pilih gambar',
      input: 'file',
      inputAttributes: { accept: 'image/*' },
      showCancelButton: true,
      confirmButtonText: 'Sisipkan',
      cancelButtonText: 'Batal'
    });
    if (!res.isConfirmed) return;
    const file = res.value;
    if (!file) return;
    try {
      await handleInlineImageUpload(file);
    } catch (e) {
      console.error(e);
      Swal.fire('Gagal', 'Tidak bisa mengunggah gambar', 'error');
    }
  };

  const handleInlineImageUploadForOption = async (letter, file) => {
    if (!file) return;
    try {
      const fd = new FormData();
      const finalSubjectId = subjectId || editingQuestion?.subject_id;
      fd.append('image', file);
      fd.append('subject_id', String(finalSubjectId));
      const res = await api.post('/questions/upload-image', fd, {
        headers: { 'Content-Type': 'multipart/form-data', Authorization: `Bearer ${localStorage.getItem('token')}` }
      });
      const url = res?.data?.url || res?.data?.path || '';
      if (!url) return;
      const wRes = await Swal.fire({
        title: 'Ukuran gambar (%)',
        input: 'range',
        inputAttributes: { min: 5, max: 160, step: 1 },
        inputValue: 100,
        showCancelButton: true,
        confirmButtonText: 'OK',
        cancelButtonText: 'Batal'
      });
      const w = Math.max(5, Math.min(160, Number(wRes.isConfirmed ? wRes.value : 100)));
      const rich = richOptRefs[letter]?.current;
      if (rich && typeof rich.insertImageWithOptions === 'function') {
        rich.insertImageWithOptions(url, w);
      } else if (rich && typeof rich.insertImage === 'function') {
        rich.insertImage(url);
      } else {
        insertAtOptionCursor(letter, ` ![img|w=${w}%](${url}) `);
      }
    } catch (e) {
      console.error(e);
      alert('Gagal memproses gambar untuk opsi');
    }
  };

  const choosePictureForOption = async (letter) => {
    const res = await Swal.fire({
      title: `Pilih gambar untuk Opsi ${letter}`,
      input: 'file',
      inputAttributes: { accept: 'image/*' },
      showCancelButton: true,
      confirmButtonText: 'Sisipkan',
      cancelButtonText: 'Batal'
    });
    if (!res.isConfirmed) return;
    const file = res.value;
    if (!file) return;
    try {
      await handleInlineImageUploadForOption(letter, file);
    } catch (e) {
      console.error(e);
      Swal.fire('Gagal', 'Tidak bisa mengunggah gambar', 'error');
    }
  };

  const removeInlineImagesFromOption = (letter) => {
    const curr = opts[letter] ?? '';
    const next = curr.replace(/!\[[^\]]*\]\([^\)]+\)/g, '').replace(/\s{2,}/g,' ');
    setOpts(prev => ({ ...prev, [letter]: next.trim() }));
  };

  const wrapOptionSelection = (letter, tag) => {
    const el = optRefs[letter]?.current;
    const curr = opts[letter] ?? '';
    const open = `[${tag}]`;
    const close = `[/${tag}]`;
    if (!el || typeof el.selectionStart !== 'number') {
      setOpts(prev => ({ ...prev, [letter]: curr + open + close }));
      return;
    }
    const start = el.selectionStart ?? curr.length;
    const end = el.selectionEnd ?? curr.length;
    const sel = curr.slice(start, end);
    const before = curr.slice(0, start);
    const after = curr.slice(end);
    const next = before + open + sel + close + after;
    setOpts(prev => ({ ...prev, [letter]: next }));
    setTimeout(() => {
      const target = optRefs[letter]?.current;
      if (target) {
        const pos = start + open.length + sel.length + close.length;
        target.selectionStart = pos;
        target.selectionEnd = pos;
        target.focus();
      }
    }, 0);
  };

  const removeInlineImagesFromText = () => {
    // Hapus semua token markdown gambar: ![alt](url)
    const next = text.replace(/!\[[^\]]*\]\([^\)]+\)/g, '').replace(/\s{2,}/g,' ');
    setText(next.trim());
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setFormErrors({});
    setErrorMessage("");
    const wasEditing = !!editingQuestion;
    // Konfirmasi sebelum menyimpan saat menambah question (bukan edit)
    if (!editingQuestion) {
      const confirm = await Swal.fire({
        title: 'Konfirmasi',
        text: `Kunci Jawaban saat ini: ${key}. Lanjutkan menambah?`,
        icon: 'question',
        showCancelButton: true,
        confirmButtonText: 'OK',
        cancelButtonText: 'Batal',
      });
      if (!confirm.isConfirmed) return;
    }
    // Validasi pra-submit untuk menghindari 422 dari backend
    const errors = {};
    const isBlankOrDash = (v) => { const t = String(v || '').trim(); return t === '' || t === '-'; };
    const upperKey = String(key || '').trim().toUpperCase();
    if (!isEditingEssay) {
      if (isBlankOrDash(text)) errors.text = 'Teks pertanyaan wajib diisi';
      ['A','B','C','D'].forEach(letter => {
        if (isBlankOrDash(opts[letter])) errors[`option_${letter.toLowerCase()}`] = `Opsi ${letter} wajib diisi`;
      });
      if (!['A','B','C','D','E'].includes(upperKey)) {
        errors.key_answer = 'Pilih kunci jawaban A–E';
      } else if (isBlankOrDash(opts[upperKey])) {
        errors.key_answer = `Kunci ${upperKey} tidak boleh kosong`;
      }
    } else {
      if (isBlankOrDash(text)) errors.text = 'Teks pertanyaan wajib diisi';
      const essayKey = String(essayAnswer || '').trim();
      if (!essayKey) {
        errors.key_answer = 'Kunci essay wajib diisi';
      }
    }
    if (!subjectId && !editingQuestion?.subject_id) {
      errors.subject_id = 'Mata pelajaran tidak terdeteksi. Buka dari menu Mata Pelajaran.';
    }
    if (Object.keys(errors).length > 0) {
      setFormErrors(errors);
      setErrorMessage('Periksa kembali isian form.');
      return;
    }
    setSubmitting(true);
    try {
      const fd = new FormData();
      const valOrSanitized = (s) => (typeof s === 'string' && looksLatex(s)) ? sanitizeLatex(s) : String(s || '');
      fd.append('text', valOrSanitized(text));
      fd.append('option_a', valOrSanitized(opts.A));
      fd.append('option_b', valOrSanitized(opts.B));
      fd.append('option_c', valOrSanitized(opts.C));
      fd.append('option_d', valOrSanitized(opts.D));
      fd.append('option_e', valOrSanitized(opts.E));
      fd.append('key_answer', isEditingEssay ? valOrSanitized(essayAnswer) : String(key || 'A').trim().toUpperCase());
      // Pastikan selalu mengirim subject_id yang valid.
      const finalSubjectId = subjectId || editingQuestion?.subject_id;
      if (!finalSubjectId) {
        throw { response: { data: { message: 'Pilih mata pelajaran: akses dari menu Mata Pelajaran → Kelola Soal' } } };
      }
      fd.append('subject_id', String(finalSubjectId));
      let res;
      if (editingQuestion) {
        // Gunakan method override agar Laravel memproses multipart dengan boundary konsisten
        fd.append('_method', 'PUT');
        res = await api.post(`/questions/${editingQuestion.id}`, fd);
        const updated = normalizeQuestion(res.data);
        setList(prev => prev.map(q => q.id === editingQuestion.id ? updated : q));
      } else {
        // Biarkan axios mengatur boundary untuk multipart
        res = await api.post('/questions', fd);
        const added = normalizeQuestion(res.data);
        setList(prev => [added, ...prev]);
      }
      // SweetAlert keberhasilan setelah menyimpan
      await Swal.fire({
        title: 'Berhasil',
        text: editingQuestion ? 'Perubahan berhasil disimpan' : 'Pertanyaan berhasil ditambahkan',
        icon: 'success',
      });
      // reset
      setText("");
      setOpts({ A: "", B: "", C: "", D: "", E: "" });
      setKey("A");
      setEssayAnswer("");
      
      if (wasEditing) setShowForm(false);
      setEditingQuestion(null);
      setFormErrors({});
      setErrorMessage("");
    } catch (err) {
      const status = err?.response?.status;
      const data = err?.response?.data;
      if (status === 422 && data?.errors) {
        setFormErrors(data.errors || {});
        setErrorMessage(data?.message || 'Periksa kembali isian form.');
      } else {
        setErrorMessage(data?.message || 'Gagal menyimpan pertanyaan');
      }
    } finally {
      setSubmitting(false);
    }
  };

  return (
    <ErrorBoundaryQuestions>
    <div className="container mt-4">
      
      <h3 className="mb-1">{subjectName ? `Soal • ${subjectName}` : 'Questions'}</h3>
      {subjectName && (
        <div className="text-muted mb-3">Mata Pelajaran: {subjectName}</div>
      )}

      <div className="d-flex justify-content-between align-items-center mb-3 gap-2 flex-wrap">
        <div className="d-flex gap-2 align-items-center">
          <div className="mb-3">
            <button className="btn btn-success" onClick={() => { setShowForm(prev => !prev); setFormErrors({}); setErrorMessage(""); }}>
              {showForm ? (editingQuestion ? 'Tutup Edit' : 'Tutup Form') : (editingQuestion ? 'Edit Soal' : 'Tambah Soal')}
            </button>
          </div>

          <div className="mb-3">
            <button className="btn btn-warning d-flex align-items-center" onClick={() => navigate('/subjects')}>
              <FaArrowLeft className="me-2" /> Kembali
            </button>
          </div>
          <div className="mb-3">
            <button className="btn btn-outline-secondary d-flex align-items-center" onClick={handlePrint} title="Print daftar soal">
              <FaPrint className="me-2" /> Print
            </button>
          </div>
          <div className="mb-3">
            <button className="btn btn-primary d-flex align-items-center" onClick={handleExportWord} title="Export Word (.doc)">
              <FaFileWord className="me-2" /> Export Word
            </button>
          </div>
          
        </div>
      </div>

      {showForm && (
      <>
      <div ref={formTopRef} />
      <form onSubmit={handleSubmit} className="mb-4">
        {errorMessage && (
          <div className="alert alert-danger" role="alert">{errorMessage}</div>
        )}
        {/* Section: Soal */}
        <div className="card mb-3">
          <div className="card-header">Soal #{currentNumber}</div>
          <div className="card-body">
            <div className="d-flex justify-content-between align-items-center">
              <label className="form-label mb-0">Teks Pertanyaan Pada Soal !</label>
              <div className="form-check">
                <input className="form-check-input" type="checkbox" id="arabicQuestion" checked={arabicQuestion} onChange={e => setArabicQuestion(e.target.checked)} />
                <label className="form-check-label" htmlFor="arabicQuestion">Mode Arab</label>
              </div>
            </div>
            <RichTextEditor
              ref={richTextRef}
              value={text}
              onChange={setText}
              className={`form-control ${formErrors.text ? 'is-invalid' : ''}`}
              style={arabicQuestion ? { minHeight: 150, textAlign: 'right', fontFamily: 'Noto Naskh Arabic, Amiri, serif' } : { minHeight: 150 }}
              placeholder="Ketik pertanyaan di sini. Pilih teks lalu klik Bold/Italic/Underline."
              dir={arabicQuestion ? 'rtl' : 'auto'}
              onMathSelect={(payload) => {
                // Buka panel MathLive dan isi dengan LaTeX dari rumus yang diklik
                setShowMathLive(true);
                setEditingActiveLatex(true);
                setTimeout(() => {
                  try {
                    const mf = mathLiveFieldRef.current;
                    if (!mf) return;
                    const latex = (payload && typeof payload === 'object' && 'latex' in payload) ? String(payload.latex || '') : '';
                    if (latex) {
                      if (typeof mf.setValue === 'function') mf.setValue(latex, 'latex');
                    }
                    if (typeof mf.focus === 'function') mf.focus();
                  } catch(_) {}
                }, 0);
              }}
            />
            {formErrors.text && (
              <div className="invalid-feedback">
                {Array.isArray(formErrors.text) ? formErrors.text.join(' ') : String(formErrors.text)}
              </div>
            )}
            <div className="mt-2 p-2 border rounded bg-light">
              <div className="d-flex align-items-center gap-2 flex-wrap">
                <span className="text-muted">Format:</span>
                <FormattingToolbar onWrap={formatSoal} onInsert={insertToSoal} />
              </div>
              <div className="d-flex align-items-center gap-2 flex-wrap mt-2">
                <span className="text-muted">Insert:</span>
                <button type="button" className="btn btn-outline-primary btn-sm rounded-pill" onClick={insertEquation} title="Tampilkan panel Equation">∑ Equation</button>
                <button type="button" className="btn btn-outline-primary btn-sm rounded-pill" onClick={choosePictureForQuestion} title="Pilih & sisipkan gambar">🖼️ Picture</button>
                <span className="text-muted ms-2">Symbols:</span>
                <div className="d-flex align-items-center flex-wrap gap-1" style={{ maxWidth:'100%', overflowX:'auto' }}>
                  {LATEX_SYMBOLS.map((s) => (
                    <button
                      key={s.code}
                      type="button"
                      className="btn btn-outline-secondary btn-sm rounded-pill"
                      title={s.code}
                      onClick={() => insertLatexSymbol(s.code)}
                    >{s.label}</button>
                  ))}
                </div>
              </div>
            </div>
            {showMathLive && (
              <div className="mt-2">
                <small className="text-muted">Equation (MathLive):</small>
                <div ref={mathLiveContainerRef} className="mt-1" />
                <div className="mt-2 d-flex gap-2 align-items-center">
                  <button type="button" className="btn btn-primary btn-sm" onClick={insertFromMathLive}>Sisipkan</button>
                  <button type="button" className="btn btn-outline-secondary btn-sm" onClick={() => setShowMathLive(false)}>Tutup</button>
                  {mathLiveError && (
                    <span className="text-danger small">{mathLiveError}</span>
                  )}
                </div>
              </div>
            )}
            <div className="mt-2">
              <small className="text-muted">Preview Soal:</small>
              <div className="p-2 border rounded bg-light" style={arabicQuestion ? { whiteSpace: 'pre-wrap', textAlign: 'right', fontFamily: 'Noto Naskh Arabic, Amiri, serif', direction: 'rtl' } : { whiteSpace: 'pre-wrap' }}>
                <RenderMixed str={text} opts={{ imgStyle: PREVIEW_IMG_STYLE }} />
              </div>
            </div>
          </div>
        </div>

        {/* Section: Opsi Jawaban */}
        {!isEditingEssay && (
        <div className="card mb-3">
          <div className="card-header">Opsi Jawaban</div>
          <div className="card-body">
            <div className="row">
              {(['A','B','C','D','E']).map(letter => {
                const errorKey = `option_${letter.toLowerCase()}`;
                const hasError = !!formErrors[errorKey];
                return (
                  <div key={letter} className="col-md-6 mb-3">
                    <div className="p-2 border rounded">
                      <label className="form-label d-flex align-items-center justify-content-between">
                        <span className="d-flex align-items-center gap-2">
                          <span className={`badge ${optionBadgeClasses[letter]}`}>{letter}</span>
                          <span style={{ color: optionColors[letter], fontWeight: 600 }}>Opsi {letter}</span>
                        </span>
                      </label>
                      <RichTextEditor
                        ref={richOptRefs[letter]}
                        value={opts[letter]}
                        onChange={(v) => setOpts({ ...opts, [letter]: v })}
                        className={`form-control ${hasError ? 'is-invalid' : ''}`}
                        style={arabicOption[letter] ? { minHeight: 60, textAlign: 'right', fontFamily: 'Noto Naskh Arabic, Amiri, serif' } : { minHeight: 60 }}
                        placeholder={letter === 'A' ? 'Contoh: الإجابة pertama' : `Opsi ${letter}`}
                        dir={arabicOption[letter] ? 'rtl' : 'auto'}
                        onMathSelect={(payload) => {
                          // Buka panel MathLive opsi dan isi latex dari rumus yang diklik
                          setMathLiveOptTarget(letter);
                          setEditingActiveOptionLatex(prev => ({ ...prev, [letter]: true }));
                          setTimeout(() => {
                            try {
                              const mf = mathLiveOptionFieldRef.current;
                              if (!mf) return;
                              const latex = (payload && typeof payload === 'object' && 'latex' in payload) ? String(payload.latex || '') : '';
                              if (latex) {
                                if (typeof mf.setValue === 'function') mf.setValue(latex, 'latex');
                              }
                              if (typeof mf.focus === 'function') mf.focus();
                            } catch(_) {}
                          }, 0);
                        }}
                      />
                      {hasError && (
                        <div className="invalid-feedback">
                          {Array.isArray(formErrors[errorKey]) ? formErrors[errorKey].join(' ') : String(formErrors[errorKey])}
                        </div>
                      )}
                    <div className="form-check mt-2">
                      <input className="form-check-input" type="checkbox" id={`arabic-${letter}`} checked={!!arabicOption[letter]} onChange={(e) => setArabicOption(prev => ({ ...prev, [letter]: e.target.checked }))} />
                      <label className="form-check-label" htmlFor={`arabic-${letter}`}>Mode Arab untuk Opsi {letter}</label>
                    </div>
                    <div className="mt-2 d-flex flex-wrap gap-2 align-items-center">
                      <span className="text-muted">Format:</span>
                      <FormattingToolbar onWrap={(tag) => {
                        const ref = richOptRefs[letter]?.current;
                        if (ref && typeof ref.applyFormat === 'function') ref.applyFormat(tag);
                        else wrapOptionSelection(letter, tag);
                      }} onInsert={(snippet) => {
                        const ref = richOptRefs[letter]?.current;
                        if (ref && typeof ref.insertSnippet === 'function') ref.insertSnippet(snippet);
                        else insertAtOptionCursor(letter, snippet);
                      }} allowYoutube={false} />
                    </div>
                    <div className="mt-2 p-2 border rounded bg-light">
                      <div className="d-flex align-items-center gap-2 flex-wrap">
                        <span className="text-muted">Insert:</span>
                        <button type="button" className="btn btn-outline-primary btn-sm rounded-pill" onClick={() => toggleOptionEquation(letter)} title={`Tampilkan panel Equation untuk Opsi ${letter}`}>∑ Equation</button>
                        <button type="button" className="btn btn-outline-primary btn-sm rounded-pill" onClick={() => choosePictureForOption(letter)} title={`Pilih & sisipkan gambar untuk Opsi ${letter}`}>🖼️ Picture</button>
                        <span className="text-muted ms-2">Symbols:</span>
                        <div className="d-flex align-items-center flex-wrap gap-1" style={{ maxWidth:'100%', overflowX:'auto' }}>
                          {LATEX_SYMBOLS.map((s) => (
                            <button
                              key={`${letter}-${s.code}`}
                              type="button"
                              className="btn btn-outline-secondary btn-sm rounded-pill"
                              title={`${s.code} (Opsi ${letter})`}
                              onClick={() => insertOptionLatexSymbol(letter, s.code)}
                            >{s.label}</button>
                          ))}
                        </div>
                      </div>
                      {mathLiveOptTarget === letter && (
                        <div className="mt-2">
                          <small className="text-muted">Equation (MathLive) — Opsi {letter}:</small>
                          <div ref={mathLiveOptionContainerRefs[letter]} className="mt-1" />
                          <div className="mt-2 d-flex gap-2 align-items-center">
                            <button type="button" className="btn btn-primary btn-sm" onClick={insertFromMathLiveOption}>Sisipkan</button>
                            <button type="button" className="btn btn-outline-secondary btn-sm" onClick={() => setMathLiveOptTarget(null)}>Tutup</button>
                            {mathLiveOptError && (
                              <span className="text-danger small">{mathLiveOptError}</span>
                            )}
                          </div>
                        </div>
                      )}
                    </div>
                    <div className="mt-2 d-flex gap-2">
                      <button type="button" className="btn btn-outline-danger btn-sm" onClick={() => removeInlineImagesFromOption(letter)}>
                        Hapus Gambar dari Opsi {letter}
                      </button>
                    </div>
                    <div className="mt-1">
                      <small className="text-muted">Preview Opsi:</small>
                      <div className="p-2 border rounded bg-light" style={arabicOption[letter] ? { whiteSpace: 'pre-wrap', textAlign: 'right', fontFamily: 'Noto Naskh Arabic, Amiri, serif', direction: 'rtl' } : { whiteSpace: 'pre-wrap' }}>
                        {opts[letter]
                          ? <RenderMixed str={opts[letter]} opts={{ imgStyle: OPTION_PREVIEW_IMG_STYLE, disableYoutube: true }} />
                          : <span className="text-muted">Masukkan teks</span>}
                      </div>
                    </div>
                  </div>
                </div>
              );})}
            </div>
            <small className="text-muted">Opsi E opsional; jika kosong tidak akan ditampilkan di daftar.</small>
          </div>
        </div>
        )}

        {/* Section: Kunci Jawaban */}
        {!isEditingEssay && (
        <div className="card mb-3">
          <div className="card-header">Kunci Jawaban</div>
          <div className="card-body">
            <div className="d-flex align-items-center gap-2">
              <select className={`form-select ${formErrors.key_answer ? 'is-invalid' : ''}`} style={{maxWidth:'120px'}} value={key} onChange={e => setKey(e.target.value)}>
                {['A','B','C','D','E'].map(k => (
                  <option key={k} value={k}>{k}</option>
                ))}
              </select>
              <span className={`badge ${optionBadgeClasses[key]}`}>Kunci saat ini: {key}</span>
              {formErrors.key_answer && (
                <div className="invalid-feedback d-block">
                  {Array.isArray(formErrors.key_answer) ? formErrors.key_answer.join(' ') : String(formErrors.key_answer)}
                </div>
              )}
            </div>
          </div>
        </div>
         )}

        {isEditingEssay && (
        <div className="card mb-3">
          <div className="card-header">Kunci Essay</div>
          <div className="card-body">
            <RichTextEditor
              value={essayAnswer}
              onChange={(v) => setEssayAnswer(v)}
              className={`form-control ${formErrors.key_answer ? 'is-invalid' : ''}`}
              placeholder="Tuliskan kunci jawaban essay di sini"
            />
            {formErrors.key_answer && (
              <div className="invalid-feedback d-block">
                {Array.isArray(formErrors.key_answer) ? formErrors.key_answer.join(' ') : String(formErrors.key_answer)}
              </div>
            )}
          </div>
        </div>
        )}

        {/* Section: Gambar Lampiran dihapus sesuai permintaan */}

        {/* Section: Preview Komprehensif */}
        <div className="card mb-3">
          <div className="card-header">Preview Soal</div>
          <div className="card-body">
            <div className="mb-2" style={{ whiteSpace: 'pre-wrap' }}><RenderMixed str={text} opts={{ imgStyle: PREVIEW_IMG_STYLE }} /></div>
            <ul className="list-group list-group-flush">
              <li className="list-group-item">
                <span>
                  <span className={`badge ${optionBadgeClasses.A} me-2`}>A</span>
                  <div className="mt-1 p-2 border rounded bg-light" style={{ whiteSpace: 'pre-wrap' }}>
                    {opts.A ? <RenderMixed str={opts.A} opts={{ imgStyle: PREVIEW_IMG_STYLE }} /> : <span className="text-muted">-</span>}
                  </div>
                </span>
              </li>
              <li className="list-group-item">
                <span>
                  <span className={`badge ${optionBadgeClasses.B} me-2`}>B</span>
                  <div className="mt-1 p-2 border rounded bg-light" style={{ whiteSpace: 'pre-wrap' }}>
                    {opts.B ? <RenderMixed str={opts.B} opts={{ imgStyle: PREVIEW_IMG_STYLE }} /> : <span className="text-muted">-</span>}
                  </div>
                </span>
              </li>
              <li className="list-group-item">
                <span>
                  <span className={`badge ${optionBadgeClasses.C} me-2`}>C</span>
                  <div className="mt-1 p-2 border rounded bg-light" style={{ whiteSpace: 'pre-wrap' }}>
                    {opts.C ? <RenderMixed str={opts.C} opts={{ imgStyle: PREVIEW_IMG_STYLE }} /> : <span className="text-muted">-</span>}
                  </div>
                </span>
              </li>
              <li className="list-group-item">
                <span>
                  <span className={`badge ${optionBadgeClasses.D} me-2`}>D</span>
                  <div className="mt-1 p-2 border rounded bg-light" style={{ whiteSpace: 'pre-wrap' }}>
                    {opts.D ? <RenderMixed str={opts.D} opts={{ imgStyle: PREVIEW_IMG_STYLE }} /> : <span className="text-muted">-</span>}
                  </div>
                </span>
              </li>
              {(opts.E && opts.E.trim() !== '') && (
                <li className="list-group-item">
                  <span>
                    <span className={`badge ${optionBadgeClasses.E} me-2`}>E</span>
                    <div className="mt-1 p-2 border rounded bg-light" style={{ whiteSpace: 'pre-wrap' }}>
                      <RenderMixed str={opts.E} opts={{ imgStyle: PREVIEW_IMG_STYLE }} />
                    </div>
                  </span>
                </li>
              )}
            </ul>
          </div>
        </div>

        <div className="d-flex gap-2">
          <button className="btn btn-primary" type="submit" disabled={submitting}>
            {submitting ? 'Menyimpan...' : (editingQuestion ? 'Simpan Perubahan' : 'Tambah Question')}
          </button>
          {editingQuestion && (
            <button type="button" className="btn btn-outline-secondary" onClick={() => { setEditingQuestion(null); setShowForm(false); }}>
              Batal Edit
            </button>
          )}
        </div>
      </form>
      </>
      )}

      {!showForm && (
        <>
          {/* Pisahkan daftar menjadi pilihan jamak vs essay */}
          {(() => {
            const isEssay = (q) => {
              if (!q || typeof q !== 'object') return false;
              const blank = (v) => {
                const t = String(v || '').trim();
                return t === '' || t === '-';
              };
              const bdBlank = blank(q?.option_b) && blank(q?.option_c) && blank(q?.option_d);
              const allBlank = ['option_a','option_b','option_c','option_d','option_e'].every(k => blank(q[k]));
              return bdBlank || allBlank;
            };
            const mcqs = Array.isArray(list) ? list.filter(q => !isEssay(q)) : [];
            const essays = Array.isArray(list) ? list.filter(q => isEssay(q)) : [];

            // Jika tidak ada satupun, tampilkan kosong
            if ((mcqs.length + essays.length) === 0) {
              return <div className="text-muted">Belum ada pertanyaan.</div>;
            }

            return (
              <>
                {/* Bagian Pilihan Jamak */}
                {mcqs.length > 0 && (
                  <>
                    <h5 className="mb-2">Daftar Pertanyaan (Pilihan Jamak)</h5>
                    <div ref={listRef}>
                      {mcqs.map((q, idx) => (
                        <div key={q.id} className="card mb-3 shadow-sm">
                          <div className="card-header d-flex justify-content-between align-items-center">
                            <strong>Soal #{idx + 1}</strong>
                            <span className={`badge ${optionBadgeClasses[q.key_answer] || 'bg-secondary'}`}>Kunci: {q.key_answer}</span>
                          </div>
                          <div className="card-body">
                <div className="mb-2" style={{ whiteSpace: 'pre-wrap' }}>
                  <RenderMixed str={q.text} opts={{ imgStyle: LIST_INLINE_IMG_STYLE }} />
                </div>
                            <div className="row">
                              <div className="col-md-8">
                                <ul className="list-group list-group-flush">
                                  <li className="list-group-item p-2">
                                    <span className={`badge ${optionBadgeClasses.A} me-2`}>A</span>
                                    <RenderMixed str={q.option_a} opts={{ imgStyle: LIST_INLINE_IMG_STYLE, disableYoutube: true }} />
                                  </li>
                                  <li className="list-group-item p-2"><span className={`badge ${optionBadgeClasses.B} me-2`}>B</span> <RenderMixed str={q.option_b} opts={{ imgStyle: LIST_INLINE_IMG_STYLE, disableYoutube: true }} /></li>
                                  <li className="list-group-item p-2"><span className={`badge ${optionBadgeClasses.C} me-2`}>C</span> <RenderMixed str={q.option_c} opts={{ imgStyle: LIST_INLINE_IMG_STYLE, disableYoutube: true }} /></li>
                                  <li className="list-group-item p-2"><span className={`badge ${optionBadgeClasses.D} me-2`}>D</span> <RenderMixed str={q.option_d} opts={{ imgStyle: LIST_INLINE_IMG_STYLE, disableYoutube: true }} /></li>
                                  {(q.option_e && q.option_e.trim() !== '') && (
                                      <li className="list-group-item p-2"><span className={`badge ${optionBadgeClasses.E} me-2`}>E</span> <RenderMixed str={q.option_e} opts={{ imgStyle: LIST_INLINE_IMG_STYLE, disableYoutube: true }} /></li>
                                    )}
                                </ul>
                              </div>
                              <div className="col-md-4">
                                {q.image_path && (
                                  <img
                                    src={resolveImageUrl(q.image_path)}
                                    alt="question"
                                    style={{ width: 140, height: 140, objectFit: 'cover', display: 'block', margin: '0 auto', borderRadius: 6 }}
                                    onError={(e) => {
                                      const raw = String(q.image_path || '').trim();
                                      const clean = raw.replace(/^\/+/, '');
                                      const base = String(BACKEND_BASE || '').replace(/\/+$/, '');
                                      let alt = null;
                                      if (/^images\//.test(clean)) alt = `${base}/storage/${clean}`;
                                      else if (/^storage\//.test(clean)) alt = `${base}/${clean.replace(/^storage\//,'')}`;
                                      if (alt) { e.currentTarget.onerror = null; e.currentTarget.src = alt; }
                                    }}
                                  />
                                )}
                              </div>
                            </div>
                          </div>
                          <div className="card-footer d-flex gap-2">
                            <button className="btn btn-sm btn-primary" onClick={() => {
                              // Buka form edit lokal untuk soal pilihan jamak
                              setEditingQuestion(q);
                              setShowForm(true);
                              setFormErrors({});
                              setErrorMessage("");
                              const sanitizeIf = (v) => (typeof v === 'string' && looksLatex(v)) ? sanitizeLatex(v) : String(v || '');
                              setText(sanitizeIf(q.text));
                              setOpts({
                                A: sanitizeIf(q.option_a),
                                B: sanitizeIf(q.option_b),
                                C: sanitizeIf(q.option_c),
                                D: sanitizeIf(q.option_d),
                                E: sanitizeIf(q.option_e),
                              });
                              const nextKey = String(q.key_answer || 'A').trim().toUpperCase();
                              setKey(nextKey === '' ? 'A' : nextKey);
                              const blank = (v) => { const t = String(v || '').trim(); return t === '' || t === '-'; };
                              const isEssayServer = (!['A','B','C','D','E'].includes(nextKey)) || (blank(q.option_b) && blank(q.option_c) && blank(q.option_d));
                              setEssayAnswer(isEssayServer ? sanitizeIf(q.key_answer || q.option_a || '') : '');
                            }}>Edit</button>
                            <button className="btn btn-sm btn-danger" onClick={async () => {
                              try {
                                const resConfirm = await Swal.fire({
                                  title: 'Hapus Soal Ini?',
                                  text: 'Tindakan ini tidak dapat dibatalkan.',
                                  icon: 'warning',
                                  showCancelButton: true,
                                  confirmButtonColor: '#d33',
                                  cancelButtonColor: '#6c757d',
                                  confirmButtonText: 'Ya, Hapus',
                                  cancelButtonText: 'Batal'
                                });
                                if (!resConfirm.isConfirmed) return;

                                await api.delete(`/questions/${q.id}`);
                                setList(prev => prev.filter(item => item.id !== q.id));

                                await Swal.fire({
                                  title: 'Terhapus',
                                  text: 'Soal berhasil dihapus.',
                                  icon: 'success'
                                });
                              } catch (err) {
                                await Swal.fire({
                                  title: 'Gagal',
                                  text: err.response?.data?.message || 'Gagal menghapus pertanyaan',
                                  icon: 'error'
                                });
                              }
                            }}>Hapus</button>
                          </div>
                        </div>
                      ))}
                    </div>
                  </>
                )}

                {/* Bagian Essay disembunyikan sesuai permintaan: tidak tampil jika opsi A–E = '-' */}
              </>
            );
          })()}
        </>
      )}
    </div>
    </ErrorBoundaryQuestions>
  );
}



// Util: potong string menjadi segmen teks vs token LaTeX berdasarkan berbagai delimiter
function splitLatexTokens(str){
  if (!str || typeof str !== 'string') return [{ type: 'text', value: '' }];
  const regex = /(\$\$[\s\S]*?\$\$|\\\[[\s\S]*?\\\]|\\\([\s\S]*?\\\)|\$[^$]*\$|\[tex\][\s\S]*?\[\/tex\]|\[math\][\s\S]*?\[\/math\]|<tex>[\s\S]*?<\/tex>|<katex>[\s\S]*?<\/katex>)/g;
  const segments = [];
  let lastIndex = 0; let m;
  while ((m = regex.exec(str))) {
    const before = str.slice(lastIndex, m.index);
    if (before) segments.push({ type: 'text', value: before });
    const token = m[0];
    if (token.startsWith('$$') || token.startsWith('\\[')) {
      segments.push({ type: 'latex_block', value: token });
    } else {
      segments.push({ type: 'latex_inline', value: token });
    }
    lastIndex = m.index + token.length;
  }
  const tail = str.slice(lastIndex);
  if (tail) segments.push({ type: 'text', value: tail });
  return segments;
}

function stripLatexDelimiters(token){
  if (!token) return '';
  if (token.startsWith('$$')) return token.replace(/^\$\$/, '').replace(/\$\$$/, '');
  if (token.startsWith('$')) return token.replace(/^\$/, '').replace(/\$$/, '');
  if (token.startsWith('\\[')) return token.replace(/^\\\[/, '').replace(/\\\]$/, '');
  if (token.startsWith('\\(')) return token.replace(/^\\\(/, '').replace(/\\\)$/, '');
  if (token.startsWith('[tex]')) return token.replace(/^\[tex\]/, '').replace(/\[\/tex\]$/, '');
  if (token.startsWith('[math]')) return token.replace(/^\[math\]/, '').replace(/\[\/math\]$/, '');
  if (token.startsWith('<tex>')) return token.replace(/^<tex>/, '').replace(/<\/tex>$/, '');
  if (token.startsWith('<katex>')) return token.replace(/^<katex>/, '').replace(/<\/katex>$/, '');
  return token;
}

// Tambahan: bungkus otomatis LaTeX jika tidak ada delimiter
const autoWrapLatexIfMissing = (s) => {
  const str = String(s || '');
  const hasDelim = /(\$\$[\s\S]*?\$\$|\\\[[\s\S]*?\\\]|\\\([\s\S]*?\\\)|\$[^$]*\$|\[tex\][\s\S]*?\[\/tex\]|\[math\][\s\S]*?\[\/math\]|<tex>[\s\S]*?<\/tex>|<katex>[\s\S]*?<\/katex>)/.test(str);
  const looksLatex = /\\(dfrac|frac|sqrt|cos|sin|tan|left|right|int|sum|prod|partial|nabla|pi|theta|alpha|beta|gamma|lim|bmatrix|pmatrix|begin|end|cdot|times|div|le|ge|neq|approx|equiv|infty|angle)/.test(str);
  return (!hasDelim && looksLatex) ? `\\(${str}\\)` : str;
};

function renderTextOrLatex(s, key){
  if (!s || typeof s !== 'string') {
    return <span key={key} />;
  }
  // decode entity HTML khusus untuk token LaTeX
  const decodeHtmlEntities = (input) => {
    return String(input || '')
      .replace(/&amp;/g, '&')
      .replace(/&lt;/g, '<')
      .replace(/&gt;/g, '>')
      .replace(/&quot;/g, '"')
      .replace(/&#039;/g, "'")
      .replace(/&#39;/g, "'");
  };
  // bersihkan zero-width/combining + perbaiki aksen teks di math mode
  const cleanLatex = (input) => sanitizeLatex(input);

  const nodes = [];
  const inputLatexAware = autoWrapLatexIfMissing(s);
  const segments = splitLatexTokens(inputLatexAware);
  let idx = 0;
  for (const seg of segments) {
    if (seg.type === 'text') {
      nodes.push(
        <span key={`${key}-text-${idx++}`} style={{ whiteSpace: 'inherit' }} dangerouslySetInnerHTML={{ __html: seg.value }} />
      );
    } else if (seg.type === 'latex_inline') {
      const math = cleanLatex(decodeHtmlEntities(stripLatexDelimiters(seg.value)));
      nodes.push(
        <span key={`${key}-latex-${idx++}`} style={{ direction: 'ltr', unicodeBidi: 'isolate' }}>
          <InlineMath errorColor="#cc0000" throwOnError={false}>{math}</InlineMath>
        </span>
      );
    } else if (seg.type === 'latex_block') {
      const math = cleanLatex(decodeHtmlEntities(stripLatexDelimiters(seg.value)));
      nodes.push(
        <div key={`${key}-block-${idx++}`} style={{ direction: 'ltr', unicodeBidi: 'isolate' }}>
          <BlockMath errorColor="#cc0000" throwOnError={false}>{math}</BlockMath>
        </div>
      );
    }
  }
  return <>{nodes}</>;
}

function renderMathOnly(s, key = 'math-only'){
  if (!s || typeof s !== 'string') {
    return <span key={key} />;
  }
  const decodeHtmlEntities = (input) => {
    return String(input || '')
      .replace(/&amp;/g, '&')
      .replace(/&lt;/g, '<')
      .replace(/&gt;/g, '>')
      .replace(/&quot;/g, '"')
      .replace(/&#039;/g, "'")
      .replace(/&#39;/g, "'");
  };
  const cleanLatex = (input) => sanitizeLatex(input);
  const wrapped = autoWrapLatexIfMissing(s);
  const isBlock = /(\$\$[\s\S]*\$\$|\\\[[\s\S]*\\\])/m.test(wrapped);
  const inner = stripLatexDelimiters(wrapped);
  const math = cleanLatex(decodeHtmlEntities(inner));
  const wrapperStyle = { direction: 'ltr', unicodeBidi: 'isolate' };
  return isBlock
    ? <div key={`${key}-block`} style={wrapperStyle}><BlockMath errorColor="#cc0000" throwOnError={false}>{math}</BlockMath></div>
    : <span key={`${key}-inline`} style={wrapperStyle}><InlineMath errorColor="#cc0000" throwOnError={false}>{math}</InlineMath></span>;
}

// Util: konversi style object ke string inline
const styleObjectToString = (obj) => {
  if (!obj || typeof obj !== 'object') return '';
  return Object.entries(obj).map(([key, val]) => {
    const cssKey = key.replace(/[A-Z]/g, (m) => '-' + m.toLowerCase());
    const cssVal = typeof val === 'number' ? `${val}px` : String(val);
    return `${cssKey}: ${cssVal};`;
  }).join(' ');
};

// Parser BBCode → HTML untuk [b], [i], [u], [s], [br], [img], align, table, video
const bbcodeToHtml = (input, opts = {}) => {
  if (!input || typeof input !== 'string') return '';
  let s = input;
  // Teks format
  s = s.replace(/\[b\](.*?)\[\/b\]/gis, '<strong>$1</strong>');
  s = s.replace(/\[i\](.*?)\[\/i\]/gis, '<em>$1</em>');
  s = s.replace(/\[u\](.*?)\[\/u\]/gis, '<u>$1</u>');
  s = s.replace(/\[s\](.*?)\[\/s\]/gis, '<s>$1</s>');
  s = s.replace(/\[br\s*\/?\]/gi, '<br/>');
  // Alignment
  s = s.replace(/\[center\](.*?)\[\/center\]/gis, (m, p1) => `<div style="text-align:center;">${p1}</div>`);
  s = s.replace(/\[left\](.*?)\[\/left\]/gis, (m, p1) => `<div style="text-align:left;">${p1}</div>`);
  s = s.replace(/\[right\](.*?)\[\/right\]/gis, (m, p1) => `<div style="text-align:right;">${p1}</div>`);
  s = s.replace(/\[align=(left|center|right)\](.*?)\[\/align\]/gis, (m, align, body) => `<div style="text-align:${align};">${body}</div>`);
  // Gambar
  s = s.replace(/\[img\](.*?)\[\/img\]/gis, (m, p1) => {
    const raw = String(p1 || '').trim();
    const url = resolveImageUrl(raw);
    const style = styleObjectToString(opts.imgStyle || { maxWidth: '100%', maxHeight: '240px', objectFit: 'contain', borderRadius: 6, margin: '6px 0' });
    const safeUrl = String(url || '').replace(/\"/g, '&quot;');
    const base = String(BACKEND_BASE || '').replace(/\/+$/, '');
    const clean = raw.replace(/^\/+/, '');
    let alt = '';
    if (/^images\//.test(clean)) alt = `${base}/storage/${clean}`;
    else if (/^storage\//.test(clean)) alt = `${base}/${clean.replace(/^storage\//,'')}`;
    const onErrAttr = alt ? ` onerror="if(this._trFallback){return;}this._trFallback=1;this.src='${alt}'"` : '';
    return `<img src="${safeUrl}" alt="img" style="${style}"${onErrAttr} />`;
  });
  // YouTube
  if (!opts.disableYoutube) {
    const extractYouTubeId = (raw) => {
      const s2 = String(raw || '').trim();
      if (/^[A-Za-z0-9_-]{11}$/.test(s2)) return s2;
      // Dukung partial path: /watch?v=ID atau watch?v=ID
      const mWatch = s2.match(/^\/?watch\?v=([A-Za-z0-9_-]{11})/i);
      if (mWatch) return mWatch[1];
      // Dukung shorts/ dan embed/
      const mShorts = s2.match(/shorts\/([A-Za-z0-9_-]{11})/i);
      if (mShorts) return mShorts[1];
      const mEmbed = s2.match(/embed\/([A-Za-z0-9_-]{11})/i);
      if (mEmbed) return mEmbed[1];
      try {
        const u = new URL(s2);
        if (u.hostname.includes('youtu.be')) {
          const id = (u.pathname || '').replace(/^\//, '').split('/')[0];
          return id || null;
        }
        if (u.hostname.includes('youtube.com')) {
          const id = u.searchParams.get('v');
          if (id) return id;
          // Path lain: /shorts/ID
          const p = (u.pathname || '').toLowerCase();
          const m2 = p.match(/shorts\/([A-Za-z0-9_-]{11})/);
          if (m2) return m2[1];
        }
      } catch (_) {}
      return null;
    };
    s = s.replace(/\[youtube\](.*?)\[\/youtube\]/gis, (m, p1) => {
      const id = extractYouTubeId(p1);
      if (!id) return p1;
      const src = `https://www.youtube.com/embed/${id}`;
      return `<div class="ratio ratio-16x9" style="max-width: 720px; margin: 6px auto;"><iframe src="${src}" title="YouTube" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" allowfullscreen style="width:100%; height:100%;"></iframe></div>`;
    });
  } else {
    // Hapus tag YouTube saat dinonaktifkan (contoh: opsi A–E)
    s = s.replace(/\[youtube\](.*?)\[\/youtube\]/gis, '');
  }
  // Video
  const buildVideoEmbed = (raw) => {
    const style = 'max-width: 720px; width: 100%; margin: 6px auto;';
    try {
      const u = new URL(String(raw || '').trim());
      const host = u.hostname.toLowerCase();
      if (host.includes('youtu.be') || host.includes('youtube.com')) {
        const id = extractYouTubeId(raw);
        if (!id) return '';
        const src = `https://www.youtube.com/embed/${id}`;
        return `<div class="ratio ratio-16x9" style="${style}"><iframe src="${src}" title="YouTube" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" allowfullscreen style="width:100%; height:100%;"></iframe></div>`;
      }
      if (host.includes('vimeo.com')) {
        const pathParts = u.pathname.split('/').filter(Boolean);
        const id = pathParts[pathParts.length - 1];
        const src = id ? `https://player.vimeo.com/video/${id}` : u.toString();
        return `<div class="ratio ratio-16x9" style="${style}"><iframe src="${src}" title="Vimeo" frameborder="0" allow="autoplay; fullscreen; picture-in-picture" allowfullscreen style="width:100%; height:100%;"></iframe></div>`;
      }
      const safeUrl = u.toString().replace(/\"/g, '&quot;');
      const vStyle = 'max-width: 720px; width: 100%; max-height: 405px; border-radius: 6px;';
      return `<video src="${safeUrl}" controls style="${vStyle}"></video>`;
    } catch (_) {
      return '';
    }
  };
  s = s.replace(/\[video\](.*?)\[\/video\]/gis, (m, p1) => buildVideoEmbed(p1));
  // Tabel
  s = s.replace(/\[table\](.*?)\[\/table\]/gis, (m, body) => {
    let inner = body || '';
    inner = inner
      .replace(/\[th\](.*?)\[\/th\]/gis, (mm, cell) => `<th style="border:1px solid #dee2e6; padding:6px;">${cell}</th>`)
      .replace(/\[td\](.*?)\[\/td\]/gis, (mm, cell) => `<td style="border:1px solid #dee2e6; padding:6px;">${cell}</td>`)
      .replace(/\[tr\](.*?)\[\/tr\]/gis, (mm, row) => `<tr>${row}</tr>`);
    const tblStyle = 'border-collapse: collapse; width: 100%; margin: 6px 0;';
    return `<table style="${tblStyle}"><tbody>${inner}</tbody></table>`;
  });
  // Konversi newline agar preview mengikuti enter/linebreak
  s = s.replace(/\r?\n/g, '<br/>');
  return s;
};

// Komponen: RenderMixed — render HTML hasil bbcodeToHtml dan auto-render LaTeX
function RenderMixed({ str, opts = {} }){
  const containerRef = React.useRef(null);
  const imgStyle = opts.imgStyle || { maxWidth: '100%', maxHeight: '240px', objectFit: 'contain', borderRadius: 6, margin: '6px 0' };

  React.useEffect(() => {
    if (containerRef.current) {
      try {
        renderMathInElement(containerRef.current, {
          delimiters: [
            { left: '$$', right: '$$', display: true },
            { left: '\\[', right: '\\]', display: true },
            { left: '\\(', right: '\\)', display: false },
            { left: '$', right: '$', display: false },
          ],
          throwOnError: false,
        });
      } catch (err) {
        console.warn('Math render error:', err);
      }
    }
  }, [str]);

  if (!str) return <span className="text-muted">Masukkan teks untuk preview</span>;
  // Pramuat BBCode → HTML
  let processed = bbcodeToHtml(str, { imgStyle, disableYoutube: !!opts.disableYoutube });
  // Konversi Markdown gambar: ![alt](url) → <img src="..." alt="..." style="...">
  // Mendukung lebar via alt: contoh alt "img|w=80%" → style width:80%
  processed = processed.replace(/!\[([^\]]*)\]\(([^)]+)\)/g, (m, alt, url) => {
    const full = resolveImageUrl(String(url || '').trim());
    const altRaw = String(alt || '').trim();
    const altBase = altRaw.replace(/\|.*$/, '').trim() || 'img';
    // Ambil semua w=xx% dan gunakan yang terakhir untuk mengatasi duplikasi
    const widthMatches = altRaw.match(/\bw\s*=\s*(\d{1,3})\s*%/gi);
    let widthPct = null;
    if (widthMatches && widthMatches.length > 0) {
      const lastMatch = widthMatches[widthMatches.length - 1];
      const mW = lastMatch.match(/\bw\s*=\s*(\d{1,3})\s*%/i);
      widthPct = mW ? Math.max(5, Math.min(160, Number(mW[1]))) : null;
    }
    const baseStyleStr = styleObjectToString(imgStyle);
    const styleStr = widthPct ? `${baseStyleStr} width: ${widthPct}%;` : baseStyleStr;
    return `<img src="${full}" alt="${altBase}" style="${styleStr}">`;
  });

  return (
    <div
      ref={containerRef}
      className="preview-soal"
      style={{ whiteSpace: 'pre-wrap' }}
      dangerouslySetInnerHTML={{ __html: processed }}
    />
  );
}

// MathToolbar dihapus: Desmos/GeoGebra tidak didukung lagi

function FormattingToolbar({ onWrap, onInsert, allowTable = true, allowYoutube = true }){
  if (!onWrap) return null;
  const insert = (s) => onInsert && onInsert(s);
  const insertTable = async () => {
    const { value: formValues } = await Swal.fire({
      title: 'Sisipkan Tabel',
      html: `
        <div class="mb-2 text-start">
          <label class="form-label">Jumlah Kolom</label>
          <input id="tbl-cols" type="number" min="1" value="2" class="form-control" />
        </div>
        <div class="mb-2 text-start">
          <label class="form-label">Jumlah Baris</label>
          <input id="tbl-rows" type="number" min="1" value="2" class="form-control" />
        </div>
      `,
      focusConfirm: false,
      showCancelButton: true,
      confirmButtonText: 'Sisipkan',
      cancelButtonText: 'Batal',
      preConfirm: () => {
        const cols = parseInt(document.getElementById('tbl-cols').value, 10);
        const rows = parseInt(document.getElementById('tbl-rows').value, 10) || 2;
        if (!cols || cols < 1) {
          Swal.showValidationMessage('Kolom minimal 1');
          return false;
        }
        return { cols, rows };
      }
    });
    if (!formValues) return;
    const { cols, rows } = formValues;
    const header = Array.from({ length: cols }, (_, i) => `<th>Kolom ${i+1}</th>`).join('');
    const bodyRows = Array.from({ length: rows }, () => `<tr>${Array.from({ length: cols }, () => '<td>-</td>').join('')}</tr>`).join('');
    insert(`<table border="1" style="border-collapse:collapse"><tr>${header}</tr>${bodyRows}</table>`);
  };
  const insertYoutube = async () => {
    const { value: link } = await Swal.fire({
      title: 'Sisipkan Video YouTube',
      input: 'text',
      inputLabel: 'Masukkan URL atau ID video',
      inputAttributes: {
        autocomplete: 'off'
      },
      inputPlaceholder: 'contoh: https://youtu.be/VIDEO_ID atau VIDEO_ID',
      showCancelButton: true,
      confirmButtonText: 'Sisipkan',
      cancelButtonText: 'Batal',
      preConfirm: (val) => {
        const s = String(val || '').trim();
        if (!s) { Swal.showValidationMessage('Tautan/ID tidak boleh kosong'); return false; }
        if (/^[A-Za-z0-9_-]{11}$/.test(s)) return s;
        try {
          const u = new URL(s);
          if (u.hostname.includes('youtu.be') || u.hostname.includes('youtube.com')) return s;
        } catch (_) {}
        Swal.showValidationMessage('Masukkan URL YouTube yang valid atau ID 11-karakter');
        return false;
      }
    });
    if (!link) return;
    insert(`[youtube]${String(link).trim()}[/youtube]`);
  };
  return (
    <div className="d-flex flex-wrap gap-1 align-items-center">
      <button type="button" className="btn btn-light btn-sm" onClick={() => onWrap('b')}>Bold</button>
      <button type="button" className="btn btn-light btn-sm" onClick={() => onWrap('i')}>Italic</button>
      <button type="button" className="btn btn-light btn-sm" onClick={() => onWrap('u')}>Underline</button>
      <span className="ms-2 text-muted">•</span>
      <button type="button" className="btn btn-light btn-sm" onClick={() => insert('[left][/left]')}>Rata Kiri</button>
      <button type="button" className="btn btn-light btn-sm" onClick={() => insert('[center][/center]')}>Rata Tengah</button>
      <button type="button" className="btn btn-light btn-sm" onClick={() => insert('[right][/right]')}>Rata Kanan</button>
      <span className="ms-2 text-muted">•</span>
      <button type="button" className="btn btn-light btn-sm" onClick={() => insert('[br/]')}>Baris Baru</button>
      <span className="ms-2 text-muted">•</span>
      {allowTable && (
        <button type="button" className="btn btn-light btn-sm" onClick={insertTable}>Sisipkan Tabel</button>
      )}
      {allowYoutube && (
        <button type="button" className="btn btn-light btn-sm" onClick={insertYoutube}>YouTube</button>
      )}
    </div>
  );
}

// (Dihapus) Toolbar matematika tidak ditampilkan lagi
// Helper untuk path langsung ke images/question tanpa upload API
const sanitizeFileName = (name) => {
  if (!name) return `img-${Date.now()}.png`;
  return String(name).toLowerCase().replace(/[^a-z0-9._-]+/g,'-').replace(/^-+|-+$/g,'');
};
const buildDefaultQuestionImagePath = (subjectIdParam, fileName) => {
  const safe = sanitizeFileName(fileName);
  const sid = String(subjectIdParam || 'general').replace(/[^a-z0-9._-]+/gi,'');
  return `images/question/${sid}/${safe}`;
};
const askTargetImagePath = async (subjectIdParam, file) => {
  const def = buildDefaultQuestionImagePath(subjectIdParam, file?.name || '');
  const res = await Swal.fire({
    title: 'Sisipkan gambar dari folder images/question',
    input: 'text',
    inputLabel: 'Path relatif (contoh: images/question/<mapel>/<file>)',
    inputValue: def,
    showCancelButton: true,
    confirmButtonText: 'Gunakan Path',
    cancelButtonText: 'Batal',
    inputValidator: (val) => {
      const v = String(val || '').trim();
      if (!v) return 'Path tidak boleh kosong';
      if (!/^\/?images\//.test(v)) return 'Mulai dengan "images/"';
      return undefined;
    },
  });
  return res.isConfirmed ? String(res.value || def).trim() : null;
};

// (Dihapus) Toolbar matematika tidak ditampilkan lagi
// Helper: konversi File ke data URL base64 untuk inline tanpa API
const readFileAsDataUrl = (file) => new Promise((resolve, reject) => {
  try {
    const reader = new FileReader();
    reader.onload = () => resolve(reader.result);
    reader.onerror = () => reject(new Error('FileReader failed'));
    reader.readAsDataURL(file);
  } catch (err) {
    reject(err);
  }
});