import React, { useEffect, useState, useMemo } from 'react';
import { FaTrash } from 'react-icons/fa';
import api from '../utils/axios';
import Swal from 'sweetalert2';
import DataTable from 'react-data-table-component';
import { useAuth } from '../context/AuthContext.jsx';
import { canAccess } from '../utils/permissionHelper';

export default function PermissionsList() {
  const { user } = useAuth();
  const canManageRoles = canAccess(user, 'manage-roles');
  const canViewPermissions = canAccess(user, 'view-permission') || canManageRoles;
  const [permissions, setPermissions] = useState([]);
  const [newPermission, setNewPermission] = useState('');
  const [loading, setLoading] = useState(true);
  const [showAddModal, setShowAddModal] = useState(false);
  const [filterText, setFilterText] = useState('');

  // Fetch permissions
  const fetchPermissions = async () => {
    try {
      setLoading(true);
      const res = await api.get('/permissions', {
        headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }
      });
      setPermissions(res.data);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk melihat daftar permissions.',
        });
      } else {
        Swal.fire('Error', 'Gagal mengambil data permissions!', 'error');
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    // Jika tidak punya izin view, jangan fetch untuk mencegah error dan tampilkan pesan akses ditolak
    if (!canViewPermissions) {
      setLoading(false);
      return;
    }
    fetchPermissions();
  }, [canViewPermissions]);

  const handleAddPermission = async () => {
    if (!newPermission.trim()) return;

    if (!canManageRoles) {
      Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk menambahkan permission.' });
      return;
    }

    try {
      await api.post('/permissions', { name: newPermission }, {
        headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }
      });
      setNewPermission('');
      setShowAddModal(false);
      fetchPermissions();
      Swal.fire('Berhasil', 'Permission berhasil ditambahkan', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({
          icon: 'warning',
          title: 'Akses Ditolak',
          text: 'Anda tidak memiliki izin untuk menambahkan permission.',
        });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Gagal menambahkan permission', 'error');
      }
    }
  };

  const handleDeletePermission = async (id) => {
    const result = await Swal.fire({
      title: 'Apakah yakin ingin menghapus permission ini?',
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Ya, hapus!',
      cancelButtonText: 'Batal'
    });

    if (result.isConfirmed) {
      if (!canManageRoles) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk menghapus permission.' });
        return;
      }
      try {
        await api.delete(`/permissions/${id}`, {
          headers: { Authorization: `Bearer ${localStorage.getItem('token')}` }
        });
        fetchPermissions();
        Swal.fire('Berhasil', 'Permission berhasil dihapus', 'success');
      } catch (err) {
        console.error(err);
        const status = err?.response?.status;
        if (status === 403) {
          Swal.fire({
            icon: 'warning',
            title: 'Akses Ditolak',
            text: 'Anda tidak memiliki izin untuk menghapus permission.',
          });
        } else {
          Swal.fire('Gagal', err.response?.data?.message || 'Gagal menghapus permission', 'error');
        }
      }
    }
  };

  // Filter data berdasarkan search
  const filteredPermissions = useMemo(() => {
    return permissions.filter(p =>
      p.name && p.name.toLowerCase().includes(filterText.toLowerCase())
    );
  }, [permissions, filterText]);

  // Columns untuk DataTable
  const columns = [
    { name: '#', selector: (row, index) => index + 1, width: '60px' },
    { name: 'Nama Permission', selector: row => row.name, sortable: true },
    ...(canManageRoles ? [{
      name: 'Aksi',
      cell: row => (
        <button
          className="btn btn-sm btn-danger"
          onClick={() => handleDeletePermission(row.id)}
          title="Hapus"
          aria-label="Hapus"
        >
          <FaTrash />
        </button>
      )
    }] : [])
  ];

  if (loading) return <div className="text-center mt-5">Loading permissions...</div>;
  if (!canViewPermissions) return <div className="container mt-4"><h3>Akses Ditolak</h3><p>Anda tidak memiliki izin untuk mengakses daftar permissions.</p></div>;

  return (
    <div className="container mt-4">
      <h3>Daftar Permissions</h3>

      {/* Toolbar: tombol tambah + search */}
      <div className="d-flex justify-content-between align-items-center mb-3">
        {canManageRoles && (
          <button className="btn btn-primary" onClick={() => setShowAddModal(true)}>
            Tambah Permission
          </button>
        )}
        <input
          type="text"
          placeholder="Cari permission..."
          className="form-control"
          style={{ maxWidth: '300px' }}
          value={filterText}
          onChange={e => setFilterText(e.target.value)}
        />
      </div>

      <DataTable
        columns={columns}
        data={filteredPermissions}
        pagination
        highlightOnHover
        noDataComponent="Belum ada permission"
      />

      {/* Modal Tambah Permission */}
      {showAddModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Tambah Permission</h5>
                <button type="button" className="btn-close" onClick={() => setShowAddModal(false)}></button>
              </div>
              <div className="modal-body">
                <input
                  type="text"
                  className="form-control"
                  placeholder="Nama permission baru"
                  value={newPermission}
                  onChange={e => setNewPermission(e.target.value)}
                />
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowAddModal(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleAddPermission}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}
      {showAddModal && <div className="modal-backdrop fade show"></div>}
    </div>
  );
}
