import React, { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { useParticipantAuth } from '../context/ParticipantAuthContext.jsx';
import { FaGraduationCap, FaKey, FaIdBadge, FaLock, FaEye, FaEyeSlash } from 'react-icons/fa';
import './participant-login.css';

export default function ParticipantLogin() {
  const { login, loginToken } = useParticipantAuth();
  const [nisn, setNisn] = useState('');
  const [password, setPassword] = useState('');
  const [useToken, setUseToken] = useState(true);
  const [examToken, setExamToken] = useState('');
  const [error, setError] = useState('');
  const [loading, setLoading] = useState(false);
  const [deviceFp, setDeviceFp] = useState('');
  const [showPassword, setShowPassword] = useState(false);
  const navigate = useNavigate();

  // Fingerprint perangkat sederhana (stabil) menggunakan Web Crypto
  useEffect(() => {
    const makeFingerprint = async () => {
      try {
        const data = [
          navigator.userAgent,
          navigator.language,
          navigator.platform,
          String(screen.width),
          String(screen.height),
          String(screen.colorDepth),
          String(navigator.hardwareConcurrency || 0),
          Intl.DateTimeFormat().resolvedOptions().timeZone || ''
        ].join('|');
        const enc = new TextEncoder();
        const buf = await crypto.subtle.digest('SHA-256', enc.encode(data));
        const hex = Array.from(new Uint8Array(buf)).map(b => b.toString(16).padStart(2, '0')).join('');
        setDeviceFp(hex);
      } catch (_) {
        // Fallback bila Web Crypto tidak tersedia
        const fallback = btoa(
          [navigator.userAgent, navigator.language, navigator.platform, String(Date.now())].join('|')
        );
        setDeviceFp(fallback);
      }
    };
    makeFingerprint();
  }, []);

  const submit = async (e) => {
    e.preventDefault();
    setError('');
    setLoading(true);
    let res;
    if (useToken) {
      res = await loginToken(examToken.trim(), deviceFp);
    } else {
      res = await login(nisn.trim(), password, deviceFp);
    }
    setLoading(false);
    if (!res.success) { setError(res.message || 'Login gagal'); return; }
    navigate('/exam');
  };

  return (
    <div className="participant-login-page">
      <div className="participant-login-card card shadow-lg" style={{ maxWidth: 520, width: '100%' }}>
        <div className="card-body p-4 p-md-5">
          <div className="text-center mb-4">
            <div className="brand-icon mb-3"><FaGraduationCap size={36} /></div>
            <h3 className="fw-bold mb-1">Login Peserta Ujian</h3>
            <p className="text-muted small mb-0">Masuk untuk mengikuti ujian yang tersedia</p>
          </div>

          {error && <div className="alert alert-danger">{error}</div>}

          <ul className="nav nav-pills justify-content-center mb-3 participant-login-tabs">
            <li className="nav-item">
              <button type="button" className={`nav-link ${useToken ? 'active' : ''}`} onClick={() => setUseToken(true)}>
                <FaKey className="me-2" /> Token Ujian
              </button>
            </li>
            <li className="nav-item">
              <button type="button" className={`nav-link ${!useToken ? 'active' : ''}`} onClick={() => setUseToken(false)}>
                <FaIdBadge className="me-2" /> NISN & Password
              </button>
            </li>
          </ul>

          <form onSubmit={submit} autoComplete="off">
            {useToken ? (
              <div className="mb-3">
                <label className="form-label">Token Ujian</label>
                <div className="input-group input-group-lg">
                  <span className="input-group-text"><FaKey /></span>
                  <input
                    className="form-control"
                    value={examToken}
                    onChange={e => setExamToken(e.target.value)}
                    placeholder="Masukkan Token Ujian"
                    autoComplete="off"
                    autoCapitalize="off"
                    spellCheck={false}
                    name="no-token"
                    data-lpignore="true"
                    data-1p-ignore="true"
                  />
                </div>
                <small className="text-muted">Token unik per siswa, tidak boleh dibagikan.</small>

              </div>
            ) : (
              <>
                <div className="mb-3">
                  <label className="form-label">NISN</label>
                  <div className="input-group input-group-lg">
                    <span className="input-group-text"><FaIdBadge /></span>
                    <input
                      className="form-control"
                      value={nisn}
                      onChange={e => setNisn(e.target.value)}
                      placeholder="Masukkan NISN"
                      autoComplete="off"
                      name="no-username"
                      inputMode="numeric"
                      autoCapitalize="off"
                      spellCheck={false}
                      data-lpignore="true"
                      data-1p-ignore="true"
                    />
                  </div>
                </div>
                <div className="mb-3">
                  <label className="form-label">Password</label>
                  <div className="input-group input-group-lg">
                    <span className="input-group-text"><FaLock /></span>
                    <input
                      type={showPassword ? 'text' : 'password'}
                      className="form-control"
                      value={password}
                      onChange={e => setPassword(e.target.value)}
                      placeholder="Masukkan Password"
                      autoComplete="off"
                      autoCapitalize="off"
                      spellCheck={false}
                      data-lpignore="true"
                      data-1p-ignore="true"
                      name="no-password"
                    />
                    <button
                      type="button"
                      className="btn btn-outline-secondary"
                      onClick={() => setShowPassword(v => !v)}
                      aria-label="Tampilkan/sembunyikan password"
                    >
                      {showPassword ? <FaEyeSlash /> : <FaEye />}
                    </button>
                  </div>
                </div>
              </>
            )}

            <button className="btn btn-primary w-100 btn-lg" disabled={loading}>
              {loading ? (
                <>
                  <span className="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></span>
                  Memproses...
                </>
              ) : 'Masuk'}
            </button>
          </form>

          <div className="mt-3 text-center">
            {useToken ? (
              <small className="text-muted">Token akan mengikat perangkat saat login pertama.</small>

            ) : (
              <small className="text-muted">Gunakan NISN dan password default bila belum diubah.</small>
            )}


          </div>
        </div>
      </div>
    </div>
  );
}