import React, { useEffect, useMemo, useState } from 'react';
import DataTable from 'react-data-table-component';
import Swal from 'sweetalert2';
import api from '../utils/axios';
import { useAuth } from '../context/AuthContext.jsx';

export default function MonitorParticipants() {
  const { hasRole } = useAuth();
  const isSuperadmin = !!(hasRole && hasRole('superadmin'));
  const [exams, setExams] = useState([]);
  const [selectedExamId, setSelectedExamId] = useState('');
  const [statusData, setStatusData] = useState({ in_progress: [], finished: [], not_started: [], inactive: [], counts: { in_progress: 0, finished: 0, not_started: 0, inactive: 0 }, exam: null });
  const [loading, setLoading] = useState(false);

  // Autocomplete exam select with highlight & limit
  const SearchableExamSelect = ({ items = [], value, onChange, placeholder = '-- pilih ujian --', maxItems = 30, maxHeight = 300 }) => {
    const [open, setOpen] = useState(false);
    const [query, setQuery] = useState('');
    const [highlight, setHighlight] = useState(0);
    const rootRef = React.useRef(null);

    const filtered = useMemo(() => {
      const q = String(query).trim().toLowerCase();
      const list = Array.isArray(items) ? items : [];
      if (!q) return list;
      return list.filter(it => {
        const name = String(it.name || '').toLowerCase();
        const code = String(it.code || '').toLowerCase();
        return name.includes(q) || code.includes(q);
      });
    }, [items, query]);

    const limited = useMemo(() => filtered.slice(0, maxItems), [filtered, maxItems]);
    const selectedItem = useMemo(() => (Array.isArray(items) ? items.find(it => String(it.id) === String(value)) : null), [items, value]);

    const escHtml = (s) => String(s).replace(/[&<>]/g, c => ({ '&': '&amp;', '<': '&lt;', '>': '&gt;' }[c]));

    const highlightMatch = (text, q) => {
      const t = String(text || '');
      const qq = String(q || '').trim();
      if (!qq) return escHtml(t);
      const idx = t.toLowerCase().indexOf(qq.toLowerCase());
      if (idx < 0) return escHtml(t);
      const before = escHtml(t.slice(0, idx));
      const mid = escHtml(t.slice(idx, idx + qq.length));
      const after = escHtml(t.slice(idx + qq.length));
      return `${before}<mark>${mid}</mark>${after}`;
    };

    const doSelect = (it) => {
      if (!it) return;
      onChange && onChange(String(it.id));
      setOpen(false);
      setQuery('');
      setHighlight(0);
    };

    const onFocus = () => { setOpen(true); setQuery(''); };
    const onChangeInput = (e) => { setQuery(e.target.value); if (!open) setOpen(true); };
    const onKeyDown = (e) => {
      if (!open) return;
      if (e.key === 'ArrowDown') { e.preventDefault(); setHighlight(h => Math.min(h + 1, limited.length - 1)); }
      else if (e.key === 'ArrowUp') { e.preventDefault(); setHighlight(h => Math.max(h - 1, 0)); }
      else if (e.key === 'Enter') { e.preventDefault(); if (limited[highlight]) doSelect(limited[highlight]); }
      else if (e.key === 'Escape') { setOpen(false); }
    };

    React.useEffect(() => {
      const onDocMouseDown = (e) => {
        if (rootRef.current && !rootRef.current.contains(e.target)) {
          setOpen(false);
        }
      };
      document.addEventListener('mousedown', onDocMouseDown);
      return () => document.removeEventListener('mousedown', onDocMouseDown);
    }, []);

    return (
      <div className="position-relative" ref={rootRef}>
        <input
          type="text"
          className="form-control form-control-sm"
          placeholder={placeholder}
          value={open ? query : (selectedItem ? `${selectedItem.name} (${selectedItem.code || '-'})` : '')}
          onFocus={onFocus}
          onChange={onChangeInput}
          onKeyDown={onKeyDown}
        />
        {open && (
          <div className="dropdown-menu show w-100" style={{ maxHeight: `${maxHeight}px`, overflowY: 'auto' }}>
            {limited.length === 0 && (
              <div className="dropdown-item text-muted">Tidak ada hasil</div>
            )}
            {limited.map((it, idx) => (
              <button
                type="button"
                key={it.id}
                className={`dropdown-item ${idx === highlight ? 'active' : ''}`}
                onMouseEnter={() => setHighlight(idx)}
                onClick={() => doSelect(it)}
                dangerouslySetInnerHTML={{ __html: highlightMatch(`${it.name} (${it.code || '-'})`, query) }}
              />
            ))}
          </div>
        )}
      </div>
    );
  };
  const [filterStatus, setFilterStatus] = useState('in_progress');
  const [search, setSearch] = useState('');
  const [page, setPage] = useState(1);
  const [perPage, setPerPage] = useState(10);
  const [filterSchoolId, setFilterSchoolId] = useState('');
  const [filterClassId, setFilterClassId] = useState('');
  const [serverSide, setServerSide] = useState(false);
  const [serverItems, setServerItems] = useState([]);
  const [serverTotal, setServerTotal] = useState(0);
  const [selectedRows, setSelectedRows] = useState([]);
  const tokenHeader = useMemo(() => ({ headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } }), []);

  useEffect(() => {
    const fetchExams = async () => {
      try {
        // Tampilkan hanya ujian hari ini
        const resToday = await api.get('/exams/today', tokenHeader);
        setExams(Array.isArray(resToday.data) ? resToday.data : []);
      } catch (err) {
        console.error(err);
        // Fallback: tampilkan semua ujian bila endpoint "today" gagal
        try {
          const resAll = await api.get('/exams', tokenHeader);
          setExams(Array.isArray(resAll.data) ? resAll.data : []);
        } catch (err2) {
          console.error(err2);
          setExams([]);
        }
      }
    };
    fetchExams();
  }, [tokenHeader]);



  useEffect(() => {
    const fetchStatus = async () => {
      if (!selectedExamId) return;
      setLoading(true);
      try {
        const res = await api.get(`/exams/${selectedExamId}/participants-status`, {
          params: {
            search: search || undefined,
            // Ambil banyak data untuk mendukung pagination client-side
            limit: 5000,
            // Kirim school_id untuk menghitung totals berbasis sekolah.
            // Jangan kirim class_id di sini agar opsi kelas tetap lengkap.
            school_id: filterSchoolId || undefined,
          },
          ...tokenHeader,
        });
        setStatusData({
          exam: res.data?.exam || null,
          in_progress: Array.isArray(res.data?.in_progress) ? res.data.in_progress : [],
          finished: Array.isArray(res.data?.finished) ? res.data.finished : [],
          not_started: Array.isArray(res.data?.not_started) ? res.data.not_started : [],
          inactive: Array.isArray(res.data?.inactive) ? res.data.inactive : [],
          counts: res.data?.counts || { in_progress: 0, finished: 0, not_started: 0, inactive: 0 },
          totals: res.data?.totals || { in_progress: 0, finished: 0, not_started: 0, inactive: 0 },
        });
      } catch (err) {
        console.error(err);
        Swal.fire('Error', 'Gagal memuat status peserta', 'error');
      } finally {
        setLoading(false);
      }
    };
    fetchStatus();
  }, [selectedExamId, tokenHeader, search, perPage, filterSchoolId, filterClassId]);

  const schoolOptions = useMemo(() => {
    const rows = [
      ...(statusData.in_progress || []),
      ...(statusData.finished || []),
      ...(statusData.not_started || []),
      ...(statusData.inactive || []),
    ];
    const map = new Map();
    rows.forEach((r) => { const s = r?.school; if (s?.id) map.set(String(s.id), s.nama || s.name || `Sekolah #${s.id}`); });
    return Array.from(map, ([id, nama]) => ({ id, nama })).sort((a,b) => String(a.nama).localeCompare(String(b.nama)));
  }, [statusData]);
  const classOptions = useMemo(() => {
    const rows = [
      ...(statusData.in_progress || []),
      ...(statusData.finished || []),
      ...(statusData.not_started || []),
      ...(statusData.inactive || []),
    ];
    const map = new Map();
    rows.forEach((r) => { const c = r?.class; if (c?.id) map.set(String(c.id), c.name || `Kelas #${c.id}`); });
    return Array.from(map, ([id, name]) => ({ id, name })).sort((a,b) => String(a.name).localeCompare(String(b.name)));
  }, [statusData]);

  const filterAndSort = React.useCallback((rows) => {
    const base = Array.isArray(rows) ? rows : [];
    const s = search.trim().toLowerCase();
    let filtered = s
      ? base.filter((row) => {
          const p = row?.participant || {};
          const kelas = row?.class || {};
          return (
            String(p.nama || '').toLowerCase().includes(s) ||
            String(p.nisn || '').toLowerCase().includes(s) ||
            String(kelas.name || '').toLowerCase().includes(s)
          );
        })
      : base;
    if (filterSchoolId) {
      // Cocokkan berdasarkan school.id terlebih dulu, fallback ke class.id_school
      filtered = filtered.filter((row) => {
        const schoolId = row?.school?.id ?? row?.class?.id_school;
        return String(schoolId || '') === String(filterSchoolId);
      });
    }
    if (filterClassId) {
      filtered = filtered.filter((row) => String(row?.class?.id || '') === String(filterClassId));
    }
    return filtered.slice().sort((a,b) => String(a?.participant?.nama || '').localeCompare(String(b?.participant?.nama || '')));
  }, [search, filterSchoolId, filterClassId]);

  const filteredInProgress = useMemo(() => filterAndSort(statusData.in_progress), [filterAndSort, statusData]);
  const filteredFinished = useMemo(() => filterAndSort(statusData.finished), [filterAndSort, statusData]);
  const filteredNotStarted = useMemo(() => filterAndSort(statusData.not_started), [filterAndSort, statusData]);
  const filteredInactive = useMemo(() => filterAndSort(statusData.inactive), [filterAndSort, statusData]);

  const list = useMemo(() => (filterStatus === 'finished' ? filteredFinished : filteredInProgress), [filterStatus, filteredFinished, filteredInProgress]);

  const paginated = useMemo(() => {
    const start = (page - 1) * perPage;
    return list.slice(start, start + perPage);
  }, [list, page, perPage]);

  useEffect(() => {
    setPage(1);
  }, [filterStatus, search, filterSchoolId, filterClassId]);

  useEffect(() => {
    const fetchServerList = async () => {
      if (!serverSide || !selectedExamId) return;
      setLoading(true);
      try {
        const params = {
          status: filterStatus,
          page,
          per_page: perPage,
          search: search || undefined,
          id_school: filterSchoolId || undefined,
          id_class: filterClassId || undefined,
        };
        const res = await api.get(`/exams/${selectedExamId}/participants-status-list`, { params, ...tokenHeader });
        const items = Array.isArray(res.data?.items) ? res.data.items : [];
        items.sort((a,b) => String(a?.participant?.nama || '').localeCompare(String(b?.participant?.nama || '')));
        setServerItems(items);
        setServerTotal(Number(res.data?.total || 0));
      } catch (err) {
        console.error(err);
        Swal.fire('Error', 'Gagal memuat data halaman (server-side)', 'error');
      } finally {
        setLoading(false);
      }
    };
    fetchServerList();
  }, [serverSide, selectedExamId, filterStatus, page, perPage, search, filterSchoolId, filterClassId, tokenHeader]);

  const reloadStatus = async () => {
    if (!selectedExamId) return;
    const res = await api.get(`/exams/${selectedExamId}/participants-status`, {
      params: {
        search: search || undefined,
        // Ambil banyak data agar pagination client-side berfungsi
        limit: 5000,
        // Hanya kirim school_id, jangan class_id agar opsi kelas tetap lengkap.
        school_id: filterSchoolId || undefined,
      },
      ...tokenHeader,
    });
    setStatusData({
      exam: res.data?.exam || null,
      in_progress: Array.isArray(res.data?.in_progress) ? res.data.in_progress : [],
      finished: Array.isArray(res.data?.finished) ? res.data.finished : [],
      not_started: Array.isArray(res.data?.not_started) ? res.data.not_started : [],
      inactive: Array.isArray(res.data?.inactive) ? res.data.inactive : [],
      counts: res.data?.counts || { in_progress: 0, finished: 0, not_started: 0, inactive: 0 },
      totals: res.data?.totals || { in_progress: 0, finished: 0, not_started: 0, inactive: 0 },
    });
  };

  const forceFinishSession = async (row) => {
    if (!selectedExamId) return;
    const sessionId = row?.session_id;
    if (!sessionId) return;
    const confirm = await Swal.fire({
      icon: 'warning',
      title: 'Paksa Selesai Sesi',
      text: `Akhiri sesi ujian untuk ${row?.participant?.nama || 'peserta'} dan proses hasil sekarang?`,
      showCancelButton: true,
      confirmButtonText: 'Ya, paksa selesai',
      cancelButtonText: 'Batal',
    });
    if (!confirm.isConfirmed) return;
    try {
      const payload = { session_ids: [sessionId], only_in_progress: true };
      const res = await api.post(`/exams/${selectedExamId}/finalize-sessions`, payload, tokenHeader);
      const queued = res?.status === 202 || res?.data?.queued_to === 'finalize';
      Swal.fire(queued ? 'Diantre' : 'Terkirim', 'Finalisasi sesi diantre untuk diproses', 'success');
      await reloadStatus();
    } catch (err) {
      console.error(err);
      const msg = err?.response?.data?.message || 'Gagal memaksa selesai sesi';
      Swal.fire('Gagal', msg, 'error');
    }
  };

  const resetSession = async (row) => {
    if (!selectedExamId) return;
    const participantId = row?.participant?.id;
    if (!participantId) return;
    const confirm = await Swal.fire({
      icon: 'question',
      title: 'Reset Sesi Peserta',
      text: `Reset sesi untuk ${row?.participant?.nama || 'peserta'} agar bisa ujian lagi?`,
      showCancelButton: true,
      confirmButtonText: 'Ya, reset',
      cancelButtonText: 'Batal',
    });
    if (!confirm.isConfirmed) return;
    try {
      // Tambahkan force=true agar juga bisa menghapus sesi yang sedang berjalan
      await api.delete(`/exams/${selectedExamId}/participants/${participantId}/session`, { ...tokenHeader, params: { force: true } });
      Swal.fire('Berhasil', 'Sesi peserta direset', 'success');
      await reloadStatus();
      if (serverSide) {
        // refresh current page
        const params = {
          status: filterStatus,
          page,
          per_page: perPage,
          search: search || undefined,
          id_school: filterSchoolId || undefined,
          id_class: filterClassId || undefined,
        };
        const resPage = await api.get(`/exams/${selectedExamId}/participants-status-list`, { params, ...tokenHeader });
        setServerItems(Array.isArray(resPage.data?.items) ? resPage.data.items : []);
        setServerTotal(Number(resPage.data?.total || 0));
      }
    } catch (err) {
      console.error(err);
      const msg = err?.response?.data?.message || 'Gagal reset sesi';
      Swal.fire('Gagal', msg, 'error');
    }
  };

  const unbindDevice = async (row) => {
    const participantId = row?.participant?.id;
    if (!participantId) return;
    const confirm = await Swal.fire({
      icon: 'question',
      title: 'Reset Perangkat Peserta',
      text: `Izinkan ${row?.participant?.nama || 'peserta'} login dari perangkat baru?`,
      showCancelButton: true,
      confirmButtonText: 'Ya, reset perangkat',
      cancelButtonText: 'Batal',
    });
    if (!confirm.isConfirmed) return;
    try {
      await api.put(`/exam-participants/${participantId}/unbind-device`, {}, tokenHeader);
      Swal.fire('Berhasil', 'Perangkat peserta direset', 'success');
    } catch (err) {
      console.error(err);
      const msg = err?.response?.data?.message || 'Gagal reset perangkat';
      Swal.fire('Gagal', msg, 'error');
    }
  };

  const bulkResetSession = async () => {
    if (!isSuperadmin) {
      Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Hanya superadmin yang boleh melakukan aksi ini.' });
      return;
    }
    if (!selectedExamId || selectedRows.length === 0) return;
    const confirm = await Swal.fire({
      icon: 'question', title: 'Bulk Reset Sesi',
      text: `Reset sesi untuk ${selectedRows.length} peserta?`,
      showCancelButton: true, confirmButtonText: 'Ya, reset', cancelButtonText: 'Batal'
    });
    if (!confirm.isConfirmed) return;
    try {
      const ops = selectedRows.map((row) => api.delete(`/exams/${selectedExamId}/participants/${row?.participant?.id}/session`, tokenHeader));
      const results = await Promise.allSettled(ops);
      const success = results.filter(r => r.status === 'fulfilled').length;
      const failed = results.length - success;
      Swal.fire('Selesai', `Berhasil: ${success}, Gagal: ${failed}`, failed ? 'warning' : 'success');
      await reloadStatus();
      if (serverSide) {
        const params = { status: filterStatus, page, per_page: perPage, search: search || undefined, id_school: filterSchoolId || undefined, id_class: filterClassId || undefined };
        const resPage = await api.get(`/exams/${selectedExamId}/participants-status-list`, { params, ...tokenHeader });
        setServerItems(Array.isArray(resPage.data?.items) ? resPage.data.items : []);
        setServerTotal(Number(resPage.data?.total || 0));
      }
    } catch (err) {
      console.error(err);
      Swal.fire('Error', 'Gagal bulk reset sesi', 'error');
    }
  };

  const bulkUnbindDevice = async () => {
    if (!isSuperadmin) {
      Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Hanya superadmin yang boleh melakukan aksi ini.' });
      return;
    }
    if (selectedRows.length === 0) return;
    const confirm = await Swal.fire({
      icon: 'question', title: 'Bulk Reset Perangkat',
      text: `Reset perangkat untuk ${selectedRows.length} peserta?`,
      showCancelButton: true, confirmButtonText: 'Ya, reset', cancelButtonText: 'Batal'
    });
    if (!confirm.isConfirmed) return;
    try {
      const ops = selectedRows.map((row) => api.put(`/exam-participants/${row?.participant?.id}/unbind-device`, {}, tokenHeader));
      const results = await Promise.allSettled(ops);
      const success = results.filter(r => r.status === 'fulfilled').length;
      const failed = results.length - success;
      Swal.fire('Selesai', `Berhasil: ${success}, Gagal: ${failed}`, failed ? 'warning' : 'success');
    } catch (err) {
      console.error(err);
      Swal.fire('Error', 'Gagal bulk reset perangkat', 'error');
    }
  };

  const columns = useMemo(() => [
    { name: 'Nama', selector: (row) => row?.participant?.nama || '', sortable: true },
    { name: 'NISN', selector: (row) => row?.participant?.nisn || '', sortable: true },
    { name: 'Kelas', selector: (row) => row?.class?.name || row?.participant?.name_class || '', sortable: true },
    { name: 'Sekolah', selector: (row) => row?.school?.nama || '-', sortable: true },
    { name: 'Aktivitas Terakhir', selector: (row) => row?.participant?.last_activity || '-', sortable: true },
    // Tambahkan kolom status dengan badge
    { name: 'Status', sortable: false, cell: (row) => {
        const s = String(row?.status || '');
        let label = s ? s.charAt(0).toUpperCase() + s.slice(1) : '-';
        let cls = 'bg-secondary';
        if (s === 'success') cls = 'bg-success';
        else if (s === 'end_pending') cls = 'bg-warning text-dark';
        else if (s === 'selesai') cls = 'bg-success';
        else if (s === 'expired') cls = 'bg-danger';
        else if (s === 'mulai') cls = 'bg-primary';
        return (
          <div className="d-flex align-items-center gap-2">
            <span className={`badge ${cls}`}>{label}</span>
            {typeof row?.submissions_count === 'number' && row.submissions_count > 0 && (
              <span className="text-muted small">({row.submissions_count}x submit)</span>
            )}
            {typeof row?.entry_attempts === 'number' && (
              <span className="badge bg-light text-dark border">Masuk: {row.entry_attempts}</span>
            )}
          </div>
        );
      }
    },
    ...(filterStatus === 'finished'
      ? [
          { name: 'Selesai', selector: (row) => row?.finished_at || '-', sortable: true },
          { name: 'Skor', selector: (row) => row?.result?.score ?? '-', sortable: true },
        ]
      : [
          { name: 'Mulai', selector: (row) => row?.started_at || '-', sortable: true },
        ]
    ),
    {
      name: 'Aksi',
      cell: (row) => (
        <div className="d-flex gap-2">
          <button className="btn btn-sm btn-outline-secondary" title="Reset Perangkat" onClick={() => unbindDevice(row)}>
            📱
          </button>
          {filterStatus !== 'finished' && (
            <button
              className="btn btn-sm btn-outline-warning"
              title="Reset Attempt (kembalikan ke 1)"
              onClick={async () => {
                if (!selectedExamId) return;
                const participantId = row?.participant?.id;
                if (!participantId) return;
                const confirm = await Swal.fire({
                  icon: 'question',
                  title: 'Reset Attempt Peserta',
                  text: `Kembalikan kesempatan masuk untuk ${row?.participant?.nama || 'peserta'} menjadi 1?`,
                  showCancelButton: true,
                  confirmButtonText: 'Ya, reset',
                  cancelButtonText: 'Batal',
                });
                if (!confirm.isConfirmed) return;
                try {
                  await api.put(`/exams/${selectedExamId}/participants/${participantId}/reset-entry-attempts`, {}, tokenHeader);
                  Swal.fire('Berhasil', 'Kesempatan masuk direset menjadi 1', 'success');
                  await reloadStatus();
                } catch (err) {
                  console.error(err);
                  const msg = err?.response?.data?.message || 'Gagal reset attempt';
                  Swal.fire('Gagal', msg, 'error');
                }
              }}
            >
              ♻️
            </button>
          )}
          {filterStatus === 'in_progress' && isSuperadmin && (
            <button
              className="btn btn-sm btn-outline-danger"
              title="Paksa Selesai Sesi"
              onClick={() => forceFinishSession(row)}
            >
              <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
                <rect x="6" y="6" width="12" height="12" rx="2" ry="2" />
              </svg>
            </button>
          )}
          {filterStatus === 'in_progress' && (
            <button
              className="btn btn-sm btn-outline-danger"
              title="Hapus Sesi di Ujian Ini"
              onClick={async () => {
                if (!selectedExamId) return;
                const participantId = row?.participant?.id;
                if (!participantId) return;
                const confirm = await Swal.fire({
                  icon: 'warning',
                  title: 'Hapus Sesi Peserta',
                  text: `Hapus sesi ujian untuk ${row?.participant?.nama || 'peserta'}? Data sesi (exam_sessions) akan dihapus.`,
                  showCancelButton: true,
                  confirmButtonText: 'Ya, hapus',
                  cancelButtonText: 'Batal',
                });
                if (!confirm.isConfirmed) return;
                try {
                  // Gunakan endpoint reset sesi dengan force=true untuk menghapus peserta dari ujian ini
                  await api.delete(`/exams/${selectedExamId}/participants/${participantId}/session`, { ...tokenHeader, params: { force: true } });
                  Swal.fire('Berhasil', 'Sesi peserta dihapus', 'success');
                  await reloadStatus();
                } catch (err) {
                  console.error(err);
                  const msg = err?.response?.data?.message || 'Gagal menghapus sesi';
                  Swal.fire('Gagal', msg, 'error');
                }
              }}
            >
              🗑️
            </button>
          )}
          {filterStatus === 'finished' && (
            <button className="btn btn-sm btn-outline-danger" title="Reset Sesi" onClick={() => resetSession(row)}>↺</button>
          )}
        </div>
      ),
    },
  ], [filterStatus]);

  return (
    <div className="container py-4">
      <div className="d-flex align-items-center justify-content-between mb-3">
        <h3 className="mb-0">Monitor Peserta</h3>
        <div className="text-muted">Pantau status ujian dan lakukan tindakan admin</div>
      </div>

      <div className="row g-3 mb-3">
        <div className="col-md-3">
          <label className="form-label">Pilih Ujian</label>
          {/* Replace select with searchable autocomplete */}
          <SearchableExamSelect items={exams} value={selectedExamId} onChange={setSelectedExamId} />
        </div>
        <div className="col-md-2">
          <label className="form-label">Status</label>
          <div className="btn-group btn-group-sm w-100" role="group" aria-label="Filter status">
            <button type="button" className={filterStatus === 'in_progress' ? 'btn btn-primary' : 'btn btn-outline-primary'} onClick={() => setFilterStatus('in_progress')}>Sedang</button>
            <button type="button" className={filterStatus === 'finished' ? 'btn btn-primary' : 'btn btn-outline-primary'} onClick={() => setFilterStatus('finished')}>Selesai</button>
          </div>
        </div>
        <div className="col-md-2">
          <label className="form-label">Sekolah</label>
          <select className="form-select form-select-sm" value={filterSchoolId} onChange={(e) => setFilterSchoolId(e.target.value)}>
            <option value="">Semua</option>
            {schoolOptions.map((s) => (
              <option key={s.id} value={s.id}>{s.nama}</option>
            ))}
          </select>
        </div>
        <div className="col-md-2">
          <label className="form-label">Kelas</label>
          <select className="form-select form-select-sm" value={filterClassId} onChange={(e) => setFilterClassId(e.target.value)}>
            <option value="">Semua</option>
            {classOptions.map((c) => (
              <option key={c.id} value={c.id}>{c.name}</option>
            ))}
          </select>
        </div>
        <div className="col-md-2">
          <label className="form-label">Cari (Nama/NISN/Kelas)</label>
          <input type="text" className="form-control form-control-sm" placeholder="kata kunci..." value={search} onChange={(e) => setSearch(e.target.value)} />
        </div>
        <div className="col-md-1">
          <label className="form-label">Per halaman</label>
          <input
            type="number"
            min="5"
            max={String(serverSide ? Math.max(5, serverTotal || 100) : Math.max(5, (filterStatus === 'finished' ? filteredFinished.length : filteredInProgress.length) || 100))}
            className="form-control form-control-sm"
            style={{ maxWidth: '90px' }}
            value={perPage}
            onChange={(e) => {
              const maxVal = serverSide ? (serverTotal || 100) : (filterStatus === 'finished' ? filteredFinished.length : filteredInProgress.length);
              const next = Number(e.target.value || 10);
              setPerPage(Math.max(5, Math.min(maxVal || 100, next)));
            }}
          />
        </div>
      </div>

      {selectedExamId && (
        <div className="mb-3">
          <div className="alert alert-light border mb-0 py-2">
            <div className="d-flex gap-2 align-items-center flex-wrap">
              <span className="badge rounded-pill text-bg-primary">In Progress: {statusData?.totals?.in_progress ?? filteredInProgress.length}</span>
              <span className="badge rounded-pill text-bg-success">Finished: {statusData?.totals?.finished ?? filteredFinished.length}</span>
              <span className="badge rounded-pill text-bg-secondary">Belum Mulai: {statusData?.totals?.not_started ?? filteredNotStarted.length}</span>
              <span className="badge rounded-pill text-bg-warning text-dark">Tidak Aktif: {statusData?.totals?.inactive ?? filteredInactive.length}</span>
            </div>
          </div>
        </div>
      )}

      <div className="card">
        <div className="card-body">
          <div className="d-flex align-items-center justify-content-between mb-2 flex-wrap gap-2">
            <div className="d-flex gap-2">
              <button className="btn btn-sm btn-outline-secondary" disabled={!selectedRows.length || !isSuperadmin} onClick={bulkUnbindDevice}>Bulk Reset Perangkat</button>
              {filterStatus === 'finished' && (
                <button className="btn btn-sm btn-outline-danger" disabled={!selectedRows.length || !isSuperadmin} onClick={bulkResetSession}>Bulk Reset Sesi</button>
              )}
            </div>
            <div className="form-check">
              <input className="form-check-input" type="checkbox" id="serverSideToggle" checked={serverSide} onChange={(e) => { setServerSide(e.target.checked); setPage(1); }} />
              <label className="form-check-label" htmlFor="serverSideToggle">Pagination server-side</label>
            </div>
          </div>
          {loading ? (
            <div className="text-center py-4">Memuat...</div>
          ) : (
            <>
              <DataTable
                columns={columns}
                data={serverSide ? serverItems : list}
                pagination
                paginationServer={serverSide}
                paginationTotalRows={serverSide ? serverTotal : list.length}
                paginationPerPage={perPage}
                paginationRowsPerPageOptions={[10, 20, 50, 100, Math.max(1, serverSide ? serverTotal : list.length)]}
                onChangePage={(p) => setPage(p)}
                onChangeRowsPerPage={(newPerPage) => setPerPage(newPerPage)}
                highlightOnHover
                dense
                selectableRows
                onSelectedRowsChange={({ selectedRows }) => setSelectedRows(selectedRows)}
              />
              {(serverSide ? serverItems.length === 0 : list.length === 0) && (
                <div className="text-center text-muted py-3">Tidak ada data untuk filter saat ini</div>
              )}
            </>
          )}
        </div>
      </div>
    </div>
  );
}