import React, { useEffect, useMemo, useState } from 'react';
import DataTable from 'react-data-table-component';
import Swal from 'sweetalert2';
import { FaPlus, FaEdit, FaTrash } from 'react-icons/fa';
import api from '../utils/axios';
import { useAuth } from '../context/AuthContext.jsx';
import { canAccess } from '../utils/permissionHelper';

export default function Grades() {
  const { user } = useAuth();
  const [grades, setGrades] = useState([]);
  const [schools, setSchools] = useState([]);
  const [loading, setLoading] = useState(true);
  const [filterText, setFilterText] = useState('');
  const [selectedSchoolId, setSelectedSchoolId] = useState('');
  const [showAddModal, setShowAddModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [form, setForm] = useState({ grade: '', id_school: '' });
  const [editing, setEditing] = useState(null);

  const tokenHeader = { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } };

  const canViewGrade = canAccess(user, 'view-grade') || canAccess(user, 'manage-grades');
  const canCreateGrade = canAccess(user, 'create-grade') || canAccess(user, 'manage-grades');
  const canEditGrade = canAccess(user, 'edit-grade') || canAccess(user, 'manage-grades');
  const canDeleteGrade = canAccess(user, 'delete-grade') || canAccess(user, 'manage-grades');
  const canViewSchools = canAccess(user, 'view-school') || canAccess(user, 'manage-schools');

  const fetchSchools = async () => {
    try {
      const res = await api.get('/schools', tokenHeader);
      setSchools(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk melihat daftar sekolah.' });
      } else {
        Swal.fire('Error', 'Gagal memuat daftar sekolah', 'error');
      }
    }
  };

  const fetchGrades = async (schoolId = '') => {
    setLoading(true);
    try {
      const url = schoolId ? `/grades?id_school=${schoolId}` : '/grades';
      const res = await api.get(url, tokenHeader);
      setGrades(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk melihat daftar grade.' });
      } else {
        Swal.fire('Error', 'Gagal memuat daftar grade', 'error');
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (canViewSchools) fetchSchools();
    if (canViewGrade) {
      fetchGrades();
    } else {
      setLoading(false);
    }
  }, [canViewSchools, canViewGrade]);

  useEffect(() => {
    if (canViewGrade) fetchGrades(selectedSchoolId || '');
  }, [selectedSchoolId, canViewGrade]);

  const filtered = useMemo(() => {
    if (!filterText) return grades;
    const q = filterText.toLowerCase();
    return grades.filter(g => String(g.grade || '').toLowerCase().includes(q));
  }, [filterText, grades]);

  const openAdd = () => {
    setForm({ grade: '', id_school: '' });
    setShowAddModal(true);
  };

  const openEdit = (row) => {
    setEditing(row);
    setForm({ grade: row.grade ?? '', id_school: row.id_school ?? '' });
    setShowEditModal(true);
  };

  const handleAdd = async () => {
    if (!String(form.grade).trim() || !String(form.id_school).trim()) return;
    try {
      await api.post('/grades', { grade: Number(form.grade), id_school: Number(form.id_school) }, tokenHeader);
      setShowAddModal(false);
      await fetchGrades(selectedSchoolId || '');
      Swal.fire('Berhasil', 'Grade ditambahkan', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk menambah grade.' });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Gagal menambah grade', 'error');
      }
    }
  };

  const handleEditSave = async () => {
    if (!editing) return;
    if (!String(form.grade).trim() || !String(form.id_school).trim()) return;
    try {
      await api.put(`/grades/${editing.id}`, { grade: Number(form.grade), id_school: Number(form.id_school) }, tokenHeader);
      setShowEditModal(false);
      setEditing(null);
      await fetchGrades(selectedSchoolId || '');
      Swal.fire('Berhasil', 'Grade diperbarui', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk memperbarui grade.' });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Gagal memperbarui grade', 'error');
      }
    }
  };

  const handleDelete = async (row) => {
    const ok = await Swal.fire({
      title: 'Hapus grade?',
      text: `Anda yakin menghapus grade "${row.grade}"?`,
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Ya, hapus',
      cancelButtonText: 'Batal'
    }).then(r => r.isConfirmed);
    if (!ok) return;
    try {
      await api.delete(`/grades/${row.id}`, tokenHeader);
      await fetchGrades(selectedSchoolId || '');
      Swal.fire('Berhasil', 'Grade dihapus', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk menghapus grade.' });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Gagal menghapus grade', 'error');
      }
    }
  };

  const getSchoolName = (id) => schools.find(s => s.id === id)?.nama || '-';

  const columns = [
    { name: '#', selector: (row, index) => index + 1, width: '60px' },
    { name: 'Grade', selector: row => row.grade, sortable: true },
    { name: 'Sekolah', selector: row => getSchoolName(row.id_school), sortable: true, wrap: true },
    {
      name: 'Aksi',
      cell: row => (
        <div className="d-flex gap-2 flex-wrap">
          {canEditGrade && (
            <button className="btn btn-sm btn-primary" title="Edit" onClick={() => openEdit(row)}>
              <FaEdit />
            </button>
          )}
          {canDeleteGrade && (
            <button className="btn btn-sm btn-danger" title="Hapus" onClick={() => handleDelete(row)}>
              <FaTrash />
            </button>
          )}
        </div>
      ),
      width: '160px'
    }
  ];
  if (!canViewGrade) {
    return (
      <div className="container mt-4">
        <h3>Akses Ditolak</h3>
        <p>Anda tidak memiliki izin untuk melihat daftar grade.</p>
      </div>
    );
  }
  if (loading) return <div className="text-center mt-5">Memuat grade...</div>;

  return (
    <div className="container mt-4">
      <h3>Daftar Grade</h3>

      <div className="d-flex justify-content-between align-items-center mb-3 gap-2 flex-wrap">
        <div className="d-flex gap-2 align-items-center">
          {canCreateGrade && (
            <button className="btn btn-primary" onClick={openAdd}>
              <FaPlus className="me-1" /> Tambah
            </button>
          )}
          <select
            className="form-select"
            style={{ minWidth: '220px' }}
            value={selectedSchoolId}
            onChange={e => setSelectedSchoolId(e.target.value)}
          >
            <option value="">Semua Sekolah</option>
            {schools.map(s => (
              <option key={s.id} value={s.id}>{s.nama}</option>
            ))}
          </select>
        </div>

        <input
          type="text"
          placeholder="Cari grade..."
          className="form-control"
          style={{ maxWidth: '280px' }}
          value={filterText}
          onChange={e => setFilterText(e.target.value)}
        />
      </div>

      <DataTable
        columns={columns}
        data={filtered}
        pagination
        highlightOnHover
        striped
        noHeader
      />

      {/* Modal Add */}
      {showAddModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Tambah Grade</h5>
                <button type="button" className="btn-close" onClick={() => setShowAddModal(false)}></button>
              </div>
              <div className="modal-body">
                <input
                  type="number"
                  className="form-control mb-2"
                  placeholder="Grade (angka)"
                  value={form.grade}
                  onChange={e => setForm({ ...form, grade: e.target.value })}
                />
                <select
                  className="form-select"
                  value={form.id_school}
                  onChange={e => setForm({ ...form, id_school: e.target.value })}
                >
                  <option value="">Pilih Sekolah</option>
                  {schools.map(s => (
                    <option key={s.id} value={s.id}>{s.nama}</option>
                  ))}
                </select>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowAddModal(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleAdd}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Edit */}
      {showEditModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Edit Grade</h5>
                <button type="button" className="btn-close" onClick={() => setShowEditModal(false)}></button>
              </div>
              <div className="modal-body">
                <input
                  type="number"
                  className="form-control mb-2"
                  placeholder="Grade (angka)"
                  value={form.grade}
                  onChange={e => setForm({ ...form, grade: e.target.value })}
                />
                <select
                  className="form-select"
                  value={form.id_school}
                  onChange={e => setForm({ ...form, id_school: e.target.value })}
                >
                  <option value="">Pilih Sekolah</option>
                  {schools.map(s => (
                    <option key={s.id} value={s.id}>{s.nama}</option>
                  ))}
                </select>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowEditModal(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleEditSave}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}