import React, { useEffect, useMemo, useState } from 'react';
import DataTable from 'react-data-table-component';
import Swal from 'sweetalert2';
import { useParams, Link } from 'react-router-dom';
import { FaPlus, FaEdit, FaTrash, FaInfoCircle, FaChartBar, FaCog } from 'react-icons/fa';
import api from '../utils/axios';
import { useAuth } from '../context/AuthContext.jsx';
import { canAccess } from '../utils/permissionHelper';

export default function Exams() {
  const { subjectId } = useParams();
  const { user, hasRole } = useAuth();
  const isSuperadmin = !!(hasRole && hasRole('superadmin'));
  const canManageExams = isSuperadmin || canAccess(user, 'manage-exams');
  const canCreateExam = canManageExams || canAccess(user, 'create-exam');
  const canEditExam = canManageExams || canAccess(user, 'edit-exam');
  const canDeleteExam = canManageExams || canAccess(user, 'delete-exam');
  // Tombol "Hasil" hanya terlihat jika memiliki izin spesifik: view-score
  const canViewScoreLink = canAccess(user, 'view-score');
  const [exams, setExams] = useState([]);
  const [subjects, setSubjects] = useState([]);
  const [loading, setLoading] = useState(true);
  const [filterText, setFilterText] = useState('');
  const [filterDateFrom, setFilterDateFrom] = useState('');
  const [filterDateTo, setFilterDateTo] = useState('');
  const [selectedSubjectId, setSelectedSubjectId] = useState(subjectId || '');
  const [showAddModal, setShowAddModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [showDetailModal, setShowDetailModal] = useState(false);
  const [form, setForm] = useState({ name: '', description: '', id_subject: '' });
  const [editing, setEditing] = useState(null);
  const [detailItem, setDetailItem] = useState(null);
  const tokenHeader = { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } };
  // Admin modal state
  const [showAdminModal, setShowAdminModal] = useState(false);
  const [adminExam, setAdminExam] = useState(null);
  const [adminLoading, setAdminLoading] = useState(false);
  const [adminStatus, setAdminStatus] = useState({ exam: null, in_progress: [], finished: [], counts: { in_progress: 0, finished: 0, not_started: 0, inactive: 0 } });
  const [adminTotals, setAdminTotals] = useState({ in_progress: 0, finished: 0, not_started: 0, inactive: 0 });
  // Pagination untuk modal Admin
  const [adminPageInProgress, setAdminPageInProgress] = useState(1);
  const [adminPerPageInProgress, setAdminPerPageInProgress] = useState(10);
  const [adminPageFinished, setAdminPageFinished] = useState(1);
  const [adminPerPageFinished, setAdminPerPageFinished] = useState(10);
  // Schools & Grades state (untuk form ujian)
  const canViewSchools = canAccess(user, 'view-school') || canAccess(user, 'manage-schools');
  const canViewGrades = canAccess(user, 'view-grade') || canAccess(user, 'manage-grades');
  const [schools, setSchools] = useState([]);
  const [gradesForm, setGradesForm] = useState([]);
  const [gradesAll, setGradesAll] = useState([]);

  const fetchExams = async (idSubject = '') => {
    setLoading(true);
    try {
      const res = await api.get('/exams', { ...tokenHeader, params: idSubject ? { id_subject: idSubject } : {} });
      setExams(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      Swal.fire('Error', 'Gagal memuat daftar ujian', 'error');
    } finally {
      setLoading(false);
    }
  };

  const fetchSubjects = async () => {
    try {
      const res = await api.get('/subjects', tokenHeader);
      setSubjects(Array.isArray(res.data) ? res.data : []);
    } catch (err) { console.error(err); }
  };

  const fetchSchools = async () => {
    try {
      const res = await api.get('/schools', tokenHeader);
      setSchools(Array.isArray(res.data) ? res.data : []);
    } catch (err) { console.error(err); }
  };

  const fetchGradesForForm = async (schoolId) => {
    try {
      const res = await api.get('/grades', { ...tokenHeader, params: { id_school: schoolId || '' } });
      setGradesForm(Array.isArray(res.data) ? res.data : []);
    } catch (err) { console.error(err); }
  };

  useEffect(() => { fetchSubjects(); }, []);
  useEffect(() => { fetchExams(selectedSubjectId || ''); }, [selectedSubjectId]);
  useEffect(() => { if (subjectId) setSelectedSubjectId(subjectId); }, [subjectId]);
  useEffect(() => { if (canViewSchools) fetchSchools(); }, [canViewSchools]);
  useEffect(() => { if (canViewGrades) fetchGradesForForm(form.id_school || ''); }, [form.id_school, canViewGrades]);
  useEffect(() => { (async () => { try { const res = await api.get('/grades', tokenHeader); setGradesAll(Array.isArray(res.data) ? res.data : []); } catch (err) { console.error(err); } })(); }, []);

  const toDateOnly = (dtStr) => {
    if (!dtStr) return '';
    try {
      const s = String(dtStr).includes('T') ? String(dtStr) : String(dtStr).replace(' ', 'T');
      const d = new Date(s);
      const pad = (n) => String(n).padStart(2, '0');
      return `${d.getFullYear()}-${pad(d.getMonth() + 1)}-${pad(d.getDate())}`;
    } catch (e) { return ''; }
  };

  const filtered = useMemo(() => {
    const base = exams;
    const q = (filterText || '').toLowerCase();
    const hasDateFrom = !!filterDateFrom;
    const hasDateTo = !!filterDateTo;
    return base.filter(e => {
      const textOk = !q || (e.name || '').toLowerCase().includes(q) || (e.description || '').toLowerCase().includes(q) || (e.code || '').toLowerCase().includes(q);
      if (!textOk) return false;
      if (!hasDateFrom && !hasDateTo) return true;
      const dateOnly = toDateOnly(e.scheduled_at);
      if (!dateOnly) return false;
      if (hasDateFrom && dateOnly < filterDateFrom) return false;
      if (hasDateTo && dateOnly > filterDateTo) return false;
      return true;
    });
  }, [filterText, exams, filterDateFrom, filterDateTo]);

  const getSubjectName = (id) => subjects.find(s => s.id === id)?.name || '-';
  const getSchoolName = (id) => {
    if (!id) return '-';
    return schools.find(sc => sc.id === id)?.nama || '-';
  };
  const getGradeName = (id) => {
    if (!id) return '-';
    return gradesForm.find(gr => gr.id === id)?.grade || '-';
  };

  const formatDateTime = (dtStr) => {
    if (!dtStr) return '-';
    try {
      const s = String(dtStr).includes('T') ? String(dtStr) : String(dtStr).replace(' ', 'T');
      const d = new Date(s);
      const pad = (n) => String(n).padStart(2, '0');
      const dd = pad(d.getDate());
      const mm = pad(d.getMonth() + 1);
      const yyyy = d.getFullYear();
      const HH = pad(d.getHours());
      const MM = pad(d.getMinutes());
      return `${dd}-${mm}-${yyyy} ${HH}:${MM}`;
    } catch (e) {
      const clean = String(dtStr).replace('T', ' ');
      const m = clean.match(/(\d{4})-(\d{2})-(\d{2})\s(\d{2}):(\d{2})/);
      if (m) return `${m[3]}-${m[2]}-${m[1]} ${m[4]}:${m[5]}`;
      return clean.slice(0, 16);
    }
  };

  const columns = [
    { name: '#', selector: (row, index) => index + 1, width: '60px' },
    { name: 'Nama Ujian', selector: row => row.name, sortable: true },
    { name: 'Mata Pelajaran', selector: row => getSubjectName(row.id_subject), sortable: true },
    { name: 'Kode', selector: row => row.code || '-', sortable: true, width: '120px' },
    { name: 'Durasi (menit)', selector: row => row.duration_minutes ?? '-', sortable: true, width: '110px' },
    { name: 'Tanggal', selector: row => formatDateTime(row.scheduled_at), sortable: true, wrap: true, width: '260px' },
    {
      name: 'Aksi',
      cell: row => (
        <div className="d-flex gap-2">
          {canEditExam && (
            <button className="btn btn-sm btn-primary" title="Edit" onClick={() => openEdit(row)}>
              <FaEdit />
            </button>
          )}
          <button className="btn btn-sm btn-outline-info" title="Detail" onClick={() => openDetail(row)}>
            <FaInfoCircle />
          </button>
          {canViewScoreLink && (
            <Link to={`/exams/${row.id}/hasil`} className="btn btn-sm btn-success" title="Hasil">
              <FaChartBar />
            </Link>
          )}
          {canDeleteExam && (
            <button className="btn btn-sm btn-danger" title="Hapus" onClick={() => handleDelete(row)}>
              <FaTrash />
            </button>
          )}
          {isSuperadmin && (
            <button className="btn btn-sm btn-outline-secondary" title="Admin" onClick={() => openAdmin(row)}>
              <FaCog />
            </button>
          )}
        </div>
      ),
      width: '200px'
    }
  ];

  const toLocalInput = (dtStr) => {
    if (!dtStr) return '';
    try {
      const s = String(dtStr).includes('T') ? String(dtStr) : String(dtStr).replace(' ', 'T');
      const d = new Date(s);
      const pad = (n) => String(n).padStart(2, '0');
      return `${d.getFullYear()}-${pad(d.getMonth() + 1)}-${pad(d.getDate())}T${pad(d.getHours())}:${pad(d.getMinutes())}`;
    } catch (e) { return ''; }
  };

  const openAdd = () => {
    setForm({ name: '', description: '', id_subject: selectedSubjectId || '', duration_minutes: '', scheduled_at: '', scheduled_date: '', scheduled_time: '', id_school: '', id_grade: '' });
    setShowAddModal(true);
  };

  const openEdit = (row) => {
    setEditing(row);
    const s = toLocalInput(row.scheduled_at);
    const parts = s ? s.split('T') : ['', ''];
    const scheduled_date = parts[0] || '';
    const scheduled_time = parts[1] || '';
    setForm({
      name: row.name || '',
      description: row.description || '',
      id_subject: row.id_subject || '',
      duration_minutes: row.duration_minutes ?? '',
      scheduled_date,
      scheduled_time,
      scheduled_at: s,
      id_school: row.id_school || '',
      id_grade: row.id_grade || '',
    });
    setShowEditModal(true);
  };

  // Admin handlers (inside component)
  const openAdmin = async (row) => {
    setAdminExam(row);
    setShowAdminModal(true);
    // Reset halaman saat membuka modal
    setAdminPageInProgress(1);
    setAdminPageFinished(1);
    await loadAdminStatus(row.id);
  };

  const loadAdminStatus = async (examId) => {
    try {
      setAdminLoading(true);
      const res = await api.get(`/exams/${examId}/participants-status`, { params: { limit: 5000 }, ...tokenHeader });
      setAdminStatus({
        exam: res.data?.exam || null,
        in_progress: Array.isArray(res.data?.in_progress) ? res.data.in_progress : [],
        finished: Array.isArray(res.data?.finished) ? res.data.finished : [],
        counts: res.data?.counts || { in_progress: 0, finished: 0, not_started: 0, inactive: 0 },
      });
      setAdminTotals(res.data?.totals || { in_progress: 0, finished: 0, not_started: 0, inactive: 0 });
    } catch (err) {
      console.error(err);
      Swal.fire('Error', 'Gagal memuat status peserta', 'error');
    } finally {
      setAdminLoading(false);
    }
  };


  const handleSyncAdmin = async () => {
    if (!adminExam) return;
    const confirm = await Swal.fire({
      icon: 'question',
      title: 'Sinkronisasi',
      text: 'Anda yakin ingin menyinkronkan sesi dan hasil?',
      showCancelButton: true,
      confirmButtonText: 'Ya, lanjut',
      cancelButtonText: 'Batal',
    });
    if (!confirm.isConfirmed) return;
    try {
      setAdminLoading(true);
      const res = await api.post(`/exams/${adminExam.id}/finalize-sessions`, { only_in_progress: true }, tokenHeader);
      const s = res.data?.stats || {};
      const queuedTo = res.data?.queued_to || 'finalize';
      const queuedCount = (typeof s.queued === 'number') ? s.queued : (typeof s.sessions_count === 'number' ? s.sessions_count : 0);
      const msg = `Sesi diantre: ${queuedCount}\nQueue: ${queuedTo}\nPemrosesan dilakukan oleh worker/cronjob.`;
      Swal.fire(res.status === 202 ? 'Diantre' : 'Info', msg, res.status === 202 ? 'success' : 'info');
      await loadAdminStatus(adminExam.id);
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', err.response?.data?.message || 'Sinkronisasi gagal', 'error');
    } finally {
      setAdminLoading(false);
    }
  };

  const openDetail = (row) => {
    setDetailItem(row);
    setShowDetailModal(true);
  };

  const handleAdd = async () => {
    if (!form.name.trim() || !String(form.id_subject).trim()) return;
    if (!canCreateExam) {
      Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk menambah ujian.' });
      return;
    }
    try {
      // Pastikan scheduled_at terbentuk dari input tanggal & jam
      const scheduledDate = form.scheduled_date || '';
      const scheduledTime = form.scheduled_time || '';
      const scheduledAt = scheduledDate ? `${scheduledDate}T${scheduledTime || '00:00'}` : (form.scheduled_at || '');
      await api.post('/exams', {
        name: form.name,
        description: form.description,
        id_subject: Number(form.id_subject),
        duration_minutes: Number(form.duration_minutes),
        scheduled_at: scheduledAt,
        id_school: form.id_school ? Number(form.id_school) : null,
        id_grade: form.id_grade ? Number(form.id_grade) : null,
      }, tokenHeader);
      setShowAddModal(false);
      await fetchExams(selectedSubjectId || '');
      Swal.fire('Berhasil', 'Ujian ditambahkan beserta kode ujian', 'success');
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', err.response?.data?.message || 'Gagal menambah ujian', 'error');
    }
  };

  const handleUpdate = async () => {
    if (!editing) return;
    if (!form.name.trim() || !String(form.id_subject).trim()) return;
    if (!canEditExam) {
      Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk mengedit ujian.' });
      return;
    }
    try {
      // Pastikan scheduled_at terbentuk dari input tanggal & jam
      const scheduledDate = form.scheduled_date || '';
      const scheduledTime = form.scheduled_time || '';
      const scheduledAt = scheduledDate ? `${scheduledDate}T${scheduledTime || '00:00'}` : (form.scheduled_at || '');
      await api.put(`/exams/${editing.id}`, {
        name: form.name,
        description: form.description,
        id_subject: Number(form.id_subject),
        duration_minutes: Number(form.duration_minutes),
        scheduled_at: scheduledAt,
        id_school: form.id_school ? Number(form.id_school) : null,
        id_grade: form.id_grade ? Number(form.id_grade) : null,
      }, tokenHeader);
      setShowEditModal(false);
      setEditing(null);
      await fetchExams(selectedSubjectId || '');
      Swal.fire('Berhasil', 'Ujian diperbarui', 'success');
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', err.response?.data?.message || 'Gagal memperbarui ujian', 'error');
    }
  };

  const handleDelete = async (row) => {
    const confirm = await Swal.fire({
      title: 'Hapus Ujian?',
      text: `Anda akan menghapus: ${row.name}`,
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Ya, hapus',
      cancelButtonText: 'Batal'
    });
    if (!confirm.isConfirmed) return;
    if (!canDeleteExam) {
      Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk menghapus ujian.' });
      return;
    }
    try {
      await api.delete(`/exams/${row.id}`, tokenHeader);
      await fetchExams(selectedSubjectId || '');
      Swal.fire('Berhasil', 'Ujian dihapus', 'success');
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', err.response?.data?.message || 'Gagal menghapus ujian', 'error');
    }
  };

  const handleDeleteAll = async () => {
    const scopeText = selectedSubjectId ? `semua ujian pada mapel: ${getSubjectName(Number(selectedSubjectId))}` : 'semua ujian';
    const confirm = await Swal.fire({
      title: 'Hapus Massal Ujian?',
      text: `Anda akan menghapus ${scopeText}. Tindakan ini akan menghapus sesi & hasil ujian terkait dan tidak dapat dibatalkan.`,
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Ya, hapus semua',
      cancelButtonText: 'Batal'
    });
    if (!confirm.isConfirmed) return;
    try {
      const res = await api.delete('/exams', { ...tokenHeader, params: selectedSubjectId ? { id_subject: Number(selectedSubjectId) } : {} });
      await fetchExams(selectedSubjectId || '');
      const msg = res?.data?.message || 'Berhasil menghapus ujian massal';
      Swal.fire('Berhasil', msg, 'success');
    } catch (err) {
      console.error(err);
      Swal.fire('Gagal', err.response?.data?.message || 'Gagal menghapus ujian massal', 'error');
    }
  };

  if (loading) return <div className="text-center mt-5">Memuat ujian...</div>;

  return (
    <div className="container mt-4">
      <h3>Daftar Ujian</h3>

      <div className="d-flex justify-content-between align-items-center mb-3 gap-2">
        <div className="d-flex align-items-center gap-2">
          {canCreateExam && (
            <button className="btn btn-sm btn-primary d-flex align-items-center" onClick={openAdd}>
              <FaPlus className="me-1" /> Tambah
            </button>
          )}
          <select
            className="form-select"
            style={{ maxWidth: '280px' }}
            value={selectedSubjectId}
            onChange={e => setSelectedSubjectId(e.target.value)}
          >
            <option value="">Semua Mata Pelajaran</option>
            {subjects.map(s => (
              <option key={s.id} value={s.id}>{s.name}</option>
            ))}
          </select>
          {isSuperadmin && (
            <button className="btn btn-sm btn-outline-danger d-flex align-items-center" onClick={handleDeleteAll} title={selectedSubjectId ? 'Hapus semua ujian pada mapel terpilih' : 'Hapus semua ujian'}>
              <FaTrash className="me-1" /> Semua
            </button>
          )}
        </div>
        <div className="d-flex align-items-center gap-2">
          <input
            type="text"
            placeholder="Cari nama/deskripsi..."
            className="form-control"
            style={{ maxWidth: '300px' }}
            value={filterText}
            onChange={e => setFilterText(e.target.value)}
          />
          <input
            type="date"
            className="form-control"
            style={{ maxWidth: '170px' }}
            value={filterDateFrom}
            onChange={e => setFilterDateFrom(e.target.value)}
            placeholder="Dari"
            title="Filter dari tanggal (YYYY-MM-DD)"
          />
          <span>—</span>
          <input
            type="date"
            className="form-control"
            style={{ maxWidth: '170px' }}
            value={filterDateTo}
            onChange={e => setFilterDateTo(e.target.value)}
            placeholder="Sampai"
            title="Filter sampai tanggal (YYYY-MM-DD)"
          />
          {(filterDateFrom || filterDateTo) && (
            <button
              type="button"
              className="btn btn-sm btn-outline-secondary"
              onClick={() => { setFilterDateFrom(''); setFilterDateTo(''); }}
              title="Reset filter tanggal"
            >
              Reset
            </button>
          )}
        </div>
      </div>

      <DataTable
        columns={columns}
        data={filtered}
        pagination
        paginationPerPage={10}
        paginationRowsPerPageOptions={[10, 20, 50, 100, Math.max(1, filtered.length)]}
        highlightOnHover
        striped
        noHeader
      />

      {/* Modal Add */}
      {showAddModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Tambah Ujian</h5>
                <button type="button" className="btn-close" onClick={() => setShowAddModal(false)}></button>
              </div>
              <div className="modal-body">
                <div className="mb-2">
                  <label className="form-label">Mata Pelajaran</label>
                  <SearchableSubjectSelect
                    subjects={subjects}
                    value={form.id_subject}
                    onChange={(val) => setForm({ ...form, id_subject: val })}
                    grades={gradesAll}
                  />
                </div>
                <input className="form-control mb-2" placeholder="Nama Ujian" value={form.name} onChange={e => setForm({ ...form, name: e.target.value })} />
                <div className="row g-2 mb-2">
                  <div className="col-4">
                    <input type="number" min="1" className="form-control" placeholder="Durasi (menit)" value={form.duration_minutes} onChange={e => setForm({ ...form, duration_minutes: e.target.value })} />
                  </div>
                  <div className="col-5">
                    <input
                      type="date"
                      className="form-control"
                      placeholder="Tanggal Ujian"
                      value={form.scheduled_date || ''}
                      onChange={e => {
                        const scheduled_date = e.target.value;
                        const scheduled_at = scheduled_date && form.scheduled_time ? `${scheduled_date}T${form.scheduled_time}` : '';
                        setForm({ ...form, scheduled_date, scheduled_at });
                      }}
                    />
                  </div>
                  <div className="col-3">
                    <input
                      type="time"
                      className="form-control"
                      placeholder="Jam"
                      value={form.scheduled_time || ''}
                      onChange={e => {
                        const scheduled_time = e.target.value;
                        const scheduled_at = form.scheduled_date && scheduled_time ? `${form.scheduled_date}T${scheduled_time}` : '';
                        setForm({ ...form, scheduled_time, scheduled_at });
                      }}
                    />
                  </div>
                </div>
                {/* Sekolah & Grade (opsional) */}
                <select
                  className="form-select mb-2"
                  value={form.id_school}
                  onChange={e => setForm({ ...form, id_school: e.target.value, id_grade: '' })}
                >
                  <option value="">Pilih Sekolah (opsional)</option>
                  {schools.map(s => (
                    <option key={s.id} value={s.id}>{s.nama}</option>
                  ))}
                </select>
                <select
                  className="form-select mb-2"
                  value={form.id_grade}
                  onChange={e => setForm({ ...form, id_grade: e.target.value })}
                  disabled={!form.id_school}
                >
                  <option value="">Pilih Grade (opsional)</option>
                  {gradesForm.map(g => (
                    <option key={g.id} value={g.id}>{g.grade}</option>
                  ))}
                </select>
                
                <textarea className="form-control" rows="3" placeholder="Deskripsi (opsional)" value={form.description} onChange={e => setForm({ ...form, description: e.target.value })}></textarea>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowAddModal(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleAdd}>OK</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Edit */}
      {showEditModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Edit Ujian</h5>
                <button type="button" className="btn-close" onClick={() => setShowEditModal(false)}></button>
              </div>
              <div className="modal-body">
                <div className="mb-2">
                  <label className="form-label">Mata Pelajaran</label>
                  <SearchableSubjectSelect
                    subjects={subjects}
                    value={form.id_subject}
                    onChange={(val) => setForm({ ...form, id_subject: val })}
                    grades={gradesAll}
                  />
                </div>
                <input className="form-control mb-2" placeholder="Nama Ujian" value={form.name} onChange={e => setForm({ ...form, name: e.target.value })} />
                <div className="row g-2 mb-2">
                  <div className="col-4">
                    <input
                      type="number"
                      min="1"
                      className="form-control"
                      placeholder="Durasi (menit)"
                      value={form.duration_minutes}
                      onChange={e => setForm({ ...form, duration_minutes: e.target.value })}
                    />
                  </div>
                  <div className="col-5">
                    <input
                      type="date"
                      className="form-control"
                      placeholder="Tanggal Ujian"
                      value={form.scheduled_date || ''}
                      onChange={e => {
                        const scheduled_date = e.target.value;
                        const scheduled_at = scheduled_date && form.scheduled_time ? `${scheduled_date}T${form.scheduled_time}` : '';
                        setForm({ ...form, scheduled_date, scheduled_at });
                      }}
                    />
                  </div>
                  <div className="col-3">
                    <input
                      type="time"
                      className="form-control"
                      placeholder="Jam"
                      value={form.scheduled_time || ''}
                      onChange={e => {
                        const scheduled_time = e.target.value;
                        const scheduled_at = form.scheduled_date && scheduled_time ? `${form.scheduled_date}T${scheduled_time}` : '';
                        setForm({ ...form, scheduled_time, scheduled_at });
                      }}
                    />
                  </div>
                </div>
                {/* Sekolah & Grade (opsional) */}
                <select
                  className="form-select mb-2"
                  value={form.id_school}
                  onChange={e => setForm({ ...form, id_school: e.target.value, id_grade: '' })}
                >
                  <option value="">Pilih Sekolah (opsional)</option>
                  {schools.map(s => (
                    <option key={s.id} value={s.id}>{s.nama}</option>
                  ))}
                </select>
                <select
                  className="form-select mb-2"
                  value={form.id_grade}
                  onChange={e => setForm({ ...form, id_grade: e.target.value })}
                  disabled={!form.id_school}
                >
                  <option value="">Pilih Grade (opsional)</option>
                  {gradesForm.map(g => (
                    <option key={g.id} value={g.id}>{g.grade}</option>
                  ))}
                </select>
                
                <textarea className="form-control" rows="3" placeholder="Deskripsi (opsional)" value={form.description} onChange={e => setForm({ ...form, description: e.target.value })}></textarea>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowEditModal(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleUpdate}>OK</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Detail */}
      {showDetailModal && detailItem && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Detail Ujian</h5>
                <button type="button" className="btn-close" onClick={() => { setShowDetailModal(false); setDetailItem(null); }}></button>
              </div>
              <div className="modal-body">
                 <div className="mb-2"><strong>Nama:</strong> {detailItem.name || '-'}</div>
                 <div className="mb-2"><strong>Mata Pelajaran:</strong> {getSubjectName(detailItem.id_subject)}</div>
                 <div className="mb-2"><strong>Sekolah:</strong> {getSchoolName(detailItem.id_school)}</div>
                 <div className="mb-2"><strong>Grade:</strong> {getGradeName(detailItem.id_grade)}</div>
                 <div className="mb-3 d-flex flex-wrap gap-2">
                   <span className="badge bg-secondary">Kode: {detailItem.code || '-'}</span>
                   <span className="badge bg-info text-dark">Durasi: {detailItem.duration_minutes ?? '-'} menit</span>
                   <span className="badge bg-light text-dark">Tanggal: {formatDateTime(detailItem.scheduled_at)}</span>
                 </div>
                 <div className="mb-2"><strong>Deskripsi:</strong></div>
                 <div className="p-2 bg-light rounded text-secondary" style={{ whiteSpace: 'pre-wrap' }}>
                   {detailItem.description || '-'}
                 </div>
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => { setShowDetailModal(false); setDetailItem(null); }}>Tutup</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Admin: Detail & Sinkronisasi */}
      {showAdminModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog modal-xl">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Admin: Detail & Sinkronisasi Ujian</h5>
                <button type="button" className="btn-close" onClick={() => setShowAdminModal(false)}></button>
              </div>
              <div className="modal-body">
                {adminLoading ? (
                  <div className="text-muted">Memuat data...</div>
                ) : (
                  <>
                    {adminStatus?.exam && (
                      <div className="mb-2">
                        <div><strong>Ujian:</strong> {adminStatus.exam.name} ({adminStatus.exam.code || '-'})</div>
                        <div><strong>Mata Pelajaran:</strong> {adminStatus.exam.subject_name || '-'} ({adminStatus.exam.subject_code || '-'})</div>
                        <div><strong>Ringkasan:</strong> Sedang: {adminTotals?.in_progress || adminStatus.in_progress.length} • Selesai: {adminTotals?.finished || adminStatus.finished.length}</div>
                      </div>
                    )}
                    <div className="row">
                      <div className="col-md-6">
                        <div className="card">
                          <div className="card-header">Sedang Mengerjakan ({adminTotals?.in_progress || adminStatus.in_progress.length})</div>
                          <div className="card-body p-0">
                            <div className="d-flex flex-wrap gap-2 align-items-center p-2">
                              <select className="form-select form-select-sm" style={{ maxWidth: 160 }} value={adminPerPageInProgress} onChange={(e) => { const n = Number(e.target.value); setAdminPerPageInProgress(n); setAdminPageInProgress(1); }}>
                                {[10,20,50,100,200,500].map(n => (<option key={n} value={n}>Per Halaman: {n}</option>))}
                              </select>
                              <div className="d-flex align-items-center gap-2">
                                <button className="btn btn-sm btn-outline-secondary" disabled={adminPageInProgress <= 1} onClick={() => { const np = Math.max(1, adminPageInProgress-1); setAdminPageInProgress(np); }}>« Prev</button>
                                <span className="small">Halaman {adminPageInProgress} dari {Math.max(1, Math.ceil((adminStatus.in_progress.length||0)/(adminPerPageInProgress||1)))} • Total: {adminStatus.in_progress.length || 0}</span>
                                <button className="btn btn-sm btn-outline-secondary" disabled={adminPageInProgress >= Math.ceil((adminStatus.in_progress.length||0)/(adminPerPageInProgress||1))} onClick={() => { const np = adminPageInProgress+1; setAdminPageInProgress(np); }}>Next »</button>
                              </div>
                            </div>
                            <div className="table-responsive">
                              <table className="table table-sm mb-0">
                                <thead>
                                  <tr>
                                    <th>NISN</th>
                                    <th>Nama</th>
                                    <th>Kelas</th>
                                    <th>Sekolah</th>
                                    <th>Mulai</th>
                                  </tr>
                                </thead>
                                <tbody>
                                  {adminStatus.in_progress.slice((adminPageInProgress-1)*adminPerPageInProgress, adminPageInProgress*adminPerPageInProgress).map(row => (
                                    <tr key={row.session_id}>
                                      <td>{row.participant?.nisn || '-'}</td>
                                      <td>{row.participant?.nama || '-'}</td>
                                      <td>{row.class?.name || '-'}</td>
                                      <td>{row.school?.nama || '-'}</td>
                                      <td>{row.started_at || '-'}</td>
                                    </tr>
                                  ))}
                                </tbody>
                              </table>
                            </div>
                          </div>
                        </div>
                      </div>
                      <div className="col-md-6">
                        <div className="card">
                          <div className="card-header">Sudah Selesai ({adminTotals?.finished || adminStatus.finished.length})</div>
                          <div className="card-body p-0">
                            <div className="d-flex flex-wrap gap-2 align-items-center p-2">
                              <select className="form-select form-select-sm" style={{ maxWidth: 160 }} value={adminPerPageFinished} onChange={(e) => { const n = Number(e.target.value); setAdminPerPageFinished(n); setAdminPageFinished(1); }}>
                                {[10,20,50,100,200,500].map(n => (<option key={n} value={n}>Per Halaman: {n}</option>))}
                              </select>
                              <div className="d-flex align-items-center gap-2">
                                <button className="btn btn-sm btn-outline-secondary" disabled={adminPageFinished <= 1} onClick={() => { const np = Math.max(1, adminPageFinished-1); setAdminPageFinished(np); }}>« Prev</button>
                                <span className="small">Halaman {adminPageFinished} dari {Math.max(1, Math.ceil((adminStatus.finished.length||0)/(adminPerPageFinished||1)))} • Total: {adminStatus.finished.length || 0}</span>
                                <button className="btn btn-sm btn-outline-secondary" disabled={adminPageFinished >= Math.ceil((adminStatus.finished.length||0)/(adminPerPageFinished||1))} onClick={() => { const np = adminPageFinished+1; setAdminPageFinished(np); }}>Next »</button>
                              </div>
                            </div>
                            <div className="table-responsive">
                              <table className="table table-sm mb-0">
                                <thead>
                                  <tr>
                                    <th>NISN</th>
                                    <th>Nama</th>
                                    <th>Kelas</th>
                                    <th>Sekolah</th>
                                    <th>Mulai</th>
                                    <th>Selesai</th>
                                  </tr>
                                </thead>
                                <tbody>
                                  {adminStatus.finished.slice((adminPageFinished-1)*adminPerPageFinished, adminPageFinished*adminPerPageFinished).map(row => (
                                    <tr key={row.session_id}>
                                      <td>{row.participant?.nisn || '-'}</td>
                                      <td>{row.participant?.nama || '-'}</td>
                                      <td>{row.class?.name || '-'}</td>
                                      <td>{row.school?.nama || '-'}</td>
                                      <td>{row.started_at || '-'}</td>
                                      <td>{row.finished_at || '-'}</td>
                                    </tr>
                                  ))}
                                </tbody>
                              </table>
                            </div>
                          </div>
                        </div>
                      </div>
                    </div>
                  </>
                )}
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowAdminModal(false)}>Tutup</button>
        <button
          className="btn btn-primary"
          onClick={handleSyncAdmin}
          disabled={adminLoading || !adminStatus || (adminStatus?.in_progress?.length || 0) === 0}
          title={adminLoading ? 'Memuat status...' : ((adminStatus?.in_progress?.length || 0) === 0 ? 'Tidak ada peserta yang sedang mengerjakan' : 'Sinkronisasi sesi dan hasil')}
        >
          Sinkronisasi
        </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}


function SearchableSubjectSelect({ subjects, value, onChange, placeholder = 'Pilih atau cari mata pelajaran...', grades = [] }) {
  const [open, setOpen] = React.useState(false);
  const [query, setQuery] = React.useState('');
  const [highlight, setHighlight] = React.useState(0);
  const inputRef = React.useRef(null);
  const maxItems = 30; // batas jumlah item tampil
  const maxHeight = 300; // tinggi maksimum dropdown (px)
  const gradesMap = React.useMemo(() => {
    const m = new Map();
    (Array.isArray(grades) ? grades : []).forEach(g => { if (g?.id) m.set(Number(g.id), g.grade || g.name || String(g.id)); });
    return m;
  }, [grades]);

  const selected = React.useMemo(() => subjects.find(s => s.id === Number(value)), [subjects, value]);
  const filtered = React.useMemo(() => {
    const q = query.trim().toLowerCase();
    if (!q) return subjects;
    return subjects.filter(s => {
      const name = String(s.name || '').toLowerCase();
      const code = String(s.code || '').toLowerCase();
      const gradeStr = String(gradesMap.get(Number(s.id_grade)) || '').toLowerCase();
      return name.includes(q) || code.includes(q) || gradeStr.includes(q);
    });
  }, [subjects, query]);
  const limited = React.useMemo(() => filtered.slice(0, maxItems), [filtered]);

  React.useEffect(() => { setHighlight(0); }, [query, open]);

  const displayValue = open ? query : (selected ? `${selected.name} (${selected.code || '-'}, Grade: ${gradesMap.get(Number(selected.id_grade)) || '-'})` : '');

  const handleInputFocus = () => {
    setOpen(true);
    setQuery('');
  };
  const handleInputChange = (e) => {
    setQuery(e.target.value);
    if (!open) setOpen(true);
  };
  const handleKeyDown = (e) => {
    if (!open) return;
    if (e.key === 'ArrowDown') {
      e.preventDefault();
      setHighlight(h => Math.min(h + 1, limited.length - 1));
    } else if (e.key === 'ArrowUp') {
      e.preventDefault();
      setHighlight(h => Math.max(h - 1, 0));
    } else if (e.key === 'Enter') {
      e.preventDefault();
      if (limited[highlight]) doSelect(limited[highlight]);
    } else if (e.key === 'Escape') {
      setOpen(false);
    }
  };
  const doSelect = (item) => {
    onChange(String(item.id));
    setOpen(false);
    setQuery('');
  };
  const clear = () => {
    onChange('');
    setQuery('');
    setOpen(false);
    inputRef.current && inputRef.current.focus();
  };
const highlightMatch = (text, q) => {
    const str = String(text || '');
    const needle = q.trim();
    if (!needle) return str;
    const lower = str.toLowerCase();
    const idx = lower.indexOf(needle.toLowerCase());
    if (idx === -1) return str;
    const before = str.slice(0, idx);
    const matched = str.slice(idx, idx + needle.length);
    const after = str.slice(idx + needle.length);
    return (
      <>
        {before}
        <span style={{ backgroundColor: '#ffe58f' }}>{matched}</span>
        {after}
      </>
    );
  };

  return (
    <div className="position-relative">
      <div className="input-group">
        <input
          ref={inputRef}
          type="text"
          className="form-control"
          placeholder={placeholder}
          value={displayValue}
          onFocus={handleInputFocus}
          onChange={handleInputChange}
          onKeyDown={handleKeyDown}
        />
        <button type="button" className="btn btn-outline-secondary" onClick={() => setOpen(o => !o)}>
          {open ? 'Tutup' : 'Cari'}
        </button>
        <button type="button" className="btn btn-outline-secondary" onClick={clear} disabled={!value}>
          Clear
        </button>
      </div>
      {open && (
        <div className="dropdown-menu show w-100" style={{ maxHeight, overflowY: 'auto' }}>
          {limited.length === 0 ? (
             <span className="dropdown-item text-muted">Tidak ada hasil</span>
           ) : (
            limited.map((s, idx) => (
               <button
                 key={s.id}
                 type="button"
                 className={`dropdown-item ${idx === highlight ? 'active' : ''}`}
                 onMouseEnter={() => setHighlight(idx)}
                 onClick={() => doSelect(s)}
               >
                {highlightMatch(s.name, query)} <span className="text-muted">({highlightMatch(s.code || '-', query)}, Grade: {highlightMatch(gradesMap.get(Number(s.id_grade)) || '-', query)})</span>
               </button>
             ))
           )}
         </div>
       )}
    </div>
  );
}