import React, { useEffect, useState, useMemo } from 'react';
import { useParams, Link } from 'react-router-dom';
import api from '../utils/axios';
import { useAuth } from '../context/AuthContext.jsx';
import { canAccess } from '../utils/permissionHelper';

export default function ExamResults() {
  const { examId } = useParams();
  const { user } = useAuth();
  // Akses halaman hasil ujian ditentukan secara ketat oleh permission view-score
  const canViewScore = canAccess(user, 'view-score');
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [exam, setExam] = useState(null);
  const [finished, setFinished] = useState([]);
  const [inProgress, setInProgress] = useState([]);
  const [counts, setCounts] = useState({ in_progress: 0, finished: 0 });
  const [subjectId, setSubjectId] = useState(null);
  const [questionsAll, setQuestionsAll] = useState([]);
  const [searchTerm, setSearchTerm] = useState('');
  // Server-side pagination untuk hasil selesai
  const [page, setPage] = useState(1);
  const [perPage, setPerPage] = useState(50);
  const [serverItems, setServerItems] = useState([]);
  const [serverTotal, setServerTotal] = useState(0);
  const [totals, setTotals] = useState({ in_progress: 0, finished: 0, not_started: 0, inactive: 0 });
  const [exportingAnalysis, setExportingAnalysis] = useState(false);
  const tokenHeader = { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } };

  // Muat ringkasan meta & counts (tanpa search, ambil banyak dengan limit besar)
  useEffect(() => {
    const loadMeta = async () => {
      setLoading(true); setError('');
      if (!canViewScore) {
        setError('Akses Ditolak: Anda tidak memiliki izin untuk melihat hasil nilai.');
        setLoading(false);
        return;
      }
      try {
        const res = await api.get(`/exams/${examId}/participants-status`, { params: { limit: 5000 }, ...tokenHeader });
        setExam(res.data.exam || null);
        setSubjectId(res.data?.exam?.id_subject || null);
        setCounts(res.data.counts || { in_progress: 0, finished: 0 });
        setTotals(res.data.totals || { in_progress: 0, finished: 0, not_started: 0, inactive: 0 });
        const fin = Array.isArray(res.data?.finished) ? res.data.finished.slice() : [];
        fin.sort((a,b) => String(a?.participant?.nama || '').localeCompare(String(b?.participant?.nama || '')));
        setFinished(fin);
        const prog = Array.isArray(res.data?.in_progress) ? res.data.in_progress : [];
        setInProgress(prog);
      } catch (e) {
        console.error('Gagal memuat ringkasan hasil ujian', e);
        setError('Gagal memuat hasil ujian. Pastikan Anda login.');
      } finally {
        setLoading(false);
      }
    };
    loadMeta();
  }, [examId, canViewScore]);

  // Pagination lokal: slice dari data finished yang sudah dimuat
  useEffect(() => {
    try {
      const q = String(searchTerm || '').trim().toLowerCase();
      const filtered = q
        ? finished.filter(r => {
            const name = String(r?.participant?.nama || '').toLowerCase();
            const nisn = String(r?.participant?.nisn || '').toLowerCase();
            return name.includes(q) || nisn.includes(q);
          })
        : finished;
      const total = filtered.length;
      const start = (page - 1) * perPage;
      const end = start + perPage;
      setServerItems(filtered.slice(start, end));
      setServerTotal(total);
    } catch (e) {
      console.warn('Gagal memproses pagination lokal hasil ujian:', e?.message || e);
    }
  }, [finished, page, perPage, searchTerm]);

  // Fetch semua soal berdasarkan subject untuk urutan kanonik (paginate 20 per halaman)
  useEffect(() => {
    const fetchAllQuestions = async () => {
      try {
        if (!subjectId) return;
        const acc = [];
        let page = 1; let lastPage = 1;
        do {
          const res = await api.get(`/questions?subject_id=${subjectId}&page=${page}`, tokenHeader);
          const data = res.data;
          if (Array.isArray(data)) {
            // Non-paginated fallback
            acc.push(...data);
            lastPage = page; // break
          } else {
            const list = Array.isArray(data?.data) ? data.data : [];
            acc.push(...list);
            lastPage = Number(data?.last_page || 1);
          }
          page += 1;
        } while (page <= lastPage && page < 200); // safety cap
        setQuestionsAll(acc);
      } catch (e) {
        console.warn('Gagal memuat soal untuk mapping jawaban:', e?.response?.data || e.message);
      }
    };
    fetchAllQuestions();
  }, [subjectId]);

  const filteredFinished = useMemo(() => serverItems, [serverItems]);

  const exportItemAnalysisCsv = async () => {
    try {
      setExportingAnalysis(true);
      const res = await api.get(`/exams/${examId}/export-item-analysis`, { responseType: 'blob', ...tokenHeader });
      const blob = new Blob([res.data], { type: 'text/csv;charset=utf-8;' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      const cd = res.headers && res.headers['content-disposition'];
      let filename = `analisis-butir-exam-${examId}.csv`;
      if (cd) {
        const m = /filename="?([^";]+)"?/i.exec(cd);
        if (m && m[1]) filename = m[1];
      }
      a.href = url;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    } catch (e) {
      console.error('Gagal export analisis butir', e);
      alert('Gagal mengekspor analisis butir soal. Pastikan ujian memiliki sesi selesai dan soal pada mapel terkait.');
    } finally {
      setExportingAnalysis(false);
    }
  };

  const exportCsv = () => {
    const headers = [
      'NISN','Nama','Kelas','Sekolah','Mulai','Selesai',
      'Benar Pilihan','Salah Pilihan','Total Soal Pilihan','Nilai Pilihan','No Benar Pilihan','No Salah Pilihan','No Essay Benar','No Essay Salah','Total Soal Essay','Nilai Essay',
      'Jawaban (mapel urutan kanonik)','Jawaban Essay (pos=jawaban)'
    ];
    const csvEscape = (v) => {
      const s = (v === null || v === undefined) ? '' : String(v);
      const escaped = s.replace(/"/g, '""');
      return '"' + escaped + '"';
    };
    const lines = [headers.map(csvEscape).join(',')];

    // Util untuk normalisasi teks essay (dipakai jika essay_details tidak tersedia)
    const normalizeText = (t) => {
      const s = (t === null || t === undefined) ? '' : String(t);
      const noTags = s.replace(/<[^>]*>/g, '');
      return noTags.replace(/\s+/g, ' ').trim().toLowerCase();
    };

    // Peta urutan kanonik: index 1-based → question
    const canonical = questionsAll;
    // Buat lookup agar cepat memetakan id → posisi kanonik dan id → objek soal
    const idToPos = new Map();
    const idToQ = new Map();
    canonical.forEach((q, i) => {
      const key = String(q.id);
      idToPos.set(key, i + 1);
      idToQ.set(key, q);
    });

    filteredFinished.forEach(row => {
      const correctIdx = Array.isArray(row.result?.correct_order_indexes) ? row.result.correct_order_indexes.join('; ') : '';
      const wrongIdx = Array.isArray(row.result?.wrong_order_indexes) ? row.result.wrong_order_indexes.join('; ') : '';

      // Bangun string jawaban (hanya pilihan jamak) sesuai urutan kanonik mapel
      let answersStr = '';
      const ansMap = row.result?.answers || {};
      const essayDetails = row.result?.essay_details || {};

      const parts = [];
      canonical.forEach((q, i) => {
        const id = String(q.id);
        const pos = i + 1;
        const keyUp = String(q?.key_answer || '').trim().toUpperCase();
        const keyIsMC = ['A','B','C','D','E'].includes(keyUp);
        const isBlankOrDash = (v) => { const t = String(v || '').trim(); return t === '' || t === '-'; };
        const bdBlank = isBlankOrDash(q?.option_b) && isBlankOrDash(q?.option_c) && isBlankOrDash(q?.option_d);
        const isEssay = (!keyIsMC) && bdBlank;
        if (!isEssay) {
          const rawAns = ansMap[id];
          const mc = /^[A-E]$/i.test(String(rawAns || '')) ? String(rawAns).toUpperCase() : '-';
          parts.push(`${pos}. ${mc}`);
        }
      });
      answersStr = parts.join(' | ');

      // Hitung essay benar/salah
      let essayCorrectCount = Number(row.result?.essay_correct_count ?? 0) || 0;
      let essayWrongCount = Number(row.result?.essay_wrong_count ?? 0) || 0;
      if ((essayCorrectCount + essayWrongCount) === 0 && essayDetails && typeof essayDetails === 'object') {
        const vals = Object.values(essayDetails);
        essayCorrectCount = vals.filter(d => String(d?.status || '').toLowerCase() === 'correct').length;
        essayWrongCount = vals.filter(d => String(d?.status || '').toLowerCase() === 'wrong').length;
      }
      const essayTotal = essayCorrectCount + essayWrongCount;
      const essayScore = essayTotal > 0 ? Math.round((essayCorrectCount / essayTotal) * 10000) / 100 : '-';

      // Hitung soal pilihan (total - essay)
      const totalCount = row.result?.total_count ?? 0;
      const mcqCorrect = Number(row.result?.correct_count ?? 0) || 0;
      const mcqWrong = Number(row.result?.wrong_count ?? 0) || 0;
      const mcqCount = totalCount - essayTotal;
      const mcqScore = mcqCount > 0 ? Math.round((mcqCorrect / mcqCount) * 10000) / 100 : '-';

      const line = [
        row.participant?.nisn || '-',
        row.participant?.nama || '-',
        row.class?.name || '-',
        row.school?.nama || '-',
        row.started_at || '-',
        row.finished_at || '-',
        mcqCorrect,
        mcqWrong,
        mcqCount > 0 ? mcqCount : '-',
        mcqScore,
        correctIdx,
        wrongIdx,
        Array.isArray(row.result?.correct_orders_essays) ? row.result.correct_orders_essays.join('; ') : '-',
        Array.isArray(row.result?.wrong_orders_essay) ? row.result.wrong_orders_essay.join('; ') : '-',
        essayTotal || '-',
        essayScore,
        answersStr,
        (row.result?.essay_details ? Object.entries(row.result.essay_details).map(([id, d]) => {
          const pos = idToPos.get(String(id)) || '?';
          const ans = String(d?.answer || '').trim();
          return `${pos}. ${ans}`;
        }).join(' | ') : ''),
      ].map(csvEscape).join(',');
      lines.push(line);
    });
    const csv = lines.join('\n');
    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    const safeName = String(exam?.name || 'hasil-akhir').trim().toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/^-+|-+$/g, '');
    const filename = `hasil-akhir-${safeName || 'hasil-akhir'}${exam?.code ? '-' + exam.code : ''}-page-${page}.csv`;
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  return (
    <div className="container mt-4">
      <div className="d-flex justify-content-between align-items-center mb-3">
        <h3>Hasil Ujian</h3>
        <div className="d-flex gap-2">
          <Link to="/exams" className="btn btn-outline-secondary">Kembali ke Daftar Ujian</Link>
          <button className="btn btn-outline-success" onClick={exportItemAnalysisCsv} disabled={exportingAnalysis}>
            {exportingAnalysis ? 'Mengekspor...' : 'Export Analisis Butir Soal (CSV)'}
          </button>
        </div>
      </div>

      {exam && (
        <div className="mb-3">
          <div><strong>Ujian:</strong> {exam.name} ({exam.code || '-'})</div>
          <div><strong>Mata Pelajaran:</strong> {exam.subject_name || '-'} ({exam.subject_code || '-'})</div>
          <div><strong>Ringkasan:</strong> Sedang: {inProgress.length} • Selesai: {finished.length} {totals.finished ? `(total: ${totals.finished})` : ''}</div>
          <div className="text-muted small">Halaman ini menampilkan peserta yang sudah selesai sebagai hasil ujian.</div>
        </div>
      )}

      <div className="d-flex flex-wrap gap-2 align-items-center mb-2">
        <input
          type="text"
          className="form-control form-control-sm"
          placeholder="Cari nama/NISN peserta..."
          value={searchTerm}
          onChange={(e) => { setPage(1); setSearchTerm(e.target.value); }}
          style={{ maxWidth: 260 }}
        />
        <select className="form-select form-select-sm" value={perPage} onChange={(e) => { setPage(1); setPerPage(Number(e.target.value)); }} style={{ maxWidth: 160 }}>
          {[20, 50, 100, 200, 500].map(n => (<option key={n} value={n}>Per Halaman: {n}</option>))}
        </select>
        <div className="d-flex align-items-center gap-2">
          <button className="btn btn-sm btn-outline-secondary" disabled={page <= 1} onClick={() => setPage(p => Math.max(1, p - 1))}>« Prev</button>
          <span className="small">Halaman {page} dari {Math.max(1, Math.ceil((serverTotal || 0) / (perPage || 1)))} • Total: {serverTotal}</span>
          <button className="btn btn-sm btn-outline-secondary" disabled={page >= Math.ceil((serverTotal || 0) / (perPage || 1))} onClick={() => setPage(p => p + 1)}>Next »</button>
        </div>
        <button className="btn btn-sm btn-primary" onClick={exportCsv}>Export Hasil Akhir (CSV - halaman ini)</button>
      </div>

      {loading ? (
        <div className="text-muted">Memuat data...</div>
      ) : error ? (
        <div className="alert alert-danger">{error}</div>
      ) : (
        <div className="card">
          <div className="card-header">Peserta Selesai ({serverTotal})</div>
          <div className="card-body p-0">
            <div className="table-responsive">
              <table className="table table-striped table-hover table-sm mb-0">
                <thead>
                  <tr>
                    <th>NISN</th>
                    <th>Nama</th>
                    <th>Kelas</th>
                    <th>Sekolah</th>
                    <th>Mulai</th>
                    <th>Selesai</th>
                    <th>Benar Pilihan</th>
                    <th>Salah Pilihan</th>
                    <th>Total Soal Pilihan</th>
                    <th>Nilai Pilihan</th>
                    <th>No Benar Pilihan</th>
                    <th>No Salah Pilihan</th>
                    <th>No Essay Benar</th>
                    <th>No Essay Salah</th>
                    <th>Total Soal Essay</th>
                    <th>Nilai Essay</th>
                  </tr>
                </thead>
                <tbody>
                  {filteredFinished.map(row => {
                    const totalCount = Number(row.result?.total_count ?? 0) || 0;
                    const essayCorrectCount = Number(row.result?.essay_correct_count ?? 0) || 0;
                    const essayWrongCount = Number(row.result?.essay_wrong_count ?? 0) || 0;
                    const essayTotal = essayCorrectCount + essayWrongCount;
                    const mcqCorrect = Number(row.result?.correct_count ?? 0) || 0;
                    const mcqWrong = Number(row.result?.wrong_count ?? 0) || 0;
                    const mcqCount = totalCount - essayTotal;
                    const mcqScore = mcqCount > 0 ? Math.round((mcqCorrect / mcqCount) * 10000) / 100 : '-';
                    const essayScore = essayTotal > 0 ? Math.round((essayCorrectCount / essayTotal) * 10000) / 100 : '-';
                    return (
                      <tr key={row.session_id}>
                        <td>{row.participant?.nisn || '-'}</td>
                        <td>{row.participant?.nama || '-'}</td>
                        <td>{row.class?.name || '-'}</td>
                        <td>{row.school?.nama || '-'}</td>
                        <td>{row.started_at || '-'}</td>
                        <td>{row.finished_at || '-'}</td>
                        <td>{mcqCorrect}</td>
                        <td>{mcqWrong}</td>
                        <td>{mcqCount > 0 ? mcqCount : '-'}</td>
                        <td>{mcqScore}</td>
                        <td>{Array.isArray(row.result?.correct_order_indexes) ? row.result.correct_order_indexes.join(', ') : '-'}</td>
                        <td>{Array.isArray(row.result?.wrong_order_indexes) ? row.result.wrong_order_indexes.join(', ') : '-'}</td>
                        <td>{Array.isArray(row.result?.correct_orders_essays) ? row.result.correct_orders_essays.join(', ') : '-'}</td>
                        <td>{Array.isArray(row.result?.wrong_orders_essay) ? row.result.wrong_orders_essay.join(', ') : '-'}</td>
                        <td>{essayTotal || '-'}</td>
                        <td>{essayScore}</td>
                      </tr>
                    );
                  })}
                </tbody>
              </table>
            </div>
            <small className="text-muted">Data ditampilkan dengan pagination lokal dari `/exams/{examId}/participants-status`.</small>
          </div>
        </div>
      )}
    </div>
  );
}