import React, { useEffect, useMemo, useState } from "react";
import { useAuth } from "../context/AuthContext.jsx";
import api from "../utils/axios";

export default function Dashboard() {
  const { user } = useAuth();
  const [stats, setStats] = useState(null);
  const [loading, setLoading] = useState(true);
  const [loadingToday, setLoadingToday] = useState(true);
  const [todayExams, setTodayExams] = useState([]);
  const [participants, setParticipants] = useState([]);
  const [classesAll, setClassesAll] = useState([]);
  const [schools, setSchools] = useState([]);
  const [grades, setGrades] = useState([]);
  const [subjects, setSubjects] = useState([]);
const [selectedSchoolId, setSelectedSchoolId] = useState('');
  const [selectedGradeId, setSelectedGradeId] = useState('');

  const fetchStats = async () => {
    setLoading(true);
    try {
      const res = await api.get('/dashboard/stats');
      setStats(res.data);
    } catch (e) {
      console.error('Gagal mengambil statistik dashboard', e);
    } finally {
      setLoading(false);
    }
  };

  const fetchTodayData = async () => {
    setLoadingToday(true);
    try {
      const examsRes = await api.get('/exams/today');
      setTodayExams(Array.isArray(examsRes.data) ? examsRes.data : []);
      // Endpoint lain mungkin membutuhkan login; bila gagal, biarkan kosong agar UI tetap jalan
      try {
        const partRes = await api.get('/exam-participants');
        setParticipants(Array.isArray(partRes.data) ? partRes.data : []);
      } catch {}
      try {
        const classesRes = await api.get('/classes');
        setClassesAll(Array.isArray(classesRes.data) ? classesRes.data : []);
      } catch {}
      try {
        const schoolsRes = await api.get('/schools');
        setSchools(Array.isArray(schoolsRes.data) ? schoolsRes.data : []);
      } catch {}
      try {
        const gradesRes = await api.get('/grades');
        setGrades(Array.isArray(gradesRes.data) ? gradesRes.data : []);
      } catch {}
      try {
        const subjectsRes = await api.get('/subjects');
        setSubjects(Array.isArray(subjectsRes.data) ? subjectsRes.data : []);
      } catch {}
    } catch (e) {
      console.error('Gagal mengambil data ujian hari ini', e);
    } finally {
      setLoadingToday(false);
    }
  };

  const refreshAll = async () => {
    await Promise.all([fetchStats(), fetchTodayData()]);
  };

  useEffect(() => {
    fetchStats();
    fetchTodayData();
  }, []);

  const getSchoolNameById = (id) => {
    if (!id) return '-';
    const s = schools.find(ss => ss.id === id);
    return s?.nama || '-';
  };

  const getGradeNameById = (id) => {
    if (!id) return '-';
    const g = grades.find(gg => gg.id === id);
    return (g?.grade ?? '-') + (g ? '' : '');
  };

  const getSubjectNameById = (id) => {
    if (!id) return '-';
    const subj = subjects.find(s => s.id === id);
    return subj?.name || '-';
  };

  const todaySummary = useMemo(() => {
    if (!Array.isArray(todayExams) || todayExams.length === 0) return [];
    return todayExams.map(ex => {
      const schoolId = ex.exam_school_id || ex.subject_school_id;
      const gradeId = ex.exam_grade_id || ex.subject_grade_id;
      const schoolName = getSchoolNameById(schoolId);
      const gradeName = getGradeNameById(gradeId);
      const subjectName = getSubjectNameById(ex.id_subject);
      const thresholdMs = 15 * 60 * 1000;
      const nowMs = Date.now();
      const matched = participants.filter(p => {
        const c = classesAll.find(x => x.id === p.id_kelas);
        if (!c) return false;
        const matchSchool = schoolId ? c.id_school === schoolId : true;
        const matchGrade = gradeId ? c.id_grade === gradeId : true;
        return matchSchool && matchGrade;
      });
      const onlineCount = matched.filter(p => p.last_activity && (nowMs - new Date(p.last_activity).getTime()) <= thresholdMs).length;
      return {
        examId: ex.id,
        name: ex.name,
        code: ex.code,
        schoolName,
        gradeName,
        subjectName,
        subject_school_id: ex.subject_school_id || null,
        subject_grade_id: ex.subject_grade_id || null,
        exam_school_id: ex.exam_school_id || null,
        exam_grade_id: ex.exam_grade_id || null,
        count: matched.length,
        onlineCount,
        inProgress: ex?.in_progress_count ?? 0,
        finished: ex?.finished_count ?? 0,
      };
    });
  }, [todayExams, participants, classesAll, schools, grades, subjects]);

  const gradesBySchool = useMemo(() => {
    if (!selectedSchoolId) return grades;
    return grades.filter(g => String(g.id_school) === String(selectedSchoolId));
  }, [grades, selectedSchoolId]);

  const todaySummaryFiltered = useMemo(() => {
    let list = todaySummary;
    if (selectedSchoolId) {
      list = list.filter(row => {
        const schoolKey = row.exam_school_id || row.subject_school_id || '';
        return String(schoolKey) === String(selectedSchoolId);
      });
    }
    if (selectedGradeId) {
      list = list.filter(row => {
        const gradeKey = row.exam_grade_id || row.subject_grade_id || '';
        return String(gradeKey) === String(selectedGradeId);
      });
    }
    return list;
  }, [todaySummary, selectedSchoolId, selectedGradeId]);
  // (Bagian peserta online dihapus sesuai permintaan)

  const Card = ({ title, online = 0, total = 0, color = '#0d6efd' }) => (
    <div className="card text-white" style={{ backgroundColor: color, minWidth: 240 }}>
      <div className="card-body">
        <h5 className="card-title">{title}</h5>
        <p className="card-text" style={{ fontSize: 20 }}>
          <strong>{online}</strong> / {total}
        </p>
        <small>Online / Total</small>
      </div>
    </div>
  );

  return (
    <div className="container mt-5">
      <div className="d-flex justify-content-between align-items-center mb-3">
        <h2 className="mb-0">Dashboard</h2>
        <button className="btn btn-outline-secondary btn-sm" onClick={refreshAll} disabled={loading || loadingToday}>
          {loading || loadingToday ? 'Memuat...' : 'Refresh'}
        </button>
      </div>
      {user && (
        <div className="mb-3">
          <p>Selamat datang, <strong>{user.name}</strong>!</p>
          <p>Role Anda: {user.roles && user.roles.length > 0 ? user.roles.map(role => role.name).join(', ') : 'Tidak ada role'}</p>
        </div>
      )}

      {loading ? (
        <div className="text-center">Memuat statistik...</div>
      ) : (
        stats ? (
          <div className="d-flex gap-3 flex-wrap">
            <Card title="Pegawai" online={stats?.pegawai?.online} total={stats?.pegawai?.total} color="#6f42c1" />
            <Card title="Guru" online={stats?.guru?.online} total={stats?.guru?.total} color="#198754" />
            <Card title="Murid" online={stats?.murid?.online} total={stats?.murid?.total} color="#0d6efd" />
            <Card title="Peserta Ujian" online={stats?.peserta?.online} total={stats?.peserta?.total} color="#fd7e14" />
          </div>
        ) : (
          <div className="text-muted">Statistik memerlukan login.</div>
        )
      )}

      {/* (Tabel peserta online dihapus) */}

      {/* Ringkasan Ujian Hari Ini */}
      <div className="mt-4">
        <h5>Ujian Hari Ini</h5>
        <div className="d-flex gap-2 align-items-end mb-2">
          <div className="input-group input-group-sm" style={{ maxWidth: '240px' }}>
            <span className="input-group-text">Sekolah</span>
            <select className="form-select" value={selectedSchoolId} onChange={e => { setSelectedSchoolId(e.target.value); setSelectedGradeId(''); }}>
              <option value="">Semua</option>
              {schools.map(s => (
                <option key={s.id} value={s.id}>{s.nama}</option>
              ))}
            </select>
          </div>
          <div className="input-group input-group-sm" style={{ maxWidth: '220px' }}>
            <span className="input-group-text">Grade</span>
            <select className="form-select" value={selectedGradeId} onChange={e => setSelectedGradeId(e.target.value)}>
              <option value="">Semua</option>
              {gradesBySchool.map(g => (
                <option key={g.id} value={g.id}>{g.grade}</option>
              ))}
            </select>
          </div>
        </div>
        {loadingToday ? (
          <div className="text-muted">Memuat data ujian hari ini...</div>
        ) : todaySummary.length === 0 ? (
          <div className="text-muted">Tidak ada ujian yang dijadwalkan hari ini.</div>
        ) : (
          <div className="card">
            <div className="card-body">
              <div className="table-responsive">
                <table className="table table-sm">
                  <thead>
                    <tr>
                      <th>Nama Ujian</th>
                      <th>Kode</th>
                      <th>Sekolah</th>
                      <th>Grade</th>
                      <th>Mata Pelajaran</th>
                      <th>Sedang</th>
                      <th>Selesai</th>
                      <th>Online / Total</th>
                    </tr>
                  </thead>
                  <tbody>
                    {todaySummaryFiltered.map(row => (
                      <tr key={row.examId}>
                        <td>{row.name}</td>
                        <td>{row.code}</td>
                        <td>{row.schoolName}</td>
                        <td>{row.gradeName}</td>
                        <td>{row.subjectName}</td>
                        <td>{row.inProgress}</td>
                        <td>{row.finished}</td>
                        <td>
                          {participants.length > 0
                            ? (<><strong>{row.onlineCount}</strong> / {row.count}</>)
                            : ('-')}
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
              <small className="text-muted">Data dihitung dari peserta dan kelas yang cocok dengan sekolah/grade pada subject ujian.</small>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}
