import React, { useEffect, useMemo, useState } from 'react';
import DataTable from 'react-data-table-component';
import Swal from 'sweetalert2';
import { FaPlus, FaEdit, FaTrash, FaSearch } from 'react-icons/fa';
import api from '../utils/axios';
import { useAuth } from '../context/AuthContext.jsx';
import { canAccess } from '../utils/permissionHelper';

export default function Classes() {
  const { user } = useAuth();
  const [classes, setClasses] = useState([]);
  const [schools, setSchools] = useState([]);
  const [gradesFilter, setGradesFilter] = useState([]);
  const [gradesForm, setGradesForm] = useState([]);
  const [loading, setLoading] = useState(true);
  const [filterText, setFilterText] = useState('');
  const [selectedSchoolId, setSelectedSchoolId] = useState('');
  const [selectedGradeId, setSelectedGradeId] = useState('');
  const [showAddModal, setShowAddModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [form, setForm] = useState({ id_school: '', id_grade: '', name: '' });
  const [editing, setEditing] = useState(null);

  const tokenHeader = { headers: { Authorization: `Bearer ${localStorage.getItem('token')}` } };

  const canViewClass = canAccess(user, 'view-class') || canAccess(user, 'manage-classes');
  const canCreateClass = canAccess(user, 'create-class') || canAccess(user, 'manage-classes');
  const canEditClass = canAccess(user, 'edit-class') || canAccess(user, 'manage-classes');
  const canDeleteClass = canAccess(user, 'delete-class') || canAccess(user, 'manage-classes');
  const canViewSchools = canAccess(user, 'view-school') || canAccess(user, 'manage-schools');
  const canViewGrades = canAccess(user, 'view-grade') || canAccess(user, 'manage-grades');

  const fetchSchools = async () => {
    try {
      const res = await api.get('/schools', tokenHeader);
      setSchools(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk melihat daftar sekolah.' });
      } else {
        Swal.fire('Error', 'Gagal memuat daftar sekolah', 'error');
      }
    }
  };

  const fetchGradesForFilter = async (schoolId = '') => {
    try {
      const url = schoolId ? `/grades?id_school=${schoolId}` : '/grades';
      const res = await api.get(url, tokenHeader);
      setGradesFilter(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk melihat daftar grade.' });
      } else {
        Swal.fire('Error', 'Gagal memuat daftar grade', 'error');
      }
    }
  };

  const fetchGradesForForm = async (schoolId = '') => {
    try {
      if (!schoolId) { setGradesForm([]); return; }
      const url = `/grades?id_school=${schoolId}`;
      const res = await api.get(url, tokenHeader);
      setGradesForm(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk memuat grade untuk form.' });
      } else {
        Swal.fire('Error', 'Gagal memuat grade untuk form', 'error');
      }
    }
  };

  const fetchClasses = async (schoolId = '', gradeId = '') => {
    setLoading(true);
    try {
      const params = [];
      if (schoolId) params.push(`id_school=${schoolId}`);
      if (gradeId) params.push(`id_grade=${gradeId}`);
      const url = params.length ? `/classes?${params.join('&')}` : '/classes';
      const res = await api.get(url, tokenHeader);
      setClasses(Array.isArray(res.data) ? res.data : []);
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk melihat daftar kelas.' });
      } else {
        Swal.fire('Error', 'Gagal memuat daftar kelas', 'error');
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (canViewSchools) fetchSchools();
    if (canViewGrades) fetchGradesForFilter('');
    if (canViewClass) {
      fetchClasses();
    } else {
      setLoading(false);
    }
  }, [canViewSchools, canViewGrades, canViewClass]);

  // Re-fetch grade list and classes when school filter changes
  useEffect(() => {
    setSelectedGradeId('');
    if (canViewGrades) fetchGradesForFilter(selectedSchoolId || '');
    if (canViewClass) fetchClasses(selectedSchoolId || '', '');
  }, [selectedSchoolId, canViewGrades, canViewClass]);

  // Re-fetch classes when grade filter changes
  useEffect(() => {
    if (canViewClass) fetchClasses(selectedSchoolId || '', selectedGradeId || '');
  }, [selectedGradeId, canViewClass]);

  // For modals: fetch grades when form school changes
  useEffect(() => {
    if (canViewGrades) fetchGradesForForm(form.id_school || '');
    if (!form.id_school) setForm(prev => ({ ...prev, id_grade: '' }));
  }, [form.id_school, canViewGrades]);

  const getSchoolName = (id) => schools.find(s => s.id === id)?.nama || '-';
  const getGradeNumber = (id) => gradesFilter.find(g => g.id === id)?.grade
    || gradesForm.find(g => g.id === id)?.grade
    || '-';

  const filtered = useMemo(() => {
    if (!filterText) return classes;
    const q = filterText.toLowerCase();
    return classes.filter(c => (
      (c.name || '').toLowerCase().includes(q) ||
      getSchoolName(c.id_school).toLowerCase().includes(q) ||
      String(getGradeNumber(c.id_grade)).toLowerCase().includes(q)
    ));
  }, [filterText, classes, schools, gradesFilter, gradesForm]);

  const openAdd = () => {
    setForm({ id_school: '', id_grade: '', name: '' });
    setGradesForm([]);
    setShowAddModal(true);
  };

  const openEdit = (row) => {
    setEditing(row);
    setForm({ id_school: row.id_school ?? '', id_grade: row.id_grade ?? '', name: row.name ?? '' });
    setShowEditModal(true);
  };

  const handleAdd = async () => {
    if (!String(form.id_school).trim() || !String(form.id_grade).trim() || !String(form.name).trim()) return;
    try {
      await api.post('/classes', { id_school: Number(form.id_school), id_grade: Number(form.id_grade), name: form.name }, tokenHeader);
      setShowAddModal(false);
      await fetchClasses(selectedSchoolId || '', selectedGradeId || '');
      Swal.fire('Berhasil', 'Kelas ditambahkan', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk menambah kelas.' });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Gagal menambah kelas', 'error');
      }
    }
  };

  const handleEditSave = async () => {
    if (!editing) return;
    if (!String(form.id_school).trim() || !String(form.id_grade).trim() || !String(form.name).trim()) return;
    try {
      await api.put(`/classes/${editing.id}`, { id_school: Number(form.id_school), id_grade: Number(form.id_grade), name: form.name }, tokenHeader);
      setShowEditModal(false);
      setEditing(null);
      await fetchClasses(selectedSchoolId || '', selectedGradeId || '');
      Swal.fire('Berhasil', 'Kelas diperbarui', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk memperbarui kelas.' });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Gagal memperbarui kelas', 'error');
      }
    }
  };

  const handleDelete = async (row) => {
    const ok = await Swal.fire({
      title: 'Hapus kelas?',
      text: `Sekolah: ${getSchoolName(row.id_school)} | Grade: ${getGradeNumber(row.id_grade)}`,
      icon: 'warning',
      showCancelButton: true,
      confirmButtonText: 'Ya, hapus',
      cancelButtonText: 'Batal'
    }).then(r => r.isConfirmed);
    if (!ok) return;
    try {
      await api.delete(`/classes/${row.id}`, tokenHeader);
      await fetchClasses(selectedSchoolId || '', selectedGradeId || '');
      Swal.fire('Berhasil', 'Kelas dihapus', 'success');
    } catch (err) {
      console.error(err);
      const status = err?.response?.status;
      if (status === 403) {
        Swal.fire({ icon: 'warning', title: 'Akses Ditolak', text: 'Anda tidak memiliki izin untuk menghapus kelas.' });
      } else {
        Swal.fire('Gagal', err.response?.data?.message || 'Gagal menghapus kelas', 'error');
      }
    }
  };

  const columns = [
    { name: '#', selector: (row, index) => index + 1, width: '60px' },
    { name: 'Nama Kelas', selector: row => row.name || '-', sortable: true, wrap: true },
    { name: 'Sekolah', selector: row => getSchoolName(row.id_school), sortable: true, wrap: true },
    { name: 'Grade', selector: row => getGradeNumber(row.id_grade), sortable: true },
    {
      name: 'Aksi',
      cell: row => (
        <div className="d-flex gap-2 flex-wrap">
          {canEditClass && (
            <button className="btn btn-sm btn-primary" title="Edit" onClick={() => openEdit(row)}>
              <FaEdit />
            </button>
          )}
          {canDeleteClass && (
            <button className="btn btn-sm btn-danger" title="Hapus" onClick={() => handleDelete(row)}>
              <FaTrash />
            </button>
          )}
        </div>
      ),
      width: '160px'
    }
  ];
  if (!canViewClass) {
    return (
      <div className="container mt-4">
        <h3>Akses Ditolak</h3>
        <p>Anda tidak memiliki izin untuk melihat daftar kelas.</p>
      </div>
    );
  }
  if (loading) return <div className="text-center mt-5">Memuat kelas...</div>;

  return (
    <div className="container mt-4">
      <h3>Daftar Kelas</h3>

      <div className="d-flex align-items-center gap-2 mb-3 flex-wrap">
        {canCreateClass && (
          <button className="btn btn-primary btn-sm" onClick={openAdd}>
            <FaPlus className="me-1" /> Tambah
          </button>
        )}

        <div className="input-group input-group-sm" style={{ maxWidth: '280px' }}>
          <span className="input-group-text">Sekolah</span>
          <select
            className="form-select"
            value={selectedSchoolId}
            onChange={e => setSelectedSchoolId(e.target.value)}
          >
            <option value="">Semua</option>
            {schools.map(s => (
              <option key={s.id} value={s.id}>{s.nama}</option>
            ))}
          </select>
        </div>

        <div className="input-group input-group-sm" style={{ maxWidth: '220px' }}>
          <span className="input-group-text">Grade</span>
          <select
            className="form-select"
            value={selectedGradeId}
            onChange={e => setSelectedGradeId(e.target.value)}
          >
            <option value="">Semua</option>
            {gradesFilter.map(g => (
              <option key={g.id} value={g.id}>{g.grade}</option>
            ))}
          </select>
        </div>

        <div className="input-group input-group-sm" style={{ maxWidth: '300px' }}>
          <span className="input-group-text"><FaSearch /></span>
          <input
            type="text"
            placeholder="Cari nama/sekolah/grade..."
            className="form-control"
            value={filterText}
            onChange={e => setFilterText(e.target.value)}
          />
        </div>
      </div>

      <DataTable
        columns={columns}
        data={filtered}
        pagination
        highlightOnHover
        striped
        noHeader
      />

      {/* Modal Add */}
      {showAddModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Tambah Kelas</h5>
                <button type="button" className="btn-close" onClick={() => setShowAddModal(false)}></button>
              </div>
              <div className="modal-body">
                <select
                  className="form-select mb-2"
                  value={form.id_school}
                  onChange={e => setForm({ ...form, id_school: e.target.value })}
                >
                  <option value="">Pilih Sekolah</option>
                  {schools.map(s => (
                    <option key={s.id} value={s.id}>{s.nama}</option>
                  ))}
                </select>
                <select
                  className="form-select"
                  value={form.id_grade}
                  onChange={e => setForm({ ...form, id_grade: e.target.value })}
                  disabled={!form.id_school}
                >
                  <option value="">Pilih Grade</option>
                  {gradesForm.map(g => (
                    <option key={g.id} value={g.id}>{g.grade}</option>
                  ))}
                </select>
                <input
                  type="text"
                  className="form-control mt-2"
                  placeholder="Nama Kelas"
                  value={form.name}
                  onChange={e => setForm({ ...form, name: e.target.value })}
                />
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowAddModal(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleAdd}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Modal Edit */}
      {showEditModal && (
        <div className="modal show d-block" tabIndex="-1">
          <div className="modal-dialog">
            <div className="modal-content">
              <div className="modal-header">
                <h5 className="modal-title">Edit Kelas</h5>
                <button type="button" className="btn-close" onClick={() => setShowEditModal(false)}></button>
              </div>
              <div className="modal-body">
                <select
                  className="form-select mb-2"
                  value={form.id_school}
                  onChange={e => setForm({ ...form, id_school: e.target.value })}
                >
                  <option value="">Pilih Sekolah</option>
                  {schools.map(s => (
                    <option key={s.id} value={s.id}>{s.nama}</option>
                  ))}
                </select>
                <select
                  className="form-select"
                  value={form.id_grade}
                  onChange={e => setForm({ ...form, id_grade: e.target.value })}
                  disabled={!form.id_school}
                >
                  <option value="">Pilih Grade</option>
                  {gradesForm.map(g => (
                    <option key={g.id} value={g.id}>{g.grade}</option>
                  ))}
                </select>
                <input
                  type="text"
                  className="form-control mt-2"
                  placeholder="Nama Kelas"
                  value={form.name}
                  onChange={e => setForm({ ...form, name: e.target.value })}
                />
              </div>
              <div className="modal-footer">
                <button className="btn btn-secondary" onClick={() => setShowEditModal(false)}>Batal</button>
                <button className="btn btn-primary" onClick={handleEditSave}>Simpan</button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}