import React, { useEffect, useMemo, useState, useRef } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import api from '../utils/axios';
import Swal from 'sweetalert2';
import html2pdf from 'html2pdf.js';
import PrintableReport from '../components/PrintableReport';
import { useAuth } from '../context/AuthContext.jsx';

export default function ApproveExam() {
  const params = useParams();
  const navigate = useNavigate();
  const initialExamId = params.examId ? String(params.examId) : '';

  const [exams, setExams] = useState([]);
  const [selectedExamId, setSelectedExamId] = useState(initialExamId);
  const [schools, setSchools] = useState([]);
  const [classes, setClasses] = useState([]);
  const [selectedSchoolId, setSelectedSchoolId] = useState('');
  const [selectedClassId, setSelectedClassId] = useState('');
  const [search, setSearch] = useState('');
  const [perPage, setPerPage] = useState(10);
  const [page, setPage] = useState(1);
  const [loading, setLoading] = useState(false);
  const [rowsAll, setRowsAll] = useState([]);
  const [total, setTotal] = useState(0);
  const [lastPage, setLastPage] = useState(1);
  const [schoolData, setSchoolData] = useState(null);
  const [examData, setExamData] = useState(null);
  const [approverNamesById, setApproverNamesById] = useState({});
  const printRef = useRef();
  const { hasRole } = useAuth();
  const canDownloadPdf = !!(hasRole && (hasRole('superadmin') || hasRole('pegawai')));

  useEffect(() => {
    const fetchExams = async () => {
      try {
        const res = await api.get('/exams');
        const list = Array.isArray(res.data?.data) ? res.data.data : (Array.isArray(res.data) ? res.data : []);
        const todayStr = (() => {
          const d = new Date();
          const yyyy = d.getFullYear();
          const mm = String(d.getMonth() + 1).padStart(2, '0');
          const dd = String(d.getDate()).padStart(2, '0');
          return `${yyyy}-${mm}-${dd}`;
        })();
        const getYMD = (cand) => {
          if (!cand) return '';
          if (typeof cand === 'string') {
            // Normalize common formats: YYYY-MM-DD, YYYY-MM-DD HH:MM:SS, DD-MM-YYYY, YYYY/MM/DD
            const s = cand.trim();
            const m1 = s.match(/^(\d{4})[-\/](\d{2})[-\/](\d{2})/); // YYYY-MM-DD or YYYY/MM/DD
            if (m1) return `${m1[1]}-${m1[2]}-${m1[3]}`;
            const m2 = s.match(/^(\d{2})-(\d{2})-(\d{4})/); // DD-MM-YYYY
            if (m2) return `${m2[3]}-${m2[2]}-${m2[1]}`;
            // Try Date parsing
            const d = new Date(s);
            if (!isNaN(d.getTime())) {
              const yyyy = d.getFullYear();
              const mm = String(d.getMonth() + 1).padStart(2, '0');
              const dd = String(d.getDate()).padStart(2, '0');
              return `${yyyy}-${mm}-${dd}`;
            }
            return '';
          } else if (cand instanceof Date) {
            const yyyy = cand.getFullYear();
            const mm = String(cand.getMonth() + 1).padStart(2, '0');
            const dd = String(cand.getDate()).padStart(2, '0');
            return `${yyyy}-${mm}-${dd}`;
          }
          return '';
        };
        const isToday = (ex) => {
          // Gunakan scheduled_at sebagai sumber kebenaran (fallback ke schaduled_at jika ejaan berbeda)
          const cand = ex?.scheduled_at || ex?.schaduled_at || ex?.tanggal_ujian || ex?.tanggal || ex?.date || ex?.exam_date;
          const ymd = getYMD(cand);
          return ymd === todayStr;
        };
        const filtered = list.filter(isToday);
        setExams(filtered);
        // Jika kosong, tampilkan info dan jangan kosongkan pilihan secara tiba-tiba
        if (filtered.length === 0) {
          Swal.fire('Info', 'Tidak ada ujian untuk hari ini.', 'info');
        } else {
          // Pastikan ada ujian terpilih yang valid
          const exists = filtered.some(ex => String(ex.id) === String(selectedExamId));
          if (!exists) setSelectedExamId(String(filtered[0]?.id || ''));
        }
      } catch (err) {
        const status = err?.response?.status;
        if (status === 403) {
          Swal.fire('Akses Ditolak', 'Anda tidak memiliki izin untuk melihat daftar ujian.', 'error');
        }
      }
    };
    fetchExams();
  }, [initialExamId]);

  useEffect(() => {
    const fetchSchools = async () => {
      try {
        const res = await api.get('/schools');
        setSchools(Array.isArray(res.data) ? res.data : []);
      } catch (err) { /* ignore */ }
    };
    const fetchClasses = async (schoolId = '') => {
      try {
        const url = schoolId ? `/classes?id_school=${schoolId}` : '/classes';
        const res = await api.get(url);
        setClasses(Array.isArray(res.data) ? res.data : []);
      } catch (err) { /* ignore */ }
    };
    fetchSchools();
    // Jangan muat kelas sebelum memilih ujian/sekola — akan dimuat saat sekolah dipilih
    if (selectedSchoolId) {
      fetchClasses(selectedSchoolId || '');
    } else {
      setClasses([]);
    }
  }, [selectedSchoolId]);

  const fetchApprovals = async () => {
    if (!selectedExamId || !selectedSchoolId || !selectedClassId) {
      // Harus pilih ujian, sekolah, dan kelas terlebih dahulu
      setRowsAll([]);
      setTotal(0);
      setLastPage(1);
      return;
    }
    setLoading(true);
    try {
      // Ambil semua halaman (client-side pagination seperti Subjects)
      const acc = [];
      const pageSize = 100; // batas maksimum backend
      let pageLocal = 1;
      let lastPageLocal = 1;
      do {
        const params = new URLSearchParams();
        params.set('per_page', String(pageSize));
        params.set('page', String(pageLocal));
        if (search) params.set('search', search);
        if (selectedSchoolId) params.set('school_id', String(selectedSchoolId));
        if (selectedClassId) params.set('class_id', String(selectedClassId));
        const url = `/exams/${selectedExamId}/approvals?` + params.toString();
        const res = await api.get(url);
        const d = res.data || {};
        const list = Array.isArray(d?.data) ? d.data : [];
        acc.push(...list);
        lastPageLocal = Number(d?.last_page || 1);
        pageLocal += 1;
      } while (pageLocal <= lastPageLocal && pageLocal < 200);

      setRowsAll(acc);
      const totalLocal = acc.length;
      setTotal(totalLocal);
      setLastPage(Math.max(1, Math.ceil(totalLocal / perPage)));
    } catch (err) {
      const status = err?.response?.status;
      const msg = err?.response?.data?.message || 'Gagal memuat daftar approval';
      if (status === 403) {
        Swal.fire('Akses Ditolak', 'Anda tidak memiliki izin untuk melihat approval ujian.', 'error');
      } else {
        Swal.fire('Error', msg, 'error');
      }
    } finally {
      setLoading(false);
    }
  };

  // Untuk client-side pagination: fetch ulang hanya saat filter berubah
  useEffect(() => { fetchApprovals(); }, [selectedExamId, search, selectedSchoolId, selectedClassId]);

  // Selaraskan lastPage ketika total/perPage berubah, dan normalkan page agar tidak melebihi lastPage
  useEffect(() => {
    const lp = Math.max(1, Math.ceil((total || 0) / (perPage || 10)));
    setLastPage(lp);
    setPage((p) => Math.min(p, lp));
  }, [total, perPage]);

  const approveOne = async (participantId) => {
    if (!selectedExamId || !participantId) return;
    try {
      await api.post(`/exams/${selectedExamId}/approve/${participantId}`);
      Swal.fire('Berhasil', 'Peserta di-approve.', 'success');
      // Refresh row
      fetchApprovals();
    } catch (err) {
      const status = err?.response?.status;
      const msg = err?.response?.data?.message || 'Gagal meng-approve peserta';
      if (status === 403) {
        Swal.fire('Akses Ditolak', 'Anda tidak memiliki izin untuk approve peserta.', 'error');
      } else if (status === 422) {
        Swal.fire('Validasi', msg, 'warning');
      } else {
        Swal.fire('Error', msg, 'error');
      }
    }
  };

  const onlyTime = (val) => {
    if (!val) return '-';
    if (typeof val === 'string') {
      const m = val.match(/(\d{2}:\d{2}:\d{2})/);
      if (m) return m[1];
      const parts = val.split(' ');
      if (parts[1] && /\d{2}:\d{2}:\d{2}/.test(parts[1])) return parts[1].slice(0, 8);
    }
    try {
      const d = new Date(val);
      const hh = String(d.getHours()).padStart(2, '0');
      const mm = String(d.getMinutes()).padStart(2, '0');
      const ss = String(d.getSeconds()).padStart(2, '0');
      return `${hh}:${mm}:${ss}`;
    } catch {
      return '-';
    }
  };

  const fetchSchoolData = async (schoolId) => {
    try {
      if (!schoolId) return;
      const res = await api.get(`/schools/${schoolId}`);
      setSchoolData(res.data);
    } catch (err) {
      console.error('Failed to fetch school data:', err);
    }
  };

  const downloadPDF = async () => {
    if (!selectedExamId || !selectedSchoolId) {
      Swal.fire('Peringatan', 'Silakan pilih Ujian dan Sekolah terlebih dahulu.', 'warning');
      return;
    }
    if (!canDownloadPdf) {
      Swal.fire('Akses Ditolak', 'Fitur download PDF hanya untuk Superadmin atau Pegawai.', 'error');
      return;
    }
    if (rows.length === 0) {
      Swal.fire('Peringatan', 'Tidak ada data peserta untuk diunduh.', 'warning');
      return;
    }

    try {
      await fetchSchoolData(selectedSchoolId);

      // Get exam data
      const currentExam = exams.find(ex => String(ex.id) === String(selectedExamId));
      setExamData(currentExam);

      // Prefetch approver names by approved_by_user_id
      const uniqueIds = Array.from(new Set((sortedRows || [])
        .map(r => r?.session?.approved_by_user_id)
        .filter(Boolean)));
      if (uniqueIds.length > 0) {
        const nameMap = {};
        await Promise.all(uniqueIds.map(async (uid) => {
          try {
            const res = await api.get(`/users/${uid}`);
            const u = res.data || {};
            const nm = u?.name || u?.nama || u?.user?.name || String(uid);
            nameMap[String(uid)] = nm;
          } catch {
            nameMap[String(uid)] = String(uid);
          }
        }));
        setApproverNamesById(nameMap);
      } else {
        setApproverNamesById({});
      }

      // Wait a bit for state to update
      setTimeout(() => {
        const element = printRef.current;
        if (!element) return;

        const opt = {
          margin: 0.5,
          filename: `daftar-peserta-${currentExam?.name || 'ujian'}-${new Date().toISOString().split('T')[0]}.pdf`,
          image: { type: 'jpeg', quality: 0.98 },
          html2canvas: { 
            scale: 2,
            useCORS: true,
            allowTaint: true,
            backgroundColor: '#ffffff'
          },
          jsPDF: { 
            unit: 'in', 
            format: 'a4', 
            orientation: 'portrait' 
          }
        };

        html2pdf().set(opt).from(element).save();
      }, 100);

    } catch (err) {
      console.error('PDF generation error:', err);
      Swal.fire('Error', 'Gagal membuat PDF. Silakan coba lagi.', 'error');
    }
  };

  const onExamChange = (e) => {
    const id = e.target.value;
    setSelectedExamId(id);
    setPage(1);
    // Navigate ke deep link untuk konsistensi
    navigate(`/exams/${id}/approve`, { replace: true });
  };

  const selectedExam = useMemo(() => exams.find(ex => String(ex.id) === String(selectedExamId)), [exams, selectedExamId]);

  // Urutkan data: per kelas, lalu nama A-Z
  const sortedRows = useMemo(() => {
    const arr = Array.isArray(rowsAll) ? [...rowsAll] : [];
    const getClassName = (r) => (r?.class?.name || '').toString().toLowerCase();
    const getName = (r) => (
      r?.user?.name || r?.participant?.name || r?.participant?.nama || r?.name || ''
    ).toString().toLowerCase();
    arr.sort((a, b) => {
      const ac = getClassName(a), bc = getClassName(b);
      if (ac < bc) return -1; if (ac > bc) return 1;
      const an = getName(a), bn = getName(b);
      if (an < bn) return -1; if (an > bn) return 1; return 0;
    });
    return arr;
  }, [rowsAll]);

  const paginatedRows = useMemo(() => {
    const start = (page - 1) * perPage;
    const end = start + perPage;
    return sortedRows.slice(start, end);
  }, [sortedRows, page, perPage]);

  return (
    <div className="container">
      <div className="d-flex align-items-center justify-content-between mb-3">
        <h4 className="mb-0">Approve Ujian</h4>
        {selectedExamId && selectedSchoolId && selectedClassId && sortedRows.length > 0 && canDownloadPdf && (
          <button className="btn btn-success" onClick={downloadPDF}>
            <i className="fas fa-download me-2"></i>Download PDF
          </button>
        )}
      </div>

      <div className="card mb-3">
        <div className="card-body">
          {/* Baris 1: pilih ujian */}
          <div className="row g-3 align-items-end">
            <div className="col-md-6 col-lg-4">
              <label className="form-label">Pilih Ujian</label>
              <select className="form-select" value={selectedExamId || ''} onChange={onExamChange}>
                <option value="">-- pilih ujian --</option>
                {exams.map(ex => (
                  <option key={ex.id} value={ex.id}>
                    {ex.name} ({ex.code})
                  </option>
                ))}
              </select>
            </div>
          </div>

          {/* Baris 2: filter lain hanya tampil setelah memilih ujian */}
          {selectedExamId && (
            <div className="row g-3 align-items-end mt-1">
              <div className="col-md-3">
                <label className="form-label">Sekolah</label>
                <select className="form-select" value={selectedSchoolId || ''} onChange={(e) => { setSelectedSchoolId(e.target.value); setSelectedClassId(''); setPage(1); }}>
                  <option value="">-- pilih sekolah --</option>
                  {schools.map(s => (<option key={s.id} value={s.id}>{s.nama}</option>))}
                </select>
              </div>
              <div className="col-md-2">
                <label className="form-label">Kelas</label>
                <select className="form-select" value={selectedClassId || ''} onChange={(e) => { setSelectedClassId(e.target.value); setPage(1); }}>
                  <option value="">-- pilih kelas --</option>
                  {(classes || [])
                    .filter(c => !selectedSchoolId || String(c.id_school) === String(selectedSchoolId))
                    .map(c => (<option key={c.id} value={c.id}>{c.name}</option>))}
                </select>
              </div>
              <div className="col-md-2">
                <label className="form-label">Per Halaman</label>
                <select className="form-select" value={perPage} onChange={(e) => { setPerPage(Number(e.target.value)); setPage(1); }}>
                  {[10,20,50,100].map(n => (<option key={n} value={n}>{n}</option>))}
                  {total > 0 && ![10,20,50,100].includes(total) && (
                    <option value={total}>Semua</option>
                  )}
                </select>
              </div>
              <div className="col-md-5">
                <label className="form-label">Pencarian (Nama/NISN)</label>
                <input className="form-control" placeholder="Cari peserta..." value={search} onChange={(e) => { setSearch(e.target.value); setPage(1); }} />
              </div>
            </div>
          )}
        </div>
      </div>

      {(selectedExamId && selectedSchoolId && selectedClassId) ? (
      <div className="card">
        <div className="card-body">
          {loading ? (
            <div className="text-center my-4">Memuat...</div>
          ) : (
            <div className="table-responsive">
              <table className="table table-striped">
                <thead>
                  <tr>
                    <th style={{ width: '60px' }}>No</th>
                    <th>Nama</th>
                    <th style={{ width: '12%' }}>Kelas</th>
                    <th style={{ width: '16%' }}>Approved</th>
                    <th style={{ width: '16%' }}>Mulai</th>
                    <th style={{ width: '16%' }}>Selesai</th>
                    <th style={{ width: '14%' }}>Status</th>
                    <th style={{ width: '14%' }}>Aksi</th>
                  </tr>
                </thead>
                <tbody>
                  {sortedRows.length === 0 && (
                    <tr><td colSpan="8" className="text-center">Tidak ada data</td></tr>
                  )}
                  {paginatedRows.map((r, idx) => {
                    const approvedAtRaw = r?.session?.approved_at || null;
                    const startedAtRaw = r?.session?.started_at || null;
                    const finishedAtRaw = r?.session?.finished_at || null;
                    const approved = !!approvedAtRaw;
                    const approvedAt = onlyTime(approvedAtRaw);
                    const startedAt = onlyTime(startedAtRaw);
                    const finishedAt = onlyTime(finishedAtRaw);
                    const displayName = r?.user?.name || r?.participant?.name || r?.participant?.nama || r?.name || '-';
                    let status = 'Belum Login';
                    if (finishedAtRaw) status = 'Selesai';
                    else if (startedAtRaw) status = 'Sedang Ujian';
                    else if (approved) status = 'Disetujui';
                    else if (r?.session) status = 'Belum Disetujui';
                    return (
                      <tr key={r.participant?.id || `${idx}-${r.class?.name || ''}`}>
                        <td>{idx + 1 + (page - 1) * perPage}</td>
                        <td>{displayName}</td>
                        <td>{r.class?.name || '-'}</td>
                        <td>{approvedAt || '-'}</td>
                        <td>{startedAt || '-'}</td>
                        <td>{finishedAt || '-'}</td>
                        <td>{status}</td>
                        <td>
                          {!approved ? (
                            r?.session ? (
                              <button className="btn btn-sm btn-primary" onClick={() => approveOne(r.participant?.id)} disabled={!selectedExamId}>Approve</button>
                            ) : (
                              <span className="badge bg-secondary">Belum Login</span>
                            )
                          ) : (
                            <span className="badge bg-success">Sudah</span>
                          )}
                        </td>
                      </tr>
                  );
                  })}
                </tbody>
              </table>
            </div>
          )}
          <div className="d-flex justify-content-between align-items-center mt-3">
            <div>Total: {total}</div>
            <div className="btn-group">
              <button className="btn btn-outline-secondary" disabled={page <= 1} onClick={() => setPage(p => Math.max(1, p-1))}>Prev</button>
              <span className="btn btn-outline-secondary disabled">{page} / {lastPage}</span>
              <button className="btn btn-outline-secondary" disabled={page >= lastPage} onClick={() => setPage(p => Math.min(lastPage, p+1))}>Next</button>
            </div>
          </div>
        </div>
      </div>
      ) : (
        <div className="alert alert-info">Silakan pilih Ujian, Sekolah, dan Kelas terlebih dahulu untuk menampilkan daftar peserta yang perlu di-approve.</div>
      )}

      {/* Hidden component for PDF generation */}
      <div style={{ position: 'absolute', left: '-9999px', top: '-9999px' }}>
        <div ref={printRef}>
          <PrintableReport 
            examData={examData}
            participantsData={sortedRows}
            schoolData={schoolData}
            approverNamesById={approverNamesById}
          />
        </div>
      </div>
    </div>
  );
}