import React, { createContext, useState, useEffect, useContext } from "react";
import api from "../utils/axios";

export const AuthContext = createContext(null);

export const AuthProvider = ({ children }) => {
  const [user, setUser] = useState(null);
  const [loading, setLoading] = useState(true);

  // 🔹 Saat pertama kali load, ambil user dari localStorage
  useEffect(() => {
    try {
      const storedUser = localStorage.getItem("user");
      if (storedUser) setUser(JSON.parse(storedUser));
    } catch (err) {
      console.warn("Failed to load user from storage:", err);
      localStorage.removeItem("user");
    }
    setLoading(false);
  }, []);

  // 🔹 Heartbeat untuk update aktivitas user tiap 1 menit
  useEffect(() => {
    let intervalId;
    if (user) {
      intervalId = setInterval(() => {
        // Pilih endpoint berdasarkan jenis token yang aktif
        const participantToken = localStorage.getItem("participantToken");
        const url = participantToken ? "/participant/update-activity" : "/update-activity";
        api.post(url).catch(() => {});
      }, 60000);
    }
    return () => intervalId && clearInterval(intervalId);
  }, [user]);

  // 🔹 LOGIN
  const login = async (email, password) => {
    try {
      const response = await api.post("/login", { email, password });
      const { user: rawUser, token, permissions } = response.data;
      const newUser = { ...rawUser, permissions: permissions || [] };

      localStorage.setItem("token", token);
      localStorage.setItem("user", JSON.stringify(newUser));
      setUser(newUser);

      // trigger event untuk sinkronisasi komponen/sidebar
      window.dispatchEvent(new Event("user-updated"));

      return { success: true };
    } catch (err) {
      console.error("Login failed", err);
      const msg = err.response?.data?.message || "Login gagal, silakan coba lagi.";
      return { success: false, message: msg };
    }
  };

  // 🔹 LOGOUT
  const logout = () => {
    localStorage.removeItem("token");
    localStorage.removeItem("user");
    setUser(null);
    window.dispatchEvent(new Event("user-updated"));
  };

  // 🔹 UPDATE USER (misal update role/menu tanpa refresh)
  const updateUser = (partial) => {
    if (!user) return;
    const updated = { ...user, ...partial };
    localStorage.setItem("user", JSON.stringify(updated));
    setUser(updated);
    window.dispatchEvent(new Event("user-updated"));
  };

  // 🔹 REFRESH USER dari API (sinkron roles & permissions)
  const refreshUser = async () => {
    try {
      const res = await api.get("/me");
      const { user: rawUser, permissions } = res.data;
      const updated = { ...rawUser, permissions: permissions || [] };
      localStorage.setItem("user", JSON.stringify(updated));
      setUser(updated);
      window.dispatchEvent(new Event("user-updated"));
      return updated;
    } catch {
      return null;
    }
  };

  // 🔹 ROLE CHECK
  const hasRole = (roles) => {
    if (!user?.roles?.length) return false;
    const names = user.roles.map((r) => r.name);
    return Array.isArray(roles)
      ? roles.some((r) => names.includes(r))
      : names.includes(roles);
  };

  // 🔹 Listener global agar sidebar/menu ikut update otomatis
  useEffect(() => {
    const syncUser = () => {
      const storedUser = localStorage.getItem("user");
      setUser(storedUser ? JSON.parse(storedUser) : null);
    };
    window.addEventListener("user-updated", syncUser);
    return () => window.removeEventListener("user-updated", syncUser);
  }, []);

  return (
    <AuthContext.Provider
      value={{
        user,
        login,
        logout,
        loading,
        hasRole,
        updateUser,
        refreshUser,
      }}
    >
      {children}
    </AuthContext.Provider>
  );
};

export const useAuth = () => useContext(AuthContext);
