import React, { useEffect, useRef, forwardRef, useImperativeHandle } from 'react';
import katex from 'katex';
import 'katex/dist/katex.min.css';
import Swal from 'sweetalert2';
import { sanitizeLatex } from '../utils/latex';

const BACKEND_BASE = import.meta.env.VITE_BACKEND_URL || 'http://127.0.0.1:8000';
const resolveImageUrl = (urlOrPath) => {
  if (!urlOrPath) return '';
  const s = String(urlOrPath).trim();
  const base = String(BACKEND_BASE || '').replace(/\/+$/, '');
  if (/^(data|blob):/.test(s)) return s;
  // Rebase absolute URL pointing to /storage or /images so origin follows BACKEND_BASE
  if (/^https?:\/\/[^/]+\/(?:storage|images)\//.test(s)) {
    return s.replace(/^https?:\/\/[^/]+/, base);
  }
  // Any other absolute URL, keep as is
  if (/^https?:\/\//.test(s)) return s;
  // Handle leading slash storage path
  if (/^\/storage\//.test(s)) return `${base}${s}`;
  // Normalize clean relative path
  const clean = s.replace(/^\/+/, '');
  // If path starts with storage/, serve via /storage/
  if (/^storage\//i.test(clean)) {
    return `${base}/storage/${clean.replace(/^storage\//i,'')}`;
  }
  // If path starts with images/, serve directly from public/images
  if (/^images\//i.test(clean)) {
    return `${base}/${clean}`;
  }
  // raw path like "questions/uuid.ext" on public disk (under storage)
  return `${base}/storage/${clean.replace(/^public\//i,'')}`;
};

// Konversi BBCode sederhana → HTML string (untuk ditampilkan di contentEditable)
function bbcodeToHtmlString(input, opts = {}) {
  if (!input) return '';
  let s = String(input);
  const imgWidthPercent = typeof opts.imageWidthPercent === 'number' ? opts.imageWidthPercent : 100;
  // Bold/Italic/Underline
  s = s.replace(/\[b\](.*?)\[\/b\]/gis, '<strong>$1</strong>');
  s = s.replace(/\[i\](.*?)\[\/i\]/gis, '<em>$1</em>');
  s = s.replace(/\[u\](.*?)\[\/u\]/gis, '<u>$1</u>');
  // Align
  s = s.replace(/\[left\](.*?)\[\/left\]/gis, '<div style="text-align:left">$1</div>');
  s = s.replace(/\[center\](.*?)\[\/center\]/gis, '<div style="text-align:center">$1</div>');
  s = s.replace(/\[right\](.*?)\[\/right\]/gis, '<div style="text-align:right">$1</div>');
  // Line break
  s = s.replace(/\[br\/?\]/gi, '<br/>');
  // Inline image token markdown → <img> (resolved + non-editable)
  s = s.replace(/!\[([^\]]*)\]\(([^\)]+)\)/g, (m, altRaw, url) => {
    const raw = String(url || '').trim();
    const resolved = resolveImageUrl(raw);
    const alt = String(altRaw || 'img');
    const mW = alt.match(/\|\s*w\s*=\s*(\d+)\s*%/i);
    const widthPct = mW ? Math.max(5, Math.min(160, Number(mW[1] || imgWidthPercent))) : imgWidthPercent;
    const style = `width:${widthPct}%;height:auto;border-radius:6px;margin:6px 0`;
    const fallbackBase = String(BACKEND_BASE || '').replace(/\/+$/, '');
    return `<span class="rte-noneditable-img" contenteditable="false"><img src="${resolved}" data-src="${raw}" alt="${alt}" style="${style}" draggable="false" onerror="if(this._trFallback){return;}this._trFallback=1;var raw=this.getAttribute('data-src')||'';var base='${fallbackBase}';var clean=String(raw).trim();while(clean.charAt(0)=='/'){clean=clean.slice(1);}var fname=(clean.split('/').pop()||'placeholder.png');this.src=base+'/images/'+fname;"/></span>`;
  });
  // YouTube BBCode → iframe embed
  s = s.replace(/\[youtube\](.*?)\[\/youtube\]/gis, (m, p1) => {
    const raw = String(p1 || '').trim();
    const idMatch = raw.match(/(?:v=|youtu\.be\/|embed\/)([A-Za-z0-9_-]{6,})/);
    const id = idMatch ? idMatch[1] : raw;
    if (!id) return raw;
    const src = `https://www.youtube.com/embed/${id}`;
    return `<div class="ratio ratio-16x9" style="max-width: 720px; margin: 6px auto;"><iframe src="${src}" title="YouTube" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" allowfullscreen style="width:100%; height:100%;"></iframe></div>`;
  });
  // Table BBCode → HTML table
  s = s.replace(/\[table\](.*?)\[\/table\]/gis, (m, body) => {
    let inner = body || '';
    inner = inner
      .replace(/\[th\](.*?)\[\/th\]/gis, (mm, cell) => `<th style="border:1px solid #dee2e6; padding:6px;">${cell}</th>`)
      .replace(/\[td\](.*?)\[\/td\]/gis, (mm, cell) => `<td style="border:1px solid #dee2e6; padding:6px;">${cell}</td>`)
      .replace(/\[tr\](.*?)\[\/tr\]/gis, (mm, row) => `<tr>${row}</tr>`);
    const tblStyle = 'border-collapse: collapse; width: 100%; margin: 6px 0;';
    return `<table style="${tblStyle}"><tbody>${inner}</tbody></table>`;
  });

  // Render LaTeX tokens (\(...\), $...$, \[...\], $$...$$) menjadi KaTeX HTML agar editor tampil seperti preview
  const escapeAttr = (str) => String(str || '').replace(/"/g, '&quot;').replace(/</g, '&lt;').replace(/>/g, '&gt;');
  const decodeHtmlEntities = (input) => {
    return String(input || '')
      .replace(/&amp;/g, '&')
      .replace(/&lt;/g, '<')
      .replace(/&gt;/g, '>')
      .replace(/&quot;/g, '"')
      .replace(/&#039;/g, "'")
      .replace(/&#39;/g, "'");
  };
  const renderLatexTokensInHtml = (html) => {
    if (!html) return '';
    let out = String(html);
    // Inline: \( ... \)
    out = out.replace(/\\\(([\s\S]*?)\\\)/g, (m, inner) => {
      const latex = decodeHtmlEntities(String(inner || '').trim());
      try {
        const safe = sanitizeLatex(latex);
        const katexHtml = katex.renderToString(safe, { throwOnError: false, errorColor: '#cc0000' });
        return `<span class="katex-inline" data-latex="${escapeAttr(m)}">${katexHtml}</span>`;
      } catch (_) { return m; }
    });
    // Inline: $ ... $
    out = out.replace(/\$([^$]+)\$/g, (m, inner) => {
      const latex = decodeHtmlEntities(String(inner || '').trim());
      try {
        const safe = sanitizeLatex(latex);
        const katexHtml = katex.renderToString(safe, { throwOnError: false, errorColor: '#cc0000' });
        return `<span class="katex-inline" data-latex="${escapeAttr(m)}">${katexHtml}</span>`;
      } catch (_) { return m; }
    });
    // Block: \[ ... \]
    out = out.replace(/\\\[([\s\S]*?)\\\]/g, (m, inner) => {
      const latex = decodeHtmlEntities(String(inner || '').trim());
      try {
        const safe = sanitizeLatex(latex);
        const katexHtml = katex.renderToString(safe, { displayMode: true, throwOnError: false, errorColor: '#cc0000' });
        return `<div class="katex-block" data-latex="${escapeAttr(m)}">${katexHtml}</div>`;
      } catch (_) { return m; }
    });
    // Block: $$ ... $$
    out = out.replace(/\$\$([\s\S]*?)\$\$/g, (m, inner) => {
      const latex = decodeHtmlEntities(String(inner || '').trim());
      try {
        const safe = sanitizeLatex(latex);
        const katexHtml = katex.renderToString(safe, { displayMode: true, throwOnError: false, errorColor: '#cc0000' });
        return `<div class="katex-block" data-latex="${escapeAttr(m)}">${katexHtml}</div>`;
      } catch (_) { return m; }
    });
    return out;
  };

  s = renderLatexTokensInHtml(s);
  return s;
}

// Konversi HTML (dari contentEditable) → BBCode string
function htmlToBBCodeString(html) {
  if (!html) return '';
  // Gunakan DOM parsing untuk robust conversion
  const container = document.createElement('div');
  container.innerHTML = html;

  const walk = (node) => {
    if (node.nodeType === Node.TEXT_NODE) {
      return node.textContent || '';
    }
    if (node.nodeType !== Node.ELEMENT_NODE) {
      return '';
    }
    const el = node;
    const tag = el.tagName;
    const children = Array.from(el.childNodes).map(walk).join('');
    const style = el.getAttribute('style') || '';

    // Jika elemen membawa data-latex, kembalikan LaTeX aslinya agar sinkron dengan preview
    const latexData = el.getAttribute && el.getAttribute('data-latex');
    if (latexData) {
      return latexData; // token lengkap: \( ... \) atau $...$ atau \[...\] / $$...$$
    }

    const align = (() => {
      const m = style.match(/text-align\s*:\s*(left|center|right)/i);
      return m ? m[1].toLowerCase() : null;
    })();

    switch (tag) {
      case 'STRONG':
      case 'B':
        return `[b]${children}[/b]`;
      case 'EM':
      case 'I':
        return `[i]${children}[/i]`;
      case 'U':
        return `[u]${children}[/u]`;
      case 'BR':
        return `[br/]`;
      case 'IMG': {
        const src = el.getAttribute('data-src') || el.getAttribute('src') || '';
        const altBase = el.getAttribute('alt') || 'img';
        const mW = style.match(/width\s*:\s*(\d+)\s*%/i);
        const alt = mW ? `${altBase}|w=${mW[1]}%` : altBase;
        return src ? `![${alt}](${src})` : '';
      }
      case 'DIV':
      case 'P': {
        let content = children;
        if (align === 'center') content = `[center]${children}[/center]`;
        else if (align === 'right') content = `[right]${children}[/right]`;
        else if (align === 'left') content = `[left]${children}[/left]`;
        // Tambahkan break setelah block element agar Enter direkam sebagai [br/]
        return content + `[br/]`;
      }
      case 'SPAN': {
        if (align === 'center') return `[center]${children}[/center]`;
        if (align === 'right') return `[right]${children}[/right]`;
        if (align === 'left') return `[left]${children}[/left]`;
        return children;
      }
      case 'IFRAME': {
        const src = el.getAttribute('src') || '';
        const m = src.match(/youtube\.com\/embed\/([A-Za-z0-9_-]{6,})/);
        const id = m ? m[1] : '';
        if (id) return `[youtube]https://youtu.be/${id}[/youtube]`;
        return children;
      }
      case 'TABLE': {
        // Konversi tabel HTML → BBCode [table][tr]...[td][/td]...
        const rows = Array.from(el.querySelectorAll('tr'));
        const bbRows = rows.map((tr) => {
          const cells = Array.from(tr.children || []);
          const bbCells = cells.map((cell) => {
            const isHeader = cell.tagName === 'TH';
            const inner = Array.from(cell.childNodes).map(walk).join('');
            return isHeader ? `[th]${inner}[/th]` : `[td]${inner}[/td]`;
          }).join('');
          return `[tr]${bbCells}[/tr]`;
        }).join('');
        return `[table]${bbRows}[/table]`;
      }
      default:
        return children;
    }
  };

  let out = Array.from(container.childNodes).map(walk).join('');
  // Bersihkan [br/] berlebihan dan trailing
  out = out.replace(/\[br\/\](\s*\[br\/\])+?/g, '[br/]');
  out = out.replace(/(\s*\[br\/\]\s*)+$/g, '');
  return out;
}

const RichTextEditor = forwardRef(function RichTextEditor({ value, onChange, className, style, placeholder, dir = 'auto', onMathSelect, onMathInsert, imageWidthPercent = 100 }, ref) {
  const divRef = useRef(null);
  const lastFromEditorRef = useRef('');
  const activeMathSpanRef = useRef(null);
  const activeMathSpecRef = useRef(null);

  // Sinkronisasi nilai BBCode → HTML ke contentEditable
  useEffect(() => {
    const el = divRef.current;
    if (!el) return;
    const current = el.innerHTML;
    const next = bbcodeToHtmlString(value || '', { imageWidthPercent: (typeof imageWidthPercent === 'number' ? imageWidthPercent : 100) });
    // Jika nilai berasal dari editor sendiri (emitChange), jangan sync ulang agar caret tidak meloncat
    if (value === lastFromEditorRef.current) return;
    if (current !== next) {
      el.innerHTML = next || '';
      // Pasang handler edit: LaTeX dan Gambar
      attachLatexEditHandlers();
      attachImageEditHandlers();
    }
  }, [value, imageWidthPercent]);

  const focusEditor = () => {
    const el = divRef.current; if (!el) return; el.focus();
  };

  const applyFormat = (tag) => {
    focusEditor();
    switch (tag) {
      case 'b':
        document.execCommand('bold'); break;
      case 'i':
        document.execCommand('italic'); break;
      case 'u':
        document.execCommand('underline'); break;
      default:
        break;
    }
    // Setelah perubahan, kirim BBCode terbaru
    emitChange();
  };

  const insertSnippet = (snippet) => {
    focusEditor();
    const s = String(snippet || '');
    if (s === '[left][/left]') {
      document.execCommand('justifyLeft'); emitChange(); return;
    }
    if (s === '[center][/center]') {
      document.execCommand('justifyCenter'); emitChange(); return;
    }
    if (s === '[right][/right]') {
      document.execCommand('justifyRight'); emitChange(); return;
    }
    if (s === '[br/]') {
      document.execCommand('insertHTML', false, '<br/>'); emitChange(); return;
    }
    // Jika snippet berbentuk HTML, sisipkan sebagai HTML;
    // Jika BBCode (mis. [youtube] atau [table]), konversi ke HTML dulu untuk tampilan WYSIWYG.
    const looksHtml = /<\w+/.test(s);
    if (looksHtml) {
      document.execCommand('insertHTML', false, s);
    } else if (/\[youtube\]/i.test(s)) {
      // Konversi BBCode YouTube ke iframe
      const html = bbcodeToHtmlString(s);
      document.execCommand('insertHTML', false, html);
    } else if (/\[table\]/i.test(s)) {
      const html = bbcodeToHtmlString(s);
      document.execCommand('insertHTML', false, html);
    } else {
      // Sisipan plain text
      document.execCommand('insertText', false, s);
    }
    emitChange();
  };

  const insertImage = (url) => {
    const raw = String(url || '').trim();
    if (!raw) return;
    focusEditor();
    const resolved = resolveImageUrl(raw);
    const wRaw = (typeof imageWidthPercent === 'number' ? imageWidthPercent : 100);
    const w = Math.max(5, Math.min(160, Number(wRaw)));
    const alt = `img|w=${w}%`;
    const style = `width:${w}%;height:auto;border-radius:6px;margin:6px 0`;
    const html = `<span class="rte-noneditable-img" contenteditable="false"><img src="${resolved}" data-src="${raw}" alt="${alt}" style="${style}" draggable="false" onerror="if(this._trFallback){return;}this._trFallback=1;var raw=this.getAttribute('data-src')||'';var base='${String(BACKEND_BASE || '').replace(/\/+$/, '')}';var clean=String(raw).trim();while(clean.charAt(0)=='/'){clean=clean.slice(1);}var fname=(clean.split('/').pop()||'placeholder.png');this.src=base+'/images/'+fname;"/></span>`;
    document.execCommand('insertHTML', false, html);
    emitChange();
  };

  const insertImageWithOptions = (url, widthPercent) => {
    const raw = String(url || '').trim();
    if (!raw) return;
    focusEditor();
    const resolved = resolveImageUrl(raw);
    const wRaw = (typeof widthPercent === 'number' ? widthPercent : (typeof imageWidthPercent === 'number' ? imageWidthPercent : 100));
    const w = Math.max(5, Math.min(160, Number(wRaw)));
    const alt = `img|w=${w}%`;
    const style = `width:${w}%;height:auto;border-radius:6px;margin:6px 0`;
    const html = `<span class="rte-noneditable-img" contenteditable="false"><img src="${resolved}" data-src="${raw}" alt="${alt}" style="${style}" draggable="false" onerror="if(this._trFallback){return;}this._trFallback=1;var raw=this.getAttribute('data-src')||'';var base='${String(BACKEND_BASE || '').replace(/\/+$/, '')}';var clean=String(raw).trim();while(clean.charAt(0)=='/'){clean=clean.slice(1);}var fname=(clean.split('/').pop()||'placeholder.png');this.src=base+'/images/'+fname;"/></span>`;
    document.execCommand('insertHTML', false, html);
    emitChange();
  };

  const parseLatexToken = (token) => {
    const t = String(token || '');
    if (/^\$\$[\s\S]*\$\$$/.test(t)) {
      return { inner: t.replace(/^\$\$/,'').replace(/\$\$$/, ''), displayMode: true, wrap: (x) => `$$${x}$$` };
    }
    if (/^\$[^$]*\$/s.test(t)) {
      return { inner: t.replace(/^\$/,'').replace(/\$$/, ''), displayMode: false, wrap: (x) => `$${x}$` };
    }
    if (/^\\\[[\s\S]*\\\]$/.test(t)) {
      return { inner: t.replace(/^\\\[/,'').replace(/\\\]$/, ''), displayMode: true, wrap: (x) => `\\[${x}\\]` };
    }
    // default: \( ... \)
    return { inner: t.replace(/^\\\(/,'').replace(/\\\)$/, ''), displayMode: false, wrap: (x) => `\\(${x}\\)` };
  };

  const renderLatexElement = (el, token) => {
    try {
      const { inner, displayMode } = parseLatexToken(token);
      const safe = inner
        .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\|\s*\}/g, '\\$1|')
        .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\?mid\s*\}/g, '\\$1|')
        .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\(\s*\}/g, '\\$1(')
        .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\)\s*\}/g, '\\$1)')
        .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\[\s*\}/g, '\\$1[')
        .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\]\s*\}/g, '\\$1]')
        .replace(/\\bigm\\\|/g, '\\Big|');
      const html = katex.renderToString(safe, { displayMode, throwOnError: false, errorColor: '#cc0000' });
      el.setAttribute('data-latex', token);
      // Sesuaikan kelas berdasarkan mode
      if (displayMode) {
        el.classList.remove('katex-inline');
        el.classList.add('katex-block');
      } else {
        el.classList.remove('katex-block');
        el.classList.add('katex-inline');
      }
      el.innerHTML = html;
    } catch (_) {
      el.textContent = token;
    }
  };

  const attachLatexEditHandlers = () => {
    const el = divRef.current; if (!el) return;
    const nodes = el.querySelectorAll('[data-latex]');
    nodes.forEach((span) => {
      // Hindari duplikasi listener
      span.removeEventListener('click', span.__latexClickHandler);
      const handler = async () => {
        const specAttr = span.getAttribute('data-math-spec');
        if (specAttr) {
          // Jika punya spesifikasi, gunakan panel kolom di luar editor
          try {
            const spec = JSON.parse(specAttr);
            activeMathSpanRef.current = span;
            activeMathSpecRef.current = spec;
            if (typeof onMathSelect === 'function') {
              onMathSelect(spec);
            }
          } catch (_) {
            // fallback ke edit teks penuh
          }
          return;
        }
        // Jika tidak ada spec, arahkan ke MathLive (jika disediakan), fallback ke SweetAlert
        const currentLatex = span.getAttribute('data-latex') || '\\()';
        const parsed = parseLatexToken(currentLatex);
        const inner = parsed.inner;
        if (typeof onMathSelect === 'function') {
          activeMathSpanRef.current = span;
          activeMathSpecRef.current = null;
          try { onMathSelect({ type: 'raw', latex: inner }); } catch(_) {}
          return;
        }
        const { value: edited } = await Swal.fire({
          title: 'Edit Rumus',
          input: 'text',
          inputValue: inner,
          showCancelButton: true,
          confirmButtonText: 'Simpan',
          cancelButtonText: 'Batal'
        });
        if (typeof edited === 'string') {
          const wrapped = parsed.wrap(edited);
          renderLatexElement(span, wrapped);
          emitChange();
        }
      };
      span.__latexClickHandler = handler;
      span.addEventListener('click', handler);
      span.style.cursor = 'pointer';
      span.title = 'Klik untuk mengedit rumus';
    });
  };

  const attachImageEditHandlers = () => {
    const el = divRef.current; if (!el) return;
    const nodes = el.querySelectorAll('.rte-noneditable-img img');
    nodes.forEach((img) => {
      img.removeEventListener('click', img.__imgClickHandler);
      const handler = async () => {
        const altRaw = img.getAttribute('alt') || 'img';
        const m = String(altRaw).match(/\|\s*w\s*=\s*(\d+)\s*%/i);
        const currentW = m ? Math.max(5, Math.min(160, Number(m[1] || 100))) : 100;
        const { isConfirmed, value } = await Swal.fire({
          title: 'Ubah ukuran gambar (%)',
          input: 'range',
          inputAttributes: { min: 5, max: 160, step: 1 },
          inputValue: currentW,
          showCancelButton: true,
          confirmButtonText: 'Simpan',
          cancelButtonText: 'Batal'
        });
        if (!isConfirmed) return;
        const w = Math.max(5, Math.min(160, Number(value || currentW)));
        // Update style
        img.style.width = w + '%';
        img.style.height = 'auto';
        // Update alt agar tersimpan ke BBCode
        const baseAlt = String(altRaw).replace(/\|\s*w\s*=\s*\d+\s*%/i, '').trim();
        img.setAttribute('alt', (baseAlt ? baseAlt : 'img') + `|w=${w}%`);
        emitChange();
      };
      img.__imgClickHandler = handler;
      img.addEventListener('click', handler);
      img.style.cursor = 'nwse-resize';
      img.title = 'Klik untuk mengubah ukuran gambar';
    });
  };

  const getSelectionTextInEditor = () => {
    const el = divRef.current; if (!el) return '';
    const sel = window.getSelection ? window.getSelection() : null;
    if (!sel || sel.rangeCount === 0) return '';
    const range = sel.getRangeAt(0);
    // Pastikan selection berada di dalam editor
    const container = range.commonAncestorContainer;
    const isInside = el.contains(container.nodeType === Node.ELEMENT_NODE ? container : container.parentNode);
    return isInside ? (sel.toString() || '') : '';
  };

  const applyMath = (key) => {
    focusEditor();
    const sel = getSelectionTextInEditor();
    const base = sel || 'x';
    let latexInner = '';
    let spec = null; // untuk kolom matematika
    switch (key) {
      case 'frac':
        latexInner = `\\frac{\\textcolor{orange}{${'a'}}}{\\textcolor{orange}{${'b'}}}`;
        spec = { type: 'frac', fields: ['a','b'] };
        break;
      case 'sqrt':
        latexInner = `\\sqrt{\\textcolor{orange}{${sel || 'x'}}}`;
        spec = { type: 'sqrt', fields: ['x'] };
        break;
      case 'pow2':
        latexInner = `${base}^{2}`; break;
      case 'powN':
        latexInner = `${base}^{\\textcolor{orange}{n}}`;
        spec = { type: 'powN', fields: ['n'] };
        break;
      case 'degree':
        latexInner = `\\textcolor{orange}{${base}}^{\\circ}`;
        spec = { type: 'degree', fields: ['deg'] }; // kolom untuk angka derajat
        break;
      case '( )':
        latexInner = `\\left( ${sel || base} \\right)`; break;
      case '[ ]':
        latexInner = `\\left[ ${sel || base} \\right]`; break;
      case 'angle': {
        const A = '\\textcolor{orange}{A}';
        const B = '\\textcolor{orange}{B}';
        const C = '\\textcolor{orange}{C}';
        latexInner = `\\angle ${A}${B}${C}`;
        spec = { type: 'angle', fields: ['A','B','C'] };
        break;
      }
      case 'overlineAB': {
        const AB = '\\textcolor{orange}{AB}';
        latexInner = `\\overline{${AB}}`;
        spec = { type: 'overline', fields: ['AB'] };
        break;
      }
      case 'sin': {
        const X = '\\textcolor{orange}{x}';
        latexInner = `\\sin\\left(${X}\\right)`;
        spec = { type: 'func', fn: 'sin', fields: ['arg'] };
        break;
      }
      case 'cos': {
        const X = '\\textcolor{orange}{x}';
        latexInner = `\\cos\\left(${X}\\right)`;
        spec = { type: 'func', fn: 'cos', fields: ['arg'] };
        break;
      }
      case 'tan': {
        const X = '\\textcolor{orange}{x}';
        latexInner = `\\tan\\left(${X}\\right)`;
        spec = { type: 'func', fn: 'tan', fields: ['arg'] };
        break;
      }
      case 'bmatrix2': {
        const A = '\\textcolor{orange}{a}';
        const B = '\\textcolor{orange}{b}';
        const C = '\\textcolor{orange}{c}';
        const D = '\\textcolor{orange}{d}';
        latexInner = `\\begin{bmatrix}${A} & ${B} \\\\ ${C} & ${D}\\end{bmatrix}`;
        spec = { type: 'bmatrix2', fields: ['a','b','c','d'] };
        break;
      }
      default:
        latexInner = sel || base;
    }
    if (!latexInner) return;
    // Bungkus sebagai KaTeX span agar langsung WYSIWYG
    try {
      const html = katex.renderToString(latexInner.replace(/\\bigm\\{\\|\\}/g, '\\Big|').replace(/\\bigm\\\|/g, '\\Big|'), { throwOnError: false, errorColor: '#cc0000' });
      const wrapped = `\\(${latexInner}\\)`;
      const uid = 'Math-' + Date.now() + '-' + Math.random().toString(36).slice(2);
      const specAttr = spec ? ` data-math-spec='${JSON.stringify(spec)}'` : '';
      const spanHtml = `<span id="${uid}" class="katex-inline" data-latex="${wrapped}"${specAttr}>${html}</span>`;
      document.execCommand('insertHTML', false, spanHtml);
      emitChange();
      // Set aktif + panggil callback kolom
      const el = divRef.current;
      const span = el ? el.querySelector('#' + CSS.escape(uid)) : null;
      if (span && spec) {
        activeMathSpanRef.current = span;
        activeMathSpecRef.current = spec;
        if (typeof onMathInsert === 'function') onMathInsert(spec);
        if (typeof onMathSelect === 'function') onMathSelect(spec);
      }
    } catch (_) {
      // fallback sisipkan LaTeX mentah
      document.execCommand('insertText', false, '\\(' + latexInner + '\\)');
      emitChange();
    }
  };

  const updateActiveMathValues = (values) => {
    const span = activeMathSpanRef.current;
    const spec = activeMathSpecRef.current;
    if (!span || !spec) return;
    const currentLatexWrapped = span.getAttribute('data-latex') || '\\()';
    const currentInner = currentLatexWrapped.replace(/^\\\(([^]*)\\\)$/,'$1');
    const val = (k) => (values && typeof values[k] === 'string') ? values[k].trim() : '';
    const isNumeric = (s) => /^-?\d+(?:[.,]\d+)?$/.test(s);
    const colorize = (s, placeholder) => {
      const v = s || placeholder;
      return isNumeric(v) ? v : `\\textcolor{orange}{${v}}`;
    };
    let nextLatex = currentInner;
    try {
      switch (spec.type) {
        case 'frac': {
          const A = colorize(val('a'), 'a');
          const B = colorize(val('b'), 'b');
          nextLatex = `\\frac{${A}}{${B}}`;
          break;
        }
        case 'sqrt': {
          const X = colorize(val('x'), 'x');
          nextLatex = `\\sqrt{${X}}`;
          break;
        }
        case 'powN': {
          const n = colorize(val('n'), 'n');
          const m = currentInner.match(/^(.+?)\^\{[^}]*\}\s*$/);
          const base = m ? m[1] : 'x';
          nextLatex = `${base}^{${n}}`;
          break;
        }
        case 'degree': {
          const d = colorize(val('deg'), 'x');
          nextLatex = `${d}^{\\circ}`;
          break;
        }
        case 'angle': {
          const A = val('A') || 'A';
          const B = val('B') || 'B';
          const C = val('C') || 'C';
          const AA = `\\textcolor{orange}{${A}}`;
          const BB = `\\textcolor{orange}{${B}}`;
          const CC = `\\textcolor{orange}{${C}}`;
          nextLatex = `\\angle ${AA}${BB}${CC}`;
          break;
        }
        case 'overline': {
          const over = val('AB') || 'AB';
          const O = `\\textcolor{orange}{${over}}`;
          nextLatex = `\\overline{${O}}`;
          break;
        }
        case 'func': {
          const arg = colorize(val('arg'), 'x');
          const fn = (activeMathSpecRef.current && activeMathSpecRef.current.fn) || 'f';
          nextLatex = `\\${fn}\\left(${arg}\\right)`;
          break;
        }
        case 'bmatrix2': {
          const A = colorize(val('a'), 'a');
          const B = colorize(val('b'), 'b');
          const C = colorize(val('c'), 'c');
          const D = colorize(val('d'), 'd');
          nextLatex = `\\begin{bmatrix}${A} & ${B} \\\\ ${C} & ${D}\\end{bmatrix}`;
          break;
        }
        default:
          return;
      }
      renderLatexElement(span, '\\(' + nextLatex + '\\)');
      emitChange();
    } catch (_) {
      // abaikan error render, tetap tampil mentah
      span.textContent = '\\(' + nextLatex + '\\)';
      emitChange();
    }
  };

  const emitChange = () => {
    if (!onChange) return;
    const el = divRef.current; if (!el) return;
    const html = el.innerHTML;
    const bb = htmlToBBCodeString(html);
    lastFromEditorRef.current = bb;
    onChange(bb);
    // Setelah emit, pasang ulang handler latex & gambar (konten mungkin berubah)
    attachLatexEditHandlers();
    attachImageEditHandlers();
  };

  useImperativeHandle(ref, () => ({
    applyFormat,
    insertSnippet,
    insertImage,
    insertImageWithOptions,
    focus: focusEditor,
    applyMath,
    updateActiveMathValues,
    // Sisipkan LaTeX sebagai span KaTeX agar bisa di-klik dan diedit
    insertLatex: (latexInner) => {
      focusEditor();
      const inner = String(latexInner || '').trim();
      if (!inner) return;
      try {
        const safe = inner
          .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\|\s*\}/g, '\\$1|')
          .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\?mid\s*\}/g, '\\$1|')
          .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\(\s*\}/g, '\\$1(')
          .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\)\s*\}/g, '\\$1)')
          .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\[\s*\}/g, '\\$1[')
          .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\]\s*\}/g, '\\$1]')
          .replace(/\\bigm\\\|/g, '\\Big|');
        const html = katex.renderToString(safe, { displayMode: false, throwOnError: false, errorColor: '#cc0000' });
        const wrapped = `\\(${inner}\\)`;
        const uid = 'Math-' + Date.now() + '-' + Math.random().toString(36).slice(2);
        const spanHtml = `<span id="${uid}" class="katex-inline" data-latex="${wrapped}">${html}</span>`;
        document.execCommand('insertHTML', false, spanHtml);
      } catch (_) {
        document.execCommand('insertText', false, '\\(' + inner + '\\)');
      }
      emitChange();
    },
    // Ganti LaTeX pada span yang sedang aktif (diklik) dengan nilai baru
    replaceActiveLatex: (latexInner) => {
      const span = activeMathSpanRef.current;
      const inner = String(latexInner || '').trim();
      if (!span || !inner) { return; }
      try {
        const safe = inner
          .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\|\s*\}/g, '\\$1|')
          .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\?mid\s*\}/g, '\\$1|')
          .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\(\s*\}/g, '\\$1(')
          .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\)\s*\}/g, '\\$1)')
          .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\[\s*\}/g, '\\$1[')
          .replace(/\\(bigm|Bigm|biggm|Biggm)\s*\{\s*\]\s*\}/g, '\\$1]')
          .replace(/\\bigm\\\|/g, '\\Big|');
        const html = katex.renderToString(safe, { displayMode: false, throwOnError: false, errorColor: '#cc0000' });
        span.setAttribute('data-latex', `\\(${inner}\\)`);
        span.classList.remove('katex-block');
        span.classList.add('katex-inline');
        span.innerHTML = html;
      } catch (_) {
        span.textContent = '\\(' + inner + '\\)';
      }
      emitChange();
    },
  }));

  return (
    <div
      ref={divRef}
      className={className || 'form-control'}
      style={{ minHeight: 120, whiteSpace: 'pre-wrap', ...style }}
      contentEditable
      suppressContentEditableWarning
      onInput={emitChange}
      onBlur={emitChange}
      dir={dir}
      aria-label={placeholder || 'Editor Teks'}
    />
  );
});

export default RichTextEditor;