import React from 'react';

const PrintableReport = ({ examData, participantsData, schoolData, approverNamesById }) => {
  const resolveImageUrl = (imagePath) => {
    if (!imagePath) return null;
    if (imagePath.startsWith('http://') || imagePath.startsWith('https://')) return imagePath;
    if (imagePath.startsWith('/'))
      return `${import.meta.env.VITE_API_BASE_URL || 'http://localhost:8000'}${imagePath}`;
    return `${import.meta.env.VITE_API_BASE_URL || 'http://localhost:8000'}/storage/${imagePath}`;
  };

  const currentDate = (() => {
    const d = new Date();
    return `${String(d.getDate()).padStart(2, '0')}-${String(d.getMonth() + 1).padStart(2, '0')}-${d.getFullYear()}`;
  })();

  const onlyTime = (val) => {
    if (!val) return '-';
    if (typeof val === 'string') {
      const m = val.match(/(\d{2}:\d{2}:\d{2})/);
      if (m) return m[1];
      const parts = val.split(' ');
      if (parts[1] && /\d{2}:\d{2}:\d{2}/.test(parts[1])) return parts[1].slice(0, 8);
    }
    try {
      const d = new Date(val);
      const hh = String(d.getHours()).padStart(2, '0');
      const mm = String(d.getMinutes()).padStart(2, '0');
      const ss = String(d.getSeconds()).padStart(2, '0');
      return `${hh}:${mm}:${ss}`;
    } catch {
      return '-';
    }
  };

  const groupByClass = () => {
    const data = Array.isArray(participantsData) ? [...participantsData] : [];
    const groups = {};
    data.forEach(r => {
      const className = r?.class?.name || 'Tidak Ada Kelas';
      if (!groups[className]) groups[className] = [];
      groups[className].push(r);
    });
    return Object.keys(groups).sort((a, b) => a.localeCompare(b, 'id')).map(k => {
      const rows = groups[k].sort((a, b) => {
        const an = (a?.user?.name || a?.participant?.name || '').toLowerCase();
        const bn = (b?.user?.name || b?.participant?.name || '').toLowerCase();
        return an.localeCompare(bn);
      });
      // Hitung mayoritas approver per kelas
      const counts = new Map();
      rows.forEach(r => {
        const id = r?.session?.approved_by_user_id || r?.approved_by_user_id || null;
        if (id) {
          counts.set(id, (counts.get(id) || 0) + 1);
        }
      });
      let majorityId = null;
      let majorityCount = 0;
      for (const [id, c] of counts.entries()) {
        if (c > majorityCount) { majorityCount = c; majorityId = id; }
      }
      const majorityName = majorityId ? (approverNamesById?.[String(majorityId)] || String(majorityId)) : null;
      return { className: k, rows, majorityApproverId: majorityId, majorityApproverName: majorityName };
    });
  };

  const generateBarcodeDataUrl = (text, width = 240, height = 60) => {
    try {
      const canvas = document.createElement('canvas');
      canvas.width = width;
      canvas.height = height;
      const ctx = canvas.getContext('2d');
      ctx.fillStyle = '#fff';
      ctx.fillRect(0, 0, width, height);
      ctx.fillStyle = '#000';
      // Guard bars
      ctx.fillRect(5, 5, 2, height - 10);
      ctx.fillRect(width - 7, 5, 2, height - 10);
      let x = 10;
      const chars = (text || '').toString();
      for (let i = 0; i < chars.length; i++) {
        const code = chars.charCodeAt(i);
        for (let b = 0; b < 8; b++) {
          const bit = (code >> b) & 1;
          const w = bit ? 2 : 1;
          ctx.fillRect(x, 5, w, height - 10);
          x += w + 1;
          if (x > width - 10) break;
        }
        if (x > width - 10) break;
      }
      return canvas.toDataURL('image/png');
    } catch {
      return null;
    }
  };

  return (
    <>
      <style>{`
        @media print {
          @page {
            size: A4 portrait;
            margin: 15mm;
          }
          .page {
            page-break-after: always;
            overflow: hidden;
          }
          .page:last-child {
            page-break-after: auto;
          }
          table, th, td {
            border: 1px solid black !important;
            border-collapse: collapse !important;
          }
          th, td {
            padding: 4px !important;
          }
          img, h1, h2, h3, h4, h5, h6, p, table {
            page-break-inside: avoid !important;
          }
        }
      `}</style>

      {groupByClass().map(({ className, rows }, classIndex) => {
        // potong per 39 peserta per halaman (sisakan 1 baris kosong untuk space bawah)
        const chunks = [];
        for (let i = 0; i < rows.length; i += 39) {
          chunks.push(rows.slice(i, i + 39));
        }

        return chunks.map((chunk, chunkIndex) => (
          <div
            key={`${className}-${chunkIndex}`}
            className="page"
            style={{
              width: '794px',
              minHeight: '1123px',
              padding: '20px 30px',
              margin: '0 auto',
              backgroundColor: 'white',
              fontFamily: 'Arial, sans-serif',
              fontSize: '11px',
              lineHeight: '1.3',
              color: '#000',
              position: 'relative',
              boxSizing: 'border-box',
            }}
          >
            {/* Watermark Logo */}
            {schoolData?.logo_path && (
              <img
                src={resolveImageUrl(schoolData.logo_path)}
                alt="Logo Sekolah"
                style={{
                  position: 'absolute',
                  top: '20px',
                  left: '20px',
                  width: '90px',
                  opacity: 0.25,
                  zIndex: 0,
                }}
              />
            )}

            {/* Header */}
            <div
              style={{
                textAlign: 'center',
                borderBottom: '1.5px solid #000',
                paddingBottom: '10px',
                marginBottom: '15px',
                zIndex: 1,
                position: 'relative',
              }}
            >
              <h1
                style={{
                  margin: '0 0 4px 0',
                  fontSize: '16px',
                  fontWeight: 'bold',
                  textTransform: 'uppercase',
                }}
              >
                {schoolData?.nama || 'Nama Sekolah'}
              </h1>
              <p style={{ margin: 0, fontSize: '10px' }}>{schoolData?.alamat || 'Alamat Sekolah'}</p>
              <p style={{ margin: 0, fontSize: '10px' }}>
                {schoolData?.kota || 'Kota'}, {schoolData?.provinsi || 'Provinsi'}
              </p>
            </div>

            {/* Title */}
            <h2
              style={{
                margin: '0 0 10px 0',
                fontSize: '14px',
                fontWeight: 'bold',
                textDecoration: 'underline',
                textAlign: 'center',
              }}
            >
              Daftar Peserta Ujian
            </h2>

            {/* Exam Info */}
            <div style={{ textAlign: 'left', marginBottom: '10px' }}>
              <p style={{ margin: '2px 0' }}>
                <strong>Ujian:</strong> {examData?.name || examData?.nama || examData?.code || '-'}
              </p>
              <p style={{ margin: '2px 0' }}>
                <strong>Kelas:</strong> {className}
              </p>
              <p style={{ margin: '2px 0' }}>
                <strong>Tanggal:</strong> {currentDate}
              </p>
            </div>

            {/* Table */}
            <table
              style={{
                width: '90%',
                border: '1px solid #000',
                borderCollapse: 'collapse',
                fontSize: '10px',
              }}
            >
              <thead>
                <tr style={{ backgroundColor: '#f2f2f2' }}>
                  <th style={{ width: '25px' }}>No</th>
                  <th>Nama Peserta</th>
                  <th style={{ width: '80px' }}>Approved</th>
                  <th style={{ width: '80px' }}>Mulai</th>
                  <th style={{ width: '80px' }}>Selesai</th>
                  <th style={{ width: '80px' }}>Status</th>
                </tr>
              </thead>
              <tbody>
                {chunk.map((r, idx) => {       
                  const aRaw = r?.session?.approved_at || null;
                  const sRaw = r?.session?.started_at || null;
                  const fRaw = r?.session?.finished_at || null;
                  const approvedAt = onlyTime(aRaw);
                  const startedAt = onlyTime(sRaw);
                  const finishedAt = onlyTime(fRaw);
                  const name = r?.user?.name || r?.participant?.name || r?.participant?.nama || '-';
                  // Status PDF: Hadir (hijau) jika sudah mulai/selesai; Absen (merah) jika tidak ikut
                  const present = !!(sRaw || fRaw);
                  const status = present ? 'Hadir' : 'Absen';
                  const statusColor = present ? 'green' : 'red';
                  return (
                    <tr key={idx}>
                      <td style={{ textAlign: 'center' }}>{idx + 1 + chunkIndex * 39}</td>
                      <td>{name}</td>
                      <td style={{ textAlign: 'center' }}>{approvedAt}</td>
                      <td style={{ textAlign: 'center' }}>{startedAt}</td>
                      <td style={{ textAlign: 'center' }}>{finishedAt}</td>
                      <td style={{ textAlign: 'center', color: statusColor, fontWeight: 'bold' }}>{status}</td>
                    </tr>
                  );
                })}
                {/* tambahkan satu baris kosong agar ada space bawah */}
                <tr>
                  <td style={{ textAlign: 'center', height: '20px' }}></td>
                  <td></td>
                  <td></td>
                  <td></td>
                  <td></td>
                  <td></td>
                </tr>
              </tbody>
            </table>
            

          {/* Signature: QR di bawah tulisan Pengawas Ujian dan berada di tengah */}
<div
  style={{
    display: 'flex',
    justifyContent: 'center',
    marginTop: '25px',
    fontSize: '11px',
  }}
>
  {(() => {
    const majorityName = groupByClass()[classIndex]?.majorityApproverName || null;
    const qrPayload = majorityName ? `Pengawas: ${majorityName}` : '';
    const qrUrl = majorityName
      ? `https://api.qrserver.com/v1/create-qr-code/?size=120x120&data=${encodeURIComponent(qrPayload)}`
      : null;

    return (
<div style={{ textAlign: 'center', transform: 'translateX(90%)' }}>
        {/* Kota dan Tanggal */}
        <p style={{ margin: '0 0 4px 0' }}>
          {schoolData?.kota || 'Kota'}, {currentDate}
        </p>
        {/* Label di atas QR */}
        <p style={{ margin: '0 0 12px 0', fontWeight: 'bold' }}>Pengawas Ujian</p>
        {/* QR-Code di bawah tulisan Pengawas Ujian dan berada di tengah */}
        {qrUrl && (
            <div style={{ margin: '0 0 12px 0', transform: 'translateX(25%)' }}>
            <img
                    src={qrUrl}
                    alt="QR Pengawas"
                    crossOrigin="anonymous"
                    style={{
                        width: '50px',
                        height: '50px',
                        transform: 'translateX(20%)', // geser ke kanan 20%
                        marginTop: '4px',
                    }}
                    />
                
            </div>
            )}
            
            <div style={{ fontWeight: 'bold', fontSize: '10px', marginTop: '4px' }}>
               {majorityName}
            </div>
       
       
      </div>
    );
  })()}
</div>

          </div>
        ));
      })}
    </>
  );
};

export default PrintableReport;
