import React from 'react';
import { requestFullscreenEnhanced } from '../utils/fullscreenHelper';

export default function ExamStartModal({ exam, onStart, onClose, attemptsUsed = 0, attemptsLimit = 3 }) {
  if (!exam) return null;
  const start = new Date(exam.start_at_iso || exam.start_at);
  const end = new Date(exam.end_at_iso || exam.end_at);

  const formatDateFormal = (d) => {
    if (!d || isNaN(d)) return '-';
    const dd = String(d.getDate()).padStart(2, '0');
    const mm = String(d.getMonth() + 1).padStart(2, '0');
    const yyyy = d.getFullYear();
    return `${dd}-${mm}-${yyyy}`;
  };

  const formatTimeHM = (d) => {
    if (!d || isNaN(d)) return '-';
    const hh = String(d.getHours()).padStart(2, '0');
    const mm = String(d.getMinutes()).padStart(2, '0');
    return `${hh}:${mm}`;
  };

  // Pindahkan proses fullscreen ke momen mulai ujian (gesture pengguna)
  const requestFullscreen = async () => {
    try {
      await requestFullscreenEnhanced(document.documentElement, {
        showWarning: false,
        fallbackToViewport: true,
        mobileOptimization: true
      });
    } catch (error) {
      console.warn('Fullscreen request failed:', error);
    }
  };
  const styles = {
    overlay: {
      position: 'fixed', top: 0, left: 0, right: 0, bottom: 0,
      background: 'rgba(0,0,0,0.35)', backdropFilter: 'blur(2px)',
      display: 'flex', alignItems: 'center', justifyContent: 'center',
      zIndex: 1050,
      padding: '16px'
    },
    modal: {
      width: '100%', maxWidth: 640,
      background: '#ffffff', borderRadius: 12,
      boxShadow: '0 12px 32px rgba(0,0,0,0.14)',
      overflow: 'hidden'
    },
    header: {
      display: 'flex', alignItems: 'center', justifyContent: 'space-between',
      padding: '14px 16px',
      background: 'linear-gradient(90deg, #e7f1ff 0%, #f8f9fa 100%)',
      borderBottom: '1px solid #e9ecef'
    },
    headerLeft: { display: 'flex', alignItems: 'center', gap: 12 },
    icon: {
      width: 40, height: 40, borderRadius: '50%',
      background: '#e7f1ff', color: '#084298',
      display: 'flex', alignItems: 'center', justifyContent: 'center',
      fontSize: 20, fontWeight: 700
    },
    title: { color: '#0d6efd', fontWeight: 600, fontSize: 18, margin: 0 },
    subtitle: { color: '#6c757d', fontSize: 13 },
    close: {
      border: '1px solid #ced4da', background: '#ffffff', color: '#495057',
      borderRadius: 8, width: 32, height: 32, cursor: 'pointer'
    },
    body: { padding: '16px' },
    row: { display: 'flex', alignItems: 'center', justifyContent: 'space-between', gap: 12, marginBottom: 10 },
    label: { color: '#6c757d', fontWeight: 500 },
    value: { color: '#0b5ed7', fontWeight: 600 },
    code: {
      background: '#e7f1ff', color: '#084298',
      padding: '2px 8px', borderRadius: 6, border: '1px solid #cfe2ff'
    },
    note: {
      marginTop: 12, color: '#dc3545', fontSize: 14, fontWeight: 600,
      background: 'rgba(220,53,69,0.08)', border: '1px solid #f5c2c7',
      padding: '10px 12px', borderRadius: 8, textAlign: 'center'
    },
    footer: {
      display: 'flex', alignItems: 'center', justifyContent: 'flex-end',
      gap: 8, padding: '12px 16px', borderTop: '1px solid #e9ecef',
      background: '#f8f9fa'
    }
  };
  return (
    <div style={styles.overlay} role="dialog" aria-modal="true">
      <div style={styles.modal}>
        <div style={styles.header}>
          <div style={styles.headerLeft}>
            <div style={styles.icon} aria-hidden="true">🎓</div>
            <div>
              <div style={styles.title}>Konfirmasi Mulai Ujian</div>
              <div style={styles.subtitle}>Pastikan Anda siap sebelum memulai.</div>
            </div>
          </div>
          <button style={styles.close} onClick={onClose} aria-label="Tutup">×</button>
        </div>
        <div style={styles.body}>
          <div style={styles.row}>
            <span style={styles.label}>Nama Ujian</span>
            <span style={styles.value}>{exam.name}</span>
          </div>
          <div style={styles.row}>
            <span style={styles.label}>Kode</span>
            <span style={styles.value}><code style={styles.code}>{exam.code}</code></span>
          </div>
          <div style={styles.row}>
            <span style={styles.label}>Tanggal</span>
            <span style={styles.value}>{formatDateFormal(start)}</span>
          </div>
          <div style={styles.row}>
            <span style={styles.label}>Waktu</span>
            <span style={styles.value}>{formatTimeHM(start)} - {formatTimeHM(end)} WIB</span>
          </div>
          <div style={styles.row}>
            <span style={styles.label}>Durasi</span>
            <span style={styles.value}>{exam.duration_minutes} menit</span>
          </div>
          <div style={styles.row}>
            <span style={styles.label}>Kesempatan Masuk</span>
            <span style={styles.value}>
              <span className={`badge ${attemptsUsed >= attemptsLimit ? 'bg-danger' : attemptsLimit - attemptsUsed === 1 ? 'bg-warning text-dark' : 'bg-success'}`}>
                {attemptsUsed}/{attemptsLimit}
              </span>
            </span>
          </div>
          <div style={styles.note}><span aria-hidden="true" className="me-2">⚠️</span>Jangan tutup atau reload halaman saat ujian berlangsung!</div>
        </div>
        <div style={styles.footer}>
          <button className="btn btn-outline-secondary" onClick={onClose}>Batal</button>
          <button 
            className="btn btn-primary" 
            onClick={async () => { 
              await requestFullscreen(); 
              onStart(); 
            }}
          >
            Mulai Ujian
          </button>
        </div>
      </div>
    </div>
  );
}