import React from "react";
import ReactDOM from "react-dom/client";
import { BrowserRouter as Router, Routes, Route, Navigate } from "react-router-dom";
import { useAuth, AuthProvider } from "./context/AuthContext.jsx";
import { ParticipantAuthProvider, useParticipantAuth } from "./context/ParticipantAuthContext.jsx";

import MonitorParticipants from "./pages/MonitorParticipants";
import Login from "./pages/Login";
import Dashboard from "./pages/Dashboard";
import Users from "./pages/Users";
import Roles from "./pages/Roles";
import Permissions from "./pages/Permissions";
import RolePermissions from "./pages/RolePermissions";
import Questions from "./pages/Questions";
import Subjects from "./pages/Subjects";
import Exams from "./pages/Exams";
import ExamResults from "./pages/ExamResults";
import TodayPublic from "./pages/TodayPublic";
import TodayPublicDetails from "./pages/TodayPublicDetails";
import Schools from "./pages/Schools";
import SchoolDetail from "./pages/SchoolDetail";
import Grades from "./pages/Grades";
import Classes from "./pages/Classes";
import ExamParticipants from "./pages/ExamParticipants";
import Forbidden from "./pages/Forbidden";
import AdminLayout from "./components/AdminLayout";
import ParticipantLogin from "./pages/ParticipantLogin";
import ParticipantHome from "./pages/ParticipantHome";
import ParticipantExam from "./pages/ParticipantExam";
import EssayKeys from "./pages/EssayKeys";
import Scores from "./pages/Scores";
import ApproveExam from "./pages/ApproveExam";
import ErrorBoundary from "./components/ErrorBoundary.jsx";

import "bootstrap/dist/css/bootstrap.min.css";
import "bootstrap/dist/js/bootstrap.bundle.min.js";
import "./App.css";

function ProtectedRoute({ children, roles }) {
  const { user, loading, hasRole } = useAuth();

  if (loading) return <div className="text-center mt-5">Loading...</div>;
  // Ubah: non-admin atau belum login, arahkan ke /exam/login
  if (!user) return <Navigate to="/exam/login" replace />;

  // Jika roles diset, pastikan user punya role itu
  if (roles && !hasRole(roles)) {
    return <Forbidden />;
  }

  return children;
}

function ParticipantProtectedRoute({ children }) {
  const { participant, loading } = useParticipantAuth();
  if (loading) return <div className="text-center mt-5">Loading...</div>;
  if (!participant) return <Navigate to="/exam/login" replace />;
  return children;
}

export default function App() {
    const baseEnv = import.meta.env?.BASE_URL;
    const routerBasename =
      (!baseEnv || baseEnv === '/' || baseEnv === './' || !String(baseEnv).startsWith('/'))
        ? undefined
        : String(baseEnv).replace(/\/+$/, '');

    return (
      <AuthProvider>
        <ParticipantAuthProvider>
          <Router basename={routerBasename}>
            <Routes>
            {/* Public route */}
            <Route path="/login" element={<Login />} />
            <Route path="/exam/login" element={<ErrorBoundary><ParticipantLogin /></ErrorBoundary>} />

            {/* Protected routes */}
            <Route
              path="/"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <Dashboard />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />

            {/* Alias Dashboard */}
            <Route
              path="/dashboard"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <Dashboard />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />

            {/* Hanya admin */}
            <Route
              path="/users"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <Users />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />
            <Route
              path="/roles"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <Roles />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />
            <Route
              path="/permissions"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <Permissions />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />
            <Route
              path="/role-permissions"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <RolePermissions />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />

            <Route
              path="/questions"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <Questions />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />

            {/* Exams */}
            <Route
              path="/exams"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <Exams />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />
            {/* Approve Exam: general selector */}
            <Route
              path="/approve-exam"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <ApproveExam />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />
            {/* Approve Exam for specific exam */}
            <Route
              path="/exams/:examId/approve"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <ApproveExam />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />
            <Route
              path="/exams/:examId/hasil"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <ExamResults />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />
            <Route
              path="/today-public"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <TodayPublic />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />
            <Route
              path="/today-public/:examId"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <TodayPublicDetails />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />

            <Route
              path="/subjects/:subjectId/exams"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <Exams />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />

            <Route
              path="/subjects/:subjectId/questions"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <Questions />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />

            <Route
              path="/subjects/:subjectId/essay-keys"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <EssayKeys />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />

            <Route
              path="/subjects"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <Subjects />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />

            {/* Akademik */}
            <Route
              path="/schools"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <Schools />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />
            <Route
              path="/schools/:schoolId"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <SchoolDetail />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />
            <Route
              path="/grades"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <Grades />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />
            <Route
              path="/classes"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <Classes />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />
            <Route
              path="/exam-participants"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <ExamParticipants />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />
            <Route
              path="/monitor-peserta"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <MonitorParticipants />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />

            {/* Nilai (Scores) */}
            <Route
              path="/scores"
              element={
                <ProtectedRoute>
                  <AdminLayout>
                    <Scores />
                  </AdminLayout>
                </ProtectedRoute>
              }
            />

            {/* Participant portal */}
            <Route
              path="/exam"
              element={
                <ParticipantProtectedRoute>
                  <ParticipantHome />
                </ParticipantProtectedRoute>
              }
            />

            <Route
              path="/exam/session/:sessionId"
              element={
                <ParticipantProtectedRoute>
                  <ParticipantExam />
                </ParticipantProtectedRoute>
              }
            />

            {/* 403 */}
            <Route path="/forbidden" element={<Forbidden />} />

            {/* Redirect fallback */}
            {/* Ubah: semua path yang tidak dikenali, arahkan ke /exam/login */}
            <Route path="*" element={<Navigate to="/exam/login" replace />} />
            </Routes>

            {(typeof window === 'undefined' || (window.location.pathname !== '/exam/login' && !window.location.pathname.startsWith('/exam/session'))) && (
               <div className="text-center text-muted small py-3 mt-4">
                 © All Rights Reserved. IT - SQDF - HSN ™
               </div>
             )}
        </Router>
      </ParticipantAuthProvider>
    </AuthProvider>
  );
}

// Root rendering dipindahkan ke resources/js/main.jsx agar tidak terjadi duplikasi createRoot
