<?php
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\UserController;
use App\Http\Controllers\Api\RolePermissionController;
use App\Http\Controllers\Api\RoleController;
use App\Http\Controllers\Api\PermissionController;
use App\Http\Controllers\Api\DashboardController;
use App\Http\Controllers\Api\QuestionController;
use App\Http\Controllers\Api\SubjectController;
use App\Http\Controllers\Api\SchoolController;
use App\Http\Controllers\Api\GradeController;
use App\Http\Controllers\Api\SchoolClassController;
use App\Http\Controllers\Api\ExamParticipantController;
use App\Http\Controllers\Api\ExamController;
use App\Http\Controllers\Api\ParticipantAuthController;
use App\Http\Controllers\Api\ScoreController;

Route::post('/login',[AuthController::class,'login'])->middleware(['throttle:login', \App\Http\Middleware\NoCacheHeaders::class]);
// Registrasi publik: hanya guru/murid, status default Tidak Aktif
Route::post('/register',[AuthController::class,'register'])->middleware('throttle:register');
// Login peserta ujian dengan throttle untuk jaga beban server
Route::post('/participant/login', [ParticipantAuthController::class, 'login'])->middleware(['throttle:login', \App\Http\Middleware\NoCacheHeaders::class]);
Route::post('/participant/login-token', [ParticipantAuthController::class, 'loginByToken'])->middleware(['throttle:login', \App\Http\Middleware\NoCacheHeaders::class]);
// Publik: daftar ujian hari ini (ringkasan) dapat diakses tanpa login
Route::get('/exams/today', [ExamController::class, 'todayForParticipant']);

// Public: unduh template DOCX/Excel format soal tanpa autentikasi
Route::get('/subjects/import-docx-template', [SubjectController::class, 'importDocxTemplate']);
Route::get('/subjects/import-excel-template', [SubjectController::class, 'importExcelTemplate']);

Route::middleware('auth:sanctum')->group(function(){
    Route::post('/logout',[AuthController::class,'logout'])->middleware(\App\Http\Middleware\NoCacheHeaders::class);
    Route::get('/me',[AuthController::class,'me']);
    Route::post('/update-activity',[AuthController::class,'updateActivity']);
    Route::put('/me/password',[AuthController::class,'changePassword']);

    // Peserta ujian: logout & profil
    Route::post('/participant/logout', [ParticipantAuthController::class, 'logout'])->middleware(\App\Http\Middleware\NoCacheHeaders::class);
    Route::get('/participant/me', [ParticipantAuthController::class, 'me']);
    // Peserta ujian: update aktivitas diri (heartbeat)
    Route::post('/participant/update-activity', [ParticipantAuthController::class, 'updateActivity']);

    // Users CRUD per-aksi dengan izin granular
    Route::get('/users', [UserController::class, 'index'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-user');
    Route::post('/users', [UserController::class, 'store'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':create-user');
    Route::get('/users/{user}', [UserController::class, 'show'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-user');
    Route::put('/users/{user}', [UserController::class, 'update'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':edit-user');
    Route::delete('/users/{user}', [UserController::class, 'destroy'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':delete-user');
    Route::put('/users/{user}/reset-password', [UserController::class,'resetPassword'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':reset-password');

    // Role & Permission management
    // Index dapat diakses oleh view-only; aksi tulis tetap butuh manage-roles
    Route::get('/roles', [RoleController::class, 'index'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-role');
    Route::get('/permissions', [PermissionController::class, 'index'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-permission');
    Route::middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':manage-roles')->group(function() {
        Route::post('/roles', [RoleController::class, 'store']);
        Route::delete('/roles/{role}', [RoleController::class, 'destroy']);
        Route::post('/permissions', [PermissionController::class, 'store']);
        Route::delete('/permissions/{permission}', [PermissionController::class, 'destroy']);

        // Role-permissions helpers
        Route::get('/role/permissions/{role}', [RolePermissionController::class,'rolePermissions']);
        Route::post('/roles/permissions', [RolePermissionController::class,'syncPermissionsForRoles']);
        Route::post('/role/permission', [RolePermissionController::class,'assignPermissionToRole']);
        Route::post('/role/permission/revoke', [RolePermissionController::class,'revokePermissionFromRole']);
    });

    // Dashboard stats: jumlah online vs total per role (pegawai, guru, murid)
    Route::get('/dashboard/stats', [DashboardController::class, 'stats'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-dashboard');

    // Questions routes dengan izin spesifik
    Route::get('/questions', [QuestionController::class, 'index'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-question');
    Route::post('/questions', [QuestionController::class, 'store'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':create-question');
    Route::post('/questions/upload-image', [QuestionController::class, 'uploadImage'])
        ->middleware([\Spatie\Permission\Middleware\PermissionMiddleware::class.':create-question','throttle:upload']);
    Route::put('/questions/{question}', [QuestionController::class, 'update'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':edit-question');
    Route::delete('/questions/{question}', [QuestionController::class, 'destroy'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':delete-question');

    // Subjects CRUD (Mata Pelajaran) dengan izin per aksi
    Route::get('/subjects', [SubjectController::class, 'index'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-subject');
    Route::post('/subjects', [SubjectController::class, 'store'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':create-subject');
    // Import soal dari DOCX ke subject
    Route::post('/subjects/{subject}/import-docx', [SubjectController::class, 'importDocx'])
        ->middleware([\Spatie\Permission\Middleware\PermissionMiddleware::class.':create-question','throttle:import']);
    // Import soal dari Excel ke subject
    Route::post('/subjects/{subject}/import-excel', [SubjectController::class, 'importExcel'])
        ->middleware([\Spatie\Permission\Middleware\PermissionMiddleware::class.':create-question','throttle:import']);

    // Bulk delete subjects (superadmin only)
    Route::delete('/subjects', [SubjectController::class, 'destroyAll'])
        ->middleware(\Spatie\Permission\Middleware\RoleMiddleware::class.':superadmin');
    Route::get('/subjects/{subject}', [SubjectController::class, 'show'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-subject');
    Route::put('/subjects/{subject}', [SubjectController::class, 'update'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':edit-subject');
    Route::delete('/subjects/{subject}', [SubjectController::class, 'destroy'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':delete-subject');

    // Exams: rute spesifik harus ditaruh sebelum resource agar tidak tertimpa oleh {exam}
    Route::post('/exams/{exam}/start', [ExamController::class, 'startForParticipant']);
    // Approvals listing dan approve peserta untuk ujian
    Route::get('/exams/{exam}/approvals', [ExamController::class, 'approvals'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-exam');
    Route::post('/exams/{exam}/approve/{exam_participant}', [ExamController::class, 'approveParticipant'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':manage-exam-participants');
    Route::get('/exams/{exam}/participants-status', [ExamController::class, 'participantsStatus'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-exam');
    // Reset sesi peserta per ujian (admin)
    Route::delete('/exams/{exam}/participants/{exam_participant}/session', [ExamController::class, 'adminResetParticipantSession'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':manage-exam-participants');
    // Reset entry_attempts peserta (admin): kembalikan ke 1 dan hapus finish_requested_at jika ada
    Route::put('/exams/{exam}/participants/{exam_participant}/reset-entry-attempts', [ExamController::class, 'adminResetEntryAttempts'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':manage-exam-participants');
    // Superadmin: finalisasi sesi ujian & sinkronisasi hasil
    Route::post('/exams/{exam}/finalize-sessions', [ExamController::class, 'adminFinalizeSessions'])
        ->middleware(\Spatie\Permission\Middleware\RoleMiddleware::class.':superadmin');
    // Export hasil ujian ke CSV (Excel-friendly)
    Route::get('/exams/{exam}/export-results', [ExamController::class, 'exportResults'])
        ->middleware([\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-exam','throttle:export']);
    // Export analisis butir soal ke XLSX
    Route::get('/exams/{exam}/export-item-analysis', [ExamController::class, 'exportItemAnalysis'])
        ->middleware([\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-exam','throttle:export']);
    // Peserta mulai ujian: buat sesi
    Route::post('/exams/{exam}/start', [ExamController::class, 'startForParticipant']);

    Route::get('/exam-sessions/{session}', [ExamController::class, 'sessionDetail']);
    Route::post('/exam-sessions/{session}/autosave-answers', [ExamController::class, 'autosaveAnswers'])
        ->middleware('throttle:autosave-exam');
    Route::put('/exam-sessions/{session}/finish', [ExamController::class, 'finishSession']);
    Route::post('/exam-sessions/{session}/violations', [ExamController::class, 'reportViolation'])
        ->middleware('throttle:report');
    Route::post('/exam-sessions/{session}/submit-answers', [ExamController::class, 'submitAnswers'])
        ->middleware('throttle:submit-exam');
    // Bulk delete exams (superadmin only)
    Route::delete('/exams', [ExamController::class, 'destroyAll'])
        ->middleware(\Spatie\Permission\Middleware\RoleMiddleware::class.':superadmin');
    // Exams CRUD per-aksi
    Route::get('/exams', [ExamController::class, 'index'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-exam');
    Route::post('/exams', [ExamController::class, 'store'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':create-exam');
    Route::get('/exams/{exam}', [ExamController::class, 'show'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-exam');
    Route::put('/exams/{exam}', [ExamController::class, 'update'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':edit-exam');
    Route::delete('/exams/{exam}', [ExamController::class, 'destroy'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':delete-exam');

    // Schools, Grades, Classes CRUD dengan izin per aksi
    // Schools
    Route::get('/schools', [SchoolController::class, 'index'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-school');
    Route::post('/schools', [SchoolController::class, 'store'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':create-school');
    Route::get('/schools/{school}', [SchoolController::class, 'show'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-school');
    Route::put('/schools/{school}', [SchoolController::class, 'update'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':edit-school');
    Route::delete('/schools/{school}', [SchoolController::class, 'destroy'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':delete-school');

    // Grades
    Route::get('/grades', [GradeController::class, 'index'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-grade');
    Route::post('/grades', [GradeController::class, 'store'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':create-grade');
    Route::get('/grades/{grade}', [GradeController::class, 'show'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-grade');
    Route::put('/grades/{grade}', [GradeController::class, 'update'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':edit-grade');
    Route::delete('/grades/{grade}', [GradeController::class, 'destroy'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':delete-grade');

    // Classes
    Route::get('/classes', [SchoolClassController::class, 'index'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-class');
    Route::post('/classes', [SchoolClassController::class, 'store'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':create-class');
    Route::get('/classes/{class}', [SchoolClassController::class, 'show'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-class');
    Route::put('/classes/{class}', [SchoolClassController::class, 'update'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':edit-class');
    Route::delete('/classes/{class}', [SchoolClassController::class, 'destroy'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':delete-class');
    // Pastikan route spesifik non-ID didefinisikan sebelum resource agar tidak tertimpa oleh {exam_participant}
    Route::get('/exam-participants/template', [ExamParticipantController::class, 'template'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':import-exam-participants');
    Route::get('/exam-participants/template-by-name', [ExamParticipantController::class, 'templateByName'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':import-exam-participants');
    Route::get('/exam-participants/export-tokens', [ExamParticipantController::class, 'exportTokens'])
        ->middleware([\Spatie\Permission\Middleware\PermissionMiddleware::class.':export-exam-participants','throttle:export']);

    Route::get('/exam-participants/export-cards-seats', [ExamParticipantController::class, 'exportCardsSeats'])
        ->middleware([\Spatie\Permission\Middleware\PermissionMiddleware::class.':export-exam-participants-cards-seats','throttle:export']);
    Route::post('/exam-participants/import', [ExamParticipantController::class, 'import'])
        ->middleware([\Spatie\Permission\Middleware\PermissionMiddleware::class.':import-exam-participants','throttle:import']);
    Route::put('/exam-participants/reset-password', [ExamParticipantController::class, 'resetPasswordBulk'])
        ->middleware(\Spatie\Permission\Middleware\RoleMiddleware::class.':superadmin');
    Route::put('/exam-participants/{exam_participant}/reset-password', [ExamParticipantController::class, 'resetPassword'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':reset-exam-participant-password');
    // Bulk delete seluruh peserta (opsional filter berdasarkan id_kelas) — superadmin only
    Route::delete('/exam-participants', [ExamParticipantController::class, 'destroyAll'])
        ->middleware(\Spatie\Permission\Middleware\RoleMiddleware::class.':superadmin');
    // Reset binding perangkat peserta: hanya admin dengan permission manage-users
    Route::middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':manage-users')->group(function() {
        Route::put('/exam-participants/{exam_participant}/unbind-device', [ExamParticipantController::class, 'unbindDevice']);
    });
    Route::post('/exam-participants/update-activity', [ExamParticipantController::class, 'updateActivity'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':manage-exam-participants');
    Route::put('/exam-participants/generate-tokens', [ExamParticipantController::class, 'generateTokens'])
        ->middleware(\Spatie\Permission\Middleware\RoleMiddleware::class.':superadmin');
    // Exam Participants CRUD per-aksi
    Route::get('/exam-participants', [ExamParticipantController::class, 'index'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-exam-participant');
    Route::post('/exam-participants', [ExamParticipantController::class, 'store'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':create-exam-participant');
    Route::get('/exam-participants/{exam_participant}', [ExamParticipantController::class, 'show'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-exam-participant');
    Route::put('/exam-participants/{exam_participant}', [ExamParticipantController::class, 'update'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':edit-exam-participant');
    Route::delete('/exam-participants/{exam_participant}', [ExamParticipantController::class, 'destroy'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':delete-exam-participant');

    // Scores (Nilai) CRUD dan export
    Route::get('/scores', [ScoreController::class, 'index'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-score');
    Route::get('/scores/export', [ScoreController::class, 'export'])
        ->middleware([\Spatie\Permission\Middleware\PermissionMiddleware::class.':export-score','throttle:export']);
    Route::post('/scores', [ScoreController::class, 'store'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':input-score');
    Route::get('/scores/{score}', [ScoreController::class, 'show'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':view-score');
    Route::put('/scores/{score}', [ScoreController::class, 'update'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':edit-score');
    Route::delete('/scores/{score}', [ScoreController::class, 'destroy'])
        ->middleware(\Spatie\Permission\Middleware\PermissionMiddleware::class.':manage-score');
});
