<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

class SqlDumpSeeder extends Seeder
{
    public function run(): void
    {
        $dumpPath = base_path('data_mysql_full.sql');
        if (!File::exists($dumpPath)) {
            $this->command?->warn("SQL dump tidak ditemukan: {$dumpPath}");
            return;
        }

        $sql = File::get($dumpPath);

        // Bangun peta kolom per tabel dari blok CREATE TABLE
        $tables = $this->parseCreateTableColumns($sql);

        // Matikan FK sementara agar insert dapat berjalan meski urutan tidak ideal
        DB::statement('SET FOREIGN_KEY_CHECKS=0');

        // Pecah per statement untuk bisa memfilter yang mengganggu (mis. migrations)
        $statements = preg_split('/;\s*(\r?\n|$)/', $sql);
        foreach ($statements as $statement) {
            $trimmed = trim($statement);
            if ($trimmed === '') {
                continue;
            }
            // Abaikan DDL/komando non-INSERT: CREATE DATABASE/USE/SET/CREATE TABLE/INDEX, dll
            if (
                preg_match('/^(CREATE\s+DATABASE|USE\s+|SET\s+FOREIGN_KEY_CHECKS|CREATE\s+TABLE|CREATE\s+UNIQUE\s+INDEX|CREATE\s+INDEX)/i', $trimmed)
            ) {
                continue;
            }
            // Hanya jalankan INSERT agar sesuai dengan skema dari migrasi Laravel
            if (preg_match('/^INSERT\s+INTO\s+/i', $trimmed)) {
                // Abaikan insert ke tabel migrations agar tidak mengacaukan status migrasi Laravel
                if (preg_match('/^INSERT\s+INTO\s+`?migrations`?/i', $trimmed)) {
                    continue;
                }
                $normalized = $this->normalizeInsertWithColumns($trimmed, $tables);
                // Jika normalisasi gagal (tetap tanpa column list) dan terdeteksi tabel users, lewati untuk menghindari error kolom mismatch
                if ($normalized === $trimmed && preg_match('/^INSERT\s+INTO\s+`?users`?\s+VALUES\s*\(/i', $trimmed)) {
                    $this->command?->warn('Lewati INSERT ke tabel users karena mismatch kolom.');
                    continue;
                }
                // Hindari kegagalan duplikasi primary/unique key dengan IGNORE
                $normalized = preg_replace('/^INSERT\s+INTO/i', 'INSERT IGNORE INTO', $normalized, 1);
                DB::unprepared($normalized.';');
            }
        }

        DB::statement('SET FOREIGN_KEY_CHECKS=1');
        $this->command?->info('Import data dari data_mysql_full.sql selesai (DDL diabaikan, hanya INSERT).');
    }

    /**
     * Ambil daftar kolom per tabel dari definisi CREATE TABLE dalam dump SQL.
     * Mengembalikan [tableName => [col1, col2, ...]]
     */
    private function parseCreateTableColumns(string $sql): array
    {
        $tables = [];
        // Tangkap semua blok CREATE TABLE (multi-line maupun single-line)
        $pattern = '/CREATE\s+TABLE\s+(?:IF\s+NOT\s+EXISTS\s+)?`([^`]+)`\s*\((.*?)\)\s*ENGINE=/is';
        if (preg_match_all($pattern, $sql, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $m) {
                $table = $m[1];
                $body = $m[2];
                $cols = [];
                // Pecah berdasarkan koma pada level top (cukup untuk dump yang sederhana)
                $parts = preg_split('/,\s*/', $body);
                foreach ($parts as $part) {
                    $trim = trim($part);
                    if (preg_match('/^`([^`]+)`\s+/i', $trim, $cm)) {
                        $cols[] = $cm[1];
                    }
                }
                if (!empty($cols)) {
                    $tables[$table] = $cols;
                }
            }
        }
        return $tables;
    }

    /**
     * Jika INSERT tidak memiliki daftar kolom, tambahkan berdasarkan hasil parse CREATE TABLE.
     */
    private function normalizeInsertWithColumns(string $insert, array $tables): string
    {
        // Jika sudah memiliki daftar kolom, kembalikan apa adanya
        if (preg_match('/^INSERT\s+INTO\s+`?([^`\s]+)`?\s*\(/i', $insert)) {
            return $insert;
        }
        // Ubah awalan "INSERT INTO <table> VALUES(" menjadi "INSERT INTO <table> (<cols>) VALUES("
        if (!preg_match('/^INSERT\s+INTO\s+`?([^`\s]+)`?\s+VALUES\s*\(/i', $insert, $m)) {
            return $insert; // pola tidak cocok, biarkan
        }
        $table = $m[1];
        if (!isset($tables[$table]) || empty($tables[$table])) {
            return $insert; // tidak tahu kolomnya
        }
        $cols = '`' . implode('`,`', $tables[$table]) . '`';
        $prefixLen = strlen($m[0]);
        $rest = substr($insert, $prefixLen);
        return "INSERT INTO `{$table}` ({$cols}) VALUES (" . $rest;
    }
}